package com.jfinal.ext.kit;

import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jfinal.kit.StrKit;

/**
 * DateKit.
 */
public class DateTimeKit {

	public static String dateFormat = "yyyy-MM-dd";
	public static String timeFormat = "yyyy-MM-dd HH:mm:ss";
	private static LocalDate now = LocalDate.now();

	private static Logger logger = LoggerFactory.getLogger(DateTimeKit.class);

	public static void setDateFromat(String dateFormat) {
		if (StrKit.isBlank(dateFormat))
			throw new IllegalArgumentException("dateFormat can not be blank.");
		DateTimeKit.dateFormat = dateFormat;
	}

	public static void setTimeFromat(String timeFormat) {
		if (StrKit.isBlank(timeFormat))
			throw new IllegalArgumentException("timeFormat can not be blank.");
		DateTimeKit.timeFormat = timeFormat;
	}

	/**
	 * 格式化
	 * 
	 * @param date
	 * @param pattern
	 * @return
	 */
	public static String format(Date date, String pattern) {
		DateFormat format = new SimpleDateFormat(pattern);
		return format.format(date);
	}

	/**
	 * 格式化
	 * 
	 * @param date
	 * @param parsePattern
	 * @param returnPattern
	 * @return
	 */
	public static String format(String date, String parsePattern, String returnPattern) {
		return format(parse(date, parsePattern), returnPattern);
	}

	/**
	 * 解析
	 * 
	 * @param date
	 * @param pattern
	 * @return
	 */
	public static Date parse(String date, String pattern) {
		SimpleDateFormat format = new SimpleDateFormat(pattern);
		try {
			return format.parse(date);
		} catch (ParseException e) {
			logger.error("parse异常：date值" + date + "，pattern值" + pattern);
			return null;
		}
	}

	/**
	 * 解析
	 * 
	 * @param dateStr
	 * @return
	 */
	public static Date parse(String dateStr) {
		Date date = null;
		try {
			date = DateFormat.getDateTimeInstance().parse(dateStr);
		} catch (ParseException e) {
			logger.error("parse异常：date值" + date);
			return null;
		}
		return date;
	}

	/**
	 * 获取上周第一天日期
	 * 
	 * @return
	 */
	public static LocalDate getLastWeekStart() {
		return firstDayOfWeek(now.minusWeeks(1));
	}

	/**
	 * 获取上周最后一天日期
	 * 
	 * @return
	 */
	public static LocalDate getLastWeekEnd() {
		return lastDayOfWeek(now.minusWeeks(1));
	}

	/**
	 * 获取本周第一天日期
	 * 
	 * @return
	 */
	public static LocalDate getCurrernWeekStart() {
		return firstDayOfWeek(now);
	}

	/**
	 * 获取本周最后一天日期
	 * 
	 * @return
	 */
	public static LocalDate getCurrernWeekEnd() {
		return lastDayOfWeek(now);
	}

	/**
	 * 获取指定时间所在州的最后一天的日期
	 * 
	 * @param date
	 * @return
	 */
	public static LocalDate lastDayOfWeek(LocalDate date) {
		return date.with(DayOfWeek.SUNDAY);
	}

	/**
	 * 获取指定时间所在周的第一天的日期
	 * 
	 * @param date
	 * @return
	 */
	public static LocalDate firstDayOfWeek(LocalDate date) {
		return date.with(DayOfWeek.MONDAY);
	}

	/**
	 * 指定时间是否在两个时间之间
	 * 
	 * @param now
	 *            当前时间
	 * @param start
	 *            开始时间
	 * @param end
	 *            结束时间
	 * @return
	 */
	public static boolean isBetween(LocalDate now, LocalDate start, LocalDate end) {
		if (now.isEqual(start)) {
			return true;
		}

		if (now.isEqual(end)) {
			return true;
		}

		if (now.isAfter(end) && now.isBefore(start)) {
			return true;
		}

		return false;

	}

	/**
	 * 获取当前时间
	 * 
	 * @return
	 */
	public static Date getDate() {
		return new Date();
	}

	/**
	 * 返回两个日期之间隔了多少天
	 * 
	 * @param date1
	 * @param end
	 * @return
	 */
	public static int getDateDaySpace(Date start, Date end) {
		int day = (int) (getDateHourSpace(start, end) / 24);
		return day;
	}

	/**
	 * 返回两个日期之间隔了多少小时
	 * 
	 * @param date1
	 * @param end
	 * @return
	 */
	public static int getDateHourSpace(Date start, Date end) {
		int hour = (int) ((start.getTime() - end.getTime()) / 3600 / 1000);
		return hour;
	}

	/**
	 * 获取当前时间的时间戳
	 * 
	 * @return
	 */
	public static long getDateTime() {
		return new Date().getTime();
	}

	/**
	 * 主要是给jfinal使用，数据库只认java.sql.*
	 * 
	 * @param date
	 * @return
	 */
	public static Timestamp getSqlTimestamp(Date date) {
		if (null == date) {
			date = new Date();
		}
		return getSqlTimestamp(date.getTime());
	}

	/**
	 * 主要是给jfinal使用，数据库只认java.sql.*
	 * 
	 * @param time
	 * @return
	 */
	public static Timestamp getSqlTimestamp(long time) {
		return new java.sql.Timestamp(time);
	}

}
