package com.dotdashpay.api;

import com.dotdashpay.api.internal.DDPUtils;
import com.dotdashpay.api.internal.Request;
import com.dotdashpay.api.internal.Server;
import com.dotdashpay.api.internal.protobufs.AuthorizePayment;
import com.dotdashpay.api.internal.protobufs.ListenForPaymentDataThenSettle;
import com.dotdashpay.api.internal.protobufs.RefundPayment;
import com.dotdashpay.api.internal.protobufs.SettlePayment;
import com.dotdashpay.api.internal.protobufs.VoidPayment;

import java.util.Map;

/**
 * Created by colorado on 11/4/15.
 */
public class Payment {
    private Payment() {}
    private static Payment thisInstance = new Payment();
    public static Payment getInstance () {
        return thisInstance;
    }


    /**
     Authorize a payment, i.e. verify that the payment_info
     is accurate and that the desired funds can be withdrawn.

     This request does not charge the card: you will have to call
     payment.settlePayment(returnObjFromAuthorizePayment) later.
     Alternatevily, you can later call
     `voidPayment(returnObjFromAuthorizePayment)` in order to void
     the given payment without ever having charged the card.

     * @param paymentData: a Map with the following format:
        {
         "payid": {String} "payment id from waitForPaymentData callback"
         "dollars": {int}: number of dollars to charge (USD),
         "cents": {int, optional, default=0} number of cents to charge (USD)
        }
     *
     * @param responseHandler
     */
    public static void authorizePayment(Map paymentData, ResponseHandler responseHandler) {
        // TODO check the paymentData
        int dollars = (int) paymentData.get("dollars");
        int cents = (int) paymentData.get("cents"); // TODO make cents optional
        String payid = (String) paymentData.get("payid");

        AuthorizePayment authorizePaymentObject = AuthorizePayment.newBuilder()
                .setData(
                        AuthorizePayment.Data.newBuilder()
                                .setCents(cents)
                                .setDollars(dollars)
                                .setPayid(payid)
                )
                .setMeta(DDPUtils.getRequestMeta())
                .build();
        Request req = new Request(authorizePaymentObject);
        Server.sendRequest(req, responseHandler);
    }


    /**
     * Settle a payment: you may call this function either after receiving data from
     * `hardware.listenForPaymentData` or after receiving data from `payment.authorizePayment`
     * If called directly after receiving payment data from the hardware, then this immediently
     * charges the payer. If called after authorizing the payment, then this request will
     * finalize the transaction and the transaction can then not be voided.
     *
     * @param paymentData
     * @param responseHandler
     */
    public static void settlePayment (Map paymentData, ResponseHandler responseHandler) {
        // TODO check the paymentData
        int dollars = (int) paymentData.get("dollars");
        int cents = (int) paymentData.get("cents"); // TODO make cents optional
        String payid = (String) paymentData.get("payid");

        SettlePayment settlePaymentDataObject =
                SettlePayment.newBuilder()
                        .setData(
                                SettlePayment.Data.newBuilder()
                                        .setCents(cents)
                                        .setDollars(dollars)
                                        .setPayid(payid)
                        )
                        .setMeta(DDPUtils.getRequestMeta())
                        .build();

        // send the request
        Request req = new Request(settlePaymentDataObject);
        Server.sendRequest(req, responseHandler);

    }

    /**
     * This is a convenience request that combines listenForPaymentData and settlePayment
     * into a single request. Once the user interacts with a payment peripheral,
     * they are immediently charged for the amount you specify.
     * Generally, this is quicker than calling listenForPaymentData then settlePayment.
     *
     * @param paymentData TODO
     * @param responseHandler
     */
    public static void listenForPaymentDataThenSettle (Map paymentData, ResponseHandler responseHandler) {
        // TODO check the paymentData
        int dollars = (int) paymentData.get("dollars");
        int cents = (int) paymentData.get("cents"); // TODO make cents optional

        ListenForPaymentDataThenSettle listenForPaymentDataThenSettleObject = ListenForPaymentDataThenSettle.newBuilder()
                .setData(
                        ListenForPaymentDataThenSettle.Data.newBuilder()
                                .setCents(cents)
                                .setDollars(dollars)
                )
                .setMeta(DDPUtils.getRequestMeta())
                .build();

        Request req = new Request(listenForPaymentDataThenSettleObject);
        Server.sendRequest(req, responseHandler);
    }

    /**
     * Refund a settled payment.
     * You can only call this function for a payment that was successfully settled
     *
     * @param transactionId: the transaction id returned from settlePayment
     * @param responseHandler
     */
    public static void refundPayment(String transactionId, ResponseHandler responseHandler) {
        RefundPayment refundPaymentObject = RefundPayment.newBuilder()
                .setData(
                        RefundPayment.Data.newBuilder()
                        .setTransactionId(transactionId)
                )
                .setMeta(DDPUtils.getRequestMeta())
                .build();
        Request req = new Request(refundPaymentObject);
        Server.sendRequest(req, responseHandler);
    }

    /**
     * Void an authorized payment.
     * You can only call this function for a payment that was successfully authorized
     * but has not been settled. Call `refundPayment` if the payment was settled.
     *
     * @param transactionId: the transaction id returned from authorizePayment
     * @param responseHandler
     */
    public static void voidPayment(String transactionId, ResponseHandler responseHandler) {
        VoidPayment voidPaymentObject = VoidPayment.newBuilder()
                .setData(
                        VoidPayment.Data.newBuilder()
                                .setTransactionId(transactionId)
                )
                .setMeta(DDPUtils.getRequestMeta())
                .build();
        Request req = new Request(voidPaymentObject);
        Server.sendRequest(req, responseHandler);
    }

}
