package com.dotdashpay.api.internal;

import com.dotdashpay.api.Response;
import com.dotdashpay.api.internal.protobufs.CloudPaymentInitiated;
import com.dotdashpay.api.internal.protobufs.RequestMeta;
import com.dotdashpay.api.internal.protobufs.ResponseMeta;
import com.eclipsesource.json.JsonObject;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Descriptors.Descriptor;
import com.google.protobuf.DynamicMessage;
import com.google.protobuf.Message;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.Buffer;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;


/**
 * Utility functions for the DotDashPay API: mostly for manipulating protobufs and translating
 * them to/from other data objects.
 *
 * Created by colorado on 11/4/15.
 */
public final class DDPUtils {
    private DDPUtils() {}

    /**
     * Generate a random id for a request meta protobuf message
     *
     * @return RequestMeta builder with the id field set with a random UUID
     */
    public static RequestMeta.Builder getRequestMeta() {
        return RequestMeta.newBuilder().setId(UUID.randomUUID().toString());
    }

    /**
     * Read in a file and return a string of the text in that file
     *
     * @param file The file path of the file to read
     * @return A string of the text in the provided file
     * @throws IOException thrown if an error occurs while readinging in the file
     * @throws FileNotFoundException thrown if the file is not found
     */
    public static String readFile(String file) throws IOException, FileNotFoundException {

        BufferedReader bufferedReader = null;
        Writer writer = new StringWriter();
        InputStream inputStream = null;
        inputStream = DDPUtils.class.getClassLoader().getResourceAsStream(file);
        // TODO(cjrd) fix this hack that works for reading the json file in both testing and deployment
        if (inputStream != null) {
            bufferedReader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
        } else {
            FileReader fileReader = new FileReader(file);
            bufferedReader = new BufferedReader(fileReader);
        }

        // read in the file
        char[] buffer = new char[1024];
        try {
            int n;
            while ((n = bufferedReader.read(buffer)) != -1) {
                writer.write(buffer, 0, n);
            }
        } finally {
            bufferedReader.close();
            if (inputStream != null) {
                inputStream.close();
            }
        }

        String jsonString = writer.toString();
        return jsonString;

        // FileReader reads text files in the default encoding.


        // Always wrap FileReader in BufferedReader.


//        while ((line = bufferedReader.readLine()) != null) {
//            fullText += line;
//        }

        // Always close files.


//        String paath = "bubbles " + DDPUtils.class.getResource("").getPath();
//        System.out.println(paath);
    }

    /**
     * Converts a proto message into a JsonObject
     * @param protomsg The proto message to convert to a json object
     * @return The JsonObject representation of the proto message
     */
    public static JsonObject convertProtoToJsonObject(Message protomsg)  {
        JsonObject returnJSONObject = new JsonObject();
        List<Descriptors.FieldDescriptor> fields = protomsg.getDescriptorForType().getFields();
        for (Descriptors.FieldDescriptor fieldDescriptor : fields) {
            // TODO(cjrd) handle arrays...
            switch (fieldDescriptor.getJavaType()) {
                case INT:
                    returnJSONObject.set(fieldDescriptor.getName(), (int) protomsg.getField(fieldDescriptor));
                    break;
                case LONG:
                    returnJSONObject.set(fieldDescriptor.getName(), (long) protomsg.getField(fieldDescriptor));
                    break;
                case FLOAT:
                    returnJSONObject.set(fieldDescriptor.getName(), (float) protomsg.getField(fieldDescriptor));
                    break;
                case DOUBLE:
                    returnJSONObject.set(fieldDescriptor.getName(), (double) protomsg.getField(fieldDescriptor));
                    break;
                case BOOLEAN:
                    returnJSONObject.set(fieldDescriptor.getName(), (boolean) protomsg.getField(fieldDescriptor));
                    break;
                case STRING:
                    returnJSONObject.set(fieldDescriptor.getName(), (String) protomsg.getField(fieldDescriptor));
                    break;
                case BYTE_STRING:
                    returnJSONObject.set(fieldDescriptor.getName(), (String) protomsg.getField(fieldDescriptor));
                    break;
                case ENUM:
                    Descriptors.EnumValueDescriptor enumDescript = (Descriptors.EnumValueDescriptor) protomsg.getField(fieldDescriptor);
                    returnJSONObject.set(fieldDescriptor.getName(), enumDescript.getFullName());
                    break;
                case MESSAGE:
                    returnJSONObject.set(fieldDescriptor.getName(), convertProtoToJsonObject((Message) protomsg.getField(fieldDescriptor)));
                    break;
            }
        }
        return returnJSONObject;
    }

    /**
     * Convert a response protomsg into a DDP Response object
     *
     * @param protomsg the response  protomsg
     * @return The DDP Response object
     */
    public static Response convertProtoMessageToResponse(Message protomsg) {
        String name = getMessageName(protomsg);
        JsonObject jsonObject = convertProtoToJsonObject(protomsg);
        String id = jsonObject.get("meta").asObject().get("id").asString();
        return new Response(id, name, jsonObject.get("data").asObject());
    }

    /**
     * Get the message name from a proto message
     * @param protomsg the proto message
     * @return the message name
     */
    public static String getMessageName (Message protomsg) {
        return protomsg.getDescriptorForType().getFullName();
    }

    /**
     * Get the protobuf class with the given message name
     *
     * @param msgName the message name
     * @return the Class of the protobuf with the given message name
     * @throws ClassNotFoundException
     */
    public static Class<?> getProtobufClass (String msgName) throws ClassNotFoundException {
        return Class.forName("com.dotdashpay.api.internal.protobufs." + msgName);
    }


    /**
     * Get the dynamic message builder for the given message name so that you can dynamically
     * build the given protmsg
     *
     * @param msgName the message name of the dynamic message builder you want returned
     * @return the dynamic message builder corresponding to the input message name
     * @throws InvocationTargetException
     * @throws IllegalAccessException
     * @throws NoSuchMethodException
     * @throws ClassNotFoundException
     */
    private static DynamicMessage.Builder getDynamicMessageBuilder(String msgName)
            throws InvocationTargetException, IllegalAccessException, NoSuchMethodException, ClassNotFoundException{
        Class<?> MsgClass = getProtobufClass(msgName);
        Method getDescriptorMethod = MsgClass.getDeclaredMethod("getDescriptor", null);
        Descriptor msgDescriptor = (Descriptor) getDescriptorMethod.invoke(null, null);
        return DynamicMessage.newBuilder(msgDescriptor);
    }

    /**
     * Build a protobuf with the given message name
     *
     * @param msgName the message name corresponding to the the message associated with the given
     *                encoded message bytes
     * @param encodedMsg the raw message byte array
     * @return protobuf generated from the input message name and encoded byte array
     */
    public static Message buildProtobuf(String msgName, byte[] encodedMsg) {
        Message returnMsg = null;
        try{
            DynamicMessage.Builder dynamicBuilder = getDynamicMessageBuilder(msgName);
            returnMsg = dynamicBuilder.mergeFrom(encodedMsg).build();
        } catch (Exception e) {
            // TODO(cjrd) better error handling
            e.printStackTrace();
        }
        return returnMsg;
    }

    /**
     * Build a protobuf with the given message name from the input json object
     *
     * @param msgName the message name corresponding to the the message associated with the given
     *                encoded message bytes
     * @param msgJson a JsonObject describing the given protobuf
     * @return protobuf generated from the input message name and encoded byte array
     */
    public static Message buildProtobuf(String msgName, JsonObject msgJson) {
        Message returnMsg = null;
        try{
            DynamicMessage.Builder dynamicBuilder = getDynamicMessageBuilder(msgName);
            DynamicMessage dynamicMsg = dynamicBuilder.getDefaultInstanceForType();
            returnMsg = createMessageFromJsonObject(msgJson, dynamicMsg);
        } catch (Exception e) {
            // TODO(cjrd) better error handling
            e.printStackTrace();
        }
        return returnMsg;
    }

    /**
     * Internal method to recursively create a message from a json object
     *
     * @param jsonObj json object to be used to create the protobuf message
     * @param inputMsg default protobuf message
     * @return
     */
    private static Message createMessageFromJsonObject (JsonObject jsonObj, Message inputMsg) {
        List<Descriptors.FieldDescriptor> fields = inputMsg.getDescriptorForType().getFields();
        List<String> jsonFieldNames = jsonObj.names();

        Message.Builder msgBuilder = inputMsg.toBuilder();
        for (Descriptors.FieldDescriptor msgField : fields) {
            String fieldName = msgField.getName();
            if (!jsonFieldNames.contains(fieldName)) {
                continue;
            }
            // TODO what about array
            // see line 378 https://github.com/google/protobuf/blob/master/java/src/main/java/com/google/protobuf/FieldSet.java
            switch (msgField.getJavaType()) {
                case INT:
                    msgBuilder.setField(msgField, jsonObj.get(fieldName).asInt());
                    break;
                case LONG:
                    msgBuilder.setField(msgField, jsonObj.get(fieldName).asLong());
                    break;
                case FLOAT:
                    msgBuilder.setField(msgField, jsonObj.get(fieldName).asFloat());
                    break;
                case DOUBLE:
                    msgBuilder.setField(msgField, jsonObj.get(fieldName).asDouble());
                    break;
                case BOOLEAN:
                    msgBuilder.setField(msgField, jsonObj.get(fieldName).asBoolean());
                    break;
                case STRING:
                    msgBuilder.setField(msgField, jsonObj.get(fieldName).asString());
                    break;
                case BYTE_STRING:
                    throw new RuntimeException("Not yet implemented");
                case ENUM:
                    Descriptors.EnumDescriptor enumType = msgField.getEnumType();
                    Descriptors.EnumValueDescriptor enumValue = enumType.findValueByNumber(jsonObj.get(fieldName).asInt());
                    msgBuilder.setField(msgField, enumValue);
                    break;
                case MESSAGE:
                    // create the sub message builder
                    Message embeddedMsg = (Message) inputMsg.getField(msgField);
                    // recursively create the message
                    Message mergedMsg = createMessageFromJsonObject(jsonObj.get(fieldName).asObject(), embeddedMsg);
                    msgBuilder.setField(msgField, mergedMsg);
                    break;
            }
        }
        return msgBuilder.build();
    }

    /**
     * Checks if the response message is an update message
     * @param responseMsg protobuf message to be checked
     * @return True if response message is an update
     */
    public static boolean isResponseUpdate (Message responseMsg) {
        Descriptors.EnumValueDescriptor responseTypeDescriptor = (Descriptors.EnumValueDescriptor) getValueFromMessage(responseMsg, "meta.response_type");
        ResponseMeta.ResponseType responseType = ResponseMeta.ResponseType.valueOf(responseTypeDescriptor);
        return responseType == ResponseMeta.ResponseType.UPDATE;
    }

    /**
     * Get a field value from protobuf message
     *  for example if you have a protomsg with format
     *  {
     *      meta {
     *          "id": "a5"
     *          "owner": {
     *              "name": "rado",
     *              "position": "OG"
     *          }
     *      }
     *      "color": "blue"
     *  }
     *
     *  calling getValueFromMessage(msg, "color")
     *  will return "blue"
     *  and calling getValueFromMessage(msg, "meta.owner.name")
     *  will return "rado"
     *
     * @param msg the proto
     * @param accessString a string that defines how to access the protomsg: use periods to separate
     *                     embedded fields (see above example)
     * @return The value of field as accessed through the accessString parameter. You must cast
     * the Object to the appropriate return type. Returns null if the field does not exist.
     */
    public static Object getValueFromMessage(Message msg, String accessString) {
        List<String> keys = Arrays.asList(accessString.split("\\."));
        Object returnObject = null;

        int numKeys = keys.size();
        Message currentMsg = msg;
        for (int i = 0; i < numKeys ; i++) {
            try {
                returnObject = currentMsg.getField(currentMsg.getDescriptorForType().findFieldByName(keys.get(i)));
            } catch (NullPointerException npe) {
                return null;
            }
            if (i < numKeys - 1) {
                currentMsg = (Message) returnObject;
            }
        }
        return returnObject;
    }


    /**
     * Parse data from a cloud payment to an appropriate protobuf  message
     *
     * @param paymentData
     * @param paymentDataId
     * @return
     */
    public static CloudPaymentInitiated parseCloudPaymentJsonToProto(JsonObject paymentData, String paymentDataId) {
        return CloudPaymentInitiated.newBuilder()
                .setData(CloudPaymentInitiated.Data.newBuilder()
                                .setPayid(paymentData.get("transaction_id").asString())
                                .setBrand(paymentData.get("payment_method").asString())
                                .setDollars(paymentData.get("amount").asObject().get("dollars").asInt())
                                .setCents(paymentData.get("amount").asObject().get("cents").asInt())
                                .setCurrency(paymentData.get("amount").asObject().get("currency").asString())
                )
                .setMeta(
                        ResponseMeta.newBuilder()
                                .setId(paymentDataId)
                                .setResponseType(ResponseMeta.ResponseType.COMPLETION)
                )
                .build();
    }

    /**
     * Get the DDP AMQP connection URL
     * Use Shared.set("dev_ddp_cloud_id", "id") and Shared.set("dev_ddp_cloud_key", "key")
     * to set the AMQP credentials
     *
     * @return the DDP AMQP connection URL
     */
    public static String getAMQPConnectionURL () {
        //testapi
        //apitests51
        return String.format("amqp://%s:%s@dev.dotdashpay.com:5672",
                Shared.get("dev_ddp_cloud_id"), Shared.get("dev_ddp_cloud_key"));
    }
}
