package com.dotdashpay.api.internal;

import android.content.res.Resources;
import android.os.Bundle;
import android.os.Handler;

import java.io.File;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Locale;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.LinkedBlockingDeque;

import com.dotdashpay.api.R;
import com.dotdashpay.api.internal.protobufs.CloudPaymentInitiated;
import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.Envelope;

import com.eclipsesource.json.Json;
import com.eclipsesource.json.JsonArray;
import com.eclipsesource.json.JsonObject;
import com.eclipsesource.json.JsonValue;
import com.google.protobuf.Message;
import com.rabbitmq.client.QueueingConsumer;


/**
 * Created by colorado on  11/5/15.
 */
public class DotSimulator implements ServerCommunicatorInterface {

    protected JsonObject requestToResponsesJson;
    protected static HashMap<String, Message> simulatorResponses = new HashMap<>();
    protected static HashMap<String, Boolean> disabledSimulatorRequests = new HashMap<>();
    protected String lastWaitForPaymentDataId = "";
    protected Thread subscribeThread, publishThread;
    protected String opsQueueName = "ops-gaspump-test";
    protected String opsReturnQueueName = "ops-return-gaspump-test";
    private BlockingDeque<String> queue = new LinkedBlockingDeque<String>();
    private boolean operationsConnectionSetup = false;
    private ConnectionFactory factory = new ConnectionFactory();

    /**
     * Initialize the dot simulator:
     *   + read in the simulator request-to-response mappings
     */
    public DotSimulator() {
        // read in the simulated request-to-response spec
        String requestToResponseJsonString = "{}";
        try {
            String file = Shared.get("simulate_request_to_responses_file");
            requestToResponseJsonString = DDPUtils.readFile(file);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Unable to read simulation request-to-responses file");
        }
        requestToResponsesJson = Json.parse(requestToResponseJsonString).asObject();
    }

    /**
     * Handle an outgoing request
     * @param request the Request the Simulator should handle
     */
    @Override
    public void handleOutgoingRequest(Request request) {
        // go from the request to the set of responses
        if (disabledSimulatorRequests.containsKey(request.getName())) {
            // make sure we're listening to the operations server
            this.setupOperationsConnection();

            // Simulator hack so we can keep track of ListenForPaymentData requests TODO(cjrd) find a better solution
            if (request.getName().equalsIgnoreCase("ListenForPaymentData")) {
                lastWaitForPaymentDataId = request.getId();
            }
            return;
        }

        // send back the appropriate set of responses
        JsonArray responses = requestToResponsesJson.get(request.getName()).asObject().get("default").asArray();
        for (JsonValue resp : responses) {
            // pull out the response info
            JsonObject respInfo = resp.asObject();
            respInfo.get("meta").asObject().set("id", request.getId());
            String messageName = respInfo.get("message").asString();

            // construct the necessary response and send it to the server
            Server.handleResponse(DDPUtils.buildProtobuf(messageName, respInfo));
        }
        return;
    }

    /**
     * TODO
     * @param requestName
     */
    public static void disableSimulatorRequestResponses(String requestName) {
        disabledSimulatorRequests.put(requestName, true);
    }

    /**
     * TODO
     * @param requestName
     */
    public static void enableSimulatorRequestResponses(String requestName) {
        if (disabledSimulatorRequests.containsKey(requestName)) {
            disabledSimulatorRequests.remove(requestName);
        }
    }

    /**
     * Setup the connection to the DDP Rabbitmq cloud server
     * This function is idempotent and should always be called before
     * when a remote rabbitmq connection is needed
     */
    private void setupOperationsConnection () {
        if (operationsConnectionSetup) {
            return;
        }
        operationsConnectionSetup = true;

        // setup cloud based connection
        setupAMQPConnectionFactory();
        // setup RMQ publishing
        setupAMQPPublishing();
        // setup RMQ subscribing
        listenForCloudData(incomingMessageHandler);
    }

    /**
     * Publish a message to the remote rabbitmq server
     *
     * @param message the message to send to the remote rabbitmq server
     *                NOTE: the message should start with the correlation id
     *                in the format: "correlation-id data"
     *                e.g. "abc123 {\"key\": \"value\"}"
     */
    private void publishAMQPReturnMessage(String message) {
        try {
            DDPLog.debug("[q] " + message);
            queue.putLast(message);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * Connect to the remote rabbitmq publishing channel and wait for messages to send to the channel
     * NOTE: use publishAMQPReturnMessage to publish messagese
     */
    private void setupAMQPPublishing() {
        publishThread = new Thread(new Runnable() {
            @Override
            public void run() {
                while(true) {
                    try {
                        Connection connection = factory.newConnection();
                        Channel ch = connection.createChannel();
                        ch.confirmSelect();
                        ch.queueDeclare(opsReturnQueueName, false, false, false, new HashMap<String, Object>());

                        while (true) {
                            String message = queue.takeFirst();

                            // break off the correlation id
                            String[] splitMsg = message.split(" ", 2);
                            String correlationId = splitMsg[0];
                            message = splitMsg[1];

                            try{
                                AMQP.BasicProperties.Builder propsBuilder = new AMQP.BasicProperties.Builder();
                                AMQP.BasicProperties msgProps = propsBuilder.correlationId("r" + correlationId).build();
                                ch.basicPublish("APIManager", opsReturnQueueName, msgProps, message.getBytes());
                                DDPLog.debug("[s] " + message);
                                ch.waitForConfirmsOrDie();
                            } catch (Exception e) {
                                DDPLog.debug("[f] " + message);
                                queue.putFirst(message);
                                throw e;
                            }
                        }
                    } catch (InterruptedException e) {
                        break;
                    } catch (Exception e) {
                        DDPLog.debug("Connection broken: " + e.getClass().getName());
                        try {
                            Thread.sleep(5000); //sleep and then try again
                        } catch (InterruptedException e1) {
                            break;
                        }
                    }
                }
            }
        });
        publishThread.start();
    }

    /**
     * Setup the AMQP connection factory
     */
    private void setupAMQPConnectionFactory() {
            try {
            factory.setAutomaticRecoveryEnabled(false);
            factory.setUri(DDPUtils.getAMQPConnectionURL());
        } catch (KeyManagementException | NoSuchAlgorithmException | URISyntaxException e1) {
            // TODO(cjrd) handle these exceptions better
            e1.printStackTrace();
        }
    }

    /**
     *
     * @param handler
     */
    private void listenForCloudData(final Handler handler) {
        // setup rabbitmq listening
        subscribeThread = new Thread(new Runnable() {
            @Override
            public void run() {
                while(true) {
                    try {
                        Connection connection = factory.newConnection();
                        Channel channel = connection.createChannel();
                        channel.basicQos(1);

                        channel.queueDeclare(opsQueueName, false, false, false, new HashMap<String, Object>());
                        QueueingConsumer consumer = new QueueingConsumer(channel);
                        channel.basicConsume(opsQueueName, true, consumer);

                        // Process deliveries
                        while (true) {
                            QueueingConsumer.Delivery delivery = consumer.nextDelivery();
                            Envelope envelope = delivery.getEnvelope();
                            channel.basicAck(envelope.getDeliveryTag(), false);

                            String message = new String(delivery.getBody());
                            DDPLog.debug("[r] " + message);

                            android.os.Message msg = handler.obtainMessage();
                            Bundle bundle = new Bundle();

                            bundle.putString("msg", message);
                            msg.setData(bundle);

                            String correlationId = delivery.getProperties().getCorrelationId();
                            publishAMQPReturnMessage(correlationId + " {\"status\": 0}");
                            handler.sendMessage(msg);
                        }
                    } catch (InterruptedException e) {
                        break;
                    } catch (Exception e1) {
                        DDPLog.debug("Connection broken: " + e1.getClass().getName());
                        try {
                            Thread.sleep(4000); //sleep and then try again
                        } catch (InterruptedException e) {
                            break;
                        }
                    }
                }
            }
        });
        subscribeThread.start();
    }

    /**
     * Handler for incoming rabbitmq messages
     */
    final private Handler incomingMessageHandler = new Handler() {
        @Override
        public void handleMessage(android.os.Message msg) {
            String message = msg.getData().getString("msg");
            JsonObject jsonMessage = null;
            JsonObject paymentData = null;

            // try to read in the data
            try {
                jsonMessage = Json.parse(message).asObject();
                paymentData = jsonMessage.get("arguments").asArray().get(0).asObject();

            } catch (Exception ex) {
                ex.printStackTrace();
                DDPLog.debug("could not parse json message");
                // TODO send bad status reply
                return;
            }

            String msgType = jsonMessage.get("operation").asString();
            switch (msgType.toLowerCase(Locale.ENGLISH)) {
                case "cloudpayment":
                    CloudPaymentInitiated cloudPaymentInitiated = DDPUtils.parseCloudPaymentJsonToProto(paymentData, lastWaitForPaymentDataId);
                    Server.handleResponse(cloudPaymentInitiated);
                    break;
                default:
                    // Return not yet implemented error to cloud server
            }
        }
    };

}
