/*
 * Decompiled with CFR 0.152.
 */
package com.dracoon.sdk.crypto;

import com.dracoon.sdk.crypto.FileDecryptionCipher;
import com.dracoon.sdk.crypto.FileEncryptionCipher;
import com.dracoon.sdk.crypto.error.CryptoSystemException;
import com.dracoon.sdk.crypto.error.InvalidFileKeyException;
import com.dracoon.sdk.crypto.error.InvalidKeyPairException;
import com.dracoon.sdk.crypto.error.InvalidPasswordException;
import com.dracoon.sdk.crypto.internal.AesGcmFileDecryptionCipher;
import com.dracoon.sdk.crypto.internal.AesGcmFileEncryptionCipher;
import com.dracoon.sdk.crypto.internal.CryptoUtils;
import com.dracoon.sdk.crypto.internal.Validator;
import com.dracoon.sdk.crypto.model.EncryptedFileKey;
import com.dracoon.sdk.crypto.model.PlainFileKey;
import com.dracoon.sdk.crypto.model.UserKeyPair;
import com.dracoon.sdk.crypto.model.UserPrivateKey;
import com.dracoon.sdk.crypto.model.UserPublicKey;
import java.io.CharArrayReader;
import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.MGF1ParameterSpec;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.OAEPParameterSpec;
import javax.crypto.spec.PSource;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMException;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.PKCS8Generator;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.openssl.jcajce.JcaPKCS8Generator;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8DecryptorProviderBuilder;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8EncryptorBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.OutputEncryptor;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.util.Properties;
import org.bouncycastle.util.io.pem.PemGenerationException;
import org.bouncycastle.util.io.pem.PemObjectGenerator;

public class Crypto {
    private static final int HASH_ITERATION_COUNT = 1300000;
    private static final int FILE_KEY_SIZE = 32;
    private static final int IV_SIZE = 12;
    private static final String PROP_ALLOW_UNSAFE_INT = "org.bouncycastle.asn1.allow_unsafe_integer";
    private static final VersionMapping[] versionMappings;

    private Crypto() {
    }

    public static UserKeyPair generateUserKeyPair(UserKeyPair.Version version, char[] password) throws IllegalArgumentException, InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        Validator.validateNotNull("version", version);
        Validator.validateCharArray("password", password);
        KeyPair keyPair = Crypto.generateKeyPair(version);
        char[] privateKey = Crypto.encryptEncodePrivateKey(keyPair.getPrivate(), password);
        char[] publicKey = Crypto.encodePublicKey(keyPair.getPublic());
        UserPrivateKey userPrivateKey = new UserPrivateKey(version, privateKey);
        UserPublicKey userPublicKey = new UserPublicKey(version, publicKey);
        return new UserKeyPair(userPrivateKey, userPublicKey);
    }

    private static KeyPair generateKeyPair(UserKeyPair.Version version) throws InvalidKeyPairException, CryptoSystemException {
        int keySize;
        switch (version) {
            case RSA2048: {
                keySize = 2048;
                break;
            }
            case RSA4096: {
                keySize = 4096;
                break;
            }
            default: {
                throw new InvalidKeyPairException("Unknown user key pair version.");
            }
        }
        try {
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
            keyGen.initialize(keySize);
            return keyGen.generateKeyPair();
        }
        catch (NoSuchAlgorithmException e) {
            throw new CryptoSystemException("Could not generate RSA key pair. Algorithm is missing.", e);
        }
    }

    private static char[] encryptEncodePrivateKey(PrivateKey key, char[] password) throws InvalidPasswordException, CryptoSystemException {
        JcaPKCS8Generator generator;
        OutputEncryptor encryptor;
        try {
            char[] encodedPassword = CryptoUtils.toUtf8CharArray(password);
            encryptor = new JceOpenSSLPKCS8EncryptorBuilder(PKCS8Generator.AES_256_CBC).setProvider("BC").setIterationCount(1300000).setPassword(encodedPassword).build();
        }
        catch (OperatorCreationException e) {
            throw new CryptoSystemException("Could not encrypt private key. Creation of PKCS8(AES 256 CBC) encryptor failed.", e);
        }
        try {
            generator = new JcaPKCS8Generator(key, encryptor);
        }
        catch (PemGenerationException e) {
            throw new InvalidPasswordException("Could not encrypt private key. Invalid private key password.", e);
        }
        try {
            CharArrayWriter charWriter = new CharArrayWriter();
            JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)charWriter);
            pemWriter.writeObject((PemObjectGenerator)generator);
            pemWriter.close();
            return charWriter.toCharArray();
        }
        catch (IOException e) {
            throw new CryptoSystemException("Could not encrypt private key. PEM encoding failed.", e);
        }
    }

    private static PrivateKey decryptDecodePrivateKey(char[] key, char[] password) throws InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        PrivateKeyInfo pkInfo;
        Object obj;
        try {
            CharArrayReader charReader = new CharArrayReader(key);
            PEMParser pemReader = new PEMParser((Reader)charReader);
            obj = pemReader.readObject();
            pemReader.close();
        }
        catch (Exception e) {
            throw new InvalidKeyPairException("Could not decrypt private key. PEM decoding failed.", e);
        }
        if (obj instanceof PKCS8EncryptedPrivateKeyInfo) {
            PKCS8EncryptedPrivateKeyInfo epkInfo = (PKCS8EncryptedPrivateKeyInfo)obj;
            try {
                char[] encodedPassword = CryptoUtils.toUtf8CharArray(password);
                pkInfo = Crypto.decryptPrivateKey(epkInfo, encodedPassword);
            }
            catch (InvalidPasswordException e) {
                pkInfo = Crypto.decryptPrivateKey(epkInfo, password);
            }
        } else {
            throw new InvalidKeyPairException("Could not decrypt private key. Provided key is not a PKCS8 encrypted private key.");
        }
        try {
            Properties.setThreadOverride((String)PROP_ALLOW_UNSAFE_INT, (boolean)true);
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
            PrivateKey privateKey = converter.getPrivateKey(pkInfo);
            return privateKey;
        }
        catch (PEMException e) {
            throw new InvalidKeyPairException("Could not decrypted private key. PEM decoding failed.", e);
        }
        finally {
            Properties.removeThreadOverride((String)PROP_ALLOW_UNSAFE_INT);
        }
    }

    private static PrivateKeyInfo decryptPrivateKey(PKCS8EncryptedPrivateKeyInfo epkInfo, char[] password) throws InvalidPasswordException, CryptoSystemException {
        try {
            InputDecryptorProvider decryptor = new JceOpenSSLPKCS8DecryptorProviderBuilder().setProvider("BC").build(password);
            return epkInfo.decryptPrivateKeyInfo(decryptor);
        }
        catch (OperatorCreationException e) {
            throw new CryptoSystemException("Could not decrypt private key. Creation of PKCS8 decryptor failed.", e);
        }
        catch (PKCSException e) {
            throw new InvalidPasswordException("Could not decrypt private key. Invalid private key password.", e);
        }
    }

    private static char[] encodePublicKey(PublicKey key) throws InvalidKeyPairException {
        try {
            CharArrayWriter charWriter = new CharArrayWriter();
            JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)charWriter);
            pemWriter.writeObject((Object)key);
            pemWriter.close();
            return charWriter.toCharArray();
        }
        catch (IOException e) {
            throw new InvalidKeyPairException("Could not encode public key. PEM encoding failed.", e);
        }
    }

    private static PublicKey decodePublicKey(char[] key) throws InvalidKeyPairException {
        Object obj;
        try {
            CharArrayReader charReader = new CharArrayReader(key);
            PEMParser pemReader = new PEMParser((Reader)charReader);
            obj = pemReader.readObject();
            pemReader.close();
        }
        catch (Exception e) {
            throw new InvalidKeyPairException("Could not decode public key. PEM decoding failed.", e);
        }
        if (!(obj instanceof SubjectPublicKeyInfo)) {
            throw new InvalidKeyPairException("Could not decode public key. Provided key is not PKCS8 public key.");
        }
        SubjectPublicKeyInfo pkInfo = (SubjectPublicKeyInfo)obj;
        try {
            Properties.setThreadOverride((String)PROP_ALLOW_UNSAFE_INT, (boolean)true);
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
            PublicKey publicKey = converter.getPublicKey(pkInfo);
            return publicKey;
        }
        catch (PEMException e) {
            throw new InvalidKeyPairException("Could not decode public key. PEM decoding failed.", e);
        }
        finally {
            Properties.removeThreadOverride((String)PROP_ALLOW_UNSAFE_INT);
        }
    }

    public static boolean checkUserKeyPair(UserKeyPair userKeyPair, char[] password) throws InvalidKeyPairException, CryptoSystemException {
        Validator.validateNotNull("userKeyPair", userKeyPair);
        Validator.validateCharArray("password", password);
        if (password == null || password.length == 0) {
            return false;
        }
        try {
            Crypto.decryptDecodePrivateKey(userKeyPair.getUserPrivateKey().getPrivateKey(), password);
            return true;
        }
        catch (InvalidPasswordException e) {
            return false;
        }
    }

    public static EncryptedFileKey encryptFileKey(PlainFileKey plainFileKey, UserPublicKey userPublicKey) throws InvalidFileKeyException, InvalidKeyPairException, CryptoSystemException {
        byte[] eFileKey;
        Cipher cipher;
        Validator.validateNotNull("plainFileKey", plainFileKey);
        Validator.validateNotNull("userPublicKey", userPublicKey);
        EncryptedFileKey.Version encFileKeyVersion = Crypto.getEncryptedFileKeyVersion(userPublicKey.getVersion(), (PlainFileKey.Version)plainFileKey.getVersion());
        PublicKey publicKey = Crypto.decodePublicKey(userPublicKey.getPublicKey());
        try {
            cipher = Crypto.createFileKeyCipher(1, userPublicKey.getVersion(), publicKey);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new CryptoSystemException("Could not encrypt file key. Creation of cipher failed.", e);
        }
        catch (InvalidKeyException e) {
            throw new InvalidKeyPairException("Could not encrypt file key. Invalid public key.", e);
        }
        byte[] pFileKey = plainFileKey.getKey();
        try {
            eFileKey = cipher.doFinal(pFileKey);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new CryptoSystemException("Could not encrypt file key. Encryption failed.", e);
        }
        EncryptedFileKey encFileKey = new EncryptedFileKey(encFileKeyVersion, eFileKey, plainFileKey.getIv());
        encFileKey.setTag(plainFileKey.getTag());
        return encFileKey;
    }

    public static PlainFileKey decryptFileKey(EncryptedFileKey encFileKey, UserPrivateKey userPrivateKey, char[] password) throws InvalidFileKeyException, InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        byte[] dFileKey;
        Cipher cipher;
        Validator.validateNotNull("encFileKey", encFileKey);
        Validator.validateNotNull("userPrivateKey", userPrivateKey);
        Validator.validateCharArray("password", password);
        PlainFileKey.Version plainFileKeyVersion = Crypto.getPlainFileKeyVersion(userPrivateKey.getVersion(), (EncryptedFileKey.Version)encFileKey.getVersion());
        PrivateKey privateKey = Crypto.decryptDecodePrivateKey(userPrivateKey.getPrivateKey(), password);
        try {
            cipher = Crypto.createFileKeyCipher(2, userPrivateKey.getVersion(), privateKey);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new CryptoSystemException("Could not decrypt file key. Creation of cipher failed.", e);
        }
        catch (InvalidKeyException e) {
            throw new InvalidKeyPairException("Could not decrypt file key. Invalid private key.", e);
        }
        byte[] eFileKey = encFileKey.getKey();
        try {
            dFileKey = cipher.doFinal(eFileKey);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new InvalidFileKeyException("Could not decrypt file key. Encryption failed.", e);
        }
        PlainFileKey plainFileKey = new PlainFileKey(plainFileKeyVersion, dFileKey, encFileKey.getIv());
        plainFileKey.setTag(encFileKey.getTag());
        return plainFileKey;
    }

    private static Cipher createFileKeyCipher(int mode, UserKeyPair.Version version, Key key) throws InvalidKeyPairException, NoSuchAlgorithmException, NoSuchPaddingException, InvalidAlgorithmParameterException, InvalidKeyException {
        OAEPParameterSpec spec;
        String transformation;
        switch (version) {
            case RSA2048: {
                transformation = "RSA/ECB/OAEPWITHSHA-256ANDMGF1PADDING";
                spec = new OAEPParameterSpec("SHA-256", "MGF1", MGF1ParameterSpec.SHA1, PSource.PSpecified.DEFAULT);
                break;
            }
            case RSA4096: {
                transformation = "RSA/ECB/OAEPWITHSHA-256ANDMGF1PADDING";
                spec = new OAEPParameterSpec("SHA-256", "MGF1", MGF1ParameterSpec.SHA256, PSource.PSpecified.DEFAULT);
                break;
            }
            default: {
                throw new InvalidKeyPairException("Unknown user key pair version.");
            }
        }
        Cipher cipher = Cipher.getInstance(transformation);
        cipher.init(mode, key, spec);
        return cipher;
    }

    public static PlainFileKey generateFileKey(PlainFileKey.Version version) {
        Validator.validateNotNull("version", version);
        byte[] key = Crypto.generateSecureRandomByteArray(32);
        byte[] iv = Crypto.generateSecureRandomByteArray(12);
        return new PlainFileKey(version, key, iv);
    }

    private static byte[] generateSecureRandomByteArray(int size) {
        SecureRandom sr = new SecureRandom();
        byte[] bytes = new byte[size];
        sr.nextBytes(bytes);
        return bytes;
    }

    public static FileEncryptionCipher createFileEncryptionCipher(PlainFileKey fileKey) throws CryptoSystemException {
        Validator.validateNotNull("fileKey", fileKey);
        return new AesGcmFileEncryptionCipher(fileKey);
    }

    public static FileDecryptionCipher createFileDecryptionCipher(PlainFileKey fileKey) throws CryptoSystemException {
        Validator.validateNotNull("fileKey", fileKey);
        return new AesGcmFileDecryptionCipher(fileKey);
    }

    private static EncryptedFileKey.Version getEncryptedFileKeyVersion(UserKeyPair.Version keyPairVersion, PlainFileKey.Version fileKeyVersion) throws InvalidFileKeyException {
        for (VersionMapping versionMapping : versionMappings) {
            if (versionMapping.kpv != keyPairVersion || versionMapping.pfkv != fileKeyVersion) continue;
            return versionMapping.efkv;
        }
        String kpv = keyPairVersion != null ? keyPairVersion.name() : "null";
        String fkv = fileKeyVersion != null ? fileKeyVersion.name() : "null";
        throw new InvalidFileKeyException(String.format("User key pair version '%s' and plain file key version '%s' are not compatible.", kpv, fkv));
    }

    private static PlainFileKey.Version getPlainFileKeyVersion(UserKeyPair.Version keyPairVersion, EncryptedFileKey.Version fileKeyVersion) throws InvalidFileKeyException {
        for (VersionMapping versionMapping : versionMappings) {
            if (versionMapping.kpv != keyPairVersion || versionMapping.efkv != fileKeyVersion) continue;
            return versionMapping.pfkv;
        }
        String kpv = keyPairVersion != null ? keyPairVersion.name() : "null";
        String fkv = fileKeyVersion != null ? fileKeyVersion.name() : "null";
        throw new InvalidFileKeyException(String.format("User key pair version '%s' and encrypted file key version '%s' are not compatible.", kpv, fkv));
    }

    static {
        Security.insertProviderAt((Provider)new BouncyCastleProvider(), 1);
        versionMappings = new VersionMapping[]{new VersionMapping(UserKeyPair.Version.RSA2048, EncryptedFileKey.Version.RSA2048_AES256GCM, PlainFileKey.Version.AES256GCM), new VersionMapping(UserKeyPair.Version.RSA4096, EncryptedFileKey.Version.RSA4096_AES256GCM, PlainFileKey.Version.AES256GCM)};
    }

    private static class VersionMapping {
        UserKeyPair.Version kpv;
        EncryptedFileKey.Version efkv;
        PlainFileKey.Version pfkv;

        VersionMapping(UserKeyPair.Version kpv, EncryptedFileKey.Version efkv, PlainFileKey.Version pfkv) {
            this.kpv = kpv;
            this.efkv = efkv;
            this.pfkv = pfkv;
        }
    }
}

