/*
 * Decompiled with CFR 0.152.
 */
package com.dracoon.sdk.internal.http;

import com.dracoon.sdk.Log;
import com.dracoon.sdk.error.DracoonApiException;
import com.dracoon.sdk.error.DracoonNetIOException;
import com.dracoon.sdk.error.DracoonNetIOInterruptedException;
import com.dracoon.sdk.error.DracoonNetInsecureException;
import com.dracoon.sdk.internal.NullLog;
import com.dracoon.sdk.internal.http.HttpStatus;
import com.dracoon.sdk.internal.http.InterceptedIOException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.List;
import javax.net.ssl.SSLHandshakeException;
import okhttp3.Response;
import retrofit2.Call;

public class HttpHelper {
    private static final String LOG_TAG = HttpHelper.class.getSimpleName();
    private static final String HEADER_RETRY_AFTER = "Retry-After";
    protected Log mLog = new NullLog();
    private boolean mIsRetryEnabled;
    private boolean mIsRateLimitingEnabled;
    private Executor mExecutor;

    public void setLog(Log log) {
        this.mLog = log != null ? log : new NullLog();
    }

    public void setRetryEnabled(boolean isRetryEnabled) {
        this.mIsRetryEnabled = isRetryEnabled;
    }

    public void setRateLimitingEnabled(boolean isRateLimitingEnabled) {
        this.mIsRateLimitingEnabled = isRateLimitingEnabled;
    }

    public void init() {
        this.mExecutor = new NetworkExecutor();
        if (this.mIsRetryEnabled) {
            this.mExecutor = new RetryExecutor(this.mExecutor);
        }
        if (this.mIsRateLimitingEnabled) {
            this.mExecutor = new RateLimitingExecutor(this.mExecutor);
        }
        this.mExecutor = new InterceptionErrorHandlingExecutor(this.mExecutor);
    }

    public <T> retrofit2.Response<T> executeRequest(Call<T> call) throws DracoonNetIOException, DracoonApiException {
        try {
            return (retrofit2.Response)this.executeRequestInternally(call);
        }
        catch (InterruptedException e) {
            String errorText = "Server communication was interrupted.";
            this.mLog.d(LOG_TAG, errorText);
            throw new DracoonNetIOInterruptedException(errorText, e);
        }
    }

    public <T> retrofit2.Response<T> executeRequest(Call<T> call, Thread thread) throws DracoonNetIOException, DracoonApiException, InterruptedException {
        try {
            return (retrofit2.Response)this.executeRequestInternally(call);
        }
        catch (DracoonNetIOException e) {
            if (thread.isInterrupted()) {
                throw new InterruptedException();
            }
            throw e;
        }
    }

    public Response executeRequest(okhttp3.Call call) throws DracoonNetIOException, DracoonApiException {
        try {
            return (Response)this.executeRequestInternally(call);
        }
        catch (InterruptedException e) {
            String errorText = "Server communication was interrupted.";
            this.mLog.d(LOG_TAG, errorText);
            throw new DracoonNetIOInterruptedException(errorText, e);
        }
    }

    public Response executeRequest(okhttp3.Call call, Thread thread) throws DracoonNetIOException, DracoonApiException, InterruptedException {
        try {
            return (Response)this.executeRequestInternally(call);
        }
        catch (DracoonNetIOException e) {
            if (thread.isInterrupted()) {
                throw new InterruptedException();
            }
            throw e;
        }
    }

    protected Object executeRequestInternally(Object call) throws DracoonNetIOException, DracoonApiException, InterruptedException {
        try {
            return this.mExecutor.execute(call);
        }
        catch (InterceptedIOException e) {
            String errorText = "Server communication failed due to an unknown error!";
            this.mLog.d(LOG_TAG, errorText);
            throw new DracoonNetIOException(errorText, e);
        }
    }

    protected static Object executeCall(Object call) throws IOException {
        if (call instanceof Call) {
            return ((Call)call).execute();
        }
        if (call instanceof okhttp3.Call) {
            return ((okhttp3.Call)call).execute();
        }
        throw new RuntimeException("Can't execute request. Invalid call object.");
    }

    protected static Object cloneCall(Object call) {
        if (call instanceof Call) {
            return ((Call)call).clone();
        }
        if (call instanceof okhttp3.Call) {
            return ((okhttp3.Call)call).clone();
        }
        throw new RuntimeException("Can't clone request. Invalid call object.");
    }

    private static boolean isRateLimitResponse(Object response) {
        int statusCode;
        if (response instanceof retrofit2.Response) {
            retrofit2.Response r = (retrofit2.Response)response;
            statusCode = r.code();
        } else if (response instanceof Response) {
            Response r = (Response)response;
            statusCode = r.code();
        } else {
            throw new RuntimeException("Can't get response status code. Invalid response object.");
        }
        return HttpStatus.valueOf(statusCode) == HttpStatus.TOO_MANY_REQUESTS;
    }

    private static Integer getRetryAfterInterval(Object response) {
        String value;
        if (response instanceof retrofit2.Response) {
            retrofit2.Response r = (retrofit2.Response)response;
            List vs = r.headers().values(HEADER_RETRY_AFTER);
            value = vs.size() > 0 ? (String)vs.get(0) : null;
        } else if (response instanceof Response) {
            Response r = (Response)response;
            value = r.header(HEADER_RETRY_AFTER);
        } else {
            throw new RuntimeException("Can't get response header. Invalid response object.");
        }
        if (value == null) {
            return null;
        }
        try {
            return Integer.parseInt(value);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    private class NetworkExecutor
    extends Executor {
        private NetworkExecutor() {
        }

        @Override
        public Object execute(Object call) throws DracoonNetIOException, InterceptedIOException, InterruptedException {
            try {
                return HttpHelper.executeCall(call);
            }
            catch (SSLHandshakeException e) {
                String errorText = "Server SSL handshake failed!";
                HttpHelper.this.mLog.e(LOG_TAG, errorText, e);
                throw new DracoonNetInsecureException(errorText, e);
            }
            catch (IOException e) {
                if (e.getClass().equals(InterruptedIOException.class)) {
                    throw new InterruptedException();
                }
                if (e.getClass().equals(InterceptedIOException.class)) {
                    throw (InterceptedIOException)e;
                }
                String errorText = "Server communication failed!";
                HttpHelper.this.mLog.d(LOG_TAG, errorText);
                throw new DracoonNetIOException(errorText, e);
            }
        }
    }

    private static abstract class Executor {
        protected Executor mNextExecutor;

        private Executor() {
        }

        public abstract Object execute(Object var1) throws DracoonNetIOException, DracoonApiException, InterceptedIOException, InterruptedException;
    }

    private class RetryExecutor
    extends Executor {
        public RetryExecutor(Executor nextExecutor) {
            this.mNextExecutor = nextExecutor;
        }

        @Override
        public Object execute(Object call) throws DracoonNetIOException, DracoonApiException, InterceptedIOException, InterruptedException {
            int retryCnt = 0;
            while (true) {
                try {
                    return this.mNextExecutor.execute(call);
                }
                catch (DracoonNetIOException e) {
                    if (retryCnt >= 3) {
                        throw e;
                    }
                    int sleepSeconds = retryCnt++;
                    HttpHelper.this.mLog.d(LOG_TAG, String.format("Next retry in %d seconds.", sleepSeconds));
                    Thread.sleep((long)sleepSeconds * 1000L);
                    call = HttpHelper.cloneCall(call);
                    continue;
                }
                break;
            }
        }
    }

    private class RateLimitingExecutor
    extends Executor {
        public RateLimitingExecutor(Executor nextExecutor) {
            this.mNextExecutor = nextExecutor;
        }

        @Override
        public Object execute(Object call) throws DracoonNetIOException, DracoonApiException, InterceptedIOException, InterruptedException {
            int retryCnt = 0;
            Object response;
            while (HttpHelper.isRateLimitResponse(response = this.mNextExecutor.execute(call))) {
                HttpHelper.this.mLog.d(LOG_TAG, "Server communication failed due to rate limit!");
                if (retryCnt >= 3) {
                    return response;
                }
                Integer retryAfterInterval = HttpHelper.getRetryAfterInterval(response);
                int sleepSeconds = retryAfterInterval != null ? retryAfterInterval : retryCnt + 1;
                HttpHelper.this.mLog.d(LOG_TAG, String.format("Next retry in %d seconds.", sleepSeconds));
                Thread.sleep((long)sleepSeconds * 1000L);
                call = HttpHelper.cloneCall(call);
                ++retryCnt;
            }
            return response;
        }
    }

    private class InterceptionErrorHandlingExecutor
    extends Executor {
        public InterceptionErrorHandlingExecutor(Executor nextExecutor) {
            this.mNextExecutor = nextExecutor;
        }

        @Override
        public Object execute(Object call) throws DracoonNetIOException, DracoonApiException, InterceptedIOException, InterruptedException {
            try {
                return this.mNextExecutor.execute(call);
            }
            catch (InterceptedIOException e) {
                HttpHelper.this.mLog.d(LOG_TAG, "Server communication was intercepted.");
                Throwable c = e.getCause();
                if (c != null) {
                    if (c.getClass().equals(DracoonNetIOException.class)) {
                        throw (DracoonNetIOException)c;
                    }
                    if (c.getClass().equals(DracoonApiException.class)) {
                        throw (DracoonApiException)c;
                    }
                }
                throw e;
            }
        }
    }
}

