/*
 * Decompiled with CFR 0.152.
 */
package com.dslplatform.compiler.client.parameters;

import com.dslplatform.compiler.client.CompileParameter;
import com.dslplatform.compiler.client.Context;
import com.dslplatform.compiler.client.Either;
import com.dslplatform.compiler.client.ExitException;
import com.dslplatform.compiler.client.ParameterParser;
import com.dslplatform.compiler.client.Utils;
import com.dslplatform.compiler.client.json.JSON;
import com.dslplatform.compiler.client.parameters.DatabaseInfo;
import com.dslplatform.compiler.client.parameters.Download;
import com.dslplatform.compiler.client.parameters.GrantRole;
import com.dslplatform.compiler.client.parameters.Mono;
import com.dslplatform.compiler.client.parameters.TempPath;
import com.dslplatform.compiler.client.parameters.VarraySize;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.management.ManagementFactory;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public enum DslCompiler implements CompileParameter,
ParameterParser
{
    INSTANCE;

    private static final Charset UTF_8;
    private static final String DSL_COMPILER_SOCKET = "dsl-compiler-socket";

    @Override
    public String getAlias() {
        return "compiler";
    }

    @Override
    public String getUsage() {
        return "path or IP port";
    }

    public static Map<String, String> compile(Context context, String target, List<String> settings, String namespace, String version, List<File> dsls, String library) throws ExitException {
        HashMap<String, String> files = new HashMap<String, String>();
        ArrayList<String> arguments = new ArrayList<String>();
        arguments.add("target=" + target);
        if (namespace != null && namespace.length() > 0) {
            arguments.add("namespace=" + namespace);
        }
        if (version != null && version.length() > 0) {
            arguments.add("version=" + version);
        }
        if (settings != null) {
            for (String o : settings) {
                arguments.add("settings=" + o);
            }
        }
        if (library != null && library.length() > 0) {
            arguments.add("library=" + library);
        }
        for (File f : dsls) {
            arguments.add("dsl=" + f.getAbsolutePath());
        }
        arguments.add("file-extension");
        context.log("Compiling DSL to " + target + "...");
        long start = new Date().getTime();
        Either<byte[]> response = DslCompiler.runCompiler(context, arguments);
        if (!response.isSuccess()) {
            context.error(response.whyNot());
            throw new ExitException();
        }
        long end = new Date().getTime();
        context.show("Creating the source took " + (end - start) / 1000L + " second(s)");
        Either<Document> xml = Utils.readXml(new ByteArrayInputStream(response.get()));
        if (!xml.isSuccess()) {
            context.error(new String(response.get(), UTF_8));
            throw new ExitException();
        }
        NodeList nodes = xml.get().getDocumentElement().getChildNodes();
        try {
            for (int i = 0; i < nodes.getLength(); ++i) {
                Element item = (Element)nodes.item(i);
                Node key = item.getElementsByTagName("Key").item(0);
                Node value = item.getElementsByTagName("Value").item(0);
                files.put(key.getTextContent(), value.getTextContent());
            }
        }
        catch (Exception ex) {
            context.error("Invalid xml found");
            context.error(new String(response.get(), UTF_8));
            throw new ExitException();
        }
        context.notify("SOURCES", files);
        return files;
    }

    private static Either<byte[]> runCompiler(Context context, List<String> arguments) throws ExitException {
        Socket socket = (Socket)context.load(DSL_COMPILER_SOCKET);
        File compiler = new File(context.get(INSTANCE));
        arguments.add("path=" + System.getProperty("user.dir"));
        context.notify("DSL", arguments);
        return socket != null ? DslCompiler.runCompilerSocket(context, socket, arguments) : DslCompiler.runCompilerFile(context, compiler, arguments);
    }

    private static int readInt(byte[] buf) {
        int ret = 0;
        for (int i = 0; i < 4; ++i) {
            ret <<= 8;
            ret |= buf[i] & 0xFF;
        }
        return ret;
    }

    public static Either<TokenParser> setupServer(Context context, File compiler) {
        context.show("Starting DSL Platform compiler...");
        Random rnd = new Random();
        int port = rnd.nextInt(40000) + 20000;
        Either<Process> tryProcess = DslCompiler.startServerMode(context, compiler, port);
        if (!tryProcess.isSuccess()) {
            return Either.fail(tryProcess.whyNot());
        }
        return Either.success(new TokenParser(context, compiler, port, tryProcess.get()));
    }

    private static Either<Process> startServerMode(Context context, File compiler, int port) {
        ArrayList<String> arguments = new ArrayList<String>();
        arguments.add(compiler.getAbsolutePath());
        arguments.add("server-mode");
        arguments.add("port=" + port);
        try {
            if (InetAddress.getLocalHost() instanceof Inet4Address) {
                arguments.add("ip=v4");
            }
        }
        catch (UnknownHostException ignore) {
            // empty catch block
        }
        try {
            String procId = ManagementFactory.getRuntimeMXBean().getName().split("@")[0];
            arguments.add("parent=" + procId);
        }
        catch (Exception ignore) {
            // empty catch block
        }
        if (!Utils.isWindows()) {
            Either<String> mono = Mono.findMono(context);
            if (mono.isSuccess()) {
                arguments.add(0, mono.get());
            } else {
                return Either.fail("Mono is required to run DSL compiler. Mono not detected or specified.");
            }
        }
        ProcessBuilder pb = new ProcessBuilder(arguments);
        context.put(INSTANCE, Integer.toString(port));
        try {
            return Either.success(pb.start());
        }
        catch (IOException e) {
            return Either.fail(e);
        }
    }

    private static boolean hasWhitespace(String input) {
        for (int i = 0; i < input.length(); ++i) {
            if (!Character.isWhitespace(input.charAt(i))) continue;
            return true;
        }
        return false;
    }

    private static Either<byte[]> runCompilerSocket(Context context, Socket socket, List<String> arguments) throws ExitException {
        StringBuilder sb = new StringBuilder();
        for (String arg : arguments) {
            if (!arg.startsWith("\"") && DslCompiler.hasWhitespace(arg)) {
                sb.append('\"');
                sb.append(arg);
                sb.append('\"');
            } else {
                sb.append(arg);
            }
            sb.append(' ');
        }
        sb.append("include-length keep-alive\n");
        try {
            int length;
            OutputStream sos = socket.getOutputStream();
            sos.write(sb.toString().getBytes(UTF_8));
            sos.flush();
            ByteStream os = DslCompiler.getByteStream(context);
            byte[] buf = os.temp;
            InputStream is = socket.getInputStream();
            int read = is.read(buf, 0, 4);
            boolean success = read == 4 && buf[0] == 79;
            read = is.read(buf, 0, 4);
            if (read != 4) {
                return Either.fail("Invalid response from server. Expecting length.");
            }
            context.log("Response size from DSL compiler: " + length);
            os.reset();
            for (length = DslCompiler.readInt(buf); length > 0 && (read = is.read(buf)) > 0; length -= read) {
                os.write(buf, 0, read);
            }
            os.flush();
            if (!success) {
                return Either.fail(os.toString("UTF-8"));
            }
            return Either.success(os.toByteArray());
        }
        catch (IOException e) {
            context.error(e);
            throw new ExitException();
        }
    }

    private static ByteStream getByteStream(Context context) {
        ByteStream os = (ByteStream)context.load("dsl-stream");
        if (os == null) {
            os = new ByteStream();
            context.cache("dsl-stream", os);
        }
        return os;
    }

    private static Either<byte[]> runCompilerFile(Context context, File compiler, List<String> arguments) throws ExitException {
        Either<Utils.CommandResult> result;
        if (Utils.isWindows()) {
            result = Utils.runCommand(context, compiler.getAbsolutePath(), compiler.getParentFile(), arguments);
        } else {
            Either<String> mono = Mono.findMono(context);
            if (mono.isSuccess()) {
                arguments.add(0, compiler.getAbsolutePath());
                result = Utils.runCommand(context, mono.get(), compiler.getParentFile(), arguments);
                if (!result.isSuccess() && DslCompiler.promptUserMonoRetry(context)) {
                    context.warning("Retrying ...");
                    context.error(result.whyNot());
                    try {
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException ignore) {
                        throw new ExitException();
                    }
                    result = Utils.runCommand(context, mono.get(), compiler.getParentFile(), arguments);
                }
            } else {
                context.error("Mono is required to run DSL compiler. Mono not detected or specified.");
                throw new ExitException();
            }
        }
        if (!result.isSuccess()) {
            return Either.fail(result.whyNot());
        }
        if (result.get().exitCode != 0) {
            return Either.fail(result.get().output + result.get().error);
        }
        return Either.success(result.get().output.getBytes(UTF_8));
    }

    public static Either<String> migration(Context context, DatabaseInfo dbInfo, List<File> currentDsls) throws ExitException {
        ArrayList<String> arguments = new ArrayList<String>();
        arguments.add("target=" + dbInfo.database.toLowerCase() + dbInfo.dbVersion);
        if (context.contains(VarraySize.INSTANCE)) {
            arguments.add("varray=" + context.get(VarraySize.INSTANCE));
        }
        if (context.contains(GrantRole.INSTANCE)) {
            arguments.add("role=" + context.get(GrantRole.INSTANCE));
        }
        if (dbInfo.dsl != null && !dbInfo.dsl.isEmpty()) {
            StringBuilder oldDsl = new StringBuilder();
            for (String v : dbInfo.dsl.values()) {
                oldDsl.append(v);
            }
            File previousDsl = new File(TempPath.getTempProjectPath(context), "old.dsl");
            try {
                Utils.saveFile(context, previousDsl, oldDsl.toString());
            }
            catch (IOException ex) {
                context.error("Unable to save old DSL version for comparison.");
                return Either.fail(ex);
            }
            arguments.add("previous-dsl=" + previousDsl.getAbsolutePath());
            if (dbInfo.compilerVersion != null) {
                arguments.add("previous-compiler=" + dbInfo.compilerVersion);
            }
        }
        for (File f : currentDsls) {
            arguments.add("dsl=" + f.getAbsolutePath());
        }
        context.log("Creating SQL migration for " + dbInfo.database + " ...");
        Either<byte[]> result = DslCompiler.runCompiler(context, arguments);
        if (!result.isSuccess()) {
            return Either.fail(result.whyNot());
        }
        String sql = new String(result.get(), UTF_8);
        return Either.success(context.notify("MIGRATION", sql));
    }

    public static Either<Boolean> parse(Context context, List<File> dsls) throws ExitException {
        ArrayList<String> arguments = new ArrayList<String>();
        for (File f : dsls) {
            arguments.add("dsl=" + f.getAbsolutePath());
        }
        context.log("Parsing DSL...");
        Either<byte[]> response = DslCompiler.runCompiler(context, arguments);
        if (!response.isSuccess()) {
            return Either.fail(response.whyNot());
        }
        return Either.success(false);
    }

    public static File lookupDefaultPath(Context context) throws ExitException {
        String home = System.getProperty("user.home");
        if (home == null || home.isEmpty()) {
            context.log("Variable user.home not found. Will fallback to temporary path");
            return new File(TempPath.getTempRootPath(context), "dsl-compiler.exe");
        }
        File homePath = new File(home);
        if (!homePath.exists()) {
            context.log("User home not found. Will fallback to temporary path");
            return new File(TempPath.getTempRootPath(context), "dsl-compiler.exe");
        }
        String dslName = Download.isDefaultUrl(context) ? ".DSL-Platform" : ".DSL-Custom";
        File compilerFolder = new File(homePath, dslName);
        if (!compilerFolder.exists() && !compilerFolder.mkdirs()) {
            context.warning("Error creating dsl compiler path in: " + compilerFolder.getAbsolutePath() + ". Will use temporary folder instead.");
            return new File(TempPath.getTempRootPath(context), "dsl-compiler.exe");
        }
        return new File(compilerFolder, "dsl-compiler.exe");
    }

    @Override
    public Either<Boolean> tryParse(String name, String value, Context context) {
        if ("compiler".equals(name)) {
            context.put(INSTANCE, value);
            return Either.success(true);
        }
        return Either.success(false);
    }

    @Override
    public boolean check(Context context) throws ExitException {
        File path;
        boolean isEmpty;
        String value = context.contains(INSTANCE) ? context.get(INSTANCE) : null;
        boolean bl = isEmpty = value == null || value.length() == 0;
        if (!isEmpty) {
            int port = -1;
            try {
                port = Integer.parseInt(value);
            }
            catch (NumberFormatException ignore) {
                // empty catch block
            }
            if (port > 0) {
                Socket socket;
                try {
                    socket = new Socket("::1", port);
                }
                catch (Exception ex6) {
                    context.log("Unable to open socket to port on IPv6 localhost: " + value);
                    try {
                        socket = new Socket("127.0.0.1", port);
                    }
                    catch (Exception ex4) {
                        context.error("Unable to open socket to port on localhost: " + value);
                        context.error(ex4);
                        throw new ExitException();
                    }
                }
                if (socket.isConnected()) {
                    try {
                        socket.setKeepAlive(true);
                        socket.setSoTimeout(30000);
                    }
                    catch (SocketException ignore) {
                        // empty catch block
                    }
                    context.cache(DSL_COMPILER_SOCKET, socket);
                    return true;
                }
                context.error("Unable to connect to specified compiler on localhost:" + value);
                throw new ExitException();
            }
        }
        if ((path = new File(isEmpty ? "dsl-compiler.exe" : value)).isDirectory()) {
            context.error("Specified compiler path is a directory: " + path.getAbsolutePath());
            throw new ExitException();
        }
        if (!path.exists()) {
            String answer;
            File compiler;
            if (!isEmpty) {
                if (context.canInteract() || context.contains(Download.INSTANCE)) {
                    context.warning("Specified compiler path not found: " + path.getAbsolutePath());
                } else {
                    context.error("Specified compiler path not found: " + path.getAbsolutePath());
                    throw new ExitException();
                }
            }
            if ((compiler = DslCompiler.lookupDefaultPath(context)).exists() && DslCompiler.testCompiler(context, compiler)) {
                if (isEmpty) {
                    if (context.contains(Download.INSTANCE)) {
                        context.show("Checking for latest compiler version due to download option");
                        DslCompiler.checkForLatestVersion(context, path, compiler.getParentFile(), compiler);
                    }
                    context.put(INSTANCE, compiler.getAbsolutePath());
                    return true;
                }
                if (context.canInteract() && (answer = context.ask("Compiler found in default location: " + compiler.getAbsolutePath() + ". Do you wish to use it? (y/N)")).toLowerCase().equals("y")) {
                    context.put(INSTANCE, compiler.getAbsolutePath());
                    return true;
                }
            }
            if (!context.contains(Download.INSTANCE)) {
                if (context.canInteract()) {
                    answer = context.ask("Do you wish to download compiler from the Internet? (y/N)");
                    if (!answer.toLowerCase().equals("y")) {
                        throw new ExitException();
                    }
                } else {
                    throw new ExitException();
                }
            }
            context.show("Downloading DSL Platform compiler since it's not found in: " + compiler.getAbsolutePath());
            DslCompiler.downloadCompiler(context, path, compiler.getParentFile(), compiler, true);
        } else {
            if (!DslCompiler.testCompiler(context, path)) {
                context.error("Specified compiler is invalid: " + path.getAbsolutePath());
                throw new ExitException();
            }
            context.put(INSTANCE, path.getAbsolutePath());
        }
        return true;
    }

    public static void checkForLatestVersion(Context context, File path, File compilerPath, File compiler) throws ExitException {
        Either<Long> lastModified = Download.lastModified(context, "dsl-compiler", "dsl-compiler.exe", compiler.lastModified());
        if (!lastModified.isSuccess()) {
            context.warning(lastModified.whyNot());
        } else if (compiler.lastModified() != lastModified.get().longValue()) {
            DslCompiler.downloadCompiler(context, path, compilerPath, compiler, false);
        }
    }

    private static void downloadCompiler(Context context, File path, File compilerPath, File compiler, boolean failOnError) throws ExitException {
        long lastModified;
        try {
            lastModified = Download.downloadAndUnpack(context, "dsl-compiler", compilerPath);
        }
        catch (IOException ex) {
            String remoteUrl = Download.remoteUrl(context);
            if (failOnError) {
                context.error("Error downloading compiler from " + remoteUrl);
                context.error("Try specifying alternative compiler or download location.");
                context.error(ex);
                throw new ExitException();
            }
            context.warning("Error downloading compiler from " + remoteUrl);
            context.warning("Skipping...");
            return;
        }
        if (!DslCompiler.testCompiler(context, compiler)) {
            context.error("Specified compiler is invalid: " + path.getAbsolutePath());
            throw new ExitException();
        }
        if (!compiler.setLastModified(lastModified)) {
            context.warning("Unable to set matching last modified date");
        }
        context.put(INSTANCE, compiler.getAbsolutePath());
    }

    private static boolean testCompiler(Context context, File path) throws ExitException {
        if (Utils.isWindows()) {
            return Utils.testCommand(context, path.getAbsolutePath(), "DSL Platform");
        }
        Either<String> mono = Mono.findMono(context);
        if (mono.isSuccess()) {
            return Utils.testCommand(context, mono.get(), "DSL Platform", Collections.singletonList(path.getAbsolutePath()));
        }
        context.error("Mono is required to run DSL compiler. Mono not detected or specified.");
        throw new ExitException();
    }

    @Override
    public void run(Context context) {
    }

    @Override
    public String getShortDescription() {
        return "Path or localhost port to DSL Platform compiler. .NET or Mono is required to run the compiler";
    }

    @Override
    public String getDetailedDescription() {
        return "DSL Platform compiler.\nRequires .NET/Mono to run.\nIt is available for download at: https://tools.dsl-platform.com/dsl-compiler.zip\n\nExample:\n\tcompiler\n\tcompiler=/var/dsl-platform/dsl-compiler.exe\n\tcompiler=12345\n";
    }

    private static boolean promptUserMonoRetry(Context context) {
        context.warning("Running the compiler failed... ");
        if (context.canInteract()) {
            String answer = context.ask("Do you wish to retry [Y/n]?");
            return answer != null && ("y".equalsIgnoreCase(answer.trim()) || answer.trim().isEmpty());
        }
        return true;
    }

    static {
        UTF_8 = Charset.forName("UTF-8");
    }

    public static class TokenParser
    implements Closeable {
        private final Context context;
        private final File compiler;
        private final Map<String, RuleInfo> rules = new HashMap<String, RuleInfo>();
        private int port;
        private Socket socket;
        private Process process;
        private long startedOn;

        public int getPort() {
            return this.port;
        }

        TokenParser(Context context, File compiler, int port, Process process) {
            this.context = context;
            this.compiler = compiler;
            this.setupMonitor(port, process, this);
        }

        public Either<RuleInfo> findRule(String name) {
            if (name != null) {
                RuleInfo rule;
                if (this.rules.isEmpty()) {
                    try {
                        Socket sck = this.setupSocket();
                        if (sck == null) {
                            return Either.fail("Unable to setup socket.");
                        }
                        Either<List<RuleInfo>> newRules = this.loadRules(sck);
                        if (!newRules.isSuccess()) {
                            return Either.fail(newRules.whyNot());
                        }
                        for (RuleInfo ri : newRules.get()) {
                            this.rules.put(ri.rule, ri);
                        }
                    }
                    catch (Exception ex) {
                        return Either.fail("Unable to load rules: " + ex.getMessage());
                    }
                }
                if ((rule = this.rules.get(name)) != null) {
                    return Either.success(rule);
                }
                Either.fail("Unknown rule: " + name);
            }
            return Either.fail("Rule name can't be null");
        }

        private void setupMonitor(int port, final Process process, final TokenParser parser) {
            parser.port = port;
            parser.process = process;
            this.context.show("Started DSL Platform compiler at port: " + port);
            Thread waitExit = new Thread(new Runnable(){

                @Override
                public void run() {
                    try {
                        process.waitFor();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    TokenParser.this.context.show("DSL Platform compiler process stopped");
                    parser.process = null;
                }
            });
            waitExit.setDaemon(true);
            waitExit.start();
            Thread consumeOutput = new Thread(new Runnable(){

                @Override
                public void run() {
                    BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                    char[] buffer = new char[8192];
                    try {
                        int len;
                        while ((len = reader.read(buffer)) != -1) {
                            TokenParser.this.context.log(new String(buffer, 0, len));
                        }
                        reader.close();
                    }
                    catch (IOException ignore) {
                        // empty catch block
                    }
                }
            });
            consumeOutput.setDaemon(true);
            consumeOutput.start();
            this.startedOn = new Date().getTime();
        }

        public Either<ParseResult> parse(String dsl) {
            try {
                if (this.process == null) {
                    Random rnd = new Random();
                    int port = rnd.nextInt(40000) + 20000;
                    Either tryProcess = DslCompiler.startServerMode(this.context, this.compiler, port);
                    if (!tryProcess.isSuccess()) {
                        return Either.fail(tryProcess.whyNot());
                    }
                    this.setupMonitor(port, (Process)tryProcess.get(), this);
                    return Either.fail("Server restarting...");
                }
                Either<ParseResult> result = this.parseTokens(this.setupSocket(), dsl);
                if (!result.isSuccess()) {
                    this.socketCleanup(false);
                    result = this.parseTokens(this.setupSocket(), dsl);
                }
                if (!result.isSuccess()) {
                    this.socketCleanup(true);
                }
                return result;
            }
            catch (Exception ex) {
                this.socketCleanup(true);
                return Either.fail(ex);
            }
        }

        private Either<List<RuleInfo>> loadRules(Socket socket) {
            String command = "format=json rules include-length keep-alive\n";
            try {
                OutputStream sos = socket.getOutputStream();
                sos.write("format=json rules include-length keep-alive\n".getBytes(UTF_8));
                sos.flush();
                ByteStream os = DslCompiler.getByteStream(this.context);
                byte[] buf = os.temp;
                InputStream is = socket.getInputStream();
                int read = is.read(buf, 0, 4);
                if (read != 4 || buf[0] != 79) {
                    return Either.fail("Invalid response from server.");
                }
                read = is.read(buf, 0, 4);
                if (read != 4) {
                    return Either.fail("Invalid response from server. Expecting length.");
                }
                os.reset();
                for (int length = DslCompiler.readInt(buf); length > 0 && (read = is.read(buf)) > 0; length -= read) {
                    os.write(buf, 0, read);
                }
                os.flush();
                List<Object> result = JSON.readList(os.getBuffer(), os.size());
                ArrayList<RuleInfo> rules = new ArrayList<RuleInfo>(result.size());
                for (Object it : result) {
                    rules.add(new RuleInfo((Map)it));
                }
                return Either.success(rules);
            }
            catch (IOException e) {
                return Either.fail(e.getMessage());
            }
        }

        private Either<ParseResult> parseTokens(Socket socket, String dsl) throws IOException {
            byte[] dslUtf8 = dsl.getBytes(UTF_8);
            String command = "tokens=" + dslUtf8.length + " format=json include-length keep-alive\n";
            try {
                OutputStream sos = socket.getOutputStream();
                sos.write(command.getBytes(UTF_8));
                sos.write(dslUtf8);
                sos.flush();
                ByteStream os = DslCompiler.getByteStream(this.context);
                byte[] buf = os.temp;
                InputStream is = socket.getInputStream();
                int read = is.read(buf, 0, 4);
                if (read != 4 || buf[0] != 79) {
                    return Either.fail("Invalid response from server.");
                }
                read = is.read(buf, 0, 4);
                if (read != 4) {
                    return Either.fail("Invalid response from server. Expecting length.");
                }
                os.reset();
                for (int length = DslCompiler.readInt(buf); length > 0 && (read = is.read(buf)) > 0; length -= read) {
                    os.write(buf, 0, read);
                }
                os.flush();
                return Either.success(new ParseResult(JSON.readMap(os.getBuffer(), os.size())));
            }
            catch (IOException e) {
                return Either.fail(e.getMessage());
            }
        }

        private Socket setupSocket() throws ExitException, IOException {
            if (this.socket != null) {
                return this.socket;
            }
            this.context.put(INSTANCE, Integer.toString(this.port));
            if (!INSTANCE.check(this.context)) {
                this.context.error("Unable to setup socket to DSL Platform");
            }
            this.context.show("Socket connected");
            this.socket = (Socket)this.context.load(DslCompiler.DSL_COMPILER_SOCKET);
            if (this.socket != null) {
                try {
                    this.socket.setSoTimeout(10000);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            return this.socket;
        }

        private void socketCleanup(boolean restartServer) {
            Socket sock;
            long now = new Date().getTime();
            if (restartServer && now > this.startedOn + 60000L) {
                this.stopServer();
            }
            if ((sock = this.socket) != null) {
                try {
                    sock.close();
                }
                catch (Exception ignore) {
                    // empty catch block
                }
                this.socket = null;
                this.context.cache(DslCompiler.DSL_COMPILER_SOCKET, null);
            }
        }

        private void stopServer() {
            Process proc = this.process;
            if (proc != null) {
                this.context.show("Stopped DSL Platform compiler");
                try {
                    proc.destroy();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                this.process = null;
            }
        }

        @Override
        public void close() {
            this.stopServer();
        }
    }

    public static class RuleInfo {
        public final String rule;
        public final String grammar;
        public final String[] children;
        public final String description;

        RuleInfo(Map<String, Object> json) {
            this.rule = (String)json.get("Rule");
            String gr = (String)json.get("Grammar");
            ArrayList ch = (ArrayList)json.get("Children");
            String de = (String)json.get("Description");
            this.grammar = gr != null ? gr : "";
            this.children = ch != null ? ch.toArray(new String[0]) : new String[]{};
            this.description = de != null ? de : "";
        }
    }

    private static class ByteStream
    extends ByteArrayOutputStream {
        private final byte[] temp = new byte[8192];

        private ByteStream() {
        }

        byte[] getBuffer() {
            return this.buf;
        }
    }

    public static class SyntaxConcept {
        public final SyntaxType type;
        public final String value;
        public final String script;
        public final int line;
        public final int column;

        SyntaxConcept(Map<String, Object> map) {
            Object value;
            this.type = map.containsKey("Type") ? ((value = map.get("Type")) instanceof String ? SyntaxType.valueOf((String)value) : SyntaxType.values()[((Number)value).intValue()]) : SyntaxType.Keyword;
            this.value = map.containsKey("Value") ? (String)map.get("Value") : "";
            this.script = map.containsKey("Script") ? (String)map.get("Script") : "";
            this.line = map.containsKey("Line") ? ((Number)map.get("Line")).intValue() : 0;
            this.column = map.containsKey("Column") ? ((Number)map.get("Column")).intValue() : 0;
        }
    }

    public static enum SyntaxType {
        Keyword,
        Identifier,
        StringQuote,
        Expression,
        Type,
        Navigation,
        RuleStart,
        RuleExtension,
        RuleEnd;

    }

    public static class ParseResult {
        public final ParseError error;
        public final List<SyntaxConcept> tokens;

        ParseResult(Map<String, Object> map) {
            Object err;
            List tokenMap = (List)map.get("Tokens");
            this.tokens = new ArrayList<SyntaxConcept>(tokenMap != null ? tokenMap.size() : 0);
            if (tokenMap != null) {
                for (Map t : tokenMap) {
                    this.tokens.add(new SyntaxConcept(t));
                }
            }
            this.error = (err = map.get("Error")) != null ? new ParseError((Map)err) : null;
        }
    }

    public static class ParseError {
        public final int line;
        public final int column;
        public final String error;

        ParseError(Map<String, Object> map) {
            this.line = map.containsKey("Line") ? ((Number)map.get("Line")).intValue() : 0;
            this.column = map.containsKey("Column") ? ((Number)map.get("Column")).intValue() : 0;
            this.error = (String)map.get("Error");
        }
    }
}

