package com.dylibso.chicory.wasm.types;

import java.util.HashMap;
import java.util.Map;

public enum OpCode {

    //unreachable
    UNREACHABLE(0x00),
    //nop
    NOP(0x01),
    //block <varuint>
    BLOCK(0x02),
    //loop <varuint>
    LOOP(0x03),
    //if <varuint>
    IF(0x04),
    //else
    ELSE(0x05),
    //end
    END(0x0B),
    //br <varuint>
    BR(0x0C),
    //br_if <varuint>
    BR_IF(0x0D),
    //br_table vec(<varuint>) <varuint>
    BR_TABLE(0x0E),
    //return
    RETURN(0x0F),
    //call <varuint>
    CALL(0x10),
    //call_indirect <varuint> <varuint>
    CALL_INDIRECT(0x11),
    //call_ref <varuint>
    CALL_REF(0x14),
    //drop
    DROP(0x1A),
    //select
    SELECT(0x1B),
    //select.t vec(<varuint>)
    SELECT_T(0x1C),
    //local.get <varuint>
    LOCAL_GET(0x20),
    //local.set <varuint>
    LOCAL_SET(0x21),
    //local.tee <varuint>
    LOCAL_TEE(0x22),
    //global.get <varuint>
    GLOBAL_GET(0x23),
    //global.set <varuint>
    GLOBAL_SET(0x24),
    //table.get <varuint>
    TABLE_GET(0x25),
    //table.set <varuint>
    TABLE_SET(0x26),
    //i32.load <varuint> <varuint>
    I32_LOAD(0x28),
    //i64.load <varuint> <varuint>
    I64_LOAD(0x29),
    //f32.load <varuint> <varuint>
    F32_LOAD(0x2A),
    //f64.load <varuint> <varuint>
    F64_LOAD(0x2B),
    //i32.load8_s <varuint> <varuint>
    I32_LOAD8_S(0x2C),
    //i32.load8_u <varuint> <varuint>
    I32_LOAD8_U(0x2D),
    //i32.load16_s <varuint> <varuint>
    I32_LOAD16_S(0x2E),
    //i32.load16_u <varuint> <varuint>
    I32_LOAD16_U(0x2F),
    //i64.load8_s <varuint> <varuint>
    I64_LOAD8_S(0x30),
    //i64.load8_u <varuint> <varuint>
    I64_LOAD8_U(0x31),
    //i64.load16_s <varuint> <varuint>
    I64_LOAD16_S(0x32),
    //i64.load16_u <varuint> <varuint>
    I64_LOAD16_U(0x33),
    //i64.load32_s <varuint> <varuint>
    I64_LOAD32_S(0x34),
    //i64.load32_u <varuint> <varuint>
    I64_LOAD32_U(0x35),
    //i32.store <varuint> <varuint>
    I32_STORE(0x36),
    //i64.store <varuint> <varuint>
    I64_STORE(0x37),
    //f32.store <varuint> <varuint>
    F32_STORE(0x38),
    //f64.store <varuint> <varuint>
    F64_STORE(0x39),
    //i32.store8 <varuint> <varuint>
    I32_STORE8(0x3A),
    //i32.store16 <varuint> <varuint>
    I32_STORE16(0x3B),
    //i64.store8 <varuint> <varuint>
    I64_STORE8(0x3C),
    //i64.store16 <varuint> <varuint>
    I64_STORE16(0x3D),
    //i64.store32 <varuint> <varuint>
    I64_STORE32(0x3E),
    //memory.size
    MEMORY_SIZE(0x3F),
    //memory.grow
    MEMORY_GROW(0x40),
    //i32.const <varsint32>
    I32_CONST(0x41),
    //i64.const <varsint64>
    I64_CONST(0x42),
    //f32.const <float32>
    F32_CONST(0x43),
    //f64.const <float64>
    F64_CONST(0x44),
    //i32.eqz
    I32_EQZ(0x45),
    //i32.eq
    I32_EQ(0x46),
    //i32.ne
    I32_NE(0x47),
    //i32.lt_s
    I32_LT_S(0x48),
    //i32.lt_u
    I32_LT_U(0x49),
    //i32.gt_s
    I32_GT_S(0x4A),
    //i32.gt_u
    I32_GT_U(0x4B),
    //i32.le_s
    I32_LE_S(0x4C),
    //i32.le_u
    I32_LE_U(0x4D),
    //i32.ge_s
    I32_GE_S(0x4E),
    //i32.ge_u
    I32_GE_U(0x4F),
    //i64.eqz
    I64_EQZ(0x50),
    //i64.eq
    I64_EQ(0x51),
    //i64.ne
    I64_NE(0x52),
    //i64.lt_s
    I64_LT_S(0x53),
    //i64.lt_u
    I64_LT_U(0x54),
    //i64.gt_s
    I64_GT_S(0x55),
    //i64.gt_u
    I64_GT_U(0x56),
    //i64.le_s
    I64_LE_S(0x57),
    //i64.le_u
    I64_LE_U(0x58),
    //i64.ge_s
    I64_GE_S(0x59),
    //i64.ge_u
    I64_GE_U(0x5A),
    //f32.eq
    F32_EQ(0x5B),
    //f32.ne
    F32_NE(0x5C),
    //f32.lt
    F32_LT(0x5D),
    //f32.gt
    F32_GT(0x5E),
    //f32.le
    F32_LE(0x5F),
    //f32.ge
    F32_GE(0x60),
    //f64.eq
    F64_EQ(0x61),
    //f64.ne
    F64_NE(0x62),
    //f64.lt
    F64_LT(0x63),
    //f64.gt
    F64_GT(0x64),
    //f64.le
    F64_LE(0x65),
    //f64.ge
    F64_GE(0x66),
    //i32.clz
    I32_CLZ(0x67),
    //i32.ctz
    I32_CTZ(0x68),
    //i32.popcnt
    I32_POPCNT(0x69),
    //i32.add
    I32_ADD(0x6A),
    //i32.sub
    I32_SUB(0x6B),
    //i32.mul
    I32_MUL(0x6C),
    //i32.div_s
    I32_DIV_S(0x6D),
    //i32.div_u
    I32_DIV_U(0x6E),
    //i32.rem_s
    I32_REM_S(0x6F),
    //i32.rem_u
    I32_REM_U(0x70),
    //i32.and
    I32_AND(0x71),
    //i32.or
    I32_OR(0x72),
    //i32.xor
    I32_XOR(0x73),
    //i32.shl
    I32_SHL(0x74),
    //i32.shr_s
    I32_SHR_S(0x75),
    //i32.shr_u
    I32_SHR_U(0x76),
    //i32.rotl
    I32_ROTL(0x77),
    //i32.rotr
    I32_ROTR(0x78),
    //i64.clz
    I64_CLZ(0x79),
    //i64.ctz
    I64_CTZ(0x7A),
    //i64.popcnt
    I64_POPCNT(0x7B),
    //i64.add
    I64_ADD(0x7C),
    //i64.sub
    I64_SUB(0x7D),
    //i64.mul
    I64_MUL(0x7E),
    //i64.div_s
    I64_DIV_S(0x7F),
    //i64.div_u
    I64_DIV_U(0x80),
    //i64.rem_s
    I64_REM_S(0x81),
    //i64.rem_u
    I64_REM_U(0x82),
    //i64.and
    I64_AND(0x83),
    //i64.or
    I64_OR(0x84),
    //i64.xor
    I64_XOR(0x85),
    //i64.shl
    I64_SHL(0x86),
    //i64.shr_s
    I64_SHR_S(0x87),
    //i64.shr_u
    I64_SHR_U(0x88),
    //i64.rotl
    I64_ROTL(0x89),
    //i64.rotr
    I64_ROTR(0x8A),
    //f32.abs
    F32_ABS(0x8B),
    //f32.neg
    F32_NEG(0x8C),
    //f32.ceil
    F32_CEIL(0x8D),
    //f32.floor
    F32_FLOOR(0x8E),
    //f32.trunc
    F32_TRUNC(0x8F),
    //f32.nearest
    F32_NEAREST(0x90),
    //f32.sqrt
    F32_SQRT(0x91),
    //f32.add
    F32_ADD(0x92),
    //f32.sub
    F32_SUB(0x93),
    //f32.mul
    F32_MUL(0x94),
    //f32.div
    F32_DIV(0x95),
    //f32.min
    F32_MIN(0x96),
    //f32.max
    F32_MAX(0x97),
    //f32.copysign
    F32_COPYSIGN(0x98),
    //f64.abs
    F64_ABS(0x99),
    //f64.neg
    F64_NEG(0x9A),
    //f64.ceil
    F64_CEIL(0x9B),
    //f64.floor
    F64_FLOOR(0x9C),
    //f64.trunc
    F64_TRUNC(0x9D),
    //f64.nearest
    F64_NEAREST(0x9E),
    //f64.sqrt
    F64_SQRT(0x9F),
    //f64.add
    F64_ADD(0xA0),
    //f64.sub
    F64_SUB(0xA1),
    //f64.mul
    F64_MUL(0xA2),
    //f64.div
    F64_DIV(0xA3),
    //f64.min
    F64_MIN(0xA4),
    //f64.max
    F64_MAX(0xA5),
    //f64.copysign
    F64_COPYSIGN(0xA6),
    //i32.wrap_i64
    I32_WRAP_I64(0xA7),
    //i32.trunc_f32_s
    I32_TRUNC_F32_S(0xA8),
    //i32.trunc_f32_u
    I32_TRUNC_F32_U(0xA9),
    //i32.trunc_f64_s
    I32_TRUNC_F64_S(0xAA),
    //i32.trunc_f64_u
    I32_TRUNC_F64_U(0xAB),
    //i64.extend_i32_s
    I64_EXTEND_I32_S(0xAC),
    //i64.extend_i32_u
    I64_EXTEND_I32_U(0xAD),
    //i64.trunc_f32_s
    I64_TRUNC_F32_S(0xAE),
    //i64.trunc_f32_u
    I64_TRUNC_F32_U(0xAF),
    //i64.trunc_f64_s
    I64_TRUNC_F64_S(0xB0),
    //i64.trunc_f64_u
    I64_TRUNC_F64_U(0xB1),
    //f32.convert_i32_s
    F32_CONVERT_I32_S(0xB2),
    //f32.convert_i32_u
    F32_CONVERT_I32_U(0xB3),
    //f32.convert_i64_s
    F32_CONVERT_I64_S(0xB4),
    //f32.convert_i64_u
    F32_CONVERT_I64_U(0xB5),
    //f32.demote_f64
    F32_DEMOTE_F64(0xB6),
    //f64.convert_i32_s
    F64_CONVERT_I32_S(0xB7),
    //f64.convert_i32_u
    F64_CONVERT_I32_U(0xB8),
    //f64.convert_i64_s
    F64_CONVERT_I64_S(0xB9),
    //f64.convert_i64_u
    F64_CONVERT_I64_U(0xBA),
    //f64.promote_f32
    F64_PROMOTE_F32(0xBB),
    //i32.reinterpret_f32
    I32_REINTERPRET_F32(0xBC),
    //i64.reinterpret_f64
    I64_REINTERPRET_F64(0xBD),
    //f32.reinterpret_i32
    F32_REINTERPRET_I32(0xBE),
    //f64.reinterpret_i64
    F64_REINTERPRET_I64(0xBF),
    //i32.extend_8_s
    I32_EXTEND_8_S(0xC0),
    //i32.extend_16_s
    I32_EXTEND_16_S(0xC1),
    //i64.extend_8_s
    I64_EXTEND_8_S(0xC2),
    //i64.extend_16_s
    I64_EXTEND_16_S(0xC3),
    //i64.extend_32_s
    I64_EXTEND_32_S(0xC4),
    //ref.null <varuint>
    REF_NULL(0xD0),
    //ref.is_null
    REF_IS_NULL(0xD1),
    //ref.func <varuint>
    REF_FUNC(0xD2),
    //i32.trunc_sat_f32_s
    I32_TRUNC_SAT_F32_S(0xFC00),
    //i32.trunc_sat_f32_u
    I32_TRUNC_SAT_F32_U(0xFC01),
    //i32.trunc_sat_f64_s
    I32_TRUNC_SAT_F64_S(0xFC02),
    //i32.trunc_sat_f64_u
    I32_TRUNC_SAT_F64_U(0xFC03),
    //i64.trunc_sat_f32_s
    I64_TRUNC_SAT_F32_S(0xFC04),
    //i64.trunc_sat_f32_u
    I64_TRUNC_SAT_F32_U(0xFC05),
    //i64.trunc_sat_f64_s
    I64_TRUNC_SAT_F64_S(0xFC06),
    //i64.trunc_sat_f64_u
    I64_TRUNC_SAT_F64_U(0xFC07),
    //memory.init <varuint> <varuint>
    MEMORY_INIT(0xFC08),
    //data.drop <varuint>
    DATA_DROP(0xFC09),
    //memory.copy <varuint> <varuint>
    MEMORY_COPY(0xFC0A),
    //memory.fill <varuint>
    MEMORY_FILL(0xFC0B),
    //table.init <varuint> <varuint>
    TABLE_INIT(0xFC0C),
    //elem.drop <varuint>
    ELEM_DROP(0xFC0D),
    //table.copy <varuint> <varuint>
    TABLE_COPY(0xFC0E),
    //table.grow <varuint>
    TABLE_GROW(0xFC0F),
    //table.size <varuint>
    TABLE_SIZE(0xFC10),
    //table.fill <varuint>
    TABLE_FILL(0xFC11);

    private final int opcode;

    OpCode(int opcode) {
        this.opcode = opcode;
    }

    public int opcode() {
        return opcode;
    }

    private static final Map<Integer, OpCode> byOpCode = new HashMap<>();

    public static OpCode byOpCode(int opcode) {
        return byOpCode.get(opcode);
    }

    private static final Map<OpCode, WasmEncoding[]> signature = new HashMap<>();

    public static WasmEncoding[] getSignature(OpCode o) {
        return signature.get(o);
    }

    static {
        for (OpCode e: OpCode.values()) { byOpCode.put(e.opcode(), e); };
        signature.put(UNREACHABLE, new WasmEncoding[] {});
        signature.put(NOP, new WasmEncoding[] {});
        signature.put(BLOCK, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(LOOP, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(IF, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(ELSE, new WasmEncoding[] {});
        signature.put(END, new WasmEncoding[] {});
        signature.put(BR, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(BR_IF, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(BR_TABLE, new WasmEncoding[] {WasmEncoding.VEC_VARUINT, WasmEncoding.VARUINT});
        signature.put(RETURN, new WasmEncoding[] {});
        signature.put(CALL, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(CALL_INDIRECT, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(CALL_REF, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(DROP, new WasmEncoding[] {});
        signature.put(SELECT, new WasmEncoding[] {});
        signature.put(SELECT_T, new WasmEncoding[] {WasmEncoding.VEC_VARUINT});
        signature.put(LOCAL_GET, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(LOCAL_SET, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(LOCAL_TEE, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(GLOBAL_GET, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(GLOBAL_SET, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(TABLE_GET, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(TABLE_SET, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(I32_LOAD, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(F32_LOAD, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(F64_LOAD, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_LOAD8_S, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_LOAD8_U, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_LOAD16_S, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_LOAD16_U, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD8_S, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD8_U, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD16_S, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD16_U, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD32_S, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_LOAD32_U, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_STORE, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_STORE, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(F32_STORE, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(F64_STORE, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_STORE8, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I32_STORE16, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_STORE8, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_STORE16, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(I64_STORE32, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(MEMORY_SIZE, new WasmEncoding[] {});
        signature.put(MEMORY_GROW, new WasmEncoding[] {});
        signature.put(I32_CONST, new WasmEncoding[] {WasmEncoding.VARSINT32});
        signature.put(I64_CONST, new WasmEncoding[] {WasmEncoding.VARSINT64});
        signature.put(F32_CONST, new WasmEncoding[] {WasmEncoding.FLOAT32});
        signature.put(F64_CONST, new WasmEncoding[] {WasmEncoding.FLOAT64});
        signature.put(I32_EQZ, new WasmEncoding[] {});
        signature.put(I32_EQ, new WasmEncoding[] {});
        signature.put(I32_NE, new WasmEncoding[] {});
        signature.put(I32_LT_S, new WasmEncoding[] {});
        signature.put(I32_LT_U, new WasmEncoding[] {});
        signature.put(I32_GT_S, new WasmEncoding[] {});
        signature.put(I32_GT_U, new WasmEncoding[] {});
        signature.put(I32_LE_S, new WasmEncoding[] {});
        signature.put(I32_LE_U, new WasmEncoding[] {});
        signature.put(I32_GE_S, new WasmEncoding[] {});
        signature.put(I32_GE_U, new WasmEncoding[] {});
        signature.put(I64_EQZ, new WasmEncoding[] {});
        signature.put(I64_EQ, new WasmEncoding[] {});
        signature.put(I64_NE, new WasmEncoding[] {});
        signature.put(I64_LT_S, new WasmEncoding[] {});
        signature.put(I64_LT_U, new WasmEncoding[] {});
        signature.put(I64_GT_S, new WasmEncoding[] {});
        signature.put(I64_GT_U, new WasmEncoding[] {});
        signature.put(I64_LE_S, new WasmEncoding[] {});
        signature.put(I64_LE_U, new WasmEncoding[] {});
        signature.put(I64_GE_S, new WasmEncoding[] {});
        signature.put(I64_GE_U, new WasmEncoding[] {});
        signature.put(F32_EQ, new WasmEncoding[] {});
        signature.put(F32_NE, new WasmEncoding[] {});
        signature.put(F32_LT, new WasmEncoding[] {});
        signature.put(F32_GT, new WasmEncoding[] {});
        signature.put(F32_LE, new WasmEncoding[] {});
        signature.put(F32_GE, new WasmEncoding[] {});
        signature.put(F64_EQ, new WasmEncoding[] {});
        signature.put(F64_NE, new WasmEncoding[] {});
        signature.put(F64_LT, new WasmEncoding[] {});
        signature.put(F64_GT, new WasmEncoding[] {});
        signature.put(F64_LE, new WasmEncoding[] {});
        signature.put(F64_GE, new WasmEncoding[] {});
        signature.put(I32_CLZ, new WasmEncoding[] {});
        signature.put(I32_CTZ, new WasmEncoding[] {});
        signature.put(I32_POPCNT, new WasmEncoding[] {});
        signature.put(I32_ADD, new WasmEncoding[] {});
        signature.put(I32_SUB, new WasmEncoding[] {});
        signature.put(I32_MUL, new WasmEncoding[] {});
        signature.put(I32_DIV_S, new WasmEncoding[] {});
        signature.put(I32_DIV_U, new WasmEncoding[] {});
        signature.put(I32_REM_S, new WasmEncoding[] {});
        signature.put(I32_REM_U, new WasmEncoding[] {});
        signature.put(I32_AND, new WasmEncoding[] {});
        signature.put(I32_OR, new WasmEncoding[] {});
        signature.put(I32_XOR, new WasmEncoding[] {});
        signature.put(I32_SHL, new WasmEncoding[] {});
        signature.put(I32_SHR_S, new WasmEncoding[] {});
        signature.put(I32_SHR_U, new WasmEncoding[] {});
        signature.put(I32_ROTL, new WasmEncoding[] {});
        signature.put(I32_ROTR, new WasmEncoding[] {});
        signature.put(I64_CLZ, new WasmEncoding[] {});
        signature.put(I64_CTZ, new WasmEncoding[] {});
        signature.put(I64_POPCNT, new WasmEncoding[] {});
        signature.put(I64_ADD, new WasmEncoding[] {});
        signature.put(I64_SUB, new WasmEncoding[] {});
        signature.put(I64_MUL, new WasmEncoding[] {});
        signature.put(I64_DIV_S, new WasmEncoding[] {});
        signature.put(I64_DIV_U, new WasmEncoding[] {});
        signature.put(I64_REM_S, new WasmEncoding[] {});
        signature.put(I64_REM_U, new WasmEncoding[] {});
        signature.put(I64_AND, new WasmEncoding[] {});
        signature.put(I64_OR, new WasmEncoding[] {});
        signature.put(I64_XOR, new WasmEncoding[] {});
        signature.put(I64_SHL, new WasmEncoding[] {});
        signature.put(I64_SHR_S, new WasmEncoding[] {});
        signature.put(I64_SHR_U, new WasmEncoding[] {});
        signature.put(I64_ROTL, new WasmEncoding[] {});
        signature.put(I64_ROTR, new WasmEncoding[] {});
        signature.put(F32_ABS, new WasmEncoding[] {});
        signature.put(F32_NEG, new WasmEncoding[] {});
        signature.put(F32_CEIL, new WasmEncoding[] {});
        signature.put(F32_FLOOR, new WasmEncoding[] {});
        signature.put(F32_TRUNC, new WasmEncoding[] {});
        signature.put(F32_NEAREST, new WasmEncoding[] {});
        signature.put(F32_SQRT, new WasmEncoding[] {});
        signature.put(F32_ADD, new WasmEncoding[] {});
        signature.put(F32_SUB, new WasmEncoding[] {});
        signature.put(F32_MUL, new WasmEncoding[] {});
        signature.put(F32_DIV, new WasmEncoding[] {});
        signature.put(F32_MIN, new WasmEncoding[] {});
        signature.put(F32_MAX, new WasmEncoding[] {});
        signature.put(F32_COPYSIGN, new WasmEncoding[] {});
        signature.put(F64_ABS, new WasmEncoding[] {});
        signature.put(F64_NEG, new WasmEncoding[] {});
        signature.put(F64_CEIL, new WasmEncoding[] {});
        signature.put(F64_FLOOR, new WasmEncoding[] {});
        signature.put(F64_TRUNC, new WasmEncoding[] {});
        signature.put(F64_NEAREST, new WasmEncoding[] {});
        signature.put(F64_SQRT, new WasmEncoding[] {});
        signature.put(F64_ADD, new WasmEncoding[] {});
        signature.put(F64_SUB, new WasmEncoding[] {});
        signature.put(F64_MUL, new WasmEncoding[] {});
        signature.put(F64_DIV, new WasmEncoding[] {});
        signature.put(F64_MIN, new WasmEncoding[] {});
        signature.put(F64_MAX, new WasmEncoding[] {});
        signature.put(F64_COPYSIGN, new WasmEncoding[] {});
        signature.put(I32_WRAP_I64, new WasmEncoding[] {});
        signature.put(I32_TRUNC_F32_S, new WasmEncoding[] {});
        signature.put(I32_TRUNC_F32_U, new WasmEncoding[] {});
        signature.put(I32_TRUNC_F64_S, new WasmEncoding[] {});
        signature.put(I32_TRUNC_F64_U, new WasmEncoding[] {});
        signature.put(I64_EXTEND_I32_S, new WasmEncoding[] {});
        signature.put(I64_EXTEND_I32_U, new WasmEncoding[] {});
        signature.put(I64_TRUNC_F32_S, new WasmEncoding[] {});
        signature.put(I64_TRUNC_F32_U, new WasmEncoding[] {});
        signature.put(I64_TRUNC_F64_S, new WasmEncoding[] {});
        signature.put(I64_TRUNC_F64_U, new WasmEncoding[] {});
        signature.put(F32_CONVERT_I32_S, new WasmEncoding[] {});
        signature.put(F32_CONVERT_I32_U, new WasmEncoding[] {});
        signature.put(F32_CONVERT_I64_S, new WasmEncoding[] {});
        signature.put(F32_CONVERT_I64_U, new WasmEncoding[] {});
        signature.put(F32_DEMOTE_F64, new WasmEncoding[] {});
        signature.put(F64_CONVERT_I32_S, new WasmEncoding[] {});
        signature.put(F64_CONVERT_I32_U, new WasmEncoding[] {});
        signature.put(F64_CONVERT_I64_S, new WasmEncoding[] {});
        signature.put(F64_CONVERT_I64_U, new WasmEncoding[] {});
        signature.put(F64_PROMOTE_F32, new WasmEncoding[] {});
        signature.put(I32_REINTERPRET_F32, new WasmEncoding[] {});
        signature.put(I64_REINTERPRET_F64, new WasmEncoding[] {});
        signature.put(F32_REINTERPRET_I32, new WasmEncoding[] {});
        signature.put(F64_REINTERPRET_I64, new WasmEncoding[] {});
        signature.put(I32_EXTEND_8_S, new WasmEncoding[] {});
        signature.put(I32_EXTEND_16_S, new WasmEncoding[] {});
        signature.put(I64_EXTEND_8_S, new WasmEncoding[] {});
        signature.put(I64_EXTEND_16_S, new WasmEncoding[] {});
        signature.put(I64_EXTEND_32_S, new WasmEncoding[] {});
        signature.put(REF_NULL, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(REF_IS_NULL, new WasmEncoding[] {});
        signature.put(REF_FUNC, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(I32_TRUNC_SAT_F32_S, new WasmEncoding[] {});
        signature.put(I32_TRUNC_SAT_F32_U, new WasmEncoding[] {});
        signature.put(I32_TRUNC_SAT_F64_S, new WasmEncoding[] {});
        signature.put(I32_TRUNC_SAT_F64_U, new WasmEncoding[] {});
        signature.put(I64_TRUNC_SAT_F32_S, new WasmEncoding[] {});
        signature.put(I64_TRUNC_SAT_F32_U, new WasmEncoding[] {});
        signature.put(I64_TRUNC_SAT_F64_S, new WasmEncoding[] {});
        signature.put(I64_TRUNC_SAT_F64_U, new WasmEncoding[] {});
        signature.put(MEMORY_INIT, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(DATA_DROP, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(MEMORY_COPY, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(MEMORY_FILL, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(TABLE_INIT, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(ELEM_DROP, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(TABLE_COPY, new WasmEncoding[] {WasmEncoding.VARUINT, WasmEncoding.VARUINT});
        signature.put(TABLE_GROW, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(TABLE_SIZE, new WasmEncoding[] {WasmEncoding.VARUINT});
        signature.put(TABLE_FILL, new WasmEncoding[] {WasmEncoding.VARUINT});
    }
}
