/**
 * Copyright 2022 Dynatrace LLC
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0
 *
 * <p>Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.dynatrace.file.util;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Properties;
import java.util.logging.Logger;

public class DynatraceFileBasedConfigurationProvider {
  // Lazy loading singleton instance.
  private static class ProviderHolder {
    private static final DynatraceFileBasedConfigurationProvider INSTANCE =
        new DynatraceFileBasedConfigurationProvider(PROPERTIES_FILENAME);
  }

  // avoid printing logs on the initial read
  private boolean alreadyInitialized = false;

  private DynatraceFileBasedConfigurationProvider(String fileName) {
    setUp(fileName, null);
  }

  private static final Logger logger =
      Logger.getLogger(DynatraceFileBasedConfigurationProvider.class.getName());

  private static final String PROPERTIES_FILENAME =
      "/var/lib/dynatrace/enrichment/endpoint/endpoint.properties";

  private FilePoller filePoller;
  private DynatraceConfiguration config;

  public static DynatraceFileBasedConfigurationProvider getInstance() {
    return ProviderHolder.INSTANCE;
  }

  private void setUp(String fileName, Duration pollInterval) {
    alreadyInitialized = false;
    config = new DynatraceConfiguration();
    FilePoller poller = null;
    try {
      if (!Files.exists(Paths.get(fileName))) {
        logger.info("File based configuration does not exist, serving default config.");
      } else {
        poller = FilePollerFactory.getDefault(fileName, pollInterval);
      }
    } catch (InvalidPathException e) {
      // This happens on Windows, when the *nix filepath is not valid.
      logger.info(
          () -> String.format("%s is not a valid file path (%s).", fileName, e.getMessage()));
    } catch (IOException | IllegalArgumentException e) {
      logger.warning(
          () -> String.format("File polling could not be initialized: %s", e.getMessage()));
    }
    filePoller = poller;
    // try to read from file
    updateConfigFromFile(fileName);
  }

  /**
   * This method should never be called by user code. It is only available for testing. When passing
   * null for the {@code fileName}, this method just shuts down the file polling mechanism.
   *
   * <p>VisibleForTesting
   *
   * @param fileName The filename of the file to watch.
   * @param pollInterval Polling interval for interval-based file pollers.
   */
  public void forceOverwriteConfig(String fileName, Duration pollInterval) {
    closePoller();
    if (fileName != null) {
      logger.warning("Overwriting config. This should ONLY happen in testing.");
      setUp(fileName, pollInterval);
    }
  }

  private void closePoller() {
    if (filePoller != null) {
      logger.warning("Shutting down file polling mechanism. This should ONLY happen in testing.");
      try {
        filePoller.close();
      } catch (IOException e) {
        logger.warning("Failed to shut down polling mechanism: " + e);
      }
      filePoller = null;
    }
  }

  private void updateConfigFromFile(String fileName) {
    if (filePoller == null) {
      // nothing to do, as no watch service is set up.
      logger.finest("No file watch set up, serving default values.");
      return;
    }
    // read the properties from the file
    try (FileInputStream inputStream = new FileInputStream(fileName)) {
      Properties props = new Properties();
      props.load(inputStream);

      final String newEndpoint = tryGetMetricsEndpoint(props);
      if (newEndpoint != null) {
        config.setMetricIngestEndpoint(newEndpoint);
      }

      final String newToken = tryGetToken(props);
      if (newToken != null) {
        config.setMetricIngestToken(newToken);
      }

      alreadyInitialized = true;
    } catch (IOException e) {
      logger.info("Failed reading properties from file.");
    }
  }

  /**
   * Tries to get the token from the {@link Properties properties} object.
   *
   * @return the new token, if it is available and different to the previous one, or null otherwise.
   */
  private String tryGetToken(Properties props) {
    final String newToken = props.getProperty("DT_METRICS_INGEST_API_TOKEN");
    if (newToken == null) {
      logger.warning("Could not read property with key 'DT_METRICS_INGEST_API_TOKEN'.");
      return null;
    }
    if (!newToken.equals(config.getMetricIngestToken())) {
      if (alreadyInitialized) {
        logger.info("API Token refreshed.");
      }
      return newToken;
    }
    return null;
  }

  /**
   * Tries to get the Endpoint from the {@link Properties properties} object.
   *
   * @return The new endpoint if it is available and different to the previous one, and null
   *     otherwise.
   */
  private String tryGetMetricsEndpoint(Properties props) {
    final String newEndpoint = props.getProperty("DT_METRICS_INGEST_URL");
    if (newEndpoint == null) {
      logger.fine("Could not read property with key 'DT_METRICS_INGEST_URL'.");
      return null;
    }
    if (!newEndpoint.equals(config.getMetricIngestEndpoint())) {
      if (alreadyInitialized) {
        logger.info(() -> String.format("Read new endpoint: %s", newEndpoint));
      }
      return newEndpoint;
    }
    return null;
  }

  private void updateConfigIfChanged() {
    if (filePoller != null && filePoller.fileContentsUpdated()) {
      updateConfigFromFile(filePoller.getWatchedFilePath());
    }
  }

  public String getMetricIngestEndpoint() {
    updateConfigIfChanged();
    return config.getMetricIngestEndpoint();
  }

  public String getMetricIngestToken() {
    updateConfigIfChanged();
    return config.getMetricIngestToken();
  }
}
