package com.hyphenate.chat;

import android.app.Activity;
import android.content.Intent;
import android.graphics.Bitmap;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;

import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.helpdesk.Error;
import com.hyphenate.helpdesk.callback.Callback;
import com.hyphenate.util.EMLog;
import com.superrtc.mediamanager.EMediaDefines;
import com.superrtc.mediamanager.EMediaEntities;
import com.superrtc.mediamanager.EMediaManager;
import com.superrtc.mediamanager.EMediaPublishConfiguration;
import com.superrtc.mediamanager.EMediaSession;
import com.superrtc.mediamanager.EMediaStream;
import com.superrtc.mediamanager.ScreenCaptureManager;
import com.superrtc.mediamanager.XClientBridger;
import com.superrtc.sdk.VideoViewRenderer;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * 音视频管理类
 */
public final class CallManager {

	static final String TAG = "callcore";
	private EMediaSession currentSession;
	private EMediaManager mediaManager;
	private EMediaSession.EMediaSessionDelegate sessionDelegate;
	private volatile int publishedCount;
	private String oldTicket;
	private JSONObject currentExtendJson;
	private static CallManager sInstance;
	final private Map<String, EMediaStream> currentStreams = Collections.synchronizedMap(new HashMap<String, EMediaStream>());
	final private Map<String, EMediaEntities.EMediaMember> mediaMemberMap = Collections.synchronizedMap(new HashMap<String, EMediaEntities.EMediaMember>());
	final private List<CallManagerDelegate> delegates = Collections.synchronizedList(new ArrayList<CallManagerDelegate>());
	final private Map<String, ReceivedStream> unPublishedStreamMaps = Collections.synchronizedMap(new HashMap<String, ReceivedStream>());
	private String callNickName;
	private JoinState currentJoinState = JoinState.UNJOIN;
	private ConcurrentLinkedQueue<TicketEntity> ticketQueue = new ConcurrentLinkedQueue<>();
	private CallSurfaceView localSurfaceView;
	private String currentMemberName;
	private String shareUid;
	private CallOption callOption = new CallOption();

	synchronized static CallManager getInstance(){
		if (sInstance == null){
			sInstance = new CallManager();
		}
		return sInstance;
	}

	/**
	 * 设置音视频选项
	 * @param callOption
	 */
	public void setCallOption (CallOption callOption) {
		if (callOption != null) {
			this.callOption = callOption;
		}
	}

	private static final String IncomingCallAction = ".action.incomingcall";

	/**
	 * 获取来电广播Action
	 * @return 广播字符串
	 */
	public String getIncomingCallBroadcastAction(){
		return ChatClient.getInstance().getContext().getPackageName() + IncomingCallAction;
	}

	private synchronized void addMember(EMediaEntities.EMediaMember eMediaMember){
		if (!mediaMemberMap.containsKey(eMediaMember.memberName)){
			mediaMemberMap.put(eMediaMember.memberName, eMediaMember);
		}
	}

	private synchronized void removeMember(EMediaEntities.EMediaMember eMediaMember){
		if (mediaMemberMap.containsKey(eMediaMember.memberName)){
			mediaMemberMap.remove(eMediaMember.memberName);
		}
	}

	void putTicket(TicketEntity ticket){
		ticketQueue.add(ticket);
		notifyJoin();
	}

	private void notifyJoin(){
		info("notifyJoin: currentJoinState = " + currentJoinState.ordinal());
		if (currentJoinState == JoinState.UNJOIN){
			info("ticketQueue: size = " + ticketQueue.size());
			if (!ticketQueue.isEmpty()){
				TicketEntity ticket = ticketQueue.poll();
				dealWithTicket(ticket.ticket, ticket.nickname, ticket.extendJson);
			}
		}
	}


	private enum JoinState{
		UNJOIN,
		JOINING,
		JOINED
	}

	public static enum HMediaNoticeCode {
		HMediaNoticeNone(0),
		HMediaNoticeStats(100),
		HMediaNoticeDisconn(120),
		HMediaNoticeReconn(121),
		HMediaNoticePoorQuality(122),
		HMediaNoticePublishSetup(123),
		HMediaNoticeSubscriptionSetup(124),
		HMediaNoticeTakeCameraPicture(125),
		HMediaNoticeCustomMsg(126),
		HMediaNoticeOpenCameraFail(201),
		HMediaNoticeOpenMicFail(202);
		public final int noticeCode;
		private HMediaNoticeCode(int noticeCode) {
			this.noticeCode = noticeCode;
		}
	}


	//============== public api =========================

	public EMediaEntities.EMediaMember getEMediaMember(String memberName){
		if (mediaMemberMap.containsKey(memberName)){
			return mediaMemberMap.get(memberName);
		}
		return null;
	}

	/**
	 * 接听通话
	 * @param videoOff 是否关闭摄像头
	 * @param mute  是否静音
	 * @param callback 回调
	 * @see #acceptCall(String, Callback)
	 */
	@Deprecated
	public void acceptCall(boolean videoOff, boolean mute, final Callback callback) {
		publish("", callback);
	}

	/**
	 * 接听通话
	 * @param callback 回调
	 * @see #acceptCall(String, Callback)
	 */
	@Deprecated
	public void acceptCall(final Callback callback){
		publish("", callback);
	}

	/**
	 * 接听通话
	 * @param nickName 通话显示的昵称
	 * @param callback 回调
	 */
	public void acceptCall(String nickName, final Callback callback){
		publish(nickName, callback);
	}


	/**
	 * 挂断通话
	 */
	public void endCall() {
		setLocalView(null);
		exit(null);
	}


	/**
	 * 实时通话时暂停语音数据传输
	 */
	public void pauseVoice()  {
		mediaManager.setMuteEnabled(true);
		callOption.isMute = true;
	}

	/**
	 * 实时通话时恢复语音数据传输
	 */
	public void resumeVoice()  {
		mediaManager.setMuteEnabled(false);
		callOption.isMute = false;
	}

	/**
	 * 实时通话时停止视频数据传输
	 */
	public void pauseVideo()  {
		mediaManager.setVideoEnabled(false);
		callOption.isVideoOff = true;
	}

	/**
	 * 实时通话时恢复视频数据传输
	 */
	public void resumeVideo()  {
		mediaManager.setVideoEnabled(true);
		callOption.isVideoOff = false;
	}



	@Deprecated
	public void setCameraFacing(int facing) throws HyphenateException {
//		EMClient.getInstance().callManager().setCameraFacing(facing);
	}

	/**
	 * 获取当前摄像头
	 * @return
	 */
	public int getCameraFacing(){
		return mediaManager.getCameraFacing();
	}

	/**
	 * 是否支持闪光灯
	 * @return
	 */
	public boolean isSupportFlashLight(){
		return mediaManager.isSupportFlashLight();
	}

	/**
	 * 切换通话中的摄像头
	 */
	public synchronized void switchCamera() {
		mediaManager.switchCamera();
	}


	/**
	 * 设置最大视频码率
	 * @param videoMaxKbps
	 */
	public void setVideoMaxKbps(int videoMaxKbps) {
		mediaManager.setVideoMaxKbps(videoMaxKbps);
	}

	/**
	 * 设置最小视频码率
	 * @param videoMinKbps
	 */
	public void setVideoMinKbps(int videoMinKbps) {
		mediaManager.setVideoMinKbps(videoMinKbps);
	}

	/**
	 * 设置最大音频码率
	 * @param audioMaxKbps
	 */
	public void setAudioMaxKbps(int audioMaxKbps) {
		mediaManager.setAudioMaxKbps(audioMaxKbps);
	}




	static class TicketEntity{
		String ticket;
		String nickname;
		JSONObject extendJson;
		TicketEntity(String ticket, String nick, JSONObject extend){
			this.ticket = ticket;
			this.nickname = nick;
			this.extendJson = extend;
		}



	}
	final Object mutex = new Object();

	void dealWithTicket(final String ticket, String callNick, JSONObject extendJson){
		callNickName = callNick;
		currentJoinState = JoinState.JOINING;
		info("current join state is joining");
		startJoin(ticket, extendJson, new Callback() {
			@Override
			public void onSuccess() {
				currentJoinState = JoinState.JOINED;
				info("current join state is joined");
				synchronized (mutex){
					mutex.notifyAll();
				}
				try {
					Intent intent = new Intent(getIncomingCallBroadcastAction());
					intent.putExtra("type", "video");
					intent.putExtra("from", currentSession.myName);
					intent.putExtra("channel", "kefu");
//					intent.putExtra("to", ChatClient.getInstance().currentUserName());
					ChatClient.getInstance().getContext().sendBroadcast(intent);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}

			@Override
			public void onError(int code, String error) {
				currentJoinState = JoinState.UNJOIN;
				info("current join state is unjoin");
				synchronized (mutex){
					mutex.notifyAll();
				}
			}

			@Override
			public void onProgress(int progress, String status) {

			}
		});
		synchronized (mutex){
			try {
				mutex.wait(3000);
			} catch (InterruptedException ignored) {
				EMLog.e(TAG, "dealWithTicker mutex wait InterruptedException");
				Thread.currentThread().interrupt();
			}
		}
		if (currentJoinState == JoinState.JOINING){
			currentJoinState = JoinState.UNJOIN;
		}
	}


	void setSpecificServerUrl(String mediaUrl, String mediaHost){
		if (mediaManager != null){
			mediaManager.setSpecificServerUrl(mediaUrl, mediaHost);
		}
	}

	private CallManager(){
		EMediaManager.initGlobal(ChatClient.getInstance().getContext());
		mediaManager = EMediaManager.getInstance();
		EMediaManager.setLoggerDelegate(new XClientBridger.Logcallbackfunc() {
			@Override
			public void onLog(int i, String s) {
				EMLog.d(TAG, "i->" + i + ", s ->" + s);
			}
		});

		sessionDelegate = new EMediaSession.EMediaSessionDelegate() {
			@Override
			public void joinMember(EMediaSession session, EMediaEntities.EMediaMember eMediaMember) {
				log("CallDelegate: onMemberJoin: instanceId="+session.instanceId+", memberName="+eMediaMember.memberName);
				info("C-> joined member ["+eMediaMember.memberName+ ",ext=" + eMediaMember.extension +"]");
				if (eMediaMember.memberName.equalsIgnoreCase(currentMemberName)){
					return;
				}
				addMember(eMediaMember);
			}

			@Override
			public void exitMember(EMediaSession session, EMediaEntities.EMediaMember eMediaMember) {
				log("CallDelegate: onMemberExit: instanceId="+session.instanceId+", memberName="+ eMediaMember.memberName);
				info("C-> exited member ["+eMediaMember.memberName + ",ext=" + eMediaMember.extension +"]");
				removeMember(eMediaMember);
			}

			@Override
			public void addStream(EMediaSession session, EMediaStream stream) {
				LOGI("CallDelegate: onAddStream: instanceId="+session.instanceId+", ["+stream.memberName+"]-["+stream.streamId+
						"]-["+stream.streamName+"], t="+stream.streamType+", v="+!stream.videoOff+", a="+!stream.audioOff + ",ext=" + stream.extension);
				if (stream.memberName.equals(currentMemberName)){
					return;
				}
				if (currentStreams.containsKey(stream.streamId)){
					return;
				}
				currentStreams.put(stream.streamId, stream);

				if (publishedCount > 0){
					notifyDelegateAddStream(session, stream);
				}else{
					synchronized (unPublishedStreamMaps){
						ReceivedStream receivedStream = new ReceivedStream();
						receivedStream.session = session;
						receivedStream.stream = stream;
						unPublishedStreamMaps.put(stream.streamId, receivedStream);
					}
				}

			}

			@Override
			public void removeStream(EMediaSession session, EMediaStream stream) {
				LOGI("CallDelegate: removeStreamWithId: instanceId="+session.instanceId+", memberName="+stream.memberName+", streamId="+stream.streamId + ",ext=" + stream.extension);
				if (currentStreams.containsKey(stream.streamId)){
					currentStreams.remove(stream.streamId);
				}
				if (publishedCount > 0){
					notifyDelegateRemoveStream(session, stream);
				}else{
					synchronized (unPublishedStreamMaps){
						if (unPublishedStreamMaps.containsKey(stream.streamId)){
							unPublishedStreamMaps.remove(stream.streamId);
						}
					}
				}
				if (currentStreams.isEmpty() && currentJoinState == JoinState.JOINED){
					notifyDelegatepassiveCloseReason(session, Error.GENERAL_ERROR, "no other stream");
				}

			}

			@Override
			public void updateStream(EMediaSession session, EMediaStream stream) {
				log("C-> upd ["+session.myName+"]-["+stream.streamId+"]-["+stream.streamName+"], t="+stream.streamType+", v="+!stream.videoOff+", a="+!stream.audioOff + ",ext=" + stream.extension);
				if (publishedCount > 0){
					notifyDelegateUpdateStream(session, stream);
				}else{
					synchronized (unPublishedStreamMaps){
						if (unPublishedStreamMaps.containsKey(stream.streamId)){
							ReceivedStream receivedStream = new ReceivedStream();
							receivedStream.session = session;
							receivedStream.stream = stream;
							unPublishedStreamMaps.put(stream.streamId, receivedStream);
						}
					}
				}
			}

			@Override
			public void passiveCloseReason(EMediaSession session, int reason, String desc) {
				LOGI("CallDelegate: onCallEnd: instanceId="+session.instanceId+", reason:"+reason+",desc:"+desc);
				if (session.instanceId != null && session.instanceId.equals(currentSession.instanceId)){
					currentJoinState = JoinState.UNJOIN;
				}
				notifyDelegatepassiveCloseReason(session, reason, desc);
			}

			@Override
			public void notice(EMediaSession session, EMediaDefines.EMediaNoticeCode code, String arg1, String arg2, Object arg3) {
				info("C-> notice [" + code + "]-[" + arg1 + "]-[" + arg2
						+ "]-[" + arg3 + "]");
				if (code == EMediaDefines.EMediaNoticeCode.EMEDIA_NOTICE_PUBLISH_SETUP){
					info("EMEDIA_NOTICE_PUBLISH_SETUP");

				}else if (code == EMediaDefines.EMediaNoticeCode.EMEDIA_NOTICE_SUBSCRIPTION_SETUP){
					info("EMEDIA_NOTICE_SUBSCRIPTION_SETUP");

				}else if (code == EMediaDefines.EMediaNoticeCode.EMEDIA_NOTICE_TAKE_CAMERA_PICTURE) {
					info("EMEDIA_NOTICE_TAKE_CAMERA_PICTURE url=" + arg3.toString());

					String toUserName = ChatClient.getInstance().chatManager().currentChatUsername();

					if (!TextUtils.isEmpty(toUserName)) {
						Message message = Message.createImageSendMessage(arg3.toString(), true, toUserName);
						JSONObject weichatJson = ChatClient.getInstance().chatManager().getLatestSendWeichat();
						if (weichatJson != null) {
							message.setAttribute(Message.KEY_WEICHAT, weichatJson);
						}
						ChatClient.getInstance().chatManager().sendMessageWithoutRecord(message);
					} else {
						EMLog.e(TAG, "lastest chat user is null");
					}
				}
				HMediaNoticeCode hNoticeCode = HMediaNoticeCode.HMediaNoticeNone;
				switch (code){
					case EMEDIA_NOTICE_NONE:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeNone;
						break;
					case EMEDIA_NOTICE_STATS:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeStats;
						break;
					case EMEDIA_NOTICE_RECONN:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeReconn;
						break;
					case EMEDIA_NOTICE_DISCONN:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeDisconn;
						break;
					case EMEDIA_NOTICE_CUSTOM_MSG:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeCustomMsg;
						break;
					case EMEDIA_NOTICE_POOR_QUALITY:
						hNoticeCode = HMediaNoticeCode.HMediaNoticePoorQuality;
						break;
					case EMEDIA_NOTICE_OPEN_MIC_FAIL:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeOpenMicFail;
						break;
					case EMEDIA_NOTICE_PUBLISH_SETUP:
						hNoticeCode = HMediaNoticeCode.HMediaNoticePublishSetup;
						break;
					case EMEDIA_NOTICE_OPEN_CAMERA_FAIL:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeOpenCameraFail;
						break;
					case EMEDIA_NOTICE_SUBSCRIPTION_SETUP:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeSubscriptionSetup;
						break;
					case EMEDIA_NOTICE_TAKE_CAMERA_PICTURE:
						hNoticeCode = HMediaNoticeCode.HMediaNoticeTakeCameraPicture;
						break;
				}
				notifyDelegateNotice(session, hNoticeCode, arg1, arg2, arg3);
			}

			@Override
			public void changeRole(EMediaSession eMediaSession) {

			}
		};
	}

	void startJoin(String ticket, JSONObject extendJson, final Callback callback){
		info("startJoin:ticket->" + ticket);
		ticket = ticket.replace("&quot;", "\"");
		oldTicket = ticket;
		currentExtendJson = extendJson;
		try {
			JSONObject jsonExt = new JSONObject();
			jsonExt.put("identity", "visitor");
			if (extendJson != null){
				jsonExt.put("extend", extendJson);
			}
			currentSession = mediaManager.newSessionWithTicket(ticket, jsonExt.toString(), sessionDelegate);
		} catch (JSONException e) {
			e.printStackTrace();
		}
		if (currentSession == null){
			currentJoinState = JoinState.UNJOIN;
			synchronized (mutex){
				mutex.notifyAll();
			}
			return;
		}
		currentJoinState = JoinState.JOINING;
		unPublishedStreamMaps.clear();
		mediaManager.join(currentSession, null, new EMediaEntities.EMediaIdBlockType(){

			@Override
			public void onDone(Object uid, EMediaEntities.EMediaError error) {
				if(currentSession == null) return; // already exit
				if (error != null) {
					currentJoinState = JoinState.UNJOIN;
					log("C-> join error " + error.code + "-["
							+ error.errorDescription + "]");
					info("C-> join error " + error.code + "-["
							+ error.errorDescription + "]");
					if (callback != null){
						callback.onError(-1, error.errorDescription);
					}
					notifyJoin();
				} else {
					currentJoinState = JoinState.JOINED;
					info("me joined Success");
					log("C-> joined me");
					if (callback != null){
						callback.onSuccess();
					}
				}
			}
		});

	}


	/**
	 * 增加消息监听 --- 通话页面初始化时调用
	 * @param listener 监听
	 */
	public void addDelegate(CallManagerDelegate listener) {
		if (!delegates.contains(listener)) {
			delegates.add(listener);
		}
	}

	/**
	 * 移除消息监听 --- 通话页面销毁的时调用
	 * @param listener
	 */
	public void removeDelegate(CallManagerDelegate listener) {
		if (delegates.contains(listener)) {
			delegates.remove(listener);
		}
	}

	private void notifyDelegateAddStream(EMediaSession session, EMediaStream stream) {
		synchronized (delegates) {
			for (CallManagerDelegate item : delegates) {
				if (item != null) {
					item.onAddStream(convertEMediaStreamToMediaStream(stream));
				}
			}
		}
	}

	private void notifyDelegateRemoveStream(EMediaSession session, EMediaStream stream) {
		synchronized (delegates) {
			for (CallManagerDelegate item : delegates) {
				if (item != null) {
					item.onRemoveStream(convertEMediaStreamToMediaStream(stream));
				}
			}
		}
	}

	private void notifyDelegateUpdateStream(EMediaSession session, EMediaStream stream) {
		synchronized (delegates) {
			for (CallManagerDelegate item : delegates) {
				if (item != null) {
					item.onUpdateStream(convertEMediaStreamToMediaStream(stream));
				}
			}
		}
	}

	private void notifyDelegatepassiveCloseReason(EMediaSession session, int reason, String desc) {
		synchronized (delegates) {
			for (CallManagerDelegate item : delegates) {
				if (item != null) {
					item.onCallEnd(reason, desc);
				}
			}
		}
	}

	private void notifyDelegateNotice(EMediaSession session, HMediaNoticeCode code, String arg1, String arg2, Object arg3){
		synchronized (delegates){
			for (CallManagerDelegate item : delegates) {
				if (item != null){
					item.onNotice(code, arg1, arg2, arg3);
				}
			}
		}
	}
	EMediaPublishConfiguration getSharePubConfig(String nickName, View view){
		EMediaPublishConfiguration pubconfig = new EMediaPublishConfiguration();
		pubconfig.setVideoOff(callOption.isVideoOff);
		pubconfig.setMute(callOption.isMute);
		pubconfig.setUseBackCamera(callOption.useBackCamera);
		pubconfig.setPubType(EMediaDefines.EMediaStreamType.EMSTREAM_TYPE_DESKTOP);
		if (view != null){
			pubconfig.setPubView(view);
		}
		if (callOption.videoHeight > 0){
			pubconfig.setVheight(callOption.videoHeight);
		}
		if (callOption.videoWidth > 0){
			pubconfig.setVwidth(callOption.videoWidth);
		}
		JSONObject jsonExt = new JSONObject();
		try {
			jsonExt.put("identity", "visitor");
			if (TextUtils.isEmpty(nickName)) {
				jsonExt.put("nickname", ChatClient.getInstance().currentUserName());
			} else {
				jsonExt.put("nickname", nickName);
			}

		} catch (JSONException e) {
			e.printStackTrace();
		}
		pubconfig.setExtension(jsonExt.toString());
		return pubconfig;
	}

	EMediaPublishConfiguration getPubConfig(String nickName){
		EMediaPublishConfiguration pubconfig = new EMediaPublishConfiguration();
		pubconfig.setVideoOff(callOption.isVideoOff);
		pubconfig.setMute(callOption.isMute);
		pubconfig.setUseBackCamera(callOption.useBackCamera);
		pubconfig.setPubType(EMediaDefines.EMediaStreamType.EMSTREAM_TYPE_NORMAL);
		if (callOption.videoHeight > 0){
			pubconfig.setVheight(callOption.videoHeight);
		}
		if (callOption.videoWidth > 0){
			pubconfig.setVwidth(callOption.videoWidth);
		}
		JSONObject jsonExt = new JSONObject();
		try {
			jsonExt.put("identity", "visitor");
			if (currentExtendJson != null){
				jsonExt.put("extend", currentExtendJson);
			}
			if (TextUtils.isEmpty(nickName)) {
				jsonExt.put("nickname", ChatClient.getInstance().currentUserName());
			} else {
				jsonExt.put("nickname", nickName);
			}

		} catch (JSONException e) {
			e.printStackTrace();
		}
		pubconfig.setExtension(jsonExt.toString());
		return pubconfig;
	}



	void publish(final String nickName, final Callback callback) {

		EMediaPublishConfiguration pubconfig = getPubConfig(nickName);


		mediaManager.publish(currentSession, pubconfig, new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object uid, EMediaEntities.EMediaError error) {
				if (error == null) {
					info("C-> pub ok");
					currentJoinState = JoinState.JOINED;
					ticketQueue.clear();
					publishedCount++;
					if (callback != null) {
						callback.onSuccess();
					}
					notifyUnPublishedStreams();
				} else {
					info("C-> pub error [" + error.code + "]-[" + error.errorDescription + "]");
					if (callback != null) {
						callback.onError(error.code.errorcode, error.errorDescription);
					}
				}
			}
		});
	}

	private void notifyUnPublishedStreams() {
		Log.e(TAG, "notifyUnPublishedStreams input:");
		synchronized (unPublishedStreamMaps) {
			for (ReceivedStream receivedStream : unPublishedStreamMaps.values()) {
				notifyDelegateAddStream(receivedStream.session, receivedStream.stream);
				Log.e(TAG, "nootifyAddStream:" + receivedStream.stream.streamId);
			}
			unPublishedStreamMaps.clear();
		}
	}


	void unPublish(String pubId, final Callback callback) {
		info("C-> start unpub");
		mediaManager.unpublish(currentSession, pubId, new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object uid, EMediaEntities.EMediaError error) {
				if (error != null) {
					info("C-> unpub error [" + error.code + "]-[" + error.errorDescription + "]");
					if (callback != null) {
						callback.onError(error.code.errorcode, error.errorDescription);
					}
				} else {
					publishedCount--;
					info("C-> unpub ok");
					if (callback != null) {
						callback.onSuccess();
					}
				}
			}
		});

	}

	/**
	 * 取消流媒体和view的关联
	 * @param streamId 流媒体Id
	 * @param callback 回调
	 */
	public void unSubscribe(final String streamId, final Callback callback){
		if (currentSession == null){
			throw new IllegalStateException("currentSession is null, please join first");
		}
		mediaManager.unsubscribe(currentSession, streamId, new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object uid, EMediaEntities.EMediaError error) {
				if (error != null){
					log("C-> unsub error ["+ error.code+"]-["+error.errorDescription+"]");
					if (callback != null){
						callback.onError(error.code.errorcode, error.errorDescription);
					}
				}else{
					log("C-> unsub ["+streamId+"]");
					if (callback != null){
						callback.onSuccess();
					}
				}
			}
		});

	}


	/**
	 * 取消流媒体和view的关联
	 * @param stream 流媒体
	 * @param callback 回调
	 */
	public void unSubscribe(final MediaStream stream, final Callback callback){
		if (stream == null){
			throw new IllegalArgumentException("stream is null");
		}
		unSubscribe(stream.streamId, callback);
	}

	/**
	 * 流媒体和View 建立关联
	 * @param stream  流媒体
	 * @param videoView 视频view
	 * @param callback 回调
	 */
	public void subscribe(final MediaStream stream, final CallSurfaceView videoView, final Callback callback){
		if (currentSession == null){
			throw new IllegalStateException("currentSession is null, please join first");
		}
		mediaManager.subscribe(currentSession, stream.streamId, (videoView == null) ? null: videoView.getRenderer(), new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object uid, EMediaEntities.EMediaError error) {
				if (error != null){
					log("C-> fail to subsr ["+stream.memberName+"]-["+stream.streamId+"]-["+stream.streamName+"], err=["+error.code+"]-["+error.errorDescription+"]");
					if (callback != null){
						callback.onError(error.code.errorcode, error.errorDescription);
					}
					return;
				}
				info("C-> subsr ["+stream.memberName+"]-["+stream.streamId+"]-["+stream.streamName+"]");
				if (callback != null){
					callback.onSuccess();
				}
			}
		});

	}

	public void updateSubscribe(final String streamId,  final CallSurfaceView videoView, final Callback callback){
		if (currentSession == null){
			throw new IllegalStateException("currentSession is null, please join first");
		}
		mediaManager.updateSubscribe(currentSession, streamId, (videoView == null) ? null: videoView.getRenderer(), new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object uid, EMediaEntities.EMediaError error) {
				if (error != null){
					log("C-> fail to updatesub ["+streamId+"], err=["+error.code+"]-["+error.errorDescription+"]");
					if (callback != null){
						callback.onError(error.code.errorcode, error.errorDescription);
					}
					return;
				}
				if (callback != null){
					callback.onSuccess();
				}
			}
		});
	}

	public void sendCustomWithRemoteStreamId(final String streamId, final String message, final Callback callback) {
		if (currentSession == null) {
			if (callback != null) {
				callback.onError(-1, "session is null");
			}
			return;
		}
		mediaManager.sendCtrlMsgByStreamId(currentSession, streamId, EMediaDefines.EMediaNoticeCode.EMEDIA_NOTICE_CUSTOM_MSG.noticeCode, null, message, new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object o, EMediaEntities.EMediaError eMediaError) {
				if (callback != null) {
					if (eMediaError != null) {
						callback.onError(eMediaError.code.errorcode, eMediaError.errorDescription);
					} else {
						callback.onSuccess();
					}
				}
			}
		});
	}

	public void sendCustomWithRemoteMemberId(final String memberId, final String message, final Callback callback) {
		if (currentSession == null) {
			if (callback != null) {
				callback.onError(-1, "session is null");
			}
			return;
		}
		mediaManager.sendCtrlMsgByMemberId(currentSession, memberId, EMediaDefines.EMediaNoticeCode.EMEDIA_NOTICE_CUSTOM_MSG.noticeCode, null, message, new EMediaEntities.EMediaIdBlockType() {
			@Override
			public void onDone(Object o, EMediaEntities.EMediaError eMediaError) {
				if (callback != null) {
					if (eMediaError != null) {
						callback.onError(eMediaError.code.errorcode, eMediaError.errorDescription);
					} else {
						callback.onSuccess();
					}
				}
			}
		});
	}

	public void publishWindow(final Activity activity, final Callback callback){
		if (currentSession == null) {
			if (callback != null) {
				callback.onError(-1, "session is null");
			}
			return;
		}
		final WeakReference<Activity> weakActivity = new WeakReference<Activity>(activity);
		if (weakActivity.get() == null) {
			LOGI("activity is finished");
			return;
		}
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
			mediaManager.publish(currentSession, getSharePubConfig(null, null), new EMediaEntities.EMediaIdBlockType(){

				@Override
				public void onDone(Object uid, EMediaEntities.EMediaError error) {
					if (error != null) {
						LOGI("Publish failed code=" + error.code + ", desc=" + error.errorDescription);
						if (callback != null) {
							callback.onError(error.code.errorcode, error.errorDescription);
						}
					} else {
						shareUid = String.valueOf(uid);
						publishedCount++;
						LOGI("Publish success Stream id - " + uid);
						if (weakActivity.get() == null) {
							LOGI("activity is finished");
							if (callback != null) {
								callback.onError(-1, "activity is finished");
							}
							return;
						}
						if (callback != null) {
							callback.onSuccess();
						}
						if (ScreenCaptureManager.getInstance().state == ScreenCaptureManager.State.IDLE) {
							ScreenCaptureManager.getInstance().init((Activity) activity);
							ScreenCaptureManager.getInstance().setScreenCaptureCallback(new ScreenCaptureManager.ScreenCaptureCallback() {
								@Override
								public void onBitmap(Bitmap bitmap) {
									mediaManager.inputExternalVideoData(bitmap);
								}
							});
						}
					}
				}
			});
		} else {
			mediaManager.publish(currentSession, getSharePubConfig(null, weakActivity.get().getWindow().getDecorView()), new EMediaEntities.EMediaIdBlockType() {
				@Override
				public void onDone(Object uid, EMediaEntities.EMediaError error) {
					if (error != null) {
						LOGI("Publish failed code = " + error.code + ", desc=" + error.errorDescription);
						if (callback != null) {
							callback.onError(error.code.errorcode, error.errorDescription);
						}
					} else {
						LOGI("Publish success Stream id - " + uid);
						if (callback != null) {
							callback.onSuccess();
						}
					}
				}
			});
		}
	}

	public void onActivityResult(int requestCode, int resultCode, Intent data){
		if (resultCode == Activity.RESULT_OK){
			if (requestCode == ScreenCaptureManager.RECORD_REQUEST_CODE) {
				if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
					ScreenCaptureManager.getInstance().start(resultCode, data);
				}
			}
		}
	}

	public void unPublishWindow(final Callback callback){
		if (shareUid == null){
			return;
		}
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
			ScreenCaptureManager.getInstance().stop();
		}
		unPublish(shareUid, callback);
	}


	/**
	 * 获取对方的昵称
	 * @return 昵称
	 */
	public String getCallNickName(){
		return callNickName;
	}

	private void exit(final Callback callback){
		callNickName = null;
		callOption.isVideoOff = false;
		callOption.isMute = false;
		currentJoinState = JoinState.UNJOIN;
		info("current join state is unjoin");
		if (currentSession != null) {
			publishedCount = 0;
			unPublishedStreamMaps.clear();
			currentStreams.clear();
			shareUid = null;
			if (ScreenCaptureManager.getInstance().state != ScreenCaptureManager.State.IDLE){
				ScreenCaptureManager.getInstance().stop();
			}

			log("C-> exiting...");
			mediaManager.exit(currentSession, new EMediaEntities.EMediaIdBlockType() {
				@Override
				public void onDone(Object uid, EMediaEntities.EMediaError error) {
					if (error != null) {
						log("C-> exit error " + error.code + "-["
								+ error.errorDescription + "]");
						if (callback != null){
							callback.onError(error.code.errorcode, error.errorDescription);
						}
					} else {
						if (callback != null){
							callback.onSuccess();
						}
						log("C-> exited me");
					}
				}
			});
		} else {
			log("currentSession is null");
		}
	}

	/**
	 * 设置本地显示
	 * @param localSurface 本地显示
	 */
	public void setLocalView(CallSurfaceView localSurface){
		if (localSurface != null) {
			mediaManager.setVideoViews(null, localSurface.getRenderer(), null, true);
		} else {
			mediaManager.setVideoViews(null, null, null, true);
		}
	}


	/**
	 * 设置远端显示
	 * @param streamId 流媒体id
	 * @param callSurfaceView 远端显示
	 */
	public void setRemoteView(String streamId, CallSurfaceView callSurfaceView){
		if (streamId == null){
			return;
		}
		if (callSurfaceView != null){
			setOtherVideoView(streamId, callSurfaceView.getRenderer());
		}else{
			setOtherVideoView(streamId, null);
			unSubscribe(streamId, null);
		}
	}


	private void setOtherVideoView(String streamId, VideoViewRenderer otherVideoViewRenderer){
		mediaManager.setVideoViews(streamId, null, otherVideoViewRenderer, false);
	}


	private void LOGI(String text){
		EMLog.d(TAG, "" + text);
	}


	private void info(String msg) {
		log(msg);
	}

	private void log(String msg) {
		LOGI(msg);
	}


	public interface CallManagerDelegate {

		void onAddStream(MediaStream stream);

		void onRemoveStream(MediaStream stream);

		void onUpdateStream(MediaStream stream);

		void onCallEnd(int reason, String desc);

		void onNotice(HMediaNoticeCode code, String arg1, String arg2, Object arg3);
	}

	private class ReceivedStream{
		EMediaSession session;
		EMediaStream stream;
	}

	private MediaStream convertEMediaStreamToMediaStream(EMediaStream stream){
		if (stream == null){
			return null;
		}
		return new MediaStream(stream.streamId, stream.memberName, stream.streamType, stream.streamName, stream.videoOff, stream.audioOff, stream.extension, stream.csrc);
	}

	/**
	 * 焦点函数
	 * @param x  x坐标点
	 * @param y  y坐标点
	 * @param width  View的宽
	 * @param height View的高
	 */
	public void manualFocus(float x, float y, int width, int height){
		if (mediaManager != null){
			mediaManager.manualFocus(x, y, width, height);
		}
	}

	/**
	 * 缩放函数
	 * @param isZoomIn  true 放大 ， false 缩小
	 * @param zoomScale 缩放比例
	 */
	public void manualZoom(boolean isZoomIn, int zoomScale){
		if (mediaManager != null){
			mediaManager.manualZoom(isZoomIn, zoomScale);
		}
	}


}
