package com.hyphenate.chat;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.content.IntentFilter;
import android.graphics.BitmapFactory;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.EMMessageListener;
import com.hyphenate.helpdesk.Error;
import com.hyphenate.helpdesk.callback.Callback;
import com.hyphenate.helpdesk.callback.ValueCallBack;
import com.hyphenate.helpdesk.httpclient.HttpClient;
import com.hyphenate.helpdesk.httpclient.HttpRequestBuilder;
import com.hyphenate.helpdesk.httpclient.HttpResponse;
import com.hyphenate.helpdesk.httpclient.HttpResponseHandler;
import com.hyphenate.helpdesk.httpclient.ProgressListener;
import com.hyphenate.helpdesk.model.MessageHelper;
import com.hyphenate.helpdesk.util.CopyDirectoryUtil;
import com.hyphenate.helpdesk.util.Log;
import com.hyphenate.helpdesk.util.ZipUtil;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.ImageUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 聊天功能管理类
 */
public class ChatManager {
    private static final String TAG = "ChatManager";

    private final List<MessageListener> messageListeners = Collections.synchronizedList(new ArrayList<MessageListener>());
    private final List<AgentInputListener> agentInputListeners = Collections.synchronizedList(new ArrayList<AgentInputListener>());
    private final List<VisitorWaitListener> visitorWaitCountListeners = Collections.synchronizedList(new ArrayList<VisitorWaitListener>());
    private ScheduledExecutorService agentInputThread;
    private ScheduledExecutorService visitorWaitThread;
    private static ChatManager instance = new ChatManager();
    private ExecutorService sendMsgThread = Executors.newCachedThreadPool();
    private ExecutorService downloadThreadPool = Executors.newCachedThreadPool();
    private Hashtable<String, Object> sendLocks;
    private static final int WAIT_TIME_OUT = 20; // send message and wait ack 20s timeout
    private static final int WAIT_TIME_ATTACHMENT_MIN = 20;
    private static final int WAIT_TIME_ATTACHEMENT_MAX = 60;
    private static final int WAIT_TIME_OUT_TYPE_TXT = 10; // send message txt and wait ack 10s timeout
    private final String SEND_EXT_MSG_ID_FOR_ACK = "msg_id_for_ack";
    private final String ACK_FOR_MSG_ID = "ack_for_msg_id";
    volatile static String currentChatUsername;
    private ExecutorService loginSingleExecutor = Executors.newSingleThreadExecutor();
    private JSONObject latestSendWeichat;

    String currentChatUsername() {
        return currentChatUsername;
    }


    private synchronized void notifySendLock(String messageId){
        if(sendLocks == null){
            return;
        }
        Object lock = sendLocks.remove(messageId);
        if(lock != null){
            synchronized(lock){
                lock.notifyAll();
            }
        }
    }


    /**
     * 获取最近一条发送的消息的weichat扩展
     * @return
     */
    JSONObject getLatestSendWeichat(){
        return latestSendWeichat;
    }


    private synchronized void addSendLock(String messageId, Object lock){
        if(sendLocks == null){
            sendLocks = new Hashtable<String, Object>();
        }
        sendLocks.put(messageId, lock);
    }



    private List<Message>  convertEMMessagesToMessages(List<EMMessage> emMessages){
        List<Message> tempMsgs = new ArrayList<Message>();
        for (final EMMessage msg : emMessages){
            final Message tempMsg = convertToMessage(msg);
            String taskId = tempMsg.getMarketingTaskId();
            MarketingHttpClient.asyncDelivered(taskId, tempMsg.from());
            tempMsgs.add(tempMsg);
        }
        return tempMsgs;
    }

    private List<Message> siftKefuMessages(List<Message> messages){
        List<Message> tempMsgs = new ArrayList<Message>();
        for (Message tempMsg: messages){
            String ackForMsgId = checkKefuMessageAck(tempMsg);
            if (ackForMsgId != null){
                notifySendLock(ackForMsgId);
            }else{
                Message message = checkIsExistMessage(tempMsg);
                if (message != null){
                    tempMsgs.add(message);
                }
            }
        }
        return tempMsgs;
    }

    synchronized Message checkIsExistMessage(Message message){
        String kefuExtMsgId = getKefuExtMsgId(message);
        if(kefuExtMsgId != null){
            boolean isExists = true;
            if (KefuDBManager.getInstance() == null) {
                return null;
            }
            isExists = KefuDBManager.getInstance().isMessageExistedByExtMsgId(kefuExtMsgId);
            EMLog.d(TAG, "im-msgid:" + message.getIMMsgId() + ", kefuExtMsgId:" + kefuExtMsgId + ",isExists:" + isExists);
            if(!isExists){
                message.setUnread(true);
                if(!MessageHelper.isNotificationMessage(message)){
                    message.setStatus(Message.Status.SUCCESS);
                    KefuConversationManager.getInstance().saveMessage(message);
                    return message;
                }
            }
        }else{
            message.setUnread(true);
            if(!MessageHelper.isNotificationMessage(message)){
                message.setStatus(Message.Status.SUCCESS);
                KefuConversationManager.getInstance().saveMessage(message);
                return message;
            } else if (MessageHelper.isTicketStatusChangedMessage(message)) {
                message.setStatus(Message.Status.SUCCESS);
                return message;
            }
        }

        return null;
    }





    private ChatManager(){
        EMClient.getInstance().chatManager()
                .addMessageListener(new EMMessageListener() {

                    @Override
                    public void onMessageReceived(List<EMMessage> messages) {
                        List<Message> tempMsgs = convertEMMessagesToMessages(messages);
                        publishNewMessage(siftKefuMessages(tempMsgs));
                    }

                    @Override
                    public void onMessageRead(
                            List<EMMessage> messages) {
                        CountDownUtils.getInstance().sendBroadcast();
                        synchronized (messageListeners) {
                            for (MessageListener listener : messageListeners) {
                                listener.onMessageStatusUpdate();
                            }
                        }
                    }

                    @Override
                    public void onMessageDelivered(
                            List<EMMessage> messages) {
                        CountDownUtils.getInstance().sendBroadcast();
                        synchronized (messageListeners) {
                            for (MessageListener listener : messageListeners) {
                                listener.onMessageStatusUpdate();
                            }
                        }

                    }

                    @Override
                    public void onMessageRecalled(List<EMMessage> list) {
                        CountDownUtils.getInstance().sendBroadcast();
                        synchronized (messageListeners) {
                            for (MessageListener listener : messageListeners) {
                                listener.onMessageStatusUpdate();
                            }
                        }
                    }

                    @Override
                    public void onMessageChanged(EMMessage message,
                                                 Object change) {
                        CountDownUtils.getInstance().sendBroadcast();
                        if(message.getBody() instanceof EMFileMessageBody){
                            Message kefuMessage = convertToMessage(message);
                            updateMessageState(kefuMessage);
                        }

                        synchronized (messageListeners) {
                            for (MessageListener listener : messageListeners) {
                                listener.onMessageStatusUpdate();
                            }
                        }
                    }

                    @Override
                    public void onCmdMessageReceived(List<EMMessage> messages) {
                        List<Message> tempMsgs = new ArrayList<Message>();
                        for (EMMessage msg : messages){
                            Message tempMsg = convertToMessage(msg);
                            if (KefuDBManager.getInstance() == null) {
                                continue;
                            }
                            if (KefuDBManager.getInstance().isExistCmdMessage(tempMsg.messageId())){
                                continue;
                            }
                            boolean isInserted = KefuDBManager.getInstance().insertCmdMessage(tempMsg);
                            if (!isInserted){
                                Log.e(TAG, "message insert failed:" + tempMsg.toString());
                                continue;
                            }
                            String ackForMsgId = checkKefuMessageAck(tempMsg);
                            if(ackForMsgId != null){
                                notifySendLock(ackForMsgId);
                            }else{
                                //是否为消息撤回的命令消息
                                if (isReCallCmdMessage(tempMsg)){
                                    notifyReCallMessage(tempMsg);
                                } else{
                                    EMCmdMessageBody cmdBody = (EMCmdMessageBody) tempMsg.body();
                                    String action = cmdBody.action();
                                    if (!TextUtils.isEmpty(action)){
                                        if (action.equals("ServiceSessionOpenedEvent")){
                                            sendDeviceInfo(tempMsg.from());
                                            if (!TextUtils.isEmpty(currentChatUsername)){
                                                showVisitorWaitCountAndAgentInputState(currentChatUsername);
                                            }
                                        }else if (action.equals("ServiceSessionClosedEvent")){
                                            // pass
                                            shutdownWaitAndInput();
                                            if (!TextUtils.isEmpty(currentChatUsername) && !currentChatUsername.contains(OfficialAccount.SEPARATOR)){
                                                getConversation(currentChatUsername).clearOfficialAccount();
                                            }
                                        }else if (action.equalsIgnoreCase("ServiceSessionCreatedEvent")){
                                            if (!TextUtils.isEmpty(currentChatUsername)){
                                                showVisitorWaitCountAndAgentInputState(currentChatUsername);
                                            }
                                        }else if (action.equalsIgnoreCase("ServiceSessionTransferedEvent")){
                                            if (!TextUtils.isEmpty(currentChatUsername)){
                                                showVisitorWaitCountAndAgentInputState(currentChatUsername);
                                            }
                                        }else if (action.equalsIgnoreCase("ServiceSessionAbortedEvent")){
                                            shutdownWaitAndInput();
                                            if (!TextUtils.isEmpty(currentChatUsername) && !currentChatUsername.contains(OfficialAccount.SEPARATOR)){
                                                getConversation(currentChatUsername).clearOfficialAccount();
                                            }
                                        }else if (action.equalsIgnoreCase("ServiceSessionTransferedToAgentQueueEvent")){
                                            if (!TextUtils.isEmpty(currentChatUsername)){
                                                showVisitorWaitCountAndAgentInputState(currentChatUsername);
                                            }
                                        }
                                    }
                                    tempMsgs.add(tempMsg);
                                }
                            }
                        }
                        publishCmdMessage(tempMsgs);
                    }
                });
    }

    private JSONObject getVideoInviteTicket(Message message){
        try {
//            return message.getJSONObjectAttribute(Message.KEY_MSGTYPE).getJSONObject("sendVisitorTicket").getString("ticket");
            return message.getJSONObjectAttribute(Message.KEY_MSGTYPE).getJSONObject("sendVisitorTicket");
        } catch (Exception ignored) {
        }
        return null;
    }

    boolean isReCallCmdMessage(Message message) {
        if (message.getType() == Message.Type.CMD) {
            EMCmdMessageBody body = ((EMCmdMessageBody) message.body());
            String action = body.action();
            if (!TextUtils.isEmpty(action) && action.equals("KEFU_MESSAGE_RECALL")) {
                return true;
            }
        }
        return false;
    }

    void notifyReCallMessage(Message message) {
        if (message.getType() == Message.Type.CMD) {
            try {
                JSONObject jsonObj = message.getJSONObjectAttribute(Message.KEY_WEICHAT);
                if (jsonObj != null && jsonObj.has("recall_msg_id")) {
                    String recallMsgId = jsonObj.getString("recall_msg_id");
                    String msgId = KefuConversationManager.getInstance().getMessageByExtMsgId(recallMsgId);
                    if (!TextUtils.isEmpty(msgId)) {
                        ChatClient.getInstance().chatManager().getConversation(message.from).removeMessage(msgId, false);
                        KefuDBManager.getInstance().recallMessage(msgId);
                    }
                    notifyMessageSent();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }


    public static synchronized ChatManager getInstance(){
        return instance;
    }

    void loginWithToken(final String username, String token, final EMCallBack callback) {
        if (TextUtils.isEmpty(token)) {
            ChatClient.getInstance().loginWithTokenFail();
            callback.onError(Error.GENERAL_ERROR, "token is null");
            return;
        }

        if(!TextUtils.isEmpty(username)){
            initDB(username);
        } else {
            ChatClient.getInstance().loginWithTokenFail();
            callback.onError(Error.GENERAL_ERROR, "username is null");
            return;
        }

        EMClient.getInstance().loginWithToken(username, token, new EMCallBack() {
            public void onSuccess() {
                loadDB();
                PreferenceUtil.getInstance().setUsername(username);
                PreferenceUtil.getInstance().setToken(EMClient.getInstance().getAccessToken());
                PreferenceUtil.getInstance().saveLoginWithToken(true);
                registerCountDown();
                CountDownUtils.getInstance().sendBroadcast();
                ChatConfig.getInstance().loadDnsConfigFromRemote(false);
                if(callback != null){
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int code, String error) {
                ChatClient.getInstance().loginWithTokenFail();
                if(callback != null){
                    callback.onError(code, error);
                }
            }

            @Override
            public void onProgress(int progress, String status) {
                if(callback != null){
                    callback.onProgress(progress, status);
                }
            }
        });
    }


    void login(final String username, final String password, final EMCallBack callback){
        String tenantId = ChatClient.getInstance().tenantId();
        if (TextUtils.isEmpty(tenantId)){
            if (callback != null){
                callback.onError(-1, "tenantid is null");
                EMLog.e(TAG, "tenantid is null, please set in option");
            }
            return;
        }
        if (!TextUtils.isDigitsOnly(tenantId)){
            if (callback != null){
                callback.onError(-1, "tenantid must be digits");
            }
            EMLog.e(TAG, "current tenantid is :" + tenantId + ", tenantid must is digits");
            return;
        }
        if(!TextUtils.isEmpty(username)){
            initDB(username);
        }
        EMClient.getInstance().login(username, password, new EMCallBack() {

            @Override
            public void onSuccess() {
                loadDB();
                if (KefuDBManager.getInstance() != null) {
                    KefuDBManager.getInstance().deleteCmdMessages(7);
                }
                PreferenceUtil.getInstance().setUsernameAndPassword(username, password);
                PreferenceUtil.getInstance().setToken(EMClient.getInstance().getAccessToken());
                PreferenceUtil.getInstance().saveLoginWithToken(false);
                if (TextUtils.isEmpty(PreferenceUtil.getInstance().getUniqueId())){
                    PreferenceUtil.getInstance().setUniqueId(getDeviceId());
                }
                registerCountDown();
                CountDownUtils.getInstance().sendBroadcast();
                ChatConfig.getInstance().loadDnsConfigFromRemote(false);
                if(callback != null){
                    callback.onSuccess();
                }
            }

            @Override
            public void onProgress(int progress, String status) {
                if(callback != null){
                    callback.onProgress(progress, status);
                }
            }

            @Override
            public void onError(int code, String error) {
                if(callback != null){
                    callback.onError(code, error);
                }
            }
        });
    }

    @SuppressLint("MissingPermission")
    private String getDeviceId() {
        String deviceId = null;
        try {
            TelephonyManager telephonyManager = (TelephonyManager) ChatClient.getInstance().getContext().getSystemService(Context.TELEPHONY_SERVICE);
	        deviceId = telephonyManager.getDeviceId();
        } catch (Exception ignored) {}
        if (TextUtils.isEmpty(deviceId)) {
            deviceId = UUID.randomUUID().toString();
        }
        return deviceId;
    }


    private int logout(boolean unbindToken){
        int error = EMClient.getInstance().logout(unbindToken);
        if(error != EMError.EM_NO_ERROR){
            return error;
        }
        KefuDBManager.closeDatabase();
        if (sendLocks != null){
            sendLocks.clear();
        }
        KefuConversationManager.getInstance().clear();
        unbindChat();
        ChatClient.getInstance().cleanCache();
        unregisterCountDown();
        return EMError.EM_NO_ERROR;
    }

    void logout(final boolean unbindToken, final EMCallBack callback) {
        new Thread(){
            @Override
            public void run(){
                int error = logout(unbindToken);

                if(error != EMError.EM_NO_ERROR){
                    if(callback != null){
                        callback.onError(error, "faild to unbind device token");
                    }
                }else{
                    unregisterCountDown();
                    kefuLogout();
                    if(callback != null){
                        callback.onSuccess();
                    }
                }
            }
        }.start();
    }


    private CountDownBroadCast broadCastReceiver;

    private void unregisterCountDown(){
        try{
            if (broadCastReceiver != null){
                EMClient.getInstance().getContext().unregisterReceiver(broadCastReceiver);
                broadCastReceiver = null;
            }
        }catch (Exception ignored){}
    }

    void registerCountDown(){
        if (broadCastReceiver == null){
            broadCastReceiver = new CountDownBroadCast();
        }
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(CountDownBroadCast.ACTION);
        EMClient.getInstance().getContext().registerReceiver(broadCastReceiver, intentFilter);
    }



    void kefuLogout(){
        KefuDBManager.closeDatabase();
        if (sendLocks != null){
            sendLocks.clear();
        }
        KefuConversationManager.getInstance().clear();
        unbindChat();
        ChatClient.getInstance().cleanCache();
        EMClient.getInstance().logout(false);
        unregisterCountDown();
    }

    void publishNewMessage(List<Message> messages){
        CountDownUtils.getInstance().sendBroadcast();
        if (messages != null && messages.size() > 0){
            synchronized (messageListeners){
                for (MessageListener messageListener : messageListeners){
                    messageListener.onMessage(messages);
                }
            }
        }
    }

    void publishCmdMessage(List<Message> allMessages){
        CountDownUtils.getInstance().sendBroadcast();
        List<Message> messages = filterNonofifyCmdMessage(allMessages);
        if (messages != null && messages.size() > 0){
            synchronized (messageListeners) {
                for (MessageListener listener : messageListeners) {
                    listener.onCmdMessage(messages);
                }
            }
        }
    }

    private boolean preLogMode;

    private List<Message> filterNonofifyCmdMessage(List<Message> allMessages) {
        List<Message> tempMessages = new ArrayList<>();
        for (Message item : allMessages) {
            EMCmdMessageBody body = (EMCmdMessageBody) item.body();
            String action = body.action();
            JSONObject inviteVideoJson = getVideoInviteTicket(item);
            if (inviteVideoJson != null){
                EMLog.d(TAG, "inviteVideoJson:" + inviteVideoJson.toString());
                String ticket = inviteVideoJson.optString("ticket");
                String callNickName = inviteVideoJson.optString("nickname");
                JSONObject extendJson = inviteVideoJson.optJSONObject("extend");
                try{
                    if (ChatClient.getInstance().callManager() != null){
                        ChatClient.getInstance().callManager().putTicket(new CallManager.TicketEntity(ticket, callNickName, extendJson));
                    }
                }catch (Exception e){
                    EMLog.e(TAG, "e->" + android.util.Log.getStackTraceString(e));
                }
                continue;
            }
            if (!TextUtils.isEmpty(action)) {
                if (action.equals("easemob_exit")){
                    int code = Error.USER_REMOVED;
                    PreferenceUtil.getInstance().removeAll();
                    ChatClient.getInstance().notifyOnDisconnected(code);
                    continue;
                }else if (action.equals("easemob_startlog")){
                    preLogMode = ChatClient.getInstance().isDebugMode();
                    ChatClient.getInstance().setDebugMode(true);
                    continue;
                }else if (action.equals("easemob_stoplog")){
                    ChatClient.getInstance().setDebugMode(preLogMode);
                    continue;
                }else if (action.equals("easemob_uploadlog")){
                    new Thread(new Runnable() {
                        @Override
                        public void run() {
                            try{
                                String tempLogDir = "/sdcard/easemoblog/";
                                String packageName = ChatClient.getInstance().getContext().getPackageName();
                                String appkey = EMClient.getInstance().getOptions().getAppKey();
                                final File tempZipFile = new File("/sdcard/easemoblog.zip");
                                CopyDirectoryUtil.copyDirectory("/sdcard/Android/data/" + packageName + "/" + appkey + "/core_log", tempLogDir);
                                CopyDirectoryUtil.copyDirectory("/data/data/" + packageName + "/databases/", tempLogDir);
                                ZipUtil.zipFolder(tempLogDir, tempZipFile.getPath());
                                ZipUtil.deleteDirectory(new File(tempLogDir));

                                HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                                HttpRequestBuilder requestBuilder = httpClient.post("https://kefu-sdk-log.easemob.com/upload?token=f2dfcdf86fc51be02a6fcb8fef9c703e");
                                requestBuilder.contentFromFile(tempZipFile.getPath());
                                requestBuilder.to(new HttpResponseHandler(){
                                    @Override
                                    public void onResponse(HttpResponse response, long contentLength) throws Exception {
                                        try {
                                            String result = response.getResponseBody();
                                            EMLog.d(TAG, "upload log zip :" + result);
                                            JSONObject jsonObject = new JSONObject(result);
                                            String urlMessage = jsonObject.getString("message");
                                            JSONObject jsonBody = new JSONObject();
                                            String username = PreferenceUtil.getInstance().getUsername();
                                            String appkey = EMClient.getInstance().getOptions().getAppKey();
                                            String logUsername = appkey.replace("#", "_") + "_" + username;
                                            EMLog.d(TAG, "logUsername:" + logUsername);
                                            jsonBody.put("username", logUsername);
                                            jsonBody.put("url", urlMessage);
                                            tempZipFile.delete();
                                            uploadLogToServer(jsonBody.toString());
                                        }catch (Exception ignored){
                                            ignored.printStackTrace();
                                        }

                                    }
                                });
                                requestBuilder.execute();
                            }catch (Exception ignored){
                                ignored.printStackTrace();
                            }
                        }
                    }).start();
                    continue;
                }
            }
            tempMessages.add(item);
        }
        return tempMessages;
    }

    private void uploadLogToServer(String jsonBody){
        try {
            HttpClient httpClient1 = new HttpClient(EMClient.getInstance().getContext());
            HttpRequestBuilder requestBuilder1 = httpClient1.post("https://kefu-sdk-log.easemob.com/applog/api/v1.0?token=f2dfcdf86fc51be02a6fcb8fef9c703e");
            requestBuilder1.content(jsonBody.getBytes(), "application/json");
            HttpResponse msgHttpResponse = requestBuilder1.execute();
            assert msgHttpResponse != null;
            int statusCode = msgHttpResponse.getStatusCode();
            if (statusCode/100 == 2){
                EMLog.d(TAG, "upload log success");
            }else{
                EMLog.d(TAG, "upload log failed");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    void sendMessageWithoutRecord(final Message msg) {
        sendMessage(msg, false, null);
    }

    /**
     * 发送消息
     * @param msg 消息
     */
    public void sendMessage(final Message msg){
        sendMessage(msg, true, null);
    }

    /**
     * 发送消息
     * @param message 消息
     * @param callback 回调
     */
    public void sendMessage(Message message, final Callback callback) {
        sendMessage(message, true, callback);
    }

    private void sendMessage(Message message, boolean isRecord, final Callback callback){
        if (TextUtils.isEmpty(currentChatUsername)){
            EMLog.e(TAG, "please invoke ChatClient.getInstance().chatManager().bindChat({imServiceNum}) method in your chat activity!");
        }

        if(message.messageId() == null){
            message.setMsgId(UUID.randomUUID().toString());
        }
        JSONObject jsonWeichat = null;
        try{
            jsonWeichat = message.getJSONObjectAttribute(Message.KEY_WEICHAT);
        }catch(Exception ignored){}
        if(jsonWeichat == null){
            jsonWeichat = new JSONObject();
        }
        latestSendWeichat = jsonWeichat;
        try {
            if (ChatClient.getInstance().hasSecondChannel) {
                jsonWeichat.put(SEND_EXT_MSG_ID_FOR_ACK, message.messageId());
            }
            // 检测添加 服务号信息
            if (message.to.contains(OfficialAccount.SEPARATOR)) {
                String officialAccountId = message.to.substring(message.to.indexOf(OfficialAccount.SEPARATOR) + 2);
                EMLog.d(TAG, "officialAccount:" + officialAccountId);
                JSONObject jsonOffAccount = new JSONObject();
                jsonOffAccount.put(OfficialAccount.KEY_ID, officialAccountId);
                jsonWeichat.put(OfficialAccount.KEY_OFFICIAL_ACCOUNT, jsonOffAccount);
                MarketingHttpClient.asyncReplied(message.to);

                Conversation conversation = getConversation(message.to);
                if (conversation != null && conversation.getMarketings() != null) {
                    JSONObject jsonMarketings = new JSONObject(conversation.getMarketings());
                    try {
                        JSONObject jsonScheduleInfo = jsonMarketings.getJSONObject("schedule_info");
                        if (jsonScheduleInfo.has("skillgroup_name")) {
                            String skillGroupName = jsonScheduleInfo.getString("skillgroup_name");
                            jsonWeichat.put("queueName", skillGroupName);
                        }
                        if (jsonScheduleInfo.has("agent_username")) {
                            String agentName = jsonScheduleInfo.getString("agent_username");
                            jsonWeichat.put("agentUsername", agentName);
                        }
                    } catch (Exception ignored) {
                    }
                }
            }
        } catch (JSONException ignored) {
        }
        message.setAttribute(Message.KEY_WEICHAT, jsonWeichat);
        EMMessageBody messageBody = message.body();
        if(messageBody == null){
            EMLog.e(TAG, "messagebody is nulll");
            if (callback != null){
                callback.onError(Error.GENERAL_ERROR, "body is null");
            }
            if (message.messageStatusCallBack != null){
                message.messageStatusCallBack.onError(Error.GENERAL_ERROR, "body is null");
            }
            return;
        }
        String currentUser = ChatClient.getInstance().currentUserName();
        if (TextUtils.isEmpty(currentUser)){
            EMLog.e(TAG, "currentUserName is empty");
            if (callback != null){
                callback.onError(Error.GENERAL_ERROR, "please login before");
            }

            if (message.messageStatusCallBack != null){
                message.messageStatusCallBack.onError(Error.GENERAL_ERROR, "please login before");
            }
            return;
        }

        message.setFrom(currentUser);


        if (message.getType() == Message.Type.TXT){
            EMTextMessageBody txtBody = (EMTextMessageBody) messageBody;
            if (txtBody.getMessage() != null && txtBody.getMessage().length() > 1500){
                if (callback != null){
                    callback.onError(Error.MESSAGE_SEND_TRAFFIC_LIMIT, "text message length could not more than 1500");
                }

                if (message.messageStatusCallBack != null){
                    message.messageStatusCallBack.onError(Error.MESSAGE_SEND_TRAFFIC_LIMIT, "text message length could not more than 1500");
                }
                return;
            }
        }

        if(isRecord && message.getType() != Message.Type.CMD){
            KefuConversationManager.getInstance().saveMessage(message);
        }

        String originPath = null;
        // If message body is image, check scale request, set size, and file name
        if(message.getType() == Message.Type.IMAGE){
            EMImageMessageBody body = (EMImageMessageBody)message.body();
            if(body == null){
                message.setStatus(Message.Status.FAIL);
                if(callback != null){
                    callback.onError(EMError.GENERAL_ERROR, "Message body cannot be null");
                }
                return;
            }
            String filePath = body.getLocalUrl();
            File file = new File(filePath);
            if(!file.exists() || !file.canRead()){
                message.setStatus(Message.Status.FAIL);
                if(callback != null){
                    callback.onError(EMError.FILE_INVALID, "File not exists or can not be read");
                }
                return;
            }
            if(!body.isSendOriginalImage()){
                String scaledImagePath = ImageUtils.getScaledImage(EMClient.getInstance().getContext(), filePath);
                if(!scaledImagePath.equals(filePath)){
                    originPath = filePath;
                    File attachment = new File(scaledImagePath);
                    long originalSize = new File(filePath).length();
                    long scaledSize = attachment.length();
                    if(originalSize == 0){
                        EMLog.d(TAG, "original image size:" + originalSize);
                        message.setStatus(Message.Status.FAIL);
                        if(callback != null){
                            callback.onError(EMError.FILE_INVALID, "original image size is 0");
                        }
                        return;
                    }
                    filePath = scaledImagePath;
                    body.setLocalUrl(filePath);
                }
            }
            // get image width and height
            BitmapFactory.Options options = ImageUtils.getBitmapOptions(filePath);
            int width = options.outWidth;
            int height = options.outHeight;
            body.setSize(width, height);
            body.setFileName(new File(filePath).getName());
        }

        message.setStatus(Message.Status.INPROGRESS);

        if (!EMClient.getInstance().isLoggedInBefore() && ChatClient.getInstance().isLoggedInBefore()){
            sendRESTMessage(message, callback);
            sendMsgThread.execute(new Runnable() {
                @Override
                public void run() {
                    asyncLoginIM();
                }
            });

        }else{
            sendIMEMMessage(message, callback);
        }
        notifyMessageSent();
        CountDownUtils.getInstance().sendBroadcast();

    }

    private final Object asyncLoginImMutex = new Object();
    private void asyncRealLoginIM(){
        if (EMClient.getInstance().isLoggedInBefore()){
            EMLog.d(TAG, "im isLoggedIn");
            notifyAsyncLoginMutex();
            return;
        }

        String uName = PreferenceUtil.getInstance().getUsername();
        String uPwd = PreferenceUtil.getInstance().getPassword();
        String uToken = PreferenceUtil.getInstance().getToken();

        if (TextUtils.isEmpty(uName)) {
            EMLog.d(TAG, "asyncLoginIM: userName is empty");
            notifyAsyncLoginMutex();
            return;
        }


        if (!TextUtils.isEmpty(uPwd)) {
            EMClient.getInstance().login(uName, uPwd, new EMCallBack() {
                @Override
                public void onSuccess() {
                    PreferenceUtil.getInstance().setToken(EMClient.getInstance().getAccessToken());
                    registerCountDown();
                    CountDownUtils.getInstance().sendBroadcast();
                    notifyAsyncLoginMutex();
                }

                @Override
                public void onError(int i, String s) {
                    notifyAsyncLoginMutex();
                }

                @Override
                public void onProgress(int i, String s) {
                }
            });
        } else if (!TextUtils.isEmpty(uToken)) {
            EMClient.getInstance().loginWithToken(uName, uToken, new EMCallBack() {
                @Override
                public void onSuccess() {
                    PreferenceUtil.getInstance().setToken(EMClient.getInstance().getAccessToken());
                    registerCountDown();
                    CountDownUtils.getInstance().sendBroadcast();
                    notifyAsyncLoginMutex();
                }

                @Override
                public void onError(int i, String s) {
                    ChatClient.getInstance().loginWithTokenFail();
                    notifyAsyncLoginMutex();
                }

                @Override
                public void onProgress(int i, String s) {
                }
            });

        }else{
            notifyAsyncLoginMutex();
        }

    }
    private void notifyAsyncLoginMutex(){
        synchronized (asyncLoginImMutex){
            asyncLoginImMutex.notifyAll();
        }
    }


    private void asyncLoginIM(){
        loginSingleExecutor.execute(new Runnable() {
            @Override
            public void run() {
                synchronized (asyncLoginImMutex){
                    asyncRealLoginIM();
                    try {
                        asyncLoginImMutex.wait(10000);
                    } catch (InterruptedException ignored) {
                        EMLog.e(TAG, "asyncLoginImMutex wait InterruptedException");
                        Thread.currentThread().interrupt();
                    }
                }
            }
        });
    }



    private void notifyMessageSent(){
        synchronized (messageListeners){
            for (MessageListener listener : messageListeners){
                listener.onMessageSent();
            }
        }
    }




    //一般用于发送消息
    private EMMessage convertToEMMessage(Message message){
        String from = message.from();
        String to = message.to();
        EMMessageBody messageBody = message.body();
        EMMessage imMessage;
        if (message.direct() == Message.Direct.SEND) {
            imMessage = EMMessage.createSendMessage(EMMessage.Type
                    .valueOf(message.getType().name()));
            if (to.contains(OfficialAccount.SEPARATOR)){
                to = to.substring(0, to.indexOf(OfficialAccount.SEPARATOR));
            }
        } else {
            imMessage = EMMessage.createReceiveMessage(EMMessage.Type
                    .valueOf(message.getType().name()));
            if (from.contains(OfficialAccount.SEPARATOR)){
                from = from.substring(0, from.indexOf(OfficialAccount.SEPARATOR));
            }
        }
        imMessage.addBody(messageBody);
        imMessage.setFrom(from);
        imMessage.setTo(to);
//        imMessage.setMsgId(message.getIMMsgId());
        imMessage.setMsgTime(message.messageTime());
        Map<String, Object> extAttrs = message.ext();
        synchronized (extAttrs) {
            for(Map.Entry<String, Object> item : extAttrs.entrySet()){
                String key = item.getKey();
                Object value = item.getValue();
                if(value instanceof Boolean){
                    imMessage.setAttribute(key, (Boolean) value);
                }else if(value instanceof Integer){
                    imMessage.setAttribute(key, (Integer) value);
                }else if(value instanceof Long){
                    imMessage.setAttribute(key, (Long) value);
                }else if(value instanceof JSONObject){
                    imMessage.setAttribute(key, (JSONObject)value);
                }else if(value instanceof JSONArray){
                    imMessage.setAttribute(key, (JSONArray)value);
                }else if(value instanceof String){
                    imMessage.setAttribute(key, (String)value);
                }
            }
        }



        return imMessage;
    }

    // 一般用于接收消息
    private Message convertToMessage(EMMessage message) {
        String from = message.getFrom();
        String to = message.getTo();
        EMMessageBody messageBody = message.getBody();
        Message msg;
        if (message.direct() == EMMessage.Direct.SEND) {
            msg = Message.createSendMessage(Message.Type.valueOf(message.getType().name()));
        } else {
            msg = Message.createReceiveMessage(Message.Type.valueOf(message.getType().name()));
        }
        msg.setBody(messageBody);
        msg.setFrom(from);
        msg.setTo(to);
        msg.setIMMsgId(message.getMsgId());
        msg.setMessageTime(message.getMsgTime());
        Map<String, Object> extAttrs = message.ext();
        synchronized (extAttrs) {
            for (Map.Entry<String, Object> item : extAttrs.entrySet()) {
                String key = item.getKey();
                Object value = item.getValue();
                if (value instanceof Boolean) {
                    msg.setAttribute(key, (Boolean) value);
                } else if (value instanceof Integer) {
                    msg.setAttribute(key, (Integer) value);
                } else if (value instanceof Long) {
                    msg.setAttribute(key, (Long) value);
                } else if (value instanceof JSONObject) {
                    msg.setAttribute(key, (JSONObject) value);
                } else if (value instanceof JSONArray) {
                    msg.setAttribute(key, (JSONArray) value);
                } else if (value instanceof String) {
                    msg.setAttribute(key, (String) value);
                }
            }
        }
        String extMsgId = getKefuExtMsgId(msg);
        if (extMsgId != null) {
            msg.setMsgId(extMsgId);
        } else {
            msg.setMsgId(message.getMsgId());
        }

        if (msg.direct() == Message.Direct.RECEIVE){
            if (msg.from() != null && !msg.from().contains(OfficialAccount.SEPARATOR)){
                OfficialAccount officialAccount = msg.getOfficialAccount();
                if (officialAccount != null && officialAccount.getId() != null && !officialAccount.getType().equals("SYSTEM")){
                    msg.setFrom(msg.from() + OfficialAccount.SEPARATOR + officialAccount.getId());
                }
            }
        }else{
            if (msg.to() != null && !msg.to().contains(OfficialAccount.SEPARATOR)){
                OfficialAccount officialAccount = msg.getOfficialAccount();
                if (officialAccount != null && officialAccount.getId() != null && !officialAccount.getType().equals("SYSTEM")){
                    msg.setTo(msg.to() + OfficialAccount.SEPARATOR + officialAccount.getId());
                }
            }
        }


        return msg;
    }



    private void sendIMEMMessage(final Message message, final EMCallBack callback){
        sendMsgThread.submit(new Runnable() {

            @Override
            public void run() {
                final EMMessage imMessage = convertToEMMessage(message);
                imMessage.setMessageStatusCallback(new EMCallBack() {
                    @Override
                    public void onSuccess() {
                        if (!ChatClient.getInstance().hasSecondChannel){
                            message.setIMMsgId(imMessage.getMsgId());
                            message.setStatus(Message.Status.SUCCESS);
                            message.setMessageTime(imMessage.getMsgTime());
                            updateMessageStatusAndMsgId(message);
                            if (callback != null){
                                callback.onSuccess();
                            }
                            if (message.messageStatusCallBack != null){
                                message.messageStatusCallBack.onSuccess();
                            }
                        }

                    }

                    @Override
                    public void onError(int i, String s) {

                    }

                    @Override
                    public void onProgress(int i, String s) {
                        if(message.status() == Message.Status.INPROGRESS){
                            if(message.messageStatusCallBack != null){
                                message.messageStatusCallBack.onProgress(i, s);
                            }
                            if(callback != null){
                                callback.onProgress(i, s);
                            }
                        }
                    }
                });
                long fileLength = 0;
                if (imMessage.getType() == EMMessage.Type.FILE){
                    EMNormalFileMessageBody fileBody = (EMNormalFileMessageBody) imMessage.getBody();
                    fileLength = fileBody.getFileSize();
                }

                if (imMessage.getType() == EMMessage.Type.VIDEO){
                    EMVideoMessageBody fileBody = (EMVideoMessageBody) imMessage.getBody();
                    fileLength = fileBody.getVideoFileLength();
                }

                EMClient.getInstance().chatManager().sendMessage(imMessage);
                if (ChatClient.getInstance().hasSecondChannel){
                    Object mutex = new Object();
                    String msgId = message.messageId();
                    addSendLock(msgId, mutex);
                    synchronized (mutex) {
                        if(sendLocks.containsKey(msgId)){
                            try {
                                if (message.getType() == Message.Type.TXT || message.getType() == Message.Type.CMD || message.getType() == Message.Type.LOCATION){
                                    mutex.wait(WAIT_TIME_OUT_TYPE_TXT * 1000L);
                                }else if (message.getType() == Message.Type.IMAGE || message.getType() == Message.Type.VOICE){
                                    mutex.wait(WAIT_TIME_OUT * 1000L);
                                }else if(message.getType() == Message.Type.FILE || message.getType() == Message.Type.VIDEO){
                                    long tempWaitTime = fileLength/1024/1024 * 3 + WAIT_TIME_OUT;
                                    long waitTime = Math.min(tempWaitTime, WAIT_TIME_ATTACHEMENT_MAX);
                                    waitTime = Math.max(waitTime, WAIT_TIME_ATTACHMENT_MIN);
                                    mutex.wait(waitTime * 1000);
                                }else{
                                    mutex.wait(WAIT_TIME_OUT * 1000L);
                                }
                            } catch (InterruptedException ignored) {
                                EMLog.e(TAG, "mutex wait InterruptedException");
                                Thread.currentThread().interrupt();
                            }
                        }
                    }
                    EMLog.d(TAG, "exit from wait msgid:" + message.messageId());
                    if(sendLocks.remove(msgId) != null){
                        sendRESTMessage(message, callback);
                    }else{
                        message.setIMMsgId(imMessage.getMsgId());
                        message.setStatus(Message.Status.SUCCESS);
                        message.setMessageTime(imMessage.getMsgTime());
                        updateMessageStatusAndMsgId(message);
                        if(message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onSuccess();
                        }
                        if(callback != null){
                            callback.onSuccess();
                        }
                    }
                }

            }
        });
    }

    private void sendRESTMessage(final Message mesage, final EMCallBack callback){
        KefuHttpClient.sendMessageByKefuRest(mesage, new EMCallBack() {
            @Override
            public void onSuccess() {
                mesage.setStatus(Message.Status.SUCCESS);
                updateMessageState(mesage);
                if(mesage.messageStatusCallBack != null){
                    mesage.messageStatusCallBack.onSuccess();
                }
                if (callback != null){
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int error, String errorMsg) {
                KefuHttpClient.sendMessageByKefuRest(mesage, new EMCallBack() {
                    @Override
                    public void onSuccess() {
                        EMLog.d(TAG, "send kefu message by rest msgid:" + mesage.messageId());
                        mesage.setStatus(Message.Status.SUCCESS);
                        updateMessageState(mesage);
                        if(mesage.messageStatusCallBack != null){
                            mesage.messageStatusCallBack.onSuccess();
                        }
                        if (callback != null){
                            callback.onSuccess();
                        }
                    }

                    @Override
                    public void onError(int error, String errorMsg) {
                        EMLog.d(TAG, "send kefu message:" + errorMsg);
                        mesage.setStatus(Message.Status.FAIL);
                        updateMessageState(mesage);
                        if(mesage.messageStatusCallBack != null){
                            mesage.messageStatusCallBack.onError(error, errorMsg);
                        }
                        if (callback != null) {
                            callback.onError(error, errorMsg);
                        }

                    }

                    @Override
                    public void onProgress(int i, String s) {

                    }
                });


            }

            @Override
            public void onProgress(int i, String s) {

            }
        });

    }

    void initDB(String userName){
        EMLog.d(TAG, "initDB - userName:"  + userName);
        Context context = EMClient.getInstance().getContext();
        if(context == null){
            return;
        }
        KefuDBManager.initDB(userName);
    }

    void loadDB(){
        KefuConversationManager.getInstance().clear();
        Thread thread = new Thread(new Runnable() {

            @Override
            public void run() {
                loadAllConversations();
            }
        });
        thread.setPriority(Thread.MAX_PRIORITY - 1);
        thread.start();
    }


    /**
     * 判断是否是silent消息
     * silent消息不需要通知前台
     * @param message 消息
     * @return 是否是silent消息状态
     */
    public boolean isSilentMessage(Message message){
        return message.getBooleanAttribute("em_ignore_notification", false);
    }

    /**
     * 下载方法 已过期
     * @param remoteUrl 远端url
     * @param localFilePath 本地url
     * @param headers 请求header
     * @param callback 回调
     */
    @Deprecated
    public void downloadFile(String remoteUrl, String localFilePath, final Map<String, String> headers, final EMCallBack callback){
        EMClient.getInstance().chatManager().downloadFile(remoteUrl, localFilePath, headers, callback);
    }

    private void shutdownVisitorWaitThread(){
        if(ChatClient.getInstance().isShowVisitorWaitCount){
            if (visitorWaitThread != null && !visitorWaitThread.isShutdown()){
                visitorWaitThread.shutdownNow();
                visitorWaitThread = null;
                if (!visitorWaitCountListeners.isEmpty()){
                    synchronized (visitorWaitCountListeners){
                        for (VisitorWaitListener listener : visitorWaitCountListeners){
                            if (listener != null){
                                listener.waitCount(0);
                            }
                        }
                    }
                }
            }
        }
    }

    private void shutdownAgentInputThread(){
        if (ChatClient.getInstance().isShowAgentInputState){
            if (agentInputThread != null && !agentInputThread.isShutdown()){
                agentInputThread.shutdownNow();
                agentInputThread = null;
                if (!agentInputListeners.isEmpty()){
                    synchronized (agentInputListeners) {
                        for (AgentInputListener listener : agentInputListeners) {
                            if (listener != null) {
                                listener.onInputState(null);
                            }
                        }
                    }
                }

            }
        }

    }

    private final Object visitorLock = new Object();
    private final Object agentInputLock = new Object();

    private void showVisitorWaitCountAndAgentInputState(final String toChatUsername){
        final Thread thread = new Thread(new Runnable() {
            @Override
            public void run() {
                synchronized (visitorLock){
                    if (ChatClient.getInstance().isShowVisitorWaitCount){
                        shutdownVisitorWaitThread();
                        if (visitorWaitCountListeners.isEmpty()){
                            return;
                        }
                        OfficialAccount officialAccount = getConversation(toChatUsername).officialAccount();
                        if (officialAccount == null){
                            officialAccount = MarketingHttpClient.getSystemOfficialAccount(toChatUsername);
                        }
                        if (officialAccount != null && officialAccount.getId() != null){
                            final ConversationInfo info = MarketingHttpClient.getLastestSession(officialAccount.getId(), toChatUsername);
                            if (info == null || info.session_id == null) {
                                EMLog.d(TAG, "sessionid is null");
                            }else if(info.state == null || !info.state.equalsIgnoreCase("wait")){
                                EMLog.d(TAG, "session is not wait state");
                            }else{
                                visitorWaitThread = Executors.newSingleThreadScheduledExecutor();
                                visitorWaitThread.scheduleAtFixedRate(new Runnable() {
                                    @Override
                                    public void run() {
                                        MarketingHttpClient.asyncGetWaitCount(info, new ValueCallBack<String>() {
                                            @Override
                                            public void onSuccess(String value) {
                                                try {
                                                    JSONObject jsonObj = new JSONObject(value);
                                                    JSONObject jsonEntty = jsonObj.getJSONObject("entity");
                                                    String number = jsonEntty.getString("visitorUserWaitingNumber");
                                                    if (!TextUtils.isEmpty(number) && TextUtils.isDigitsOnly(number)){
                                                        synchronized (visitorWaitCountListeners){
                                                            for (VisitorWaitListener listener : visitorWaitCountListeners){
                                                                if (listener != null){
                                                                    listener.waitCount(Integer.parseInt(number));
                                                                }
                                                            }
                                                        }
                                                    }else{
                                                        shutdownVisitorWaitThread();
                                                    }
                                                } catch (JSONException e) {
                                                    shutdownVisitorWaitThread();
                                                }
                                            }

                                            @Override
                                            public void onError(int error, String errorMsg) {
                                                EMLog.i(TAG, "visitor wait error -> " + errorMsg);
                                            }
                                        });
                                    }
                                }, 1000, 5000, TimeUnit.MILLISECONDS);
                            }

                        }
                    }
                }
            }
        });
        thread.start();

        Thread agentThread = new Thread(new Runnable() {
            @Override
            public void run() {
                synchronized (agentInputLock){
                    if (ChatClient.getInstance().isShowAgentInputState){
                        shutdownAgentInputThread();
                        if (agentInputListeners.isEmpty()){
                            return;
                        }
                        OfficialAccount officialAccount = getConversation(toChatUsername).officialAccount();
                        if (officialAccount == null){
                            officialAccount = MarketingHttpClient.getSystemOfficialAccount(toChatUsername);
                        }
                        if (officialAccount != null && officialAccount.getId() != null){
                            final ConversationInfo info = MarketingHttpClient.getLastestSession(officialAccount.getId(), toChatUsername);
                            if (info == null || info.session_id == null) {
                                EMLog.i(TAG, "sessionid is null");
                            }else{
                                agentInputThread = Executors.newScheduledThreadPool(1);
                                agentInputThread.scheduleAtFixedRate(new Runnable() {
                                    @Override
                                    public void run() {
                                        MarketingHttpClient.asyncGetAgentState(info, new ValueCallBack<String>() {
                                            @Override
                                            public void onSuccess(String value) {
                                                if (!TextUtils.isEmpty(value) && !value.equals("null")){
                                                    synchronized (agentInputListeners){
                                                        for (AgentInputListener listener : agentInputListeners){
                                                            if (listener != null){
                                                                listener.onInputState(value);
                                                            }
                                                        }
                                                    }
                                                }else{
                                                    synchronized (agentInputListeners) {
                                                        for (AgentInputListener listener : agentInputListeners) {
                                                            if (listener != null) {
                                                                listener.onInputState(null);
                                                            }
                                                        }
                                                    }
                                                }
                                            }

                                            @Override
                                            public void onError(int error, String errorMsg) {
                                                EMLog.i(TAG, "get agent state error->" + errorMsg);
                                            }
                                        });

                                    }
                                }, 3000, 3000, TimeUnit.MILLISECONDS);
                            }

                        }

                    }
                }
            }
        });
        agentThread.start();
    }

    /**
     * 消息预知功能发送消息内容
     * @param content 内容
     */
    public void postMessagePredict(final String content) {
        postMessagePredict(content, currentChatUsername);
    }

    /**
     * 消息预知功能发送消息内容
     * @param content 内容
     * @param toChatUsername 聊天对象UserName
     */
    public void postMessagePredict(final String content, final String toChatUsername) {
        if (!ChatClient.getInstance().isShowMessagePredict || TextUtils.isEmpty(content) || TextUtils.isEmpty(toChatUsername)) {
            return;
        }

        new Thread(new Runnable() {
            @Override
            public void run() {
                OfficialAccount officialAccount = getConversation(toChatUsername).officialAccount();
                if (officialAccount == null){
                    officialAccount = MarketingHttpClient.getSystemOfficialAccount(toChatUsername);
                }
                if (officialAccount != null && officialAccount.getId() != null) {
                    final ConversationInfo info = MarketingHttpClient.getLastestSession(officialAccount.getId(), toChatUsername);
                    if (info == null || info.session_id == null) {
                        EMLog.i(TAG, "sessionid is null");
                    } else {
                        MarketingHttpClient.postMessagePredict(content, info, new ValueCallBack<String>() {
                            @Override
                            public void onSuccess(String value) {

                            }

                            @Override
                            public void onError(int error, String errorMsg) {

                            }
                        });
                    }
                }
            }
        }).start();
    }

    /**
     * 进入聊天页面需要bind联系人
     * @param toChatUsername 聊天对象UserName
     */
    public void bindChat(final String toChatUsername) {
        if (TextUtils.isEmpty(toChatUsername)){
            return;
        }
        currentChatUsername = toChatUsername;
        KefuPolling.startPolling(toChatUsername);
        showVisitorWaitCountAndAgentInputState(toChatUsername);
    }

    /**
     * 进入聊天页面需要bind联系人
     * @param toChatUsername 聊天对象UserName
     * @see #bindChat(String)
     */
    @Deprecated
    public void bindChatUI(final String toChatUsername){
        bindChat(toChatUsername);
    }

    private void sendDeviceInfo(String target){
        if (ChatClient.getInstance().tenantId() != null){
            KefuHttpClient.asyncSendDeviceInfo(ChatClient.getInstance().tenantId(), target, new ValueCallBack<String>() {
                @Override
                public void onSuccess(String value) {
                    Log.d(TAG, "value:" + value);
                }

                @Override
                public void onError(int error, String errorMsg) {
                    Log.e(TAG, "error:" + errorMsg);
                }
            });
        }
    }

    /**
     * 离开聊天页面调用此方法
     */
    public void unbindChat() {
        KefuPolling.stopPolling();
        shutdownWaitAndInput();
        if (!TextUtils.isEmpty(MarketingHttpClient.currentConversationId)){
            getConversation(MarketingHttpClient.currentConversationId).setConversationInfo(null);
        }
        currentChatUsername = null;
    }

    /**
     * 离开聊天页面调用此方法
     * @see #unbindChat()
     */
    @Deprecated
    public void unBind(){
        unbindChat();
    }

    private void shutdownWaitAndInput(){

        shutdownVisitorWaitThread();
        shutdownAgentInputThread();



    }


    /**
     * 增加消息监听
     * @param listener 监听
     */
    public void addMessageListener(MessageListener listener) {
        if(listener == null){
            return;
        }

        if(!messageListeners.contains(listener)){
            messageListeners.add(listener);
        }
    }

    /**
     * 移除消息监听
     * @param listener 要移除的监听
     */
    public void removeMessageListener(MessageListener listener) {
        if(listener == null){
            return;
        }
        if (messageListeners.contains(listener)){
            messageListeners.remove(listener);
        }
    }

    /**
     * 管理员输入状态监听 聊天界面初始化时调用
     * @param listener 监听
     */
    public void addAgentInputListener(AgentInputListener listener){
        if (listener == null){
            return;
        }
        if (!agentInputListeners.contains(listener)){
            agentInputListeners.add(listener);
        }
    }

    /**
     * 移除管理员输入状态监听 聊天界面销毁时调用
     * @param listener 监听
     */
    public void removeAgentInputListener(AgentInputListener listener){
        if (listener == null){
            return;
        }
        if (agentInputListeners.contains(listener)){
            agentInputListeners.remove(listener);
        }
    }

    /**
     * 访客等待数监听 聊天界面初始化时调用
     * @param listener 监听
     */
    public void addVisitorWaitListener(VisitorWaitListener listener){
        if (listener == null){
            return;
        }
        if (!visitorWaitCountListeners.contains(listener)){
            visitorWaitCountListeners.add(listener);
        }
    }

    /**
     * 移除访客等待数监听 聊天界面销毁时调用
     * @param listener 监听
     */
    public void removeVisitorWaitListener(VisitorWaitListener listener){
        if (listener == null){
            return;
        }
        if (visitorWaitCountListeners.contains(listener)){
            visitorWaitCountListeners.remove(listener);
        }
    }

    private String checkKefuMessageAck(Message message){
        try {
            JSONObject jsonExt = message.getJSONObjectAttribute(Message.KEY_WEICHAT);
            if (jsonExt != null) {
                String msgIdForAck = jsonExt.getString(ACK_FOR_MSG_ID);// ACK_FOR_MSG_ID
                if (!TextUtils.isEmpty(msgIdForAck)
                        && !msgIdForAck.equals("null")) {
                    return msgIdForAck;
                }
            }
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * 第二通道收到的消息的msgid
     * @param message 消息
     * @return 第二通道消息msgid
     */
    String getKefuExtMsgId(Message message){
        try {
            JSONObject jsonExt = message.getJSONObjectAttribute(Message.KEY_WEICHAT);
            if (jsonExt != null) {
                String msgIdForAck = jsonExt.getString("msgId");
                if (!TextUtils.isEmpty(msgIdForAck)
                        && !msgIdForAck.equals("null")) {
                    return msgIdForAck;
                }
            }
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * 消息重发
     * @param msg 消息
     */
    public void resendMessage(Message msg) {
        msg.setStatus(Message.Status.CREATE);
        sendMessage(msg);
    }

    /**
     * 消息重发
     * @param msg 消息
     * @see #resendMessage(Message)
     */
    @Deprecated
    public void reSendMessage(Message msg) {
        resendMessage(msg);
    }

    /**
     * 设置语音消息已读状态
     * @param message  消息
     */
    public void setMessageListened(Message message){
        message.setListened(true);
        KefuDBManager.getInstance().updateMessageListened(message.messageId(), true);
    }

    /**
     * 通过消息id获取消息
     * @param messageId 消息id
     * @return 消息
     */
    public Message getMessage(String messageId){
        return KefuConversationManager.getInstance().getMessage(messageId);
    }

    void addMessage(Message msg){
        KefuConversationManager.getInstance().addMessage(msg);
    }

    void addMessage(Message msg, boolean unreadCountIncrease){
        KefuConversationManager.getInstance().addMessage(msg, unreadCountIncrease);
    }

    /**
     * 通过会话id获取会话
     * @param conversationId 会话id
     * @return 会话
     */
    public Conversation getConversation(final String conversationId){
        return KefuConversationManager.getInstance().getConversation(conversationId);
    }

    /**
     * 删除会话
     * @param username ToUserName
     * @param deleteMessages 是否删除消息
     * @return 成功或失败
     */
    public boolean deleteConversation(String username, boolean deleteMessages){
        EMClient.getInstance().chatManager().deleteConversation(username, deleteMessages);
        return KefuConversationManager.getInstance().removeConversation(username, deleteMessages);
    }

    /**
     * 标记所有会话消息为已读
     */
    public void markAllConversationsAsRead(){
        KefuConversationManager.getInstance().resetAllUnreadMsgCount();
    }

    private void loadAllConversations(){
        KefuConversationManager.getInstance().loadAllConversations();
    }

    /**
     * 获取所有会话
     * @return 会话列表
     */
    public Hashtable<String, Conversation> getAllConversations(){
        return KefuConversationManager.getInstance().getAllConversations();
    }

    /**
     * 获取未读消息数
     * @return 未读消息数
     */
    public int getUnreadMsgsCount(){
        return KefuConversationManager.getInstance().getUnreadMsgsCount();
    }

    /**
     * 保持信息进入数据库
     * @param message 消息
     */
    public void saveMessage(Message message){
        KefuConversationManager.getInstance().saveMessage(message);
    }

    /**
     * 更新消息体
     * @param message
     * @return
     */
    public boolean updateMessageBody(Message message){
        return KefuDBManager.getInstance().updateMessageBody(message);
    }

    private void updateMessageState(Message message){
        ContentValues cvs = new ContentValues();
        cvs.put(KefuDBManager.COLUMN_MSG_STATUS, message.status().ordinal() + "");
        KefuDBManager.getInstance().updateMessage(message.messageId(), cvs);
    }

    private void updateMessageStatusAndMsgId(Message message){
        ContentValues cvs = new ContentValues();
        cvs.put(KefuDBManager.COLUMN_MSG_STATUS, message.status().ordinal() + "");
        cvs.put(KefuDBManager.COLUMN_IM_MSG_ID, message.getIMMsgId());
        cvs.put(KefuDBManager.COLUMN_MSG_TIME, message.messageTime());
        KefuDBManager.getInstance().updateMessage(message.messageId(), cvs);
    }

//    public void downloadAttachment(final Message message){
//        EMMessage imMessage = convertToEMMessage(message);
//        if (message.messageStatusCallBack != null){
//            imMessage.setMessageStatusCallback(message.messageStatusCallBack);
//        }
//        EMClient.getInstance().chatManager().downloadAttachment(imMessage);
//    }

    /**
     * 下载消息中包含的附件
     * @param message 消息
     */
    public void downloadAttachment(final Message message){
        if (!(message.body() instanceof EMFileMessageBody)) {
            EMLog.d(TAG, "download file msg body is not FileMessageBody");
            return;
        }
        EMFileMessageBody msgBody = (EMFileMessageBody) message.body();
        final String remoteUrl = msgBody.getRemoteUrl();
        final String localPath = msgBody.getLocalUrl();
        if (TextUtils.isEmpty(remoteUrl) || TextUtils.isEmpty(localPath)){
            EMLog.d(TAG, "download file remoteUrl or localPath is empty");
            return;
        }

        final File localFile = new File(localPath);
        final File tempLocalFile = new File(localFile.getParent(), "tmp_" + localFile.getName());
        updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.DOWNLOADING, false);
        downloadThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                try {
                    requestBuilder.to(tempLocalFile, new ProgressListener() {
                        @Override
                        public void loadProgress(long progress) {
                            if (message.messageStatusCallBack != null){
                                if (progress < 100){
                                    message.messageStatusCallBack.onProgress((int) progress, "progress");
                                }
                            }

                        }
                    });
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null){
                        if (message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onError(Error.FILE_DOWNLOAD_FAILED, "rfile download failed");
                        }
                        return;
                    }
                    int status = httpResponse.getStatusCode();
                    if (status / 100 == 2){
                        tempLocalFile.renameTo(localFile);
                        updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.SUCCESSED, false);
                        updateMessageBody(message);
                        if (message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onSuccess();
                        }
                    }else{
                        updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.FAILED, false);
                        updateMessageBody(message);
                        if (message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onError(Error.FILE_DOWNLOAD_FAILED, "file download failed");
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.FAILED, false);
                    updateMessageBody(message);
                    if (message.messageStatusCallBack != null){
                        message.messageStatusCallBack.onError(Error.FILE_DOWNLOAD_FAILED, "file download failed");
                    }
                }

            }
        });

    }

    void downloadAttachments(final Message message,final boolean isThumbnail){
        if (isThumbnail){
            downloadThumbnail(message);
        }else{
            downloadAttachment(message);
        }
    }

    /**
     * 下载微缩图
     * @param message 消息
     */
    public void downloadThumbnail(final Message message){
        final String remoteUrl;
        final String localPath;

        if (message.body() instanceof EMImageMessageBody) {
            EMImageMessageBody msgBody = (EMImageMessageBody) message.body();
            remoteUrl = msgBody.getRemoteUrl();
            localPath = msgBody.thumbnailLocalPath();
        } else if (message.body() instanceof EMVideoMessageBody){
            EMVideoMessageBody msgBody = (EMVideoMessageBody) message.body();
            remoteUrl = msgBody.getThumbnailUrl();
            localPath = msgBody.getLocalThumb();
        } else {
            EMLog.d(TAG, "download file msg body is not ImageMessageBody or EMVideoMessageBody");
            return;
        }
        if (TextUtils.isEmpty(remoteUrl) || TextUtils.isEmpty(localPath)){
            EMLog.d(TAG, "download file remoteUrl or localPath is empty");
            return;
        }
        final File localFile = new File(localPath);
        final File tempLocalFile = new File(localFile.getParent(), "tmp_" + localFile.getName());
        updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.DOWNLOADING, true);
        downloadThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                requestBuilder.param("thumbnail", "true");
                try {
                    requestBuilder.to(tempLocalFile);
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null){
                        if (message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onError(Error.FILE_DOWNLOAD_FAILED, "rfile download failed");
                        }
                        return;
                    }
                    int status = httpResponse.getStatusCode();
                    if (status / 100 == 2){
                        tempLocalFile.renameTo(localFile);
                        updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.SUCCESSED, true);
                        updateMessageBody(message);
                        if (message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onSuccess();
                        }
                    }else{
                        updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.FAILED, true);
                        updateMessageBody(message);
                        if (message.messageStatusCallBack != null){
                            message.messageStatusCallBack.onError(Error.FILE_DOWNLOAD_FAILED, "file download failed");
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    updateMessageBodyDownloadStatus(message, EMFileMessageBody.EMDownloadStatus.FAILED, true);
                    updateMessageBody(message);
                    if (message.messageStatusCallBack != null){
                        message.messageStatusCallBack.onError(Error.FILE_DOWNLOAD_FAILED, "file download failed");
                    }
                }

            }
        });
    }


    private void updateMessageBodyDownloadStatus(final Message message, EMFileMessageBody.EMDownloadStatus status, boolean isThumbnail){
        switch (message.getType()){
            case FILE:
            case VOICE:
                EMFileMessageBody fileBody = (EMFileMessageBody) message.body();
                fileBody.setDownloadStatus(status);
                break;
            case IMAGE:
                EMImageMessageBody imgBody = (EMImageMessageBody) message.body();
                if (isThumbnail){
                    imgBody.setThumbnailDownloadStatus(status);
                }else{
                    imgBody.setDownloadStatus(status);
                }
                break;
            default:
                break;
        }
    }

//    public void downloadThumbnail(Message message){
//        EMMessage imMessage = convertToEMMessage(message);
//        if (message.messageStatusCallBack != null){
//            imMessage.setMessageStatusCallback(message.messageStatusCallBack);
//        }
//        EMClient.getInstance().chatManager().downloadThumbnail(imMessage);
//    }


    /**
     * 主动发起满意度评价
     * @param toUser IM服务号
     * @param callback
     */
    public void asyncSendInviteEvaluationMessage(final String toUser, final Callback callback) {
        KefuHttpClient.asyncGetInviteEvaluation(toUser, new ValueCallBack<String>() {
            @Override
            public void onSuccess(String value) {
                try {
                    JSONObject jsonObject = new JSONObject(value);
                    String status = jsonObject.getString("status");
                    if (!"OK".equalsIgnoreCase(status)) {
                        if (callback != null) {
                            callback.onError(-1, "server is error");
                        }
                        return;
                    }
                    JSONArray jsonEntites = jsonObject.getJSONArray("entities");
                    Message message = Message.createReceiveMessage(Message.Type.TXT);
                    message.addBody(new EMTextMessageBody(""));
                    JSONObject weichatJson = new JSONObject();
                    weichatJson.put("ctrlType", "inviteEnquiry");
                    JSONObject ctrlArgsJson = new JSONObject();
                    ctrlArgsJson.put("inviteId", 0);
                    ConversationInfo conversationInfo = getConversation(toUser).getConversationInfo();
                    if (conversationInfo != null){
                        ctrlArgsJson.put("serviceSessionId", conversationInfo.session_id);
                    }
                    ctrlArgsJson.put("evaluationDegree", jsonEntites);
                    weichatJson.put("ctrlArgs", ctrlArgsJson);
                    JSONObject officialAccountJson = new JSONObject();
                    OfficialAccount officialAccount = getConversation(toUser).officialAccount();
                    if (officialAccount != null && officialAccount.getId() != null) {
                        officialAccountJson.put("official_account_id", officialAccount.getId());
                    }
                    weichatJson.put("official_account", officialAccountJson);
                    message.setAttribute("weichat", weichatJson);
                    message.setMessageTime(System.currentTimeMillis());
                    message.setMsgId(UUID.randomUUID().toString());
                    message.setFrom(toUser);
                    saveMessage(message);
                    notifyMessageSent();
                    if (callback != null){
                        callback.onSuccess();
                    }
                } catch (Exception e) {
                    if (callback != null) {
                        callback.onError(-1, "invite evaluation message error:" + e.getMessage());
                    }
                }
            }

            @Override
            public void onError(int error, String errorMsg) {
                if (callback != null) {
                    callback.onError(error, errorMsg);
                }
            }
        });
    }


    /**
     * 清空会话
     * @param username ToUserName
     */
    public void clearConversation(String username){
        KefuConversationManager.getInstance().clearConversation(username);
    }


    public interface MessageListener {
        void onMessage(List<Message> msgs);
        void onCmdMessage(List<Message> msgs);
        void onMessageStatusUpdate();
        void onMessageSent();
    }

    public interface AgentInputListener{
        void onInputState(String input);
    }

    public interface VisitorWaitListener{
        void waitCount(int num);
    }

    /**
     * 获取企业欢迎语
     */
    public void getEnterpriseWelcome(final ValueCallBack<String> callback){
        KefuHttpClient.getEnterpriseWelcome(callback);
    }

    /**
     * 获取机器人欢迎语
     */
    public void getRobotWelcome(final ValueCallBack<JSONObject> callBack){
        KefuHttpClient.getRobotWelcome(callBack);
    }

    /**
     * 获取当前会话状态
     */
    public void getCurrentSessionId(final String toChatUsername, final ValueCallBack<String> callback){
        KefuHttpClient.getCurrentSessionStatus(toChatUsername, new ValueCallBack<String>() {
            @Override
            public void onSuccess(String value) {
                if (callback != null){
                    try{
                        JSONObject jsonObject = new JSONObject(value);
                        String status = jsonObject.getString("status");
                        if (status != null && status.equalsIgnoreCase("ok")){
                            JSONArray entityArr = jsonObject.getJSONArray("entities");
                            if (entityArr != null && entityArr.length() > 0){
                                callback.onSuccess(entityArr.getString(0));
                            }else{
                                callback.onSuccess("");
                            }
                        }else{
                            String description = jsonObject.getString("errorDescription");
                            callback.onError(-1, description);
                        }
                    }catch (Exception e){
                        callback.onError(-1, "request failed:" + android.util.Log.getStackTraceString(e));
                    }
                }

            }

            @Override
            public void onError(int error, String errorMsg) {
                if (callback != null){
                    callback.onError(error, errorMsg);
                }
            }
        });
    }

    public void cancelVideoConferences(final String toChatUsername, final ValueCallBack<String> callBack){
        new Thread(new Runnable() {
            @Override
            public void run() {
                OfficialAccount officialAccount = getConversation(toChatUsername).officialAccount();
                if (officialAccount == null){
                    officialAccount = MarketingHttpClient.getSystemOfficialAccount(toChatUsername);
                }
                if (officialAccount != null && officialAccount.getId() != null) {
                    final ConversationInfo info = MarketingHttpClient.getLastestSession(officialAccount.getId(), toChatUsername);
                    if (info == null || info.session_id == null) {
                        EMLog.e(TAG, "sessionid is null");
                        if (callBack != null) {
                            callBack.onError(-1, "session is null");
                        }
                    } else {
                        KefuHttpClient.asyncCancelVideoConferences(info.session_id, callBack);
                    }
                }
            }
        }).start();
    }


}
