package com.hyphenate.chat;

import android.text.TextUtils;

import com.hyphenate.helpdesk.callback.ValueCallBack;
import com.hyphenate.helpdesk.httpclient.HttpClient;
import com.hyphenate.helpdesk.httpclient.HttpRequestBuilder;
import com.hyphenate.helpdesk.httpclient.HttpResponse;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Created by liyuzhao on 20/06/2017.
 */

final class MarketingHttpClient {

	private static final String TAG = "MarketingHttpClient";

	public static int DEFAULT_CONNECTION_TIMEOUT = 20 * 1000;
	public static int DEFAULT_READ_TIMEOUT = 20 * 1000;

	private static ExecutorService sendThreadPool = Executors
			.newFixedThreadPool(10);

	private static final Object lock = new Object();

	public static String currentConversationId;

	private MarketingHttpClient(){
		throw new IllegalAccessError("not support constructor new object");
	}

	private static void sendMarketingDelivered(String marketingTaskId, final String toUser, ValueCallBack<String> callBack){
		String tenantId = ChatClient.getInstance().tenantId();
		if (TextUtils.isEmpty(tenantId)){
			if (callBack != null){
				callBack.onError(-1, "tenantid isnot nullable");
			}
			return;
		}

		String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/delivered", ChatClient.getInstance().kefuRestServer(), tenantId, marketingTaskId);
		sendMarketingStatusChanged(remoteUrl, toUser, callBack);
	}

	private static void sendMarketingOpened(String marketingTaskId, final String toUser, ValueCallBack<String> callBack){
		String tenantId = ChatClient.getInstance().tenantId();
		if (TextUtils.isEmpty(tenantId)){
			if (callBack != null){
				callBack.onError(-1, "tenantid isnot nullable");
			}
			return;
		}
		String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/opened", ChatClient.getInstance().kefuRestServer(), tenantId, marketingTaskId);
		sendMarketingStatusChanged(remoteUrl, toUser, callBack);
	}

	private static void sendMarketingReplied(String marketingTaskId, final String toUser, ValueCallBack<String> callBack){
		String tenantId = ChatClient.getInstance().tenantId();
		if (TextUtils.isEmpty(tenantId)){
			if (callBack != null){
				callBack.onError(-1, "tenantid isnot nullable");
			}
			return;
		}
		String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/replied", ChatClient.getInstance().kefuRestServer(), tenantId, marketingTaskId);
		sendMarketingStatusChanged(remoteUrl, toUser, callBack);
	}


	private static void sendMarketingStatusChanged(final String remoteUrl, final String toUser, final ValueCallBack<String> callBack){
		if (!ChatClient.getInstance().isLoggedInBefore()){
			if (callBack != null){
				callBack.onError(-1, "please login first");
			}
			return;
		}

		sendThreadPool.execute(new Runnable() {
			@Override
			public void run() {
				try {
					String visitorId;

					synchronized (lock){
						visitorId = KefuDBManager.getInstance().getVisitorUserId(toUser);
						if (TextUtils.isEmpty(visitorId)){
							visitorId = getVisitorIdFromRemote(toUser);
						}
					}
					if (TextUtils.isEmpty(visitorId)){
						if (callBack != null){
							callBack.onError(-1, "visitorid is null");
						}
						return;
					}

					String currentUser = ChatClient.getInstance().currentUserName();
					String token = ChatClient.getInstance().accessToken();
					String appkey = ChatClient.getInstance().appKey();
					JSONObject jsonObject = new JSONObject();
					jsonObject.put("visitor_id", visitorId);

					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
					httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
					HttpRequestBuilder requestBuilder = httpClient.put(remoteUrl);
					assert appkey != null;
					requestBuilder.param("orgName", appkey.split("#")[0]);
					requestBuilder.param("appName", appkey.split("#")[1]);
					requestBuilder.param("userName", currentUser);
					requestBuilder.param("token", token);
					requestBuilder.content(jsonObject.toString().getBytes(), "application/json");
					HttpResponse httpResponse = requestBuilder.execute();
					assert httpResponse != null;
					int status_code = httpResponse.getStatusCode();
					EMLog.d(TAG, "marketing status_code:" + status_code);
					if (status_code >= 200 && status_code < 300) {
						StringBuffer buffer = new StringBuffer();
						httpResponse.read(buffer);
						String result = buffer.toString();
						if (callBack != null) {
							callBack.onSuccess(result);
						}
					} else {
						if (callBack != null) {
							callBack.onError(-1, "send marketing status fail");
						}
					}

				} catch (Exception e) {
					if (callBack != null) {
						callBack.onError(-1, "error:" + e.getMessage());
					}
				}
			}
		});

	}

	private static synchronized String getVisitorIdFromRemote(String toUser){
		if (!ChatClient.getInstance().isLoggedInBefore()){
			EMLog.e(TAG, "get visitor but not login");
			return "";
		}
		String tenantId = ChatClient.getInstance().tenantId();
		if (TextUtils.isEmpty(tenantId)){
			return "";
		}
		try {
			String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors", ChatClient.getInstance().kefuRestServer(), tenantId);
			String appkey = ChatClient.getInstance().appKey();
			String token = ChatClient.getInstance().accessToken();
			String currentUser = ChatClient.getInstance().currentUserName();
			String techChannelinfo = appkey + "#" + toUser;
			HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
			httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
			httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
			HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
			assert appkey != null;
			requestBuilder.param("orgName", appkey.split("#")[0]);
			requestBuilder.param("appName", appkey.split("#")[1]);
			requestBuilder.param("userName", currentUser);
			requestBuilder.param("token", token);
			requestBuilder.param("techChannelInfo", techChannelinfo);
			HttpResponse httpResponse = requestBuilder.execute();
			if (httpResponse == null){
				return "";
			}
			int status_code = httpResponse.getStatusCode();
			EMLog.d(TAG, "getVisitorIdFromRemote status_code:" + status_code);
			if (status_code >= 200 && status_code < 300) {
				StringBuffer buffer = new StringBuffer();
				httpResponse.read(buffer);
				String result = buffer.toString();
				JSONObject resultJson = new JSONObject(result);
				if (resultJson.has("entity")){
					JSONObject entitiesJson = resultJson.getJSONObject("entity");
					String userId = entitiesJson.getString("userId");
					KefuDBManager.getInstance().saveVisitorInfo(toUser, userId);
					return userId;
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return "";
	}

	 static void asyncGetAgentState(final ConversationInfo info, final ValueCallBack<String> callBack){

		sendThreadPool.execute(new Runnable() {
			@Override
			public void run() {
				try {
					String remoteUrl = String.format("%1$s/v1/webimplugin/sessions/%2$s/agent-input-state",
							ChatClient.getInstance().kefuRestServer(), info.session_id);
					String appkey = ChatClient.getInstance().appKey();
					String token = ChatClient.getInstance().accessToken();
					String currentUser = ChatClient.getInstance().currentUserName();
					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
					httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
					HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
					assert appkey != null;
					requestBuilder.param("orgName", appkey.split("#")[0]);
					requestBuilder.param("appName", appkey.split("#")[1]);
					requestBuilder.param("userName", currentUser);
					requestBuilder.param("token", token);
					requestBuilder.param("tenantId", ChatClient.getInstance().tenantId());
					HttpResponse httpResponse = requestBuilder.execute();
					if (httpResponse == null){
						if (callBack != null){
							callBack.onError(-1, "error: response is null");
						}
						return;
					}
					int status_code = httpResponse.getStatusCode();
					EMLog.d(TAG, "asyncGetAgentState status_code:" + status_code);
					if (status_code >= 200 && status_code < 300) {
						StringBuffer buffer = new StringBuffer();
						httpResponse.read(buffer);
						String result = buffer.toString();
						JSONObject resultJson = new JSONObject(result);
						JSONObject entitiesJson = resultJson.getJSONObject("entity");
						String input_state_tips = entitiesJson.optString("input_state_tips");
						if (callBack != null){
							callBack.onSuccess(input_state_tips);
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
					if (callBack != null){
						callBack.onError(-1, "error:" + e.getMessage());
					}
				}

			}
		});
	}

	static void postMessagePredict(final String content, final ConversationInfo info, final ValueCallBack<String> callback) {

			sendThreadPool.execute(new Runnable() {
			@Override
			public void run() {
				if (!ChatClient.getInstance().isLoggedInBefore()) {
					if (callback != null) {
						callback.onError(-1, "no login");
					}
					return;
				}
				String tenantId = ChatClient.getInstance().tenantId();
				if(TextUtils.isEmpty(tenantId)){
					if (callback != null) {
						callback.onError(-1, "tenantId is null");
					}
					return;
				}

				try {
					String remoteUrl = String.format("%1$s/v1/webimplugin/servicesessions/%2$s/messagePredict", ChatClient.getInstance().kefuRestServer(), info.session_id);
					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
					requestBuilder.header("Authorization", "Easemob IM " + ChatClient.getInstance().accessToken());
					requestBuilder.header("Content-Type", "application/json");

					JSONObject postContent = new JSONObject();
					postContent.put("visitor_user_id", info.visitor_id);
					postContent.put("content", content);
					postContent.put("timestamp", System.currentTimeMillis());
					requestBuilder.content(postContent.toString().getBytes(), "application/json");

					HttpResponse httpResponse = requestBuilder.execute();
					assert httpResponse != null;
					int statusCode = httpResponse.getStatusCode();
					if (statusCode == HttpURLConnection.HTTP_OK) {
						StringBuffer sb = new StringBuffer();
						httpResponse.read(sb);
						if (callback != null) {
							callback.onSuccess(sb.toString());
						}
					} else {
						if (callback != null) {
							callback.onError(-1, "statusCode:" + statusCode);
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
					if (callback != null) {
						callback.onError(-1, e.getMessage());
					}
				}
			}
		});
	}

	static void asyncGetWaitCount(final ConversationInfo info, final ValueCallBack<String> callBack){
		sendThreadPool.execute(new Runnable() {
			@Override
			public void run() {
				try {
					String remoteUrl = String.format("%1$s/v1/visitors/waitings/data", ChatClient.getInstance().kefuRestServer());
					String tenantId = ChatClient.getInstance().tenantId();
					if (tenantId == null) {
						if (callBack != null) {
							callBack.onError(-1, "tenantid is null");
						}
						return;
					}

					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
					httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
					HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
					requestBuilder.param("tenantId", tenantId);
					requestBuilder.param("queueId", String.valueOf(info.skill_group_id));
					requestBuilder.param("serviceSessionId", info.session_id);
					HttpResponse httpResponse = requestBuilder.execute();
					if (httpResponse == null) {
						if (callBack != null) {
							callBack.onError(-1, "response is null");
						}
						return;
					}
					int status_code = httpResponse.getStatusCode();
					EMLog.d(TAG, "asyncGetWaitCount status_code:" + status_code);
					if (status_code >= 200 && status_code < 300) {
						StringBuffer buffer = new StringBuffer();
						httpResponse.read(buffer);
						String result = buffer.toString();
						if (callBack != null) {
							callBack.onSuccess(result);
						}
					}
				} catch (Exception ignored) {
					if (callBack != null) {
						callBack.onError(-1, "error->" + ignored);
					}
				}
			}
		});

	}

	public static synchronized OfficialAccount getSystemOfficialAccount(String conversationId) {
		String visitorId;
		String toUser = conversationId;
		if (toUser.contains(OfficialAccount.SEPARATOR)) {
			toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
		}
		synchronized (lock) {
			visitorId = KefuDBManager.getInstance().getVisitorUserId(conversationId);
			if (TextUtils.isEmpty(visitorId)) {
				visitorId = getVisitorIdFromRemote(toUser);
			}
		}
		if (TextUtils.isEmpty(visitorId)) {
			return null;
		}
		try {
			String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors/%3$s/official-accounts",
					ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), visitorId);
			String appkey = ChatClient.getInstance().appKey();
			String token = ChatClient.getInstance().accessToken();
			String currentUser = ChatClient.getInstance().currentUserName();
			HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
			httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
			httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
			HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
			assert appkey != null;
			requestBuilder.param("orgName", appkey.split("#")[0]);
			requestBuilder.param("appName", appkey.split("#")[1]);
			requestBuilder.param("userName", currentUser);
			requestBuilder.param("token", token);
			HttpResponse httpResponse = requestBuilder.execute();
			if (httpResponse == null) {
				return null;
			}
			int status_code = httpResponse.getStatusCode();
			EMLog.d(TAG, "getSystemOfficialAccount status_code:" + status_code);
			if (status_code >= 200 && status_code < 300) {
				StringBuffer buffer = new StringBuffer();
				httpResponse.read(buffer);
				String result = buffer.toString();
				JSONObject resultJson = new JSONObject(result);
				if (resultJson.has("entities") && !resultJson.isNull("entities")) {
					JSONArray entitiesJson = resultJson.getJSONArray("entities");
					for (int j = 0; j < entitiesJson.length(); j++) {
						JSONObject itemJson = entitiesJson.getJSONObject(j);
						String type = itemJson.getString("type");
						if (type != null && type.equals("SYSTEM")) {
							OfficialAccount officialAccount = new OfficialAccount();
							officialAccount.setId(itemJson.getString("official_account_id"));
							if (itemJson.has("img")){
								officialAccount.setImg(itemJson.getString("img"));
							}
							if (itemJson.has("name")){
								officialAccount.setName(itemJson.getString("name"));
							}
							officialAccount.setType("SYSTEM");
							ChatClient.getInstance().chatManager().getConversation(conversationId).setOfficialAccount(officialAccount);
							return officialAccount;
						}
					}
				}
			}
		} catch (Exception ignored) {
			ignored.printStackTrace();
		}
		return null;
	}

	static synchronized ConversationInfo getLastestSession(String officialAccountId, String conversationId){
		String visitorId;
		String toUser = conversationId;
		if (toUser.contains(OfficialAccount.SEPARATOR)){
			toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
		}
		synchronized (lock){
			visitorId = KefuDBManager.getInstance().getVisitorUserId(conversationId);
			if (TextUtils.isEmpty(visitorId)){
				visitorId = getVisitorIdFromRemote(toUser);
			}
		}
		if (TextUtils.isEmpty(visitorId)){
			return null;
		}
		try {
			String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors/%3$s/official-accounts/%4$s/latest-session",
					ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), visitorId, officialAccountId);
			String appkey = ChatClient.getInstance().appKey();
			String token = ChatClient.getInstance().accessToken();
			String currentUser = ChatClient.getInstance().currentUserName();
			HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
			httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
			httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
			HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
			assert appkey != null;
			requestBuilder.param("orgName", appkey.split("#")[0]);
			requestBuilder.param("appName", appkey.split("#")[1]);
			requestBuilder.param("userName", currentUser);
			requestBuilder.param("token", token);
			HttpResponse httpResponse = requestBuilder.execute();
			if (httpResponse == null){
				return null;
			}
			int status_code = httpResponse.getStatusCode();
			EMLog.d(TAG, "getLastestSession status_code:" + status_code);
			if (status_code >= 200 && status_code < 300) {
				StringBuffer buffer = new StringBuffer();
				httpResponse.read(buffer);
				String result = buffer.toString();
				JSONObject resultJson = new JSONObject(result);
				if (resultJson.has("entity") && !resultJson.isNull("entity")){
					JSONObject entitiesJson = resultJson.getJSONObject("entity");
					ConversationInfo info = new ConversationInfo();
					if (entitiesJson.has("session_id")){
						info.session_id = entitiesJson.getString("session_id");
					}
					if (entitiesJson.has("visitor_id")){
						info.visitor_id = entitiesJson.getString("visitor_id");
					}
					if (entitiesJson.has("skill_group_id")){
						info.skill_group_id = entitiesJson.getLong("skill_group_id");
					}
					if (entitiesJson.has("state")){
						info.state = entitiesJson.getString("state");
					}
					if (info.session_id != null){
						ChatClient.getInstance().chatManager().getConversation(conversationId).setConversationInfo(info);
						currentConversationId = conversationId;
					}
					return info;
				}
			}
		} catch (Exception ignored) {
			ignored.printStackTrace();
		}
		return null;
	}


	static void asyncOpened(final String conversationId){
		if (TextUtils.isEmpty(conversationId)){
			return;
		}
		String toUser = conversationId;
		if (toUser.contains(OfficialAccount.SEPARATOR)){
			toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
		}
		final String tempToUser = toUser;
		List<String> marketingIds = KefuDBManager.getInstance().getMarketingIdsByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_DELIVERED);
		if (marketingIds != null && !marketingIds.isEmpty()){
			KefuDBManager.getInstance().updateMarketingInfoStatusByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_DELIVERED, Constants.MESSAGE_MARKETING_STATUS_OPENED);
			for (final String marketingId : marketingIds){
				sendThreadPool.execute(new Runnable() {
					@Override
					public void run() {
						MarketingHttpClient.sendMarketingOpened(marketingId, tempToUser, new ValueCallBack<String>() {
							@Override
							public void onSuccess(String value) {
							}

							@Override
							public void onError(int error, String errorMsg) {
								String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/opened", ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), marketingId);
								RequestInfo info = new RequestInfo();
								info.key = RequestInfo.REQUEST_KEY_MARKETING;
								info.url = remoteUrl;
								info.params = tempToUser;
								KefuDBManager.getInstance().addRequestInfo(info);
							}
						});
					}
				});
			}
		}

	}

	static void asyncDelivered(final String taskId, final String conversationId) {
		if (TextUtils.isEmpty(taskId) || TextUtils.isEmpty(conversationId)) {
			return;
		}
		KefuDBManager.getInstance().addMarketingInfo(taskId, conversationId, Constants.MESSAGE_MARKETING_STATUS_DELIVERED);
		String toUser = conversationId;
		if (toUser.contains(OfficialAccount.SEPARATOR)) {
			toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
		}
		final String tempToUser = toUser;
		sendThreadPool.execute(new Runnable() {
			@Override
			public void run() {
				MarketingHttpClient.sendMarketingDelivered(taskId, tempToUser, new ValueCallBack<String>() {
					@Override
					public void onSuccess(String value) {
					}

					@Override
					public void onError(int error, String errorMsg) {
						String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/delivered", ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), taskId);
						RequestInfo info = new RequestInfo();
						info.key = RequestInfo.REQUEST_KEY_MARKETING;
						info.url = remoteUrl;
						info.params = tempToUser;
						KefuDBManager.getInstance().addRequestInfo(info);
					}
				});
			}
		});
	}

	static void asyncReplied(final String conversationId){
		if (TextUtils.isEmpty(conversationId)){
			return;
		}
		String toUser = conversationId;
		if (toUser.contains(OfficialAccount.SEPARATOR)){
			toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
		}
		final String tempToUser = toUser;
		List<String> marketingIds = KefuDBManager.getInstance().getMarketingIdsByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_OPENED);
		if (marketingIds != null && !marketingIds.isEmpty()){
			KefuDBManager.getInstance().deleteMarketingInfoByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_OPENED);
			for (final String marketingId : marketingIds){
				sendThreadPool.execute(new Runnable() {
					@Override
					public void run() {
						MarketingHttpClient.sendMarketingReplied(marketingId, tempToUser, new ValueCallBack<String>() {
							@Override
							public void onSuccess(String value) {
							}

							@Override
							public void onError(int error, String errorMsg) {
								String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/replied", ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), marketingId);
								RequestInfo info = new RequestInfo();
								info.key = RequestInfo.REQUEST_KEY_MARKETING;
								info.url = remoteUrl;
								info.params = tempToUser;
								KefuDBManager.getInstance().addRequestInfo(info);

							}
						});
					}
				});
			}
		}

	}

	static void asyncRequest() {
		try {
			List<RequestInfo> requestInfos = KefuDBManager.getInstance().getRequestInfoByKey(RequestInfo.REQUEST_KEY_MARKETING);
			if (requestInfos != null && !requestInfos.isEmpty()) {
				for (final RequestInfo info : requestInfos) {
					sendMarketingStatusChanged(info.url, info.params, new ValueCallBack<String>() {
						@Override
						public void onSuccess(String value) {
							KefuDBManager.getInstance().deleteRequestInfoById(info.id);
						}

						@Override
						public void onError(int error, String errorMsg) {

						}
					});
				}
			}
		} catch (Exception ignored) {
		}
	}


}
