package com.hyphenate.chat;


import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.hyphenate.EMError;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.helpdesk.callback.Callback;
import com.hyphenate.helpdesk.model.CompositeContent;
import com.hyphenate.helpdesk.model.Content;
import com.hyphenate.helpdesk.model.MessageHelper;
import com.hyphenate.helpdesk.model.ToCustomServiceInfo;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * 消息类
 */

public class Message implements Parcelable, Cloneable, Comparable {

    private final static String TAG = "Message";
    public final static String KEY_WEICHAT = "weichat";
    public final static String KEY_MSGTYPE = "msgtype";

    Callback messageStatusCallBack;

    Type type;


    private Direct direct;

    Status status = Status.CREATE;


    String from;

    String to;

    EMMessageBody body;

    private String msgId;

    private String imMsgId;

    boolean isAcked = false;

    boolean isDelivered = false;

    long msgTime;

    private transient int progress = 0;

    Hashtable<String, Object> attributes = new Hashtable<String, Object>();

    private transient boolean unread = false;

    boolean isListened;

    private boolean kefuReceived;

    @Override
    public int compareTo(Object another) {
        if (!(another instanceof Message)) {
            return 0;
        }
        if (msgTime == 0 || ((Message) another).messageTime() == 0) {
            return 0;
        }
        if (msgTime > ((Message) another).messageTime()) {
            return 1;
        } else if (msgTime < ((Message) another).messageTime()) {
            return -1;
        }
        return 0;
    }

    public enum Type {
        TXT, IMAGE, VIDEO, LOCATION, VOICE, FILE, CMD
    }

    public enum Direct{
        SEND, RECEIVE
    }

    public enum Status{
        SUCCESS, FAIL, INPROGRESS, CREATE
    }



    Message(Type type){
        this.type = type;
        this.msgTime = System.currentTimeMillis();
    }

    void setKefuReceived(boolean isKefuReceived){
        this.kefuReceived = isKefuReceived;
    }

    boolean kefuReceived(){
        return kefuReceived;
    }
    private int error = EMError.EM_NO_ERROR;

    /**
     * 获取消息类型
     * @return 类型
     */
    public Type getType(){
        return type;
    }

    /**
     * 获取消息body
     * @return body
     */
    public EMMessageBody body() {
        return body;
    }

    /**
     * 获取消息body
     * @return body
     * @see #body
     */
    @Deprecated
    public EMMessageBody getBody(){
        return body();
    }

    /**
     * 获取消息时间戳
     * @return 时间戳
     */
    public long messageTime() {
        return msgTime;
    }

    /**
     * 获取消息时间戳
     * @return 时间戳
     * @see #messageTime()
     */
    @Deprecated
    public long getMsgTime(){
        return messageTime();
    }

    /**
     * 设置消息时间戳
     * @param msgTime 时间戳
     */
    public void setMessageTime(long msgTime) {
        this.msgTime = msgTime;
    }

    /**
     * 设置消息时间戳
     * @param msgTime 时间戳
     * @see #setMessageTime(long)
     */
    @Deprecated
    public void setMsgTime(long msgTime){
        setMessageTime(msgTime);
    }

    /**
     * 创建一条发送消息
     * @param type 消息类型
     * @return 消息
     */
    public static Message createSendMessage(Type type){
        Message msg = new Message(type);
        msg.direct = Message.Direct.SEND;
        msg.from = ChatClient.getInstance().currentUserName();
        msg.setMsgId(UUID.randomUUID().toString());
        return msg;
    }

    /**
     * 创建一条接受消息
     * @param type 消息类型
     * @return 消息
     */
    public static Message createReceiveMessage(Type type){
        Message msg = new Message(type);
        msg.direct = Message.Direct.RECEIVE;
        msg.to = ChatClient.getInstance().currentUserName();
        return msg;
    }

    /**
     * 创建txt消息
     * @param content 内容
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createTxtSendMessage(String content, String toUserName){
        Message message = createSendMessage(Type.TXT);
        EMTextMessageBody textBody;
        if(TextUtils.isEmpty(content)){
            textBody = new EMTextMessageBody("");
        }else{
            textBody = new EMTextMessageBody(content);
        }
        message.setBody(textBody);
        message.setTo(toUserName);
        return message;
    }

    /**
     * 创建视频邀请
     * @param content 内容
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createVideoInviteSendMessage(String content, String toUserName){
        Message message = createSendMessage(Type.TXT);
        EMTextMessageBody textBody;
        if(TextUtils.isEmpty(content)){
            textBody = new EMTextMessageBody("");
        }else{
            textBody = new EMTextMessageBody(content);
        }
        message.setBody(textBody);
        message.setTo(toUserName);

        JSONObject jsonInvit = new JSONObject();
        try {
            JSONObject jsonMsg = new JSONObject();
            jsonMsg.put("msg", content);
            String appKeyStr[] = ChatClient.getInstance().appKey().split("#");
            jsonMsg.put("orgName", appKeyStr[0]);
            jsonMsg.put("appName", appKeyStr[1]);
            jsonMsg.put("userName", ChatClient.getInstance().currentUserName());
            jsonMsg.put("resource", "mobile");
            jsonMsg.put("isNewInvitation", true);
            jsonInvit.put("liveStreamInvitation", jsonMsg);
            message.setAttribute("msgtype", jsonInvit);
            message.setAttribute("type", "rtcmedia/video");
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return message;
    }

    /**
     * 创建语音消息
     * @param filePath 语音文件路径
     * @param timeLength 语音长度
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createVoiceSendMessage(String filePath, int timeLength, String toUserName){
        if (TextUtils.isEmpty(filePath)){
            EMLog.e(TAG, "voice filePath is null or empty");
            return null;
        }
        File voiceFile = new File(filePath);
        if(!(voiceFile.exists())){
            EMLog.e(TAG, "voice file does not exists");
            return null;
        }
        Message message = Message.createSendMessage(Message.Type.VOICE);
        EMVoiceMessageBody messageBody = new EMVoiceMessageBody(voiceFile, timeLength);
        message.setBody(messageBody);
        message.setTo(toUserName);
        return message;
    }

    /**
     * 创建图片消息
     * @param filePath 文件路径
     * @param sendOriginalImage 是否发送原始图片
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createImageSendMessage(String filePath, boolean sendOriginalImage, String toUserName){
        File imageFile = new File(filePath);
        if(!imageFile.exists()){
            EMLog.e(TAG, "image file does not exist");
            return null;
        }
        Message message = Message.createSendMessage(Message.Type.IMAGE);
        EMImageMessageBody messageBody = new EMImageMessageBody(imageFile);
        messageBody.setSendOriginalImage(sendOriginalImage);
        message.setBody(messageBody);
        message.setTo(toUserName);
        return message;
    }

    /**
     * 创建自定义表情消息
     * @param filePath 远端文件路基
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createCustomEmojiSendMessage(String filePath, String toUserName){
        Message message = Message.createSendMessage(Type.TXT);
        EMTextMessageBody messageBody = new EMTextMessageBody("");
        message.setBody(messageBody);
        message.setTo(toUserName);

        JSONObject jsonEmoji = new JSONObject();
        try {
            JSONObject jsonMsg = new JSONObject();
            jsonMsg.put("url", filePath);
            jsonEmoji.put("customMagicEmoji", jsonMsg);
            message.setAttribute("msgtype", jsonEmoji);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return message;
    }

    /**
     * 创建短视频消息
     * @param filePath 短视频文件路径
     * @param thumbImagePath 微缩图路径
     * @param fileLength 文件长度
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createVideoSendMessage(String filePath, String thumbImagePath, int fileLength, String toUserName){
        File videoFile = new File(filePath);
        if(!videoFile.exists()){
            EMLog.e(TAG, "video file does not exist");
            return null;
        }
        Message message = Message.createSendMessage(Message.Type.VIDEO);
        EMVideoMessageBody messageBody = new EMVideoMessageBody(filePath, thumbImagePath, fileLength, videoFile.length());
        message.setBody(messageBody);
        message.setTo(toUserName);
        return message;
    }

    /**
     * 创建位置消息
     * @param latitude 纬度
     * @param longitude 经度
     * @param locAddress 地址
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createLocationSendMessage(double latitude, double longitude, String locAddress, String toUserName){
        Message message = Message.createSendMessage(Message.Type.LOCATION);
        EMLocationMessageBody messageBody = new EMLocationMessageBody(locAddress, latitude, longitude);
        message.setBody(messageBody);
        message.setTo(toUserName);
        return message;
    }

    /**
     * 创建文件消息
     * @param filePath 文件路径
     * @param toUserName ToUserName
     * @return 消息
     */
    public static Message createFileSendMessage(String filePath, String toUserName){
        File file = new File(filePath);
        if(!file.exists()){
            EMLog.e(TAG, "file does not exist");
            return null;
        }
        Message message = Message.createSendMessage(Message.Type.FILE);
        EMNormalFileMessageBody messageBody = new EMNormalFileMessageBody(file);
        message.setBody(messageBody);
        message.setTo(toUserName);
        return message;
    }

    /**
     * 创建转接消息
     * @param toUserName ToUserName
     * @param info 转接信息
     * @return 消息
     */
    public static Message createTranferToKefuMessage(String toUserName, ToCustomServiceInfo info) {
        Message cmdMessage = Message.createSendMessage(Message.Type.CMD);
        cmdMessage.setTo(toUserName);
        EMCmdMessageBody cmdMessageBody = new EMCmdMessageBody("TransferToKf");
        cmdMessage.setBody(cmdMessageBody);
        cmdMessage.addContent(info);
        return cmdMessage;
    }

    /**
     * 设置msgbody
     * @param messageBody body
     */
    public void addBody(EMMessageBody messageBody) {
        setBody(messageBody);
    }

    /**
     * 设置msgbody
     * @param messageBody body
     */
    public void setBody(EMMessageBody messageBody) {
        this.body = messageBody;
    }

    /**
     * 获取消息源
     * @return 消息源
     */
    public String from() {
        return from;
    }

    /**
     * 获取消息源
     * @return 消息源
     * @see #from()
     */
    @Deprecated
    public String getFrom(){
        return from();
    }

    /**
     * 设置消息源
     * @param from 消息源
     */
    public void setFrom(String from){
        this.from = from;
    }

    /**
     * 设置目标源
     * @param toUserName 目标源
     */
    public void setTo(String toUserName){
        this.to = toUserName;
    }

    /**
     * 获取目标源
     * @return 目标源
     */
    public String to() {
        return to;
    }

    /**
     * 获取目标源
     * @return 目标源
     * @see #to()
     */
    @Deprecated
    public String getTo(){
        return to();
    }

    /**
     * 获取消息ID
     * @return 消息ID
     */
    public String messageId() {
        return msgId;
    }

    /**
     * 获取消息ID
     * @return 消息ID
     * @see #messageId()
     */
    @Deprecated
    public String getMsgId(){
        return messageId();
    }

    String getIMMsgId(){
        return imMsgId;
    }

    /**
     * 设置消息Id
     * @param msgId 消息Id
     */
    public void setMsgId(String msgId){
        this.msgId = msgId;
        if(imMsgId == null){
            imMsgId = msgId;
        }
    }

    void setIMMsgId(String imMsgId){
        this.imMsgId = imMsgId;
    }

    /**
     * 设置消息状态
     * @param status 状态
     */
    public void setStatus(Status status) {
        this.status = status;
    }

    /**
     * 获取消息状态
     * @return 状态
     */
    public Status status() {
        return status;
    }

    /**
     * 获取消息状态
     * @return 状态
     * @see #status()
     */
    @Deprecated
    public Status getStatus() {
        return status();
    }

    /**
     * 设置消息回调
     * @param callBack 回调
     */
    public void setMessageStatusCallback(Callback callBack){
        this.messageStatusCallBack = callBack;
    }

    public Callback messageStatusCallback(){
        return messageStatusCallBack;
    }

    public String toString(){
        String sb = "msg{from:" + from +
                ", to:" + to +
                " body:" + body.toString();
        return sb;
    }

    public void setAttribute(String attr, Object value){
        if(attributes == null){
            attributes = new Hashtable<String, Object>();
        }
        attributes.put(attr, value);
    }

    public boolean getBooleanAttribute(String attr) throws HyphenateException {
        Boolean val = null;
        if(attributes != null){
            Object test = attributes.get(attr);
            if(test instanceof Boolean){
                val = (Boolean) test;
            }else if(test instanceof Integer){
                int ival = (Integer) test;
                val = ival > 0;
            }
        }
        if(val == null){
            throw new HyphenateException("attribute " + attr + " not found");
        }
        return val;
    }

    public boolean getBooleanAttribute(String attr, boolean defaultVal) {
        Boolean val = null;
        try {
            val = getBooleanAttribute(attr);
        } catch (HyphenateException ignored) {
        }
        if (val == null) {
            return defaultVal;
        }
        return val;
    }

    public int getIntAttribute(String attr, int defaultVal) {
        Object val = null;
        if (attributes != null) {
            val = attributes.get(attr);
        }
        if (val == null) {
            return defaultVal;
        }
        if (val instanceof Integer) {
            return (Integer) val;
        } else if (val instanceof Long) {
            long lVal = (Long) val;
            if (lVal <= Integer.MAX_VALUE && lVal >= Integer.MIN_VALUE) {
                return (int) lVal;
            }
        }
        return defaultVal;
    }

    public long getLongAttribute(String attr, long defaultVal){
        Object val = null;
        if(attributes != null){
            val = attributes.get(attr);
        }
        if(val == null){
            return defaultVal;
        }
        if(val instanceof Integer){
            return (Integer) val;
        }else if(val instanceof Long){
            return (Long) val;
        }
        return defaultVal;
    }

    public int getIntAttribute(String attr) throws HyphenateException{
        Object val = null;
        if(attributes != null){
            val = attributes.get(attr);
        }
        if(val == null){
            throw new HyphenateException("attribute " + attr + " not found");
        }
        if(val instanceof Integer){
            return (Integer) val;
        }else if(val instanceof Long){
            long lVal = (Long) val;
            if(lVal <= Integer.MAX_VALUE && lVal >= Integer.MIN_VALUE){
                return (int)lVal;
            }
        }
        throw new HyphenateException("Value of attribute " + attr + " to int fail");
    }

    public long getLongAttribute(String attr) throws HyphenateException{
        Object val = null;
        if(attributes != null){
            val = attributes.get(attr);
        }
        if(val == null){
            throw new HyphenateException("attribute " + attr + " not found");
        }
        if(val instanceof Integer){
            return (Integer) val;
        }else if(val instanceof Long){
            return (Long) val;
        }
        throw new HyphenateException("Value of attribute " + attr + " to long fail");
    }

    public String getStringAttribute(String attr) throws HyphenateException{
        if(attributes != null && attributes.containsKey(attr)){
            Object val = attributes.get(attr);
            if(val instanceof String){
                return (String)val;
            }else if(val instanceof JSONObject){
                return ((JSONObject)val).toString();
            }else if(val instanceof JSONArray){
                return ((JSONArray)val).toString();
            }else{
                throw new HyphenateException("attribute " + attr + " not String type");
            }
        }
        throw new HyphenateException("attribute " + attr + " not found");
    }

    public String getStringAttribute(String attr, String defaultVal){
        if(attributes != null && attributes.containsKey(attr)){
            Object val = attributes.get(attr);
            if(val instanceof String){
                return (String)val;
            }else if(val instanceof JSONObject){
                return ((JSONObject)val).toString();
            }else if(val instanceof JSONArray){
                return ((JSONArray)val).toString();
            }
        }
        return defaultVal;
    }

    /**
     * 判断消息是否为通知消息
     * @return 是或否
     */
    @Deprecated
    public boolean isNotificationMessage() {
        try {
            JSONObject weichatJson = getJSONObjectAttribute("weichat");
            if (weichatJson != null && weichatJson.has("notification")) {
                return weichatJson.getBoolean("notification");
            }
        } catch (Exception e) {
//            e.printStackTrace();
        }
        return false;
    }


    public JSONObject getJSONObjectAttribute(String attr) throws HyphenateException{
        if(attributes != null){
            if(attributes.containsKey(attr)){
                Object val = attributes.get(attr);
                if(val instanceof JSONObject){
                    return ((JSONObject)val);
                }else if(val instanceof String){
                    try{
                        return new JSONObject((String)val);
                    }catch(JSONException ignored){}

                }
                throw new HyphenateException("attribute " + attr + " not JSONObject type");
            }
        }
        throw new HyphenateException("attribute " + attr + "  not found");
    }

    public JSONArray getJSONArrayAttribute(String attr) throws HyphenateException{
        if(attributes != null){
            if(attributes.containsKey(attr)){
                Object val = attributes.get(attr);
                if(val instanceof JSONArray){
                    return (JSONArray)val;
                }else if(val instanceof String){
                    try{
                        return new JSONArray((String)val);
                    }catch(JSONException ignored){}
                }
                throw new HyphenateException("attribute " + attr + " does not JSONArray type");
            }
        }
        throw new HyphenateException("attribute " + attr + " not found.");

    }




    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(type.name());
        dest.writeString(direct.name());
        dest.writeString(msgId);
        dest.writeString(imMsgId);
        dest.writeLong(msgTime);
        dest.writeString(from);
        dest.writeString(to);
        dest.writeParcelable(body, flags);

        List<Pair<String, JSONObject>> jsonObjects = new ArrayList<Pair<String, JSONObject>>();
        List<Pair<String, JSONArray>> jsonArrays  = new ArrayList<Pair<String, JSONArray>>();
        Hashtable<String, Object> tempAttrs = new Hashtable<String, Object>();
        for(Map.Entry<String, Object> entry : attributes.entrySet()){
            if (entry.getValue() == null) {
                continue;
            }
            if (entry.getValue() instanceof JSONObject) {
                jsonObjects.add(Pair.create(entry.getKey(), (JSONObject)entry.getValue()));
            } else if (entry.getValue() instanceof JSONArray) {
                jsonArrays.add(Pair.create(entry.getKey(), (JSONArray)entry.getValue()));
            } else {
                tempAttrs.put(entry.getKey(), entry.getValue());
            }
        }
        dest.writeInt(jsonObjects.size());
        for (Pair<String, JSONObject> pair : jsonObjects) {
            dest.writeString(pair.first);
            dest.writeString(pair.second.toString());
        }
        dest.writeInt(jsonArrays.size());
        for (Pair<String, JSONArray> pair : jsonArrays) {
            dest.writeString(pair.first);
            dest.writeString(pair.second.toString());
        }
        dest.writeMap(tempAttrs);
    }

    public static final Parcelable.Creator<Message> CREATOR = new Parcelable.Creator<Message>() {

        @Override
        public Message createFromParcel(Parcel source) {
            return new Message(source);
        }

        @Override
        public Message[] newArray(int size) {
            return new Message[size];
        }
    };

    private Message(Parcel source){
        type = Type.valueOf(source.readString());
        direct = Direct.valueOf(source.readString());
        msgId = source.readString();
        imMsgId = source.readString();
        msgTime = source.readLong();
        from = source.readString();
        to = source.readString();
        body = source.readParcelable(Message.class.getClassLoader());

        attributes = new Hashtable<String, Object>();

        int jsonObjectCount = source.readInt();
        for (int i  = 0; i < jsonObjectCount; i++) {
            String key = source.readString();
            JSONObject jsonObject = null;
            try {
                jsonObject = new JSONObject(source.readString());
            } catch (JSONException e) {
                e.printStackTrace();
            }
            attributes.put(key, jsonObject);
        }
        int jsonArrayCount = source.readInt();
        for (int i  = 0; i < jsonArrayCount; i++) {
            String key = source.readString();
            JSONArray jsonArray = null;
            try {
                jsonArray = new JSONArray(source.readString());
            } catch (JSONException e) {
                e.printStackTrace();
            }
            attributes.put(key, jsonArray);
        }
        Hashtable<String, Object> tempAttrs = new Hashtable<String, Object>();
        //extra parameters
        source.readMap(tempAttrs, null);
        attributes.putAll(tempAttrs);
    }

    @Override
    protected Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    boolean isUnread() {
        return unread;
    }

    void setUnread(boolean unread) {
        this.unread = unread;
    }

    /**
     * 设置消息类型
     * @param type 类型
     */
    public void setType(Type type) {
        this.type = type;
    }

    /**
     * 获取语音消息状态
     * @return true为已读 false为未读
     */
    public boolean isListened() {
        return isListened;
    }

    /**
     * 这只语音消息状态
     * @param isListened 已读或未读
     */
    public void setListened(boolean isListened) {
        this.isListened = isListened;
    }

    /**
     * 获取消息方向
     * @return 方向
     */
    public Direct direct(){
        return direct;
    }

    void setDirection(Direct dir) {
        this.direct = dir;
    }

    public String getUserName(){
        String username;
        if(from() != null && from().equals(ChatClient.getInstance().currentUserName())){
            username = to();
        }else{
            username = from();
        }
        return username;
    }

    public void setError(int error) {
        this.error = error;
    }

    public int error() {
        return error;
    }

    @Deprecated
    public int getError() {
        return error();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public Map<String, Object> ext(){
        return attributes;
    }


    public synchronized void addContent(Content content) {
        if (content == null)
            return;

        JSONObject container = MessageHelper.getContainerObject(this, content.getParentName());
        if (container == null){
            container = new JSONObject();
        }
        try {
            if (content.getString() == null) {
                if (content.getName() != null){
                    container.put(content.getName(), content.getContent());
                }
            } else {
                if (content.getName() != null) {
                    container.put(content.getName(), content.getString());
                }
            }
            if (content.getParentName() != null){
                setAttribute(content.getParentName(), container);
            }
        } catch (JSONException ex) {
            ex.printStackTrace();
            Log.e(TAG, ex.getMessage());
        }
    }

    public synchronized void addContent(CompositeContent compositeContent) {
        Collection<Content> contents = compositeContent.getContents();
        if (contents != null && contents.size() > 0){
            for (Content c : contents) {
                addContent(c);
            }
        }
    }

    OfficialAccount getOfficialAccount(){
        if (attributes != null && attributes.containsKey(KEY_WEICHAT)){
            try {
                JSONObject jsonWeiChat = getJSONObjectAttribute(KEY_WEICHAT);
                if (jsonWeiChat != null && jsonWeiChat.has(OfficialAccount.KEY_OFFICIAL_ACCOUNT)){
                    JSONObject jsonOffAccount = jsonWeiChat.getJSONObject(OfficialAccount.KEY_OFFICIAL_ACCOUNT);
                    OfficialAccount officialAccount = new OfficialAccount();
                    String id = jsonOffAccount.getString(OfficialAccount.KEY_ID);
                    officialAccount.setId(id);
                    if (jsonOffAccount.has(OfficialAccount.KEY_NAME)){
                        String name = jsonOffAccount.getString(OfficialAccount.KEY_NAME);
                        officialAccount.setName(name);
                    }
                    if (jsonOffAccount.has(OfficialAccount.KEY_TYPE)){
                        String type = jsonOffAccount.getString(OfficialAccount.KEY_TYPE);
                        officialAccount.setType(type);
                    }
                    if (jsonOffAccount.has(OfficialAccount.KEY_IMG)){
                        String img = jsonOffAccount.getString(OfficialAccount.KEY_IMG);
                        officialAccount.setImg(img);
                    }
                    return officialAccount;
                }
            } catch (Exception ignored) {
            }
        }

        return null;
    }

    String getMarketingTaskId() {
        String marketings = getMarketings();
        if (marketings != null) {
            try {
                JSONObject jsonMarketing = new JSONObject(marketings);
                return jsonMarketing.getString(OfficialAccount.KEY_MARKETING_TASK_ID);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    String getMarketings() {
        if (attributes != null && attributes.containsKey(KEY_WEICHAT)) {
            try {
                JSONObject jsonWeiChat = getJSONObjectAttribute(KEY_WEICHAT);
                if (jsonWeiChat != null && jsonWeiChat.has(OfficialAccount.KEY_MARKETING)) {
                    JSONObject jsonMarketing = jsonWeiChat.getJSONObject(OfficialAccount.KEY_MARKETING);
                    if (jsonMarketing != null && jsonMarketing.has(OfficialAccount.KEY_MARKETING_TASK_ID)) {
                        return jsonMarketing.toString();
                    }
                }
            } catch (Exception ignored) {
            }
        }
        return null;
    }

    @Deprecated
    public int getProgress() {
        return progress;
    }

    @Deprecated
    public void setProgress(int progress) {
        this.progress = progress;
    }

}
