package com.hyphenate.helpdesk.httpclient;

import android.content.Context;
import android.os.Build;

import com.hyphenate.chat.ChatClient;

import java.util.HashMap;
import java.util.Map;

/**
 * The Http client is responsible for sending Http requests.
 */
public final class HttpClient {
    static {
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.FROYO){
            //Disable connection pooling before Proyo:
            // http://stackoverflow.com/a/4261005/422906
            System.setProperty("http.keepAlive", "false");
        }
    }

    private static final String DEFAULT_USER_AGENT = getDefaultUserAgent();
    private final Context context;
    private int connectTimeout;
    private int readTimeout;
    private String userAgent;
    private final Map<String, String> inMemoryCookies = new HashMap<String, String>(8);

    /**
     * Create a new instance for this {@link Context}
     */
    public HttpClient(final Context context){
        if (context == null){
            throw new IllegalArgumentException("Context cannot be null");
        }
        this.context = context;
    }

    Context getContext(){
        return context;
    }

    /**
     * Prepare a new request with the request method <code>GET</code>
     */
    public HttpRequestBuilder get(String uri){
        if (uri == null){
            throw new IllegalArgumentException("URI cannot be null");
        }
        return new HttpRequestBuilder(this, uri, Constants.HTTP_GET);
    }

    /**
     * Prepare a new request with the request method <code>HEAD</code>
     */
    public HttpRequestBuilder header(String uri){
        if (uri == null){
            throw new IllegalArgumentException("URI cannot be null");
        }
        return new HttpRequestBuilder(this, uri, Constants.HTTP_HEAD);
    }


    /**
     * Prepare a new request with the request method <code>POST</code>
     */
    public HttpRequestBuilder post(String uri){
        if (uri == null){
            throw new IllegalArgumentException("URI cannot be null");
        }
        return new HttpRequestBuilder(this, uri, Constants.HTTP_POST);
    }

    /**
     * Prepare a new request with the request method <code>PUT</code>
     */
    public HttpRequestBuilder put(String uri){
        if (uri == null){
            throw new IllegalArgumentException("URI cannot be null");
        }
        return new HttpRequestBuilder(this, uri, Constants.HTTP_PUT);
    }


    /**
     * Prepare a new request with the request method <code>DELETE</code>
     */
    public HttpRequestBuilder delete(String uri){
        if (uri == null){
            throw new IllegalArgumentException("URI cannot be null");
        }
        return new HttpRequestBuilder(this, uri, Constants.HTTP_DELETE);
    }


    /**
     * Get the connect timeout in seconds.
     */
    public int getConnectTimeout(){
        return connectTimeout;
    }

    /**
     * Set the connect timeout in seconds.
     */
    public void setConnectTimeout(int connectTimeout){
        if (connectTimeout < 0){
            throw new IllegalArgumentException("Invalid connect timeout:" + connectTimeout);
        }
        this.connectTimeout = connectTimeout;
    }

    /**
     * Get the read timeout in seconds.
     */
    public int getReadTimeout(){
        return readTimeout;
    }


    /**
     * Set the read timeout in seconds.
     */
    public void setReadTimeout(int readTimeout){
        if (readTimeout < 0){
            throw new IllegalArgumentException("Invalid read timeout:" + readTimeout);
        }
        this.readTimeout = readTimeout;
    }

    /**
     * Get the user agent sent with every request.
     */
    public String getUserAgent(){
        if (userAgent == null){
            return DEFAULT_USER_AGENT;
        }
        return userAgent;
    }

    /**
     * Set the user agent sent with every request.
     */
    public void setUserAgent(String userAgent){
        this.userAgent = userAgent;
    }

    Map<String, String> getInMemoryCookies(){
        return inMemoryCookies;
    }


    /**
     * Get the default Http User Agent for this client.
     */
    private static String getDefaultUserAgent(){
        return "Easemob-Android V" + ChatClient.getInstance().sdkVersion() + "(" + Build.MANUFACTURER + " " + Build.MODEL + "; Android "
                + Build.VERSION.RELEASE + "/" + Build.VERSION.SDK_INT + ")";
    }







}
