package com.hyphenate.helpdesk.httpclient;

import android.os.Build;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * When the Http server receives a request from a client, the Http response is sent back.
 */
public final class HttpResponse {

    private final int statusCode;
    private final Map<String, String> cookies;
    private final Map<String, List<String>> headers;
    private InputStream payload;

    HttpResponse(final int statusCode, final InputStream payload, final Map<String, List<String>> rawHeaders, final Map<String, String> cookies) {
        this.statusCode = statusCode;
        this.payload = payload;
        this.cookies = Collections.unmodifiableMap(cookies);

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.GINGERBREAD) {
            // Before Gingerbread, Android has a bug where all headers are
            // stored in lower-case:
            // http://code.google.com/p/android/issues/detail?id=6684
            final Map<String, List<String>> newHeaders = new HashMap<String, List<String>>(rawHeaders.size());
            for (final Map.Entry<String, List<String>> e : rawHeaders.entrySet()) {
                final String key = e.getKey();
                if (key == null){
                    continue;
                }
                final int keyLen = key.length();
                final StringBuilder newKey = new StringBuilder(keyLen);
                for (int i = 0; i < keyLen; ++i) {
                    final char c = key.charAt(i);
                    final char c2;
                    if (i == 0 || key.charAt(i - 1) == '-') {
                        c2 = Character.toUpperCase(c);
                    } else {
                        c2 = c;
                    }
                    newKey.append(c2);
                }
                newHeaders.put(newKey.toString(), e.getValue());
            }
            this.headers = Collections.unmodifiableMap(newHeaders);
        } else {
            this.headers = Collections.unmodifiableMap(rawHeaders);
        }
    }


    /**
     * Get the content type for this response, or <code>null</code> if unknown.
     *
     * @return 类型
     */
    public String getContentType() {
        final String contentType = getFirstHeaderValue("Content-Type");
        if (contentType == null) {
            return null;
        }
        final int i = contentType.indexOf(';');
        return i == -1 ? contentType : contentType.substring(0, i).trim();
    }


    /**
     * Get the charset for this response, or <code>null</code> if unknown.
     */
    public String getContentCharset() {
        final String contentType = getFirstHeaderValue("Content-Type");
        if (contentType == null) {
            return null;
        }
        final int i = contentType.indexOf('=');
        return i == -1 ? null : contentType.substring(i + 1).trim();
    }


    /**
     * Get the response payload
     *
     * @return 流
     */
    public InputStream getPayload() {
        return payload;
    }

    void preload(File temp) throws IOException {
        FileOutputStream out = null;
        try {
            out = new FileOutputStream(temp);
            final byte[] inBuf = new byte[1024];
            final InputStream in = getPayload();
            for (int bytesRead = 0; (bytesRead = in.read(inBuf)) != -1; ) {
                out.write(inBuf, 0, bytesRead);
            }
            payload = new FileInputStream(temp);
        } catch (IOException e) {
            throw e;
        }finally {
            if (out != null){
                out.close();
            }
        }
    }

    public String getResponseBody() throws IOException{
        String enc = getContentCharset();
        if (enc == null){
            enc = "UTF-8";
        }
        InputStream input = getPayload();
        InputStreamReader inputStreamReader = new InputStreamReader(input, enc);
        BufferedReader bufferedReader = new BufferedReader(inputStreamReader);

        StringBuilder stringBuilder = new StringBuilder();
        String bufferedStrChunk = null;

        while ((bufferedStrChunk = bufferedReader.readLine()) != null){
            stringBuilder.append(bufferedStrChunk);
        }

        bufferedReader.close();

        inputStreamReader.close();
        return stringBuilder.toString();
    }


    public void read(StringBuffer buffer) throws IOException {
        String enc = getContentCharset();
        if (enc == null) {
            enc = "UTF-8";
        }
        final InputStream input = getPayload();
        final InputStreamReader reader = new InputStreamReader(input, enc);
        final char[] inBuf = new char[64];
        for (int charsRead; (charsRead = reader.read(inBuf)) != -1; ) {
            buffer.append(inBuf, 0, charsRead);
        }
    }

    /**
     * Get the response status code.
     *
     * @return status code
     */
    public int getStatusCode() {
        return statusCode;
    }

    /**
     * Get the response headers.
     *
     * @return header map
     */
    public Map<String, List<String>> getHeaders() {
        return headers;
    }


    /**
     * Get the first header value, or <code>null</code> if unset.
     *
     * @param name
     * @return headerValue
     */
    public String getFirstHeaderValue(String name) {
        final List<String> values = headers.get(name);
        if (values == null || values.isEmpty()) {
            return null;
        }
        return values.get(0);
    }

    /**
     * Get the response cookies.
     *
     * @return cookie map
     */
    public Map<String, String> getCookies() {
        return cookies;
    }


}
