package com.hyphenate.helpdesk.model;


import android.text.TextUtils;

import com.hyphenate.chat.ChatClient;
import com.hyphenate.chat.EMCmdMessageBody;
import com.hyphenate.chat.Message;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.helpdesk.callback.Callback;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

public class MessageHelper {
    private static final String TAG = "MessageHelper";
    private static final boolean DEBUG = false;
    public static final String TAG_ROOT = "ext";
    public static final String TAG_MSGTYPE = "msgtype";
    public static final String TAG_WEICHAT = "weichat";

    static final String MESSAGE_ATTR_IS_BIG_EXPRESSION = "em_is_big_expression";

    static final String TRANSFER_INDICATION_ACTION = "transfer";

    public enum ExtMsgType {
        EvaluationMsg, OrderMsg, TrackMsg, FormMsg,
        RobotMenuMsg, ArticlesMsg, ToCustomServiceMsg, BigExpressionMsg,
        CustomEmojiMsg,
	    GeneralMsg
    }

    public static ExtMsgType getMessageExtType(Message message) {
        if(getEvalRequest(message) != null) {
            return ExtMsgType.EvaluationMsg;
        }

        if(getOrderInfo(message) != null) {
            return ExtMsgType.OrderMsg;
        }

        if(getVisitorTrack(message) != null) {
            return ExtMsgType.TrackMsg;
        }

        if (isFormMessage(message)) {
            return ExtMsgType.FormMsg;
        }

        if(getRobotMenu(message) != null) {
            return ExtMsgType.RobotMenuMsg;
        }

        if (isArticlesMessage(message)) {
            return ExtMsgType.ArticlesMsg;
        }
        if (message.getBooleanAttribute(MESSAGE_ATTR_IS_BIG_EXPRESSION, false)) {
            return ExtMsgType.BigExpressionMsg;
        }

        if (!TextUtils.isEmpty(getCustomEmojiMessage(message))) {
            return ExtMsgType.CustomEmojiMsg;
        }
        if (getToCustomServiceInfo(message) != null) {
            return ExtMsgType.ToCustomServiceMsg;
        }

        return ExtMsgType.GeneralMsg;
    }

    public static JSONObject getContainerObject(Message message, String tag) {
        JSONObject obj = null;
        try {
            if (message != null)
                obj = message.getJSONObjectAttribute(tag);
        } catch (HyphenateException e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
        return obj;

    }

    public static EvaluationInfo getEvalRequest(Message message) {
        EvaluationInfo info = null;
        JSONObject jsonObject = getContainerObject(message, EvaluationInfo.PARENT_NAME);
        if (jsonObject != null) {
            String ctrlType = getString(jsonObject, EvaluationInfo.NAME);
            if (ctrlType != null && ctrlType.equals(EvaluationInfo.VALUE)) {
                JSONObject content = getJSONObject(jsonObject, EvaluationInfo.ARGS);
                if (content != null) {
                    info = ContentFactory.createEvaluationInfo(content);
                }
            }

        }
        return info;
    }

    public static RobotMenuInfo getRobotMenu(Message message) {
        RobotMenuInfo menuInfo = null;
        JSONObject content = extractObject(message, RobotMenuInfo.PARENT_NAME, RobotMenuInfo.NAME, false);
        if (content != null)
            menuInfo = ContentFactory.createRobotMenuInfo(content);
        if (menuInfo != null){
            if (menuInfo.has(RobotMenuInfo.NAME_ITEMS) || menuInfo.has(RobotMenuInfo.NAME_LIST)){
                return menuInfo;
            }
        }
        return null;
    }

    public static TransferIndication getTransferIndication(Message message) {
        TransferIndication indication = null;
        if (hasAction(message, TRANSFER_INDICATION_ACTION)) {
            JSONObject content = extractObject(message, TransferIndication.PARENT_NAME, null, true);
            if (content != null) {
                indication = ContentFactory.createTransferIndication(content);
                if (indication.isNull())
                    indication = null;
            }
        }
        return indication;

    }

    public static AgentInfo getAgentInfo(Message message) {
        AgentInfo agentInfo = null;
        JSONObject content = extractObject(message, AgentInfo.PARENT_NAME, AgentInfo.NAME, false);
        if (content != null)
            agentInfo = ContentFactory.createAgentInfo(content);
        return agentInfo;
    }

    public static ControlMessage getTransferToAgent(Message message) {
        return null;
    }

    public static OrderInfo getOrderInfo(Message message) {
        OrderInfo orderInfo = null;
        JSONObject content = extractObject(message, OrderInfo.PARENT_NAME, OrderInfo.NAME, false);
        if (content != null)
            orderInfo = ContentFactory.createOrderInfo(content);
        if (orderInfo != null){
            if (orderInfo.getPrice() == null && orderInfo.getDesc() == null){
                orderInfo = null;
            }
        }
        return orderInfo;
    }

    public static VisitorTrack getVisitorTrack(Message message) {
        VisitorTrack visitorTrack = null;
        JSONObject content = extractObject(message, VisitorTrack.PARENT_NAME, VisitorTrack.NAME, false);
        if (content != null)
            visitorTrack = ContentFactory.createVisitorTrack(content);
        if (visitorTrack != null){
            if (visitorTrack.getDesc() == null && visitorTrack.getPrice() == null){
                visitorTrack = null;
            }
        }
        return visitorTrack;
    }

    private static boolean hasAction(Message msg, String action) {
        if (msg.getType() == Message.Type.CMD) {
            EMCmdMessageBody body = (EMCmdMessageBody) msg.body();
            if (body.action().equals(action))
                return true;
        }
        return false;
    }

    private static JSONObject extractObject(Message message, String parentName, String name, boolean isComposite) {
        JSONObject content = null;
        JSONObject container = getContainerObject(message, parentName);
        if (!isComposite) {
            if (container != null) {
                if (container.isNull(name)) {
                    return null;
                }
                try {
                    content = container.getJSONObject(name);
                } catch (JSONException ex) {
                    if (DEBUG) {
                        ex.printStackTrace();
                    }
                }
            }
            return content;
        } else {
            return container;
        }

    }

    @SuppressWarnings("unused")
    private static String extractString(Message message, String parentName, String name) {
        String content = null;
        JSONObject container = getContainerObject(message, parentName);
        if (container != null) {
            try {
                content = container.getString(name);
            } catch (JSONException ex) {
                if (DEBUG) {
                    ex.printStackTrace();
                }
            }
        }
        return content;

    }


    public static ToCustomServiceInfo getToCustomServiceInfo(Message message) {
        ToCustomServiceInfo info = null;
        JSONObject jsonObject = getContainerObject(message, ToCustomServiceInfo.PARENT_NAME);
        if (jsonObject != null) {
            String ctrlType = getString(jsonObject, ToCustomServiceInfo.TYPE);
            if (ctrlType != null && ctrlType.equals(ToCustomServiceInfo.TYPE_VALUE)) {
                JSONObject content = getJSONObject(jsonObject, ToCustomServiceInfo.NAME);
                if (content != null) {
                    info = ContentFactory.createToCustomeServiceInfo(content);
                }
            }

        }
        return info;
    }

    public static ArticlesInfo getArticlesMessage(Message message) {
	    ArticlesInfo articlesInfo = null;
		JSONObject content = extractObject(message, ArticlesInfo.PARENT_NAME, null, true);
		if (content != null) {
			articlesInfo = ContentFactory.createArticlesInfo(content);
		}
	    return articlesInfo;
    }

    public static boolean isArticlesMessage(Message message) {
        try {
            JSONObject msgtype = message.getJSONObjectAttribute(ArticlesInfo.PARENT_NAME);
            if (msgtype != null) {
                JSONArray articles = msgtype.getJSONArray(ArticlesInfo.ITEM_NAME);
                if (articles != null) {
                    return true;
                }
            }
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
        return false;
    }

    public static String getCustomEmojiMessage(Message message) {
        try {
            JSONObject msgtype = message.getJSONObjectAttribute(MessageHelper.TAG_MSGTYPE);
            if (msgtype != null) {
                JSONObject customEmoji = msgtype.getJSONObject("customMagicEmoji");
                if (customEmoji != null && customEmoji.has("url")) {
                    return customEmoji.getString("url");
                }
            }
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
        return "";
    }

	public static FormInfo getFormMessage(Message message) {
		FormInfo formInfo = null;
		JSONObject content = extractObject(message, FormInfo.PARENT_NAME, FormInfo.NAME, false);
		if (content != null)
			formInfo = ContentFactory.createFormInfo(content);
		return formInfo;
	}

    public static boolean isFormMessage(Message message){
        if (message.getStringAttribute("type", null) != null){
            try {
                String type = message.getStringAttribute("type");
                if (type.equals("html/form")){
                    return true;
                }
            } catch (HyphenateException e) {
                if (DEBUG) {
                    e.printStackTrace();
                }
            }
        }

        return false;
    }

    public static boolean isNotificationMessage(Message message) {
        try {
            JSONObject weichatJson = message.getJSONObjectAttribute("weichat");
            if (weichatJson != null && weichatJson.has("notification")) {
                return weichatJson.getBoolean("notification");
            }
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
        return false;
    }

    public static boolean isTicketStatusChangedMessage(Message message) {
        try {
            JSONObject weichatJson = message.getJSONObjectAttribute("weichat");
            if (weichatJson != null && weichatJson.has("event")) {
                JSONObject eventJson = weichatJson.getJSONObject("event");
                if (eventJson != null && eventJson.has("eventName")) {
                    String eventName = eventJson.getString("eventName");
                    if (!TextUtils.isEmpty(eventName)){
                        if (eventName.equals("TicketStatusChangedEvent") || eventName.equals("CommentCreatedEvent")){
                            return true;
                        }
                    }
                }
            }
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
        return false;
    }


    private static JSONObject getJSONObject(JSONObject jsonObject, String name) {
        if (jsonObject.has(name) && !jsonObject.isNull(name)) {
            try {
                return jsonObject.getJSONObject(name);
            } catch (JSONException e) {
                if (DEBUG) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    private static String getString(JSONObject jsonObj, String name) {
        if (jsonObj.has(name) && !jsonObj.isNull(name)) {
            try {
                return jsonObj.getString(name);
            } catch (JSONException e) {
                if (DEBUG) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    public static void sendEvalMessage(Message message, String detail, EvaluationInfo.Degree degree, List<EvaluationInfo.TagInfo> selectedTags, final Callback callback){
        EvaluationInfo evaluationInfo = getEvalRequest(message);
        ControlMessage content = ContentFactory.createEvaluationResponse(evaluationInfo, detail, degree.getLevel(), degree.getId(), selectedTags);
        final Message messageToSend = Message.createTxtSendMessage("", message.from());
        messageToSend.addContent(content);
        ChatClient.getInstance().chatManager().sendMessage(messageToSend, new Callback() {
            @Override
            public void onSuccess() {
                ChatClient.getInstance().chatManager().getConversation(messageToSend.to()).removeMessage(messageToSend.messageId());
                if (callback != null) {
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int code, String error) {
                ChatClient.getInstance().chatManager().getConversation(messageToSend.to()).removeMessage(messageToSend.messageId());
                if (callback != null) {
                    callback.onError(code, error);
                }
            }

            @Override
            public void onProgress(int progress, String status) {

            }
        });
    }


    public static void sendEvalMessage(String msgId, String summary, String detail, final Callback callback) {
        Message message = ChatClient.getInstance().chatManager().getMessage(msgId);
        EvaluationInfo evaluationInfo = getEvalRequest(message);
        int level = 5;
        try {
            level = Integer.parseInt(summary);
        } catch (NumberFormatException e) {
            try{
                level = (int) Float.parseFloat(summary);
            }catch (NumberFormatException err){
                err.printStackTrace();
            }
        }
        ControlMessage content = ContentFactory.createEvaluationResponse(evaluationInfo, detail, level, -1, null);
        final Message messageToSend = Message.createTxtSendMessage("", message.from());
        messageToSend.addContent(content);
        ChatClient.getInstance().chatManager().sendMessage(messageToSend, new Callback() {
            @Override
            public void onSuccess() {
                ChatClient.getInstance().chatManager().getConversation(messageToSend.to()).removeMessage(messageToSend.messageId());
                if (callback != null) {
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int code, String error) {
                ChatClient.getInstance().chatManager().getConversation(messageToSend.to()).removeMessage(messageToSend.messageId());
                if (callback != null) {
                    callback.onError(code, error);
                }
            }

            @Override
            public void onProgress(int progress, String status) {

            }
        });
    }

}