package com.hyphenate.chat;


import android.text.TextUtils;


import com.hyphenate.util.EMLog;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 会话类
 */
public class Conversation  {
    private static final String TAG = "conversation";

    List<Message> messages;

    private int unreadMsgCount = 0;

    private String conversationId;

    private OfficialAccount mOfficialAccount;

    private String mMarketings;

    private ConversationInfo conversationInfo;


    //该会话全部的消息数目
    private long msgCount = 0;

    public Conversation(String conversationId){
        this.conversationId = conversationId;
        if(messages == null){
            messages = Collections.synchronizedList(new ArrayList<Message>());
        }
        if(unreadMsgCount <= 0){
            unreadMsgCount = KefuDBManager.getInstance().getConversationUnreadCount(conversationId);
        }
    }

    Conversation(String conversationId, List<Message> msgList, long msgCount){
        this.conversationId = conversationId;
        if(messages == null){
            messages = Collections.synchronizedList(msgList);
        }
        if (unreadMsgCount <= 0){
            unreadMsgCount = KefuDBManager.getInstance().getConversationUnreadCount(conversationId);
        }
        this.msgCount = msgCount;
    }

     void setOfficialAccount(OfficialAccount officialAccount){
        this.mOfficialAccount = officialAccount;
    }

    void clearOfficialAccount(){
        mOfficialAccount = null;
    }
    void setOfficialAccount(String strOfficialAccount){
        if (strOfficialAccount == null){
            this.mOfficialAccount = null;
            return;
        }
        try{
            JSONObject jsonObject = new JSONObject(strOfficialAccount);
            OfficialAccount officialAccount = null;
            if (jsonObject.has(OfficialAccount.KEY_ID)){
                String id = jsonObject.getString(OfficialAccount.KEY_ID);
                officialAccount = new OfficialAccount();
                officialAccount.setId(id);
            }
            if (jsonObject.has(OfficialAccount.KEY_TYPE)){
                String type = jsonObject.getString(OfficialAccount.KEY_TYPE);
                assert officialAccount != null;
                officialAccount.setType(type);
            }
            if (jsonObject.has(OfficialAccount.KEY_NAME)){
                String name = jsonObject.getString(OfficialAccount.KEY_NAME);
                assert officialAccount != null;
                officialAccount.setName(name);
            }
            if (jsonObject.has(OfficialAccount.KEY_IMG)){
                assert officialAccount != null;
                officialAccount.setImg(jsonObject.getString(OfficialAccount.KEY_IMG));
            }
            this.mOfficialAccount = officialAccount;
        }catch (Exception e){
            e.printStackTrace();
            this.mOfficialAccount = null;
        }

    }

    void setConversationInfo(ConversationInfo info){
        this.conversationInfo = info;
    }

    ConversationInfo getConversationInfo(){
        return this.conversationInfo;
    }


    void setMarketings(String marketings){
        this.mMarketings = marketings;
    }

    String getMarketings(){
        return mMarketings;
    }

    public OfficialAccount officialAccount() {
        return mOfficialAccount;
    }

    @Deprecated
    public OfficialAccount getOfficialAccount(){
        return officialAccount();
    }

    /**
     * 获取会话id
     * @return 会话id
     */
    public String conversationId(){
        return conversationId;
    }

    /**
     * 设置会话id
     * @param conId 会话id
     */
    public void setConversationId(String conId){
        this.conversationId = conId;
    }

    /**
     * 会话中增加一条消息
     * @param message 消息
     */
    public void addMessage(Message message){
        addMessage(message, true);
    }

    void addMessage(Message message, boolean unreadCountIncrease){
        if(messages.size() > 0){
            Message previousMsg = messages.get(messages.size() - 1);
            if(message.messageId() != null && previousMsg.messageId() != null
                    && message.messageId().equals(previousMsg.messageId())){
                return;
            }
        }

        if (KefuConversationManager.getInstance().isRecalledMessage(message.messageId())){
            return;
        }

        boolean isAdded = false;
        for(Message msg : messages){
            if(msg.messageId().equals(message.messageId())){
                isAdded = true;
                break;
            }
        }
        if(!isAdded){
            boolean isOldMsg = false;
            if (!messages.isEmpty()){
                isOldMsg = messages.get(messages.size() -1).messageTime() - message.messageTime() > 0;
            }
            messages.add(message);
            msgCount++;
            if(message.direct() == Message.Direct.RECEIVE && message.isUnread()){
                if(unreadCountIncrease){
                    unreadMsgCount++;
                    saveUnreadMsgCount(unreadMsgCount);
                }
            }
            if (isOldMsg){
                //sort msgs
                Collections.sort(messages);
            }
        }
    }

   synchronized void saveUnreadMsgCount(final int count){
        new Thread(new Runnable() {
            @Override
            public void run() {
                KefuDBManager.getInstance().saveConversationUnreadCount(conversationId, count);
            }
        }).start();
    }

    synchronized void deleteUnreadMsgCountRecord(){
        new Thread(new Runnable() {
            @Override
            public void run() {
                KefuDBManager.getInstance().deleteConversationUnreadrecord(conversationId);
            }
        }).start();
    }

    /**
     * 获取会话消息数
     * @return 消息数
     */
    public synchronized int getAllMsgCount(){
        return (int) msgCount;
    }

    /**
     * 获取会话未读消息数
     * @return 消息数
     * @see #unreadMessagesCount() ()
     */
    @Deprecated
    public int getUnreadMsgCount(){
        return unreadMessagesCount();
    }

    /**
     * 标记会话的消息为已读
     */
    public void markAllMessagesAsRead(){
        unreadMsgCount = 0;
        saveUnreadMsgCount(0);
        MarketingHttpClient.asyncOpened(conversationId);
    }

    /**
     * 获取会话消息数
     * @return 消息数
     */
    @Deprecated
    public int getMsgCount(){
        return messages.size();
    }

    /**
     * 通过位置获取消息
     * @param position
     * @return 消息
     */
    public Message getMessageOfPosition(int position){
        return getMessageOfPosition(position, true);
    }

    /**
     * 通过位置获取消息并标记
     * @param position 位置
     * @param markAsRead 是否标记为已读
     * @return 消息
     */
    public synchronized Message getMessageOfPosition(int position, boolean markAsRead){
        int msgSize = messages.size();
        if(position >= msgSize){
            EMLog.e(TAG, "outofbound, message.size:" + msgSize);
            return null;
        }
        Message message = messages.get(position);
        if(markAsRead){
            if(message != null && message.isUnread()){
                message.setUnread(false);
                if(unreadMsgCount > 0){
                    unreadMsgCount--;
                    saveUnreadMsgCount(unreadMsgCount);
                }
            }
        }
        return message;
    }

    /**
     * 从数据库加载消息
     * @param startMsgId 开始的消息id
     * @param pageSize 消息数
     * @return 消息列表
     */
    public List<Message> loadMessages(String startMsgId, int pageSize) {
        List<Message> msgList = KefuDBManager.getInstance().findMessages(conversationId, startMsgId, pageSize);
        messages.addAll(0, msgList);
        for(Message message : msgList){
            ChatManager.getInstance().addMessage(message, false);
        }
        return msgList;
    }

    /**
     * 从数据库加载消息
     * @param startMsgId 开始的消息id
     * @param pageSize 消息数
     * @return 消息列表
     * @see #loadMessages (String, int)
     */
    @Deprecated
    public List<Message> loadMoreMsgFromDB(String startMsgId, int pageSize){
        return loadMessages(startMsgId, pageSize);
    }

    /**
     * 通过消息id获取消息
     * @param messageId 消息id
     * @return 消息
     */
    public Message getMessage(String messageId){
        return getMessage(messageId, true);
    }

    /**
     * 通过消息id获取消息并标记
     * @param messageId 消息id
     * @param markAsRead 是否标记为已读
     * @return 消息
     */
    public Message getMessage(String messageId, boolean markAsRead){
        for(int i = messages.size()-1; i >= 0; i--){
            Message msg = messages.get(i);
            if(msg.messageId().equals(messageId)){
                if(markAsRead){
                    if(msg.isUnread()){
                        msg.setUnread(false);
                        if(unreadMsgCount > 0){
                            unreadMsgCount--;
                            saveUnreadMsgCount(unreadMsgCount);
                        }
                    }
                }
                return msg;
            }
        }
        return null;
    }

    /**
     * 根据id从数据库load 某条消息
     * @param msgId 消息id
     * @return 消息
     */
    public Message loadMessage(String msgId){
        if(TextUtils.isEmpty(msgId)){
            return null;
        }

        Message msg = getMessage(msgId, false);
        if(msg == null){
            msg = KefuDBManager.getInstance().getMessage(msgId);
        }
        return msg;
    }

    /**
     * 根据ids从数据库load 某些消息
     * @param msgIds 消息id列表
     * @return 消息列表
     */
    public List<Message> loadMessages(List<String> msgIds){
        if(msgIds == null){
            return null;
        }

        List<Message> msgList = new ArrayList<Message>();
        for(String msgId : msgIds){
            Message message = loadMessage(msgId);
            if(message != null){
                msgList.add(message);
            }
        }
        return msgList.size() > 0 ? msgList : null;
    }


    /**
     * 根据id标记某条消息
     * @param msgId 消息id
     */
    public void markMessageAsRead(String msgId){
        getMessage(msgId);
    }

    /**
     * 获取已加载的消息列表
     * @return 消息列表
     */
    public List<Message> getAllMessages(){
        return messages;
    }

    /**
     * 获取某条消息的位置
     * @param message 消息
     * @return 消息位置
     */
    public int getMessagePosition(Message message){
        try{
            for(Message msg : messages){
                if(message.messageId().equals(msg.messageId())){
                    return messages.indexOf(msg);
                }
            }
        }catch(Exception ignored){}
        return -1;

    }

    /**
     * 删除某条消息包括数据库
     * @param messageId 消息id
     */
    public void removeMessage(String messageId){
        removeMessage(messageId, true);
    }

    /**
     * 删除某条消息
     * @param messageId 消息id
     * @param deleteDB 是否删除数据库
     */
    public void removeMessage(String messageId, boolean deleteDB){
        EMLog.d(TAG, "remove msg from conversation:" + messageId);
        if(messageId == null){
            return;
        }
        for(int i = messages.size() -1; i >= 0; i--){
            Message msg = messages.get(i);
            if(msg.messageId().equals(messageId)){
                if(msg.isUnread()){
                    msg.setUnread(false);
                    if(unreadMsgCount > 0){
                        unreadMsgCount--;
                        saveUnreadMsgCount(unreadMsgCount);
                    }
                }
                messages.remove(i);
                if(msgCount > 0){
                    msgCount--;
                }
                if (deleteDB){
                    KefuDBManager.getInstance().deleteMessage(messageId);
                }else{
                    KefuConversationManager.getInstance().addRecalledMessage(messageId);
                }
                KefuConversationManager.getInstance().removeMessage(messageId);
                break;
            }
        }
    }

    /**
     * 获取最新的消息
     * @return 消息
     */
    public Message latestMessage() {
        if(messages.size() == 0){
            return null;
        }
        return messages.get(messages.size() - 1);
    }

    /**
     * 获取最新的消息
     * @return 消息
     * @see #latestMessage()
     */
    @Deprecated
    public Message getLastMessage(){
        return latestMessage();
    }

    /**
     * 情况会话消息 包括数据库
     */
    public void clear(){
        messages.clear();
        unreadMsgCount = 0;
        msgCount = 0;
        KefuDBManager.getInstance().deleteConversationUnreadrecord(conversationId);
    }

    @Deprecated
    public void setExtField(String ext){
        KefuDBManager.getInstance().setExtField(conversationId, ext);
    }

    @Deprecated
    public String getExtField(){
        return KefuDBManager.getInstance().getExtField(conversationId);
    }

    /**
     * 获取未读消息数
     * @return 未读消息数
     */
    public int unreadMessagesCount(){
        if(unreadMsgCount < 0){
            unreadMsgCount = 0;
        }
        return unreadMsgCount;
    }

    @Deprecated
    public int messagesCount(){
        return messages.size();
    }

}