package com.hyphenate.chat;

import com.hyphenate.helpdesk.util.Log;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

/**
 * 会话管理类（sdk外部请不要调用）
 */
public class KefuConversationManager {

    private static final String TAG = "kefu conversation";

    private static final int DEFAULT_LOAD_MESSAGE_COUNT = 20;

    List<String> recallMessageIds = Collections.synchronizedList(new ArrayList<String>());

    private static KefuConversationManager instance = new KefuConversationManager();

    private Hashtable<String, Message> allMessages = new Hashtable<String, Message>();
    private Hashtable<String, Conversation> conversations = new Hashtable<String, Conversation>();
    private Hashtable<String, Conversation> tempConversations = new Hashtable<String, Conversation>();

    private boolean allConversationsLoaded = false;

    public static KefuConversationManager getInstance(){
        return instance;
    }

    void loadAllConversations(){
        loadAllConversations(DEFAULT_LOAD_MESSAGE_COUNT);
    }

    synchronized void loadAllConversations(int messagesNum){
        if(allConversationsLoaded){
            return;
        }
        conversations.clear();
        tempConversations.clear();

        // load conversation with users
        conversations = KefuDBManager.getInstance().loadAllParticipantsWithMsgs(messagesNum);

        EMLog.d(TAG, "load converstations size:" + conversations.size());
        synchronized (conversations) {
            for(Conversation conversation : conversations.values()){
                for(Message msg : conversation.messages){
                    allMessages.put(msg.messageId(), msg);
                }
            }
        }
        loadAllRecallMessageIds(20);
        allConversationsLoaded = true;
    }

    void loadAllRecallMessageIds(int limit){
        recallMessageIds.clear();
        List<String> msgIds = KefuDBManager.getInstance().loadAllRecalledMessageIds(limit);
        recallMessageIds.addAll(msgIds);
    }

    Conversation getConversation(String conversationId) {
        if (conversationId == null){
            EMLog.e(TAG, "conversationId is null");
            return null;
        }
        Conversation chat = null;
        if (conversations != null && conversations.containsKey(conversationId)){
            chat = conversations.get(conversationId);
        }
        if (chat == null && tempConversations.containsKey(conversationId)) {
            chat = tempConversations.get(conversationId);
        }
        if (chat == null) {
            List<Message> chatHistory = new ArrayList<Message>();
            long msgCount = KefuDBManager.getInstance().getConversationMessageCount(conversationId, true);
            chat = new Conversation(conversationId, chatHistory, msgCount);
            tempConversations.put(conversationId, chat);
        }
        return chat;
    }

    public String getMessageByExtMsgId(String extMsgId){
        return KefuDBManager.getInstance().getMessageIdByExtMsgId(extMsgId);
    }

    /**
     * 删除会话
     * @param username
     * @return 是否成功
     */
    public boolean removeConversation(String username){
        return removeConversation(username, true);
    }

    public boolean isRecalledMessage(String msgId){
        for (String recallMessageId : recallMessageIds){
            if (recallMessageId.equals(msgId)){
                return true;
            }
        }
        return false;
    }


    public void addRecalledMessage(String msgId){
        if (!recallMessageIds.contains(msgId)){
            recallMessageIds.add(msgId);
        }
    }



    public boolean removeConversation(String conversationId, boolean isRemoveMessages){
        EMLog.d(TAG, "remove conversation for user:" + conversationId);
        Conversation chat = conversations.get(conversationId);
        if(chat == null){
            chat = getConversation(conversationId);
        }
        if(chat == null){
            return false;
        }
        if(isRemoveMessages){
            KefuDBManager.getInstance().deleteChatMsgs(conversationId);
        }
        KefuDBManager.getInstance().deleteConversation(conversationId);
        try{
            List<Message> convMessages = chat.getAllMessages();
            synchronized (convMessages) {
                for(Message msg : convMessages){
                    if(allMessages.containsKey(msg.messageId())){
                        allMessages.remove(msg.messageId());
                    }
                }
            }
        }catch(Exception ignored){}

        chat.clear();
        conversations.remove(conversationId);
        if(tempConversations.containsKey(conversationId)){
            tempConversations.remove(conversationId);
        }
        KefuDBManager.getInstance().deleteMarketingInfoByConversationId(conversationId);
        return true;
    }

    void resetAllUnreadMsgCount(){
        synchronized (conversations) {
            Enumeration<String> enumeration = conversations.keys();
            while (enumeration.hasMoreElements()) {
                String username = enumeration.nextElement();
                conversations.get(username).markAllMessagesAsRead();
            }
        }
    }

    public boolean clearConversation(String username){
        EMLog.d(TAG, "clear conversation for user:" + username);
        Conversation chat = conversations.get(username);
        if(chat == null){
            return false;
        }
        KefuDBManager.getInstance().deleteChatMsgs(username);
        try{
            List<Message> convMessages = chat.getAllMessages();
            synchronized (convMessages) {
                for(Message msg: convMessages){
                    if(allMessages.containsKey(msg.messageId())){
                        allMessages.remove(msg.messageId());
                    }
                }
            }
        }catch(Exception ignored){}
        chat.clear();
        return true;
    }

    /**
     * 获取会话列表
     * @return 会话列表
     */
    public Hashtable<String, Conversation> getAllConversations(){
        return conversations;
    }

    void removeMessage(String msg_id){
        allMessages.remove(msg_id);
    }

    /**
     * 根据id获取消息
     * @param messageId
     * @return 消息
     */
    public Message getMessage(String messageId){
        return allMessages.get(messageId);
    }

    void addMessage(Message msg){
        addMessage(msg, true);
    }

    void addMessage(Message msg, boolean unreadcountIncrease) {
        String msgId = msg.messageId();
        if (msgId == null){
            Log.d("chat", "messageId is null");
            return;
        }
        if (isRecalledMessage(msgId)){
            return;
        }
        if (!allMessages.containsKey(msgId)) {
            allMessages.put(msgId, msg);

            String conversationId;
            if (msg.direct() == Message.Direct.RECEIVE) {
                conversationId = msg.from();
            } else {
                conversationId = msg.to();
            }
            // add to conversation message list
            OfficialAccount officialAccount = msg.getOfficialAccount();
            String marketings = msg.getMarketings();
            Conversation conversation = getConversation(conversationId);
            conversation.addMessage(msg, unreadcountIncrease);
            if (msg.direct() == Message.Direct.RECEIVE){
                conversation.setOfficialAccount(officialAccount);
                conversation.setMarketings(marketings);
            }

            if (!conversations.containsKey(conversationId)) {
                conversations.put(conversationId, conversation);
            }
        }
    }


    private void addConversationToDB(Message message) {
        String username = "";
        if (message.from().equals(ChatClient.getInstance().currentUserName())) {
            username = message.to();
        } else {
            username = message.from();
        }
        OfficialAccount officialAccount = message.getOfficialAccount();
        String marketings = message.getMarketings();
        KefuDBManager.getInstance().addConversation(username, officialAccount, marketings, message.direct() == Message.Direct.SEND);
    }

    public void saveMessage(Message message) {
        EMLog.d(TAG, "save message:" + message.messageId());
        try {
            if(!allMessages.containsKey(message.messageId())){
                addMessage(message);
                KefuDBManager.getInstance().saveMessage(message);
            }
            addConversationToDB(message);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void saveMessage(Message message,boolean unreadcountIncrease) {
        EMLog.d(TAG, "save message:" + message.messageId());
        try {
            addMessage(message,unreadcountIncrease);
            KefuDBManager.getInstance().saveMessage(message);
            addConversationToDB(message);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public int getUnreadMsgsCount(){
        int count = 0;
        synchronized (conversations) {
            Collection<Conversation> c = conversations.values();
            Iterator<Conversation> iter = c.iterator();
            while(iter.hasNext()){
                count += iter.next().unreadMessagesCount();
            }
        }
        EMLog.d(TAG, "getUnreadMsgsCount:" + count);
        return count;
    }

    synchronized public void clear(){
        if(conversations != null){
            conversations.clear();
        }
        if(tempConversations != null){
            tempConversations.clear();
        }
        if(allMessages != null){
            allMessages.clear();
        }
        allConversationsLoaded = false;
    }
}
