package com.hyphenate.chat;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.text.TextUtils;
import android.util.Log;


import com.hyphenate.util.EMLog;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

/**
 * Created by liyuzhao on 16/10/12.
 */
class KefuDBManager {

    private static final String TAG = "KefuDBManager";
    static final String DATABASE_NAME = "_kefumsg.db";
    private static final int DATABASE_VERSION = 7;//old is 5

    private static final String COLUMN_ID = "_id";
    private static final String COLUMN_MSG_ID = "msgid";
    static final String COLUMN_IM_MSG_ID = "immsgid";
    static final String COLUMN_MSG_TIME = "msgtime";
    private static final String COLUMN_MSG_DIR = "msgdir";
    private static final String COLUMN_PARTICIPANT = "participant";
    private static final String COLUMN_MSG_BODY = "msgbody";
    private static final String COLUMN_MSG_GROUP = "groupname";
    private static final String COLUMN_MSG_ISACKED = "isacked";
    private static final String COLUMN_MSG_ISDELIVERED = "isdelivered";
    static final String COLUMN_MSG_STATUS = "status";
    private static final String COLUMN_MSG_ISLISTENED = "islistened";
    private static final String COLUMN_MSG_TYPE = "msgtype";
    private static final String COLUMN_MSG_EXT_MSG_ID = "extMsgId";
    private static final String COLUMN_MSG_RECALLED = "recalled";
    private static final String CHAT_TABLE_NAME = "chat";
    private static final String UNREAD_TABLE_NAME = "unreadcount";
    private static final String COLUMN_UNREAD_CONVERSATION_ID = "conversation_id";
    private static final String COLUMN_UNREAD_COUNT = "count";

    private static final String CONVERSATION_LIST_TABLE_NAME = "conversation_list";
    private static final String COLUMN_CONVERSATION_LIST_USERNAME = "username";
    private static final String COLUMN_CONVERSATION_OFFICIAL_ACCOUNT = "official_account";
    private static final String COLUMN_CONVERSATION_MARKETING = "marketing";
    private static final String COLUMN_CONVERSATION_LIST_EXT = "ext";

    private static final String VISITOR_INFO_TABLE_NAME = "visitor_info";
    private static final String COLUMN_VISITOR_INFO_USERNAME = "username";
    private static final String COLUMN_VISITOR_INFO_TO = "im_number";
    private static final String COLUMN_VISITOR_INFO_ID = "id";
    private static final String COLUMN_VISITOR_INFO_EXT = "ext";

    private static final String MARKETING_INFO_TABLE_NAME = "marketing_info";
    private static final String COLUMN_MARKETING_INFO_ID = "id";
    private static final String COLUMN_MARKETING_INFO_CONVERSATIONID = "conversation_id";
    private static final String COLUMN_MARKETING_INFO_STATUS = "status";

    private static final String REQUEST_INFO_TABLE_NAME = "request_info";
    private static final String COLUMN_REQUEST_INFO_ID = "id";
    private static final String COLUMN_REQUEST_INFO_URL = "url";
    private static final String COLUMN_REQUEST_INFO_PARAMS = "params";
    private static final String COLUMN_REQUEST_INFO_KEY = "keyname";

    private static final String EMOJICON_INFO_TABLE_NAME = "emojicon_info";
    private static final String COLUMN_EMOJICON_INFO_TENANTID = "tenantId";
    private static final String COLUMN_EMOJICON_INFO_ICONS_JSON = "icons_json";
    private static final String COLUMN_EMOJICON_INFO_PACKAGES_JSON = "packages_json";

    private static final String CMD_MSG_TABLE_NAME = "cmdMessages";
    private static final String COLUMN_CMD_MSG_ID = "msgid";
    private static final String COLUMN_CMD_MSG_ACTION = "actionname";
    private static final String COLUMN_CMD_MSG_FROM = "msgfrom";
    private static final String COLUMN_CMD_MSG_TO = "msgto";
    private static final String COLUMN_CMD_MSG_EXT = "ext";
    private static final String COLUMN_CMD_MSG_TIMESTAMP = "timestamp";




    private static final String CREATE_CHAT_TABLE = "create table if not exists " + CHAT_TABLE_NAME + " (" + COLUMN_ID
            + " integer primary key autoincrement, " + COLUMN_MSG_ID + " text, " + COLUMN_IM_MSG_ID + " text, " + COLUMN_MSG_TIME + " integer, " + COLUMN_MSG_DIR
            + " integer, " + COLUMN_MSG_ISACKED + " integer, " + COLUMN_MSG_ISDELIVERED + " integer, " + COLUMN_MSG_STATUS + " integer," + COLUMN_PARTICIPANT + " text not null, "
            + COLUMN_MSG_ISLISTENED + " integer, "
            + COLUMN_MSG_BODY + " text not null,"
            + COLUMN_MSG_TYPE + " integer, "
            + COLUMN_MSG_EXT_MSG_ID + " text, "
            + COLUMN_MSG_RECALLED + " bit default 0, "
            + COLUMN_MSG_GROUP + " text);";

    private static final String CREATE_UNREAD_COUNT_TABLE = "create table if not exists " + UNREAD_TABLE_NAME + " (" + COLUMN_UNREAD_CONVERSATION_ID
            + " text primary key, " + COLUMN_UNREAD_COUNT + " integer);";


    private static final String CREATE_CONVERSATION_LIST_TABLE = "create table if not exists " + CONVERSATION_LIST_TABLE_NAME + " ("
            + COLUMN_CONVERSATION_LIST_USERNAME + " text primary key, "
            + COLUMN_CONVERSATION_OFFICIAL_ACCOUNT + " text, "
            + COLUMN_CONVERSATION_MARKETING + " text, "
            + COLUMN_CONVERSATION_LIST_EXT + " text);";

    private static final String CREATE_VISITOR_INFO_TABLE = "create table if not exists " + VISITOR_INFO_TABLE_NAME + " ("
            + COLUMN_VISITOR_INFO_ID + " text primary key, "
            + COLUMN_VISITOR_INFO_USERNAME + " text, "
            + COLUMN_VISITOR_INFO_TO + " text, "
            + COLUMN_VISITOR_INFO_EXT + " text);";


    private static final String CREATE_MARKETING_INFO_TABLE = "create table if not exists " + MARKETING_INFO_TABLE_NAME + " ("
            + COLUMN_MARKETING_INFO_ID + " text primary key, "
            + COLUMN_MARKETING_INFO_CONVERSATIONID + " text, "
            + COLUMN_MARKETING_INFO_STATUS + " text);";

    private static final String CREATE_REQUEST_INFO_TABLE = "create table if not exists " + REQUEST_INFO_TABLE_NAME + " ("
            + COLUMN_REQUEST_INFO_ID + " text primary key, "
            + COLUMN_REQUEST_INFO_URL + " text, "
            + COLUMN_REQUEST_INFO_PARAMS + " text, "
            + COLUMN_REQUEST_INFO_KEY + " text);";

    private static final String CREATE_EMOJIICON_INFO_TABLE = "create table if not exists " + EMOJICON_INFO_TABLE_NAME + " ("
            + COLUMN_EMOJICON_INFO_TENANTID + " text primary key, "
            + COLUMN_EMOJICON_INFO_ICONS_JSON + " text, "
            + COLUMN_EMOJICON_INFO_PACKAGES_JSON + " text);";

    private static final String CREATE_CMD_MSG_TABLE = "create table if not exists " + CMD_MSG_TABLE_NAME + " ("
            + COLUMN_CMD_MSG_ID + " text primary key, "
            + COLUMN_CMD_MSG_ACTION + " text, "
            + COLUMN_CMD_MSG_FROM + " text, "
            + COLUMN_CMD_MSG_TO + " text, "
            + COLUMN_CMD_MSG_EXT + " text, "
            + COLUMN_CMD_MSG_TIMESTAMP + " integer);";


    private static KefuDBManager instance = null;


    private String currentUserName = null;
    private Context appContext;

    private boolean isDBClosed = true;

    private KefuDBManager(){}

    synchronized static void initDB(String username){
        EMLog.e(TAG, "initDB : " + username);
        if (instance != null) {
            if(instance.currentUserName != null && instance.currentUserName.equals(username)){
                return;
            }

            closeDatabase();
        }

        if(instance == null){
            instance = new KefuDBManager();
            instance.appContext = EMClient.getInstance().getContext();
        }

        instance.currentUserName = username;
        instance.isDBClosed = false;

    }

    public synchronized static KefuDBManager getInstance(){
        EMLog.d(TAG, "getInstance:" + instance);
        if(instance == null){
            String lastUser = getLastLoginUser();
            if (TextUtils.isEmpty(lastUser)) {
                EMLog.e(TAG, "Please login first!");
                throw new RuntimeException("Please login first");
            }else{
                initDB(lastUser);
            }
        }
        return instance;
    }


    private static String getLastLoginUser() {
	    String username = PreferenceUtil.getInstance().getUsername();
	    if (TextUtils.isEmpty(username)) {
		    username = EMClient.getInstance().getCurrentUser();
	    }
	    return username;
    }

    static synchronized void closeDatabase(){
        try{
            EMChatDBOpenHelper.closeDB();
            EMLog.d(TAG, "close msg db");
        }catch(Exception e){
            e.printStackTrace();
        }
    }


    private String getEMMessageExtMsgId(Message message){
        try {
            JSONObject jsonExt = message.getJSONObjectAttribute(Message.KEY_WEICHAT);
            if (jsonExt != null) {
                String msgIdForAck = jsonExt.getString("msgId");// msg_id_for_ack
                if (!TextUtils.isEmpty(msgIdForAck) && !msgIdForAck.equals("null")) {
                    return msgIdForAck;
                }
            }
        } catch (Exception ignored) {
        }
        return null;
    }



    boolean saveMessage(Message message){
        try {

            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if(!db.isOpen()){
                return false;
            }
            // todo, move the save msg to a thread
            ContentValues values = new ContentValues();
            values.put(COLUMN_MSG_ID, message.messageId());
            values.put(COLUMN_IM_MSG_ID, message.getIMMsgId());
            values.put(COLUMN_MSG_TIME, message.messageTime());
            values.put(COLUMN_MSG_ISACKED, message.isAcked);
            values.put(COLUMN_MSG_ISDELIVERED, message.kefuReceived());
            values.put(COLUMN_MSG_DIR, message.direct().ordinal());
            values.put(COLUMN_MSG_TYPE, 0);
            String extMsgId = getEMMessageExtMsgId(message);
            if(extMsgId != null){
                values.put(COLUMN_MSG_EXT_MSG_ID, extMsgId);
            }
            Message.Status status = message.status();
            // INPROGRESS is a transient state which is not supposed to be saved to db
            // we use CREATE instead
            if(status == Message.Status.INPROGRESS){
                status = Message.Status.CREATE;
            }

            values.put(COLUMN_MSG_STATUS, status.ordinal());

            String participant;
            if (message.from().equals(currentUserName)) {
                participant = message.to();
            } else {
                participant = message.from();
            }

            String id = participant;
            values.put(COLUMN_PARTICIPANT, participant);

            values.put(COLUMN_MSG_BODY, KefuMessageEncoder.getJSONMsg(message, true));

            values.putNull(COLUMN_MSG_GROUP);
            values.put(COLUMN_MSG_ISLISTENED, message.isListened() ? 1 : 0);

            // for message send to/from bot, that is test message, we won't save
            // to db
            if (!participant.equals("bot")) {
                db.insert(CHAT_TABLE_NAME, null, values);
            }
            String conversation = participant;
            // update conversation table
            OfficialAccount officialAccount = message.getOfficialAccount();
            String marketings = message.getMarketings();
            addConversation(conversation, officialAccount, marketings, message.direct() == Message.Direct.SEND);

            EMLog.d(TAG, "save msg to db");
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            EMLog.e(TAG, "save msg has error: " + e);
            return false;
        }
    }

    /**
     * update the message body
     * @param message
     * @return
     */
    boolean updateMessageBody(Message message){
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if (!db.isOpen()){
                return false;
            }
            ContentValues values = new ContentValues();
            String msgId = message.messageId();
            String body = KefuMessageEncoder.getJSONMsg(message, true);
            values.put(COLUMN_MSG_BODY, body);

            db.update(CHAT_TABLE_NAME, values, COLUMN_MSG_ID + " = ?", new String[] { msgId });
            EMLog.d(TAG, "update msg:" + msgId + " messagebody:" + body);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }


    /**
     * 删除一条指定消息
     *
     * @param msgId
     *            消息的id
     */
    void deleteMessage(String msgId) {
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if (!db.isOpen()){
                return;
            }
            // if the message is the only one in conversation, need also delete conversation
            // get conversation related participant first
            String username = "";
            cursor = db.rawQuery("select " + COLUMN_PARTICIPANT + ", " + COLUMN_MSG_GROUP + " from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_ID + " =? limit 1", new String[]{msgId});
            if (cursor.moveToFirst()){
                username = cursor.getString(0);
            }
            int ret = db.delete(CHAT_TABLE_NAME, COLUMN_MSG_ID + " = ?", new String[] { msgId });
            EMLog.d(TAG, "delete msg:" + msgId + " return:" + ret);

            // update conversation table
            if (!TextUtils.isEmpty(username)){
                deleteConversationIfNoMessages(username);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }

    }

    /**
     * 从消息数据库查询所有对话的用户名,并返回该用户最后一定数据的消息
     * 返回消息按照时间升序排列
     */
    Hashtable<String, Conversation> loadAllParticipantsWithMsgs(final int LIMIT) {
        Hashtable<String, Conversation> results = new Hashtable<String, Conversation>();
        Cursor cursor = null;

        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if(!db.isOpen()){
                return results;
            }

            String sqlChat = "select * from chat where "+COLUMN_MSG_RECALLED+"=0 and participant in (select username from " + CONVERSATION_LIST_TABLE_NAME +
                    ") order by participant, msgtime desc";

            cursor = db.rawQuery(sqlChat, null);
            List<Message> list = null;
            String prevUserName = null;
            long counter = 0;
            Message msg = null;
            while (cursor.moveToNext()){
                String username = cursor.getString(cursor.getColumnIndex(COLUMN_PARTICIPANT));
                if (prevUserName != null && prevUserName.equals(username)) {
                    if (list.size() < LIMIT) {
                        msg = loadMsgFromCursor(cursor);
                        if (msg != null) {
                            list.add(msg);
                        }
                    }
                    counter++;
                } else {
                    if (prevUserName == null || !prevUserName.equals(username)) {
                        if (prevUserName != null) {
                            // 需要按照时间升序返回
                            Collections.reverse(list);
                            // save previous conversation
                            Conversation conversation = new Conversation(prevUserName, list, counter);
                            attachExtToConversation(conversation);
                            results.put(prevUserName, conversation);
                        }

                        list = new LinkedList<Message>();
                        Message temp = loadMsgFromCursor(cursor);
                        if (temp != null) {
                            list.add(temp);
                        }
                        prevUserName = username;
                        counter = 1;
                    }
                }

            }

            // save the last conversation
            if (prevUserName != null) {
                // 需要按照时间升序返回
                Collections.reverse(list);
                // save previous conversation
                Conversation conversation = new Conversation(prevUserName, list, counter);
                attachExtToConversation(conversation);
                results.put(prevUserName, conversation);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        return results;
    }

    private void attachExtToConversation(Conversation conversation){
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if (!db.isOpen()){return;}
        Cursor cursor = null;
        try{
            cursor = db.rawQuery("select * from " + CONVERSATION_LIST_TABLE_NAME + " where " + COLUMN_CONVERSATION_LIST_USERNAME + "=? ", new String[]{conversation.conversationId()});
            if (cursor.moveToFirst()){
                String strOffAcc = cursor.getString(cursor.getColumnIndex(COLUMN_CONVERSATION_OFFICIAL_ACCOUNT));
                conversation.setOfficialAccount(strOffAcc);
                String strMarketings = cursor.getString(cursor.getColumnIndex(COLUMN_CONVERSATION_MARKETING));
                conversation.setMarketings(strMarketings);
            }
        }catch (Exception e){
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
    }


    /**
     * 查询会话消息总数
     */
    long getConversationMessageCount(String username, boolean isGroup) {
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return 0;
        }
        Cursor cursor = null;
        try {
            String field = !isGroup ? COLUMN_PARTICIPANT : COLUMN_MSG_GROUP;
             cursor = db.rawQuery("select count(*) as msgCount from chat where " + field + " = ?", new String[] { username });
            if (cursor.moveToFirst()){
                return cursor.getLong(0);
            }

        } catch (Exception e) {
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        return 0;
    }

    String getMessageIdByExtMsgId(String extMsgId) {
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if (!db.isOpen()){
            return null;
        }
        Cursor cursor = null;
        try {
            cursor = db.rawQuery("select " + COLUMN_MSG_ID + " from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_EXT_MSG_ID + "=?", new String[]{extMsgId});
            if (cursor.moveToFirst()) {
                return cursor.getString(0);
            }
        } catch (Exception e) {
            e.printStackTrace();
            EMLog.d(TAG, "exitMsgId: " + extMsgId + ", error:" + e.getMessage());
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return null;
    }

    void recallMessage(String msgId){
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if (!db.isOpen()){
                return;
            }
            ContentValues values = new ContentValues();
            values.put(COLUMN_MSG_RECALLED, 1);
            db.update(CHAT_TABLE_NAME, values, COLUMN_MSG_ID + " = ?", new String[] { msgId });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 根据消息Id查询消息
     * @param messageId
     * @return
     */
    public Message getMessage(String messageId) {
        Message msg = null;
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            if (!db.isOpen()) {
                return null;
            }
            cursor = db.rawQuery("select * from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_ID + " =?", new String[]{messageId});
            if (cursor.moveToFirst()) {
                msg = loadMsgFromCursor(cursor);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        EMLog.d(TAG, "load msg msgId:" + messageId);
        return msg;
    }


//    public boolean isMessageExisted(String messageId){
//        Cursor cursor = null;
//        try {
//            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
//            if (!db.isOpen()){
//                return true;
//            }
//            cursor = db.rawQuery(String.format(Locale.getDefault(), "select * from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_ID + " ='%s'", messageId), null);
//            if (!cursor.moveToFirst()) {
//                return false;
//            }
//        } catch (Exception e) {
//            e.printStackTrace();
//        }finally {
//            if (cursor != null && !cursor.isClosed()){
//                cursor.close();
//            }
//        }
//        return true;
//    }


    boolean isMessageExistedByExtMsgId(String extMsgId) {
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            if (!db.isOpen()) {
                return true;
            }
            cursor = db.rawQuery("select * from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_EXT_MSG_ID + " =?", new String[]{extMsgId});
            if (!cursor.moveToFirst()) {
                return false;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return true;
    }


//    /**
//     * 从db加载消息
//     * @param username 用户id或者群id
//     * @return
//     */
//    public List<Message> loadMessages(String username){
//        ArrayList<Message> msgList = new ArrayList<Message>();
//        Cursor cursor = null;
//        try {
//            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
//
//            cursor  = db.rawQuery(String.format(Locale.getDefault(), "select * from " + CHAT_TABLE_NAME + " where participant ='%s' and " + COLUMN_MSG_RECALLED +"=0 and groupname = null order by msgtime", username), null);
//            while (cursor.moveToNext()){
//                Message msg = loadMsgFromCursor(cursor);
//                msgList.add(msg);
//            }
//        } catch (Exception e) {
//            e.printStackTrace();
//        }finally {
//            if (cursor != null && !cursor.isClosed()){
//                cursor.close();
//            }
//        }
//        EMLog.d(TAG, "load msgs size:" + msgList.size() + " for username:" + username);
//        return msgList;
//
//    }

    List<String> loadAllRecalledMessageIds(int limit){
        List<String> msgIds = new ArrayList<>();
        Cursor cursor = null;
        try{
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            if(!db.isOpen()){
                return msgIds;
            }
            cursor = db.rawQuery("select * from " + CHAT_TABLE_NAME + " where " + COLUMN_MSG_RECALLED + "=1 order by msgtime limit ?", new String[]{String.valueOf(limit)});
            while (cursor.moveToNext()){
                String msgId = cursor.getString(cursor.getColumnIndex(COLUMN_MSG_ID));
                msgIds.add(msgId);
            }
        }catch(Exception e){
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        return msgIds;
    }



    /**
     * 查询用户指定message id前指定数量的对话消息，包括发送和接收的消息
     *
     * @param username
     *            用户名
     * @param startMsgId
     *            开始查询的消息id,传入null即取最新的pageSize条message
     * @param pageSize
     *            查询多少条
     * @return
     */
    public List<Message> findMessages(String username, String startMsgId, int pageSize) {
        // we load conversation sort by date.
        ArrayList<Message> msgList = new ArrayList<Message>();
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            if (!db.isOpen()){
                return msgList;
            }

            if (startMsgId != null) {
                Message message = ChatManager.getInstance().getMessage(startMsgId);
                if(message == null){
                    EMLog.e(TAG, "can't find message for startMsgId");
                    return msgList;
                }
                cursor = db.rawQuery("select * from " + CHAT_TABLE_NAME + " where participant =? and "+ COLUMN_MSG_RECALLED +"=0 and " + COLUMN_MSG_TIME +
                        " <? order by " + COLUMN_MSG_TIME + " desc limit ?", new String[]{username, String.valueOf(message.messageTime()), String.valueOf(pageSize)});
            } else {
                cursor = db.rawQuery("select * from " + CHAT_TABLE_NAME + " where participant =? and "+ COLUMN_MSG_RECALLED +"=0 order by "
                        + COLUMN_MSG_TIME + " desc limit ?", new String[]{username, String.valueOf(pageSize)});
            }
            while (cursor.moveToNext()){
                Message msg = loadMsgFromCursor(cursor);
                if (msg != null) {
                    msgList.add(msg);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        EMLog.d(TAG, "load msgs size:" + msgList.size() + " for participate:" + username);
        return msgList;
    }



    private Message loadMsgFromCursor(Cursor cursor) {
        String jsonString = cursor.getString(cursor.getColumnIndex(COLUMN_MSG_BODY));
        Message msg = KefuMessageEncoder.getMsgFromJson(jsonString);
        if (msg == null){
            return null;
        }

        msg.setMsgId(cursor.getString(cursor.getColumnIndex(COLUMN_MSG_ID)));
        msg.setMessageTime(cursor.getLong(cursor.getColumnIndex(COLUMN_MSG_TIME)));
        msg.setIMMsgId(cursor.getString(cursor.getColumnIndex(COLUMN_IM_MSG_ID)));
        int dir = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_DIR));
        if (dir == Message.Direct.SEND.ordinal()) {
            msg.setDirection(Message.Direct.SEND);
            msg.setTo(cursor.getString(cursor.getColumnIndex(COLUMN_PARTICIPANT)));
            msg.setFrom(currentUserName);
        } else {
            msg.setDirection(Message.Direct.RECEIVE);
            msg.setFrom(cursor.getString(cursor.getColumnIndex(COLUMN_PARTICIPANT)));
            msg.setTo(currentUserName);
        }
        int status = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_STATUS));
        if (status == Message.Status.CREATE.ordinal()) {
            msg.setStatus(Message.Status.CREATE);
        } else if (status == Message.Status.INPROGRESS.ordinal()) {
            msg.setStatus(Message.Status.INPROGRESS);
        } else if (status == EMMessage.Status.SUCCESS.ordinal()) {
            msg.setStatus(Message.Status.SUCCESS);
        } else if (status == EMMessage.Status.FAIL.ordinal()) {
            msg.setStatus(Message.Status.FAIL);
        }

        int isAcked = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_ISACKED));
        msg.isAcked = isAcked != 0;

        int isDelivered = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_ISDELIVERED));
//        msg.isDelivered = isDelivered != 0;
        msg.setKefuReceived(isDelivered != 0);

        int isListened = cursor.getInt(cursor.getColumnIndex(COLUMN_MSG_ISLISTENED));
        msg.setListened(isListened == 1);
        msg.setUnread(false);
        return msg;
    }
    /**
     * 删除和指定用户的所有对话消息
     *
     * @param participate
     *            用户名
     */
    void deleteChatMsgs(String participate) {
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if(!db.isOpen()){
                return;
            }
            int ret = db.delete(CHAT_TABLE_NAME, COLUMN_PARTICIPANT + " = ? and " + COLUMN_MSG_GROUP + " is null", new String[] { participate });
            EMLog.d(TAG, "delete chat msgs with:" + participate + " return:" + ret);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private boolean conversationExists(String conversationId) {
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if (!db.isOpen()) {
            return false;
        }
        Cursor cursor = null;
        try {
            cursor = db.rawQuery("select " + COLUMN_CONVERSATION_LIST_USERNAME + " from " + CONVERSATION_LIST_TABLE_NAME + " where " + COLUMN_CONVERSATION_LIST_USERNAME + "=?", new String[]{conversationId});
            if (cursor.moveToFirst()) {
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return false;
    }

    private boolean insertConversation(String conversationId, OfficialAccount officialAccount, String marketings){
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        ContentValues cv = new ContentValues();
        cv.put(COLUMN_CONVERSATION_LIST_USERNAME, conversationId);
        if (officialAccount != null){
            cv.put(COLUMN_CONVERSATION_OFFICIAL_ACCOUNT, officialAccount.toString());
        }
        if (marketings != null){
            cv.put(COLUMN_CONVERSATION_MARKETING, marketings);
        }
        long row = db.insertWithOnConflict(CONVERSATION_LIST_TABLE_NAME, null, cv, SQLiteDatabase.CONFLICT_IGNORE);
        return row > 0;
    }

    private boolean updateConversation(String conversationId, OfficialAccount officialAccount, String marketings){
        if (conversationId == null){
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        ContentValues cv = new ContentValues();
        cv.put(COLUMN_CONVERSATION_LIST_USERNAME, conversationId);
        if (officialAccount != null){
            cv.put(COLUMN_CONVERSATION_OFFICIAL_ACCOUNT, officialAccount.toString());
        }
        if (marketings != null){
            cv.put(COLUMN_CONVERSATION_MARKETING, marketings);
        }
        long row = db.updateWithOnConflict(CONVERSATION_LIST_TABLE_NAME, cv, COLUMN_CONVERSATION_LIST_USERNAME +"=? " , new String[]{conversationId}, SQLiteDatabase.CONFLICT_IGNORE);
        return row > 0;
    }

     void addConversation(String conversationId, OfficialAccount officialAccount, String marketings, boolean isSend){
        try{
            if (conversationExists(conversationId)){
                if (!isSend){
                    boolean boolParam = updateConversation(conversationId, officialAccount, marketings);
                    EMLog.d(TAG, "update conversation is " + boolParam);
                }
            }else{
                boolean boolParam = insertConversation(conversationId, officialAccount, marketings);
                EMLog.d(TAG, "insert conversation is " + boolParam);
            }
        }catch (Exception e){
            e.printStackTrace();
            Log.e(TAG, "addConversation-error:" + e.getMessage());
        }

    }

     boolean addMarketingInfo(String marketingId, String conversationId, String status){
        if (TextUtils.isEmpty(marketingId) || TextUtils.isEmpty(conversationId)){
            EMLog.e(TAG, "marketing or conversationId is null");
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        ContentValues cv = new ContentValues();
        cv.put(COLUMN_MARKETING_INFO_ID, marketingId);
        cv.put(COLUMN_MARKETING_INFO_CONVERSATIONID, conversationId);
        cv.put(COLUMN_MARKETING_INFO_STATUS, status);
        long row = db.insertWithOnConflict(MARKETING_INFO_TABLE_NAME, null, cv, SQLiteDatabase.CONFLICT_IGNORE);
        return row > 0;
    }

    boolean addEmojiconInfo(String tenantId, String iconsJson, String packagesJson) {
        if (TextUtils.isEmpty(tenantId)){
            EMLog.e(TAG, "tenantId is null");
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        ContentValues cv = new ContentValues();
        cv.put(COLUMN_EMOJICON_INFO_TENANTID, tenantId);
        cv.put(COLUMN_EMOJICON_INFO_ICONS_JSON, iconsJson);
        cv.put(COLUMN_EMOJICON_INFO_PACKAGES_JSON, packagesJson);
        long row = db.insertWithOnConflict(EMOJICON_INFO_TABLE_NAME, null, cv, SQLiteDatabase.CONFLICT_IGNORE);
        return row > 0;
    }

    boolean deleteEmojiconInfo(String tenantId){
        if (TextUtils.isEmpty(tenantId)){
            EMLog.e(TAG, "tenantId is null");
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        long row = db.delete(EMOJICON_INFO_TABLE_NAME, COLUMN_EMOJICON_INFO_TENANTID + "=?", new String[]{tenantId});
        return row > 0;
    }

    String getEmojiconInfoIconsJson(String tenantId) {
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            cursor = db.rawQuery("select * from " + EMOJICON_INFO_TABLE_NAME, new String[]{});
            while (cursor.moveToNext()) {
                String tempTenantId = cursor.getString(cursor.getColumnIndex(COLUMN_EMOJICON_INFO_TENANTID));
                if (tempTenantId.equals(tenantId)) {
                    return cursor.getString(cursor.getColumnIndex(COLUMN_EMOJICON_INFO_ICONS_JSON));
                }
            }
        } catch (Exception ignored) {
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return null;
    }

    String getEmojiconInfoPackagesJson(String tenantId) {
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            cursor = db.rawQuery("select * from " + EMOJICON_INFO_TABLE_NAME, new String[]{});
            while (cursor.moveToNext()) {
                String tempTenantId = cursor.getString(cursor.getColumnIndex(COLUMN_EMOJICON_INFO_TENANTID));
                if (tempTenantId.equals(tenantId)) {
                    return cursor.getString(cursor.getColumnIndex(COLUMN_EMOJICON_INFO_PACKAGES_JSON));
                }
            }
        } catch (Exception ignored) {
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return null;
    }


     boolean updateMarketingInfoStatusByConversationId(String conversationId, String fromStatus, String status){
        if (TextUtils.isEmpty(conversationId)){
            EMLog.e(TAG, "conversationId is null");
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        StringBuilder sb = new StringBuilder();
        List<String> argList = new ArrayList<>();
        sb.append(COLUMN_MARKETING_INFO_CONVERSATIONID).append("=? ");
         argList.add(conversationId);
        if (!TextUtils.isEmpty(fromStatus)){
            sb.append(" and ").append(COLUMN_MARKETING_INFO_STATUS).append("=? ");
            argList.add(fromStatus);
        }
        ContentValues cv = new ContentValues();
        cv.put(COLUMN_MARKETING_INFO_STATUS, status);
        long row = db.updateWithOnConflict(MARKETING_INFO_TABLE_NAME, cv, sb.toString(), argList.toArray(new String[argList.size()]), SQLiteDatabase.CONFLICT_IGNORE);
        return row > 0;
    }

     boolean deleteMarketingInfoByConversationId(String conversationId, String status){
        if (TextUtils.isEmpty(conversationId)){
            EMLog.e(TAG, "conversationId is null");
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        StringBuilder sb = new StringBuilder();
        List<String> argList = new ArrayList<>();
        sb.append(COLUMN_MARKETING_INFO_CONVERSATIONID).append("=? ");
        argList.add(conversationId);
         if (!TextUtils.isEmpty(status)){
             sb.append(" and ").append(COLUMN_MARKETING_INFO_STATUS).append("=? ");
             argList.add(status);
         }
        long row = db.delete(MARKETING_INFO_TABLE_NAME, sb.toString(), argList.toArray(new String[argList.size()]));
        return row > 0;
    }

    boolean deleteMarketingInfoByConversationId(String conversationId){
        return deleteMarketingInfoByConversationId(conversationId, null);
    }

     List<String> getMarketingIdsByConversationId(String conversationId, String status) {
        List<String> marketingIds = new ArrayList<>();
        if (TextUtils.isEmpty(conversationId)) {
            EMLog.e(TAG, "conversationId is null");
            return marketingIds;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if (!db.isOpen()) {
            return marketingIds;
        }
        Cursor cursor = null;
        try {
            StringBuilder sb = new StringBuilder();
            List<String> argList = new ArrayList<>();
            sb.append(COLUMN_MARKETING_INFO_CONVERSATIONID).append("=? ");
            argList.add(conversationId);
            if (!TextUtils.isEmpty(status)){
                sb.append(" and ").append(COLUMN_MARKETING_INFO_STATUS).append("=? ");
                argList.add(status);
            }
            cursor = db.rawQuery("select * from " + MARKETING_INFO_TABLE_NAME + " where " + sb.toString(), argList.toArray(new String[argList.size()]));
            while (cursor.moveToNext()) {
                String marketingId = cursor.getString(cursor.getColumnIndex(COLUMN_MARKETING_INFO_ID));
                marketingIds.add(marketingId);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return marketingIds;
    }

    boolean addRequestInfo(RequestInfo info){
        if (info == null){
            return false;
        }
        if (info.id == null){
            info.id = UUID.randomUUID().toString();
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        ContentValues cv = new ContentValues();
        cv.put(COLUMN_REQUEST_INFO_ID, info.id);
        cv.put(COLUMN_REQUEST_INFO_URL, info.url);
        cv.put(COLUMN_REQUEST_INFO_PARAMS, info.params);
        cv.put(COLUMN_REQUEST_INFO_KEY, info.key);
        long row = db.insertWithOnConflict(REQUEST_INFO_TABLE_NAME, null, cv, SQLiteDatabase.CONFLICT_IGNORE);
        return row > 0;
    }

    List<RequestInfo> getRequestInfoByKey(String key) {
        List<RequestInfo> list = new ArrayList<>();
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if (!db.isOpen()) {
            return list;
        }
        Cursor cursor = null;
        try {
            cursor = db.rawQuery("select * from " + REQUEST_INFO_TABLE_NAME + " where " + COLUMN_REQUEST_INFO_KEY + "=? ", new String[]{key});
            while (cursor.moveToNext()) {
                RequestInfo info = new RequestInfo();
                info.id = cursor.getString(cursor.getColumnIndex(COLUMN_REQUEST_INFO_ID));
                info.url = cursor.getString(cursor.getColumnIndex(COLUMN_REQUEST_INFO_URL));
                info.params = cursor.getString(cursor.getColumnIndex(COLUMN_REQUEST_INFO_PARAMS));
                info.key = cursor.getString(cursor.getColumnIndex(COLUMN_REQUEST_INFO_KEY));
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return list;

    }

    boolean deleteRequestInfoById(String id){
        if (TextUtils.isEmpty(id)){
            EMLog.e(TAG, "id cannot is null");
            return false;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        long row = db.delete(REQUEST_INFO_TABLE_NAME, COLUMN_REQUEST_INFO_ID + "=?", new String[]{id});
        return row > 0;
    }


    private void deleteConversationIfNoMessages(String username) {
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            db.execSQL("delete from " + CONVERSATION_LIST_TABLE_NAME + " where " + COLUMN_CONVERSATION_LIST_USERNAME + " =? and "
                    + " not exists (select null from " + CHAT_TABLE_NAME + " where " + COLUMN_PARTICIPANT + " =? )", new String[]{username, username});
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

     void deleteConversation(String username) {
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if(!db.isOpen()){
                return;
            }
            db.execSQL("delete from " + CONVERSATION_LIST_TABLE_NAME + " where " + COLUMN_CONVERSATION_LIST_USERNAME + " =?", new String[]{username});
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

     void updateMessageListened(String msgId, boolean isListened){
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if (!db.isOpen()){
                return;
            }
            ContentValues values = new ContentValues();
            values.put(COLUMN_MSG_ISLISTENED, isListened);
            db.update(CHAT_TABLE_NAME, values, COLUMN_MSG_ID + " = ?", new String[] { msgId });
            EMLog.d(TAG, "update msg:" + msgId + " isListened:" + isListened);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateMessageId(String oldId, String newId){
        ContentValues values = new ContentValues();
        values.put(KefuDBManager.COLUMN_MSG_ID, newId + "");
        KefuDBManager.getInstance().updateMessage(oldId, values);
    }




    /**
     * 根据msgid和values更新message
     *
     * @param values
     */
     void updateMessage(String msgId, ContentValues values) {
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return;
        }
        db.update(CHAT_TABLE_NAME, values, COLUMN_MSG_ID + " = ?", new String[] { msgId });
    }



    /**
     * get user or group unread message count
     *
     * @param conversationId
     * @return
     */
    int getConversationUnreadCount(String conversationId) {
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            if (!db.isOpen()) {
                return 0;
            }
            int count = 0;
            cursor = db.rawQuery("select " + COLUMN_UNREAD_COUNT + " from " + UNREAD_TABLE_NAME + " where " + COLUMN_UNREAD_CONVERSATION_ID
                    + " =? ", new String[]{conversationId});
            if (cursor.moveToFirst()) {
                count = cursor.getInt(cursor.getColumnIndex(COLUMN_UNREAD_COUNT));
            }
            if (count < 0)
                return 0;
            return count;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return 0;
    }



    /**
     * save or update unread msg count
     * @param conversationId
     * @param count
     */
     void saveConversationUnreadCount(String conversationId,int count){
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if (!db.isOpen()){
                return;
            }
            ContentValues values = new ContentValues();
            values.put(COLUMN_UNREAD_CONVERSATION_ID, conversationId);
            values.put(COLUMN_UNREAD_COUNT, count);
            db.replace(UNREAD_TABLE_NAME, null, values);
        } catch (Exception ignored) {
        }
    }



     void deleteConversationUnreadrecord(String conversationId){
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return;
        }
        db.delete(UNREAD_TABLE_NAME, COLUMN_UNREAD_CONVERSATION_ID + " = ?", new String[]{conversationId});
    }


    /**
     * 获取有未读数的conversations
     *
     * get list of user id or group id which have the unread messages
     * @return
     */
    public List<String> getConversationsUnread() {
        ArrayList<String> conversations = new ArrayList<String>();
        Cursor cursor = null;
        try {
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
            cursor = db.rawQuery("select * from " + UNREAD_TABLE_NAME, new String[]{});
            while (cursor.moveToNext()) {
                String userOrName = cursor.getString(0);
                int count = cursor.getInt(1);
                if (count > 0)
                    conversations.add(userOrName);
            }
        } catch (Exception ignored) {
        } finally {
            if (cursor != null && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return conversations;
    }



     void setExtField(String username, String ext) {
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return;
        }
        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_CONVERSATION_LIST_EXT, ext == null ? "" : ext);
            db.update(CONVERSATION_LIST_TABLE_NAME, values, COLUMN_CONVERSATION_LIST_USERNAME + " = ?", new String[] { username });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


     String getExtField(String conversationId) {
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if (!db.isOpen()){
            return "";
        }
         Cursor cursor = null;
        try {
            cursor = db.rawQuery("select " + COLUMN_CONVERSATION_LIST_EXT + " from " + CONVERSATION_LIST_TABLE_NAME + " where " + COLUMN_CONVERSATION_LIST_USERNAME + " =? ", new String[]{conversationId});
            if (cursor.moveToFirst()) {
                return cursor.getString(0);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        return "";
    }

     boolean saveVisitorInfo(String toUser, String visitorUserId){
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        try{
            ContentValues values = new ContentValues();
            values.put(COLUMN_VISITOR_INFO_ID, visitorUserId);
            values.put(COLUMN_VISITOR_INFO_USERNAME, ChatClient.getInstance().currentUserName());
            values.put(COLUMN_VISITOR_INFO_TO, toUser);
            long row = db.replace(VISITOR_INFO_TABLE_NAME, null, values);
            if (row > 0){
                return true;
            }
        }catch (Exception e){
            e.printStackTrace();
        }
        return false;
    }


     String getVisitorUserId(String toUser){
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if(!db.isOpen()){
            return "";
        }
        Cursor cursor = null;
        try{
            cursor = db.rawQuery("select " + COLUMN_VISITOR_INFO_ID + " from " + VISITOR_INFO_TABLE_NAME + " where " + COLUMN_VISITOR_INFO_USERNAME +"=? " +
                    "and "+COLUMN_VISITOR_INFO_TO + "=? ", new String[]{ChatClient.getInstance().currentUserName(), toUser});
            if (cursor.moveToFirst()){
                return cursor.getString(cursor.getColumnIndex(COLUMN_VISITOR_INFO_ID));
            }
        }catch (Exception e){
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        return "";
    }

    // cmdMsgs
    boolean isExistCmdMessage(String messageId){
        if (TextUtils.isEmpty(messageId)){
            return true;
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getReadableDatabase();
        if(!db.isOpen()){
            return false;
        }
        boolean isExist = false;

        String sql = "select " + COLUMN_CMD_MSG_ID + " from " + CMD_MSG_TABLE_NAME + " where " + COLUMN_CMD_MSG_ID + "=? ";
        Cursor cursor = null;
        try{
            cursor = db.rawQuery(sql, new String[]{messageId});
            if (cursor.moveToNext()){
                isExist = true;
            }
        }catch (Exception e){
            e.printStackTrace();
        }finally {
            if (cursor != null && !cursor.isClosed()){
                cursor.close();
            }
        }
        return isExist;
    }

    boolean insertCmdMessage(Message message){
        if (message == null){
            return false;
        }
        if (message.getType() != Message.Type.CMD){
            EMLog.d(TAG, "Non-command message,msgid->" + message.messageId());
            return false;
        }
        EMCmdMessageBody cmdBody = (EMCmdMessageBody) message.body();
        String messageId = message.messageId();
        String ext = KefuMessageEncoder.getJSONMsg(message, false);
        long timestamp = message.messageTime();
        if (timestamp <= 0){
            timestamp = System.currentTimeMillis();
        }
        SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
        if (!db.isOpen()){
            return false;
        }
        ContentValues cvs = new ContentValues();
        cvs.put(COLUMN_CMD_MSG_ID, messageId);
        cvs.put(COLUMN_CMD_MSG_ACTION, cmdBody.action());
        cvs.put(COLUMN_CMD_MSG_FROM, message.from());
        cvs.put(COLUMN_CMD_MSG_TO, message.to());
        cvs.put(COLUMN_CMD_MSG_EXT, ext);
        cvs.put(COLUMN_CMD_MSG_TIMESTAMP, timestamp);
        long row = db.insertWithOnConflict(CMD_MSG_TABLE_NAME, null, cvs, SQLiteDatabase.CONFLICT_IGNORE);
        if (row <= 0){
            EMLog.e(TAG, "cmd Message insert failed:msgid:" + messageId);
            return false;
        }
        return true;
    }


    void deleteCmdMessages(int numberDayBefore){
        if (numberDayBefore <= 0){
            return;
        }
        try{
            long currentTime = System.currentTimeMillis();
            int oneDay = 24 * 60 * 60;
            long theDateTime = currentTime - oneDay * numberDayBefore;
            SQLiteDatabase db = EMChatDBOpenHelper.getInstance(appContext, currentUserName).getWritableDatabase();
            if (!db.isOpen()){
                return;
            }
            long row = db.delete(CMD_MSG_TABLE_NAME, COLUMN_CMD_MSG_TIMESTAMP + "<? ", new String[]{String.valueOf(theDateTime)});
        }catch (Exception ignored){
        }
    }




    private static class EMChatDBOpenHelper extends SQLiteOpenHelper{
        private static EMChatDBOpenHelper instance = null;
        @SuppressWarnings("unused")
        private String username;

        private EMChatDBOpenHelper(Context context, String username){
            super(context, username + DATABASE_NAME, null, DATABASE_VERSION);
            this.username = username;
            EMLog.e(TAG, "create kefu chatdb for:" + username);
        }

        public synchronized static EMChatDBOpenHelper getInstance(Context context, String username){
            if(instance == null){
                instance = new EMChatDBOpenHelper(context, username);
            }
            return instance;
        }


        synchronized static void closeDB(){
            if(instance != null){
                try {
                    SQLiteDatabase db = instance.getWritableDatabase();
                    db.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
                instance = null;
            }
        }


        private void dropTableDB(SQLiteDatabase db){
//            db.execSQL("drop table if exists " + CHAT_TABLE_NAME);
            db.execSQL("drop table if exists " + UNREAD_TABLE_NAME);
            db.execSQL("drop table if exists " + CONVERSATION_LIST_TABLE_NAME);
            db.execSQL("drop table if exists " + VISITOR_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + MARKETING_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + REQUEST_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + EMOJICON_INFO_TABLE_NAME);
        }

        private void dropAllTableDB(SQLiteDatabase db){
            db.execSQL("drop table if exists " + CHAT_TABLE_NAME);
            db.execSQL("drop table if exists " + UNREAD_TABLE_NAME);
            db.execSQL("drop table if exists " + CONVERSATION_LIST_TABLE_NAME);
            db.execSQL("drop table if exists " + VISITOR_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + MARKETING_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + REQUEST_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + EMOJICON_INFO_TABLE_NAME);
            db.execSQL("drop table if exists " + CMD_MSG_TABLE_NAME);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            db.execSQL(CREATE_CHAT_TABLE);
            db.execSQL(CREATE_UNREAD_COUNT_TABLE);
            db.execSQL(CREATE_CONVERSATION_LIST_TABLE);
            db.execSQL(CREATE_VISITOR_INFO_TABLE);
            db.execSQL(CREATE_MARKETING_INFO_TABLE);
            db.execSQL(CREATE_REQUEST_INFO_TABLE);
            db.execSQL(CREATE_EMOJIICON_INFO_TABLE);
            db.execSQL(CREATE_CMD_MSG_TABLE);
        }


        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            Log.w(TAG, "Upgrading from version " + oldVersion + " to " + newVersion);
            if (oldVersion < 5) {
                dropTableDB(db);
                onCreate(db);
            }else if (oldVersion < 7){
                db.execSQL("drop table if exists " + REQUEST_INFO_TABLE_NAME);
                db.execSQL("drop table if exists " + CMD_MSG_TABLE_NAME);
                db.execSQL(CREATE_CMD_MSG_TABLE);
                db.execSQL(CREATE_REQUEST_INFO_TABLE);
            }

        }
    }

}
