package com.hyphenate.chat;

import android.net.Uri;
import android.text.TextUtils;

import com.hyphenate.util.EMLog;
import com.hyphenate.util.UriUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Iterator;

/**
 * 消息转换类（sdk外部请不要调用）
 */
public class KefuMessageEncoder {

    private static final String TAG = "kefu encoder";

    public final static String ATTR_TYPE = "type";
    public final static String ATTR_TO = "to";
    public final static String ATTR_FROM = "from";
    public final static String ATTR_MSG = "msg";
    public final static String ATTR_URL = "url";
    public final static String ATTR_LOCALURL = "localurl";
    public final static String ATTR_THUMB_LOCALURL = "thumblocalurl";
    public final static String ATTR_FILENAME = "filename";
    public final static String ATTR_THUMBNAIL = "thumb";
    public final static String ATTR_SECRET = "secret";
    public final static String ATTR_SIZE = "size";
    public final static String ATTR_IMG_WIDTH = "width";
    public final static String ATTR_IMG_HEIGHT = "height";
    public final static String ATTR_THUMBNAIL_SECRET = "thumb_secret";
    public final static String ATTR_LENGTH = "length";
    public final static String ATTR_ADDRESS = "addr";
    public final static String ATTR_LATITUDE = "lat";
    public final static String ATTR_LONGITUDE = "lng";
    public final static String ATTR_ACTION = "action";
    public final static String ATTR_FILE_LENGTH = "file_length";

    // optional json obj for additional user defined attributes
    public final static String ATTR_EXT = "ext";

    private final static String ATTR_TYPE_TXT = "txt";
    private final static String ATTR_TYPE_IMG = "img";
    private final static String ATTR_TYPE_VOICE = "audio";
    private final static String ATTR_TYPE_VIDEO = "video";
    private final static String ATTR_TYPE_LOCATION = "loc";
    private final static String ATTR_TYPE_CMD = "cmd";
    private final static String ATTR_TYPE_file = "file";


    public static String getJSONMsg(Message msg, boolean all){
        StringBuffer jsonString = new StringBuffer();
        jsonString.append("{");
        jsonString.append("\"" + ATTR_FROM + "\":\"" + msg.from() + "\",");
        jsonString.append("\"" + ATTR_TO + "\":\"" + msg.to() + "\",");
        jsonString.append("\"bodies\":[{");
        if (msg.type == Message.Type.TXT) {
            addTextBody(jsonString, msg);
        } else if (msg.type == Message.Type.IMAGE) {
            addImageBody(jsonString, msg, all);
        } else if (msg.type == Message.Type.VOICE) {
            addVoiceBody(jsonString, msg, all);
        } else if (msg.type == Message.Type.LOCATION) {
            addLocationBody(jsonString, msg);
        } else if (msg.type == Message.Type.CMD) {
            addCmdBody(jsonString, msg);
        } else if (msg.type == Message.Type.VIDEO) {
            addVideoBody(jsonString, msg, all);
        } else if (msg.type == Message.Type.FILE) {
            addFileBody(jsonString, msg, all);
        }
        jsonString.append("}]");
        if (msg.attributes != null) {
            addExtAttr(jsonString, msg);
        }
        jsonString.append("}");
        return jsonString.toString();
    }

    private static void addExtAttr(StringBuffer jsonString, Message msg){
        jsonString.append(",");
        jsonString.append("\"" + ATTR_EXT + "\":");
        JSONObject extValueObj = new JSONObject();
        try {
            synchronized (msg.attributes) {
                for (String attr : msg.attributes.keySet()) {
                    Object value = msg.attributes.get(attr);
                    if(value instanceof String){
                        String valStr = (String) value;
                        if ((valStr.startsWith("{") && valStr.endsWith("}") && valStr.contains(":"))) {
                            // it is json string
                            value = new JSONObject(valStr);
                        } else if ((valStr.startsWith("[{") && valStr.endsWith("}]") && valStr.contains(":"))) {
                            value = new JSONArray(valStr);
                        }
                    }
                    extValueObj.put(attr, value);
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        String extObjStr = extValueObj.toString();
        jsonString.append(extObjStr);
    }

    private static void addImageBody(StringBuffer jsonString, Message msg, boolean all){
        jsonString.append("\"" + ATTR_TYPE + "\":\"img\",");
        EMImageMessageBody imgBody = (EMImageMessageBody)msg.body();
        String remoteUrl = imgBody.getRemoteUrl();
        if(remoteUrl != null){
            jsonString.append("\"" + ATTR_URL + "\":" + "\"" + remoteUrl + "\",");
        }
        if (all) {
            String localUrl = imgBody.getLocalUrl();
            if(localUrl != null){
                jsonString.append("\"" + ATTR_LOCALURL + "\":" + "\"" + localUrl + "\",");
            }
            String localThumb = imgBody.thumbnailLocalPath();
            if (localThumb != null){
                jsonString.append("\"" + ATTR_THUMB_LOCALURL + "\":" + "\"" + localThumb + "\",");
            }
        }

        String fileName = imgBody.getFileName();
        if(fileName != null){
            jsonString.append("\"" + ATTR_FILENAME + "\":" + "\"" + fileName + "\",");
        }

        String thumbnailUrl = imgBody.getThumbnailUrl();

        if (thumbnailUrl != null) {
            jsonString.append("\"" + ATTR_THUMBNAIL + "\":" + "\""
                    + thumbnailUrl + "\",");
        }
        String secret = imgBody.getSecret();

        if(secret != null){
            jsonString.append("\"" + ATTR_SECRET + "\":" + "\"" + secret + "\",");
        }

        jsonString.append("\"" + ATTR_SIZE + "\":{" + "\"" + ATTR_IMG_WIDTH +
                "\":" + imgBody.getWidth() + ",\"" + ATTR_IMG_HEIGHT + "\":" + imgBody.getHeight() +"}");

        String thumbnailSecret = imgBody.getThumbnailSecret();
        if (thumbnailSecret != null) {
            jsonString.append(",\"" + ATTR_THUMBNAIL_SECRET + "\":" + "\"" + thumbnailSecret+ "\"");
        }
    }

    private static void addVideoBody(StringBuffer jsonString, Message msg, boolean all){
        jsonString.append("\"" + ATTR_TYPE + "\":\"video\",");
        EMVideoMessageBody videoBody = (EMVideoMessageBody)msg.body();
        jsonString.append("\"" + ATTR_URL + "\":" + "\"" + videoBody.getRemoteUrl() + "\",");
        if(all){
            String localUrl = videoBody.getLocalUrl();
            String localThumb = videoBody.getLocalThumb();
            if(localUrl != null){
                jsonString.append("\"" + ATTR_LOCALURL + "\":" + "\"" + localUrl + "\",");
            }
            if(localThumb != null){
                jsonString.append("\"" + ATTR_THUMB_LOCALURL + "\":" + "\"" + localThumb + "\",");
            }
        }

        String fileName = videoBody.getFileName();
        if(fileName != null){
            jsonString.append("\"" + ATTR_FILENAME + "\":" + "\"" + fileName + "\",");
        }
        String thumbnailUrl = videoBody.getThumbnailUrl();
        if(thumbnailUrl != null){
            jsonString.append("\"" + ATTR_THUMBNAIL + "\":" + "\"" + thumbnailUrl + "\",");
        }
        jsonString.append("\"" + ATTR_LENGTH + "\":" + videoBody.getDuration() + ",");
        jsonString.append("\"" + ATTR_FILE_LENGTH + "\":" + videoBody.getVideoFileLength() + ",");
        String secret = videoBody.getSecret();
        if(secret != null){
            jsonString.append("\"" + ATTR_SECRET + "\":" + "\"" + secret + "\"");
        }
        String thumbSecret = videoBody.getThumbnailSecret();
        if(thumbSecret != null){
            jsonString.append(",\"" + ATTR_THUMBNAIL_SECRET + "\":" + "\"" + thumbSecret + "\"");
        }
    }

    private static void addTextBody(StringBuffer jsonString, Message msg) {
        jsonString.append("\"" + ATTR_TYPE + "\":\"txt\",");
        EMTextMessageBody textBody = (EMTextMessageBody)msg.body();
        String msgString = textBody.getMessage();
        msgString = JSONObject.quote(msgString);
        if ((msgString.startsWith("{") && msgString.endsWith("}"))
                || (msgString.startsWith("[") && msgString.endsWith("]"))) {
            // to support test message with json body, we encode " to %22
            // following url encoding rule
            // so the msg string will be a valid json string
            msgString = msgString.replaceAll("\"", "%22");
            jsonString
                    .append("\"" + ATTR_MSG + "\":" + "\"" + msgString + "\"");
        }
        jsonString.append("\"" + ATTR_MSG + "\":" +  msgString);
    }

    private static void addCmdBody(StringBuffer jsonString, Message msg){
        jsonString.append("\"" + ATTR_TYPE + "\":\"cmd\",");
        EMCmdMessageBody cmdBody = (EMCmdMessageBody) msg.body();
        jsonString.append("\"" + ATTR_ACTION + "\":" + "\"" + cmdBody.action() + "\"]");
    }

    private static void addVoiceBody(StringBuffer jsonString, Message msg, boolean all){
        jsonString.append("\"" + ATTR_TYPE + "\":\"audio\",");
        EMVoiceMessageBody voiceBody = (EMVoiceMessageBody) msg.body();
        String remoteUrl = voiceBody.getRemoteUrl();
        if(remoteUrl != null){
            jsonString.append("\"" + ATTR_URL + "\":" + "\"" + remoteUrl + "\",");
        }
        if(all){
            String localUrl = voiceBody.getLocalUrl();
            if(localUrl != null){
                jsonString.append("\"" + ATTR_LOCALURL + "\":" + "\"" + localUrl + "\",");
            }
        }

        jsonString.append("\"" + ATTR_FILENAME + "\":" + "\"" + voiceBody.getFileName() + "\",");
        jsonString.append("\"" + ATTR_LENGTH + "\":" + voiceBody.getLength() + ",");
        String secret = voiceBody.getSecret();
        if(secret != null){
            jsonString.append("\"" + ATTR_SECRET + "\":" + "\"" + secret + "\"");
        }
    }

    private static void addFileBody(StringBuffer jsonString, Message msg, boolean all) {
        jsonString.append("\"" + ATTR_TYPE + "\":\"file\",");
        EMNormalFileMessageBody fileMessageBody = (EMNormalFileMessageBody) msg.body();
        String remoteUrl = fileMessageBody.getRemoteUrl();
        if(remoteUrl != null){
            jsonString.append("\"" + ATTR_URL + "\":" + "\"" + remoteUrl + "\",");
        }
        if(all){
            String localUrl = fileMessageBody.getLocalUrl();
            if(localUrl != null){
                jsonString.append("\"" + ATTR_LOCALURL + "\":" + "\"" + localUrl + "\",");
            }
        }
        String fileName = fileMessageBody.getFileName();
        if(fileName != null){
            jsonString.append("\"" + ATTR_FILENAME + "\":" + "\"" + fileName + "\",");
        }
        jsonString.append("\"" + ATTR_FILE_LENGTH + "\":" + fileMessageBody.getFileSize() + ",");
        String secret = fileMessageBody.getSecret();
        if(secret != null){
            jsonString.append("\"" + ATTR_SECRET + "\":" + "\"" + secret + "\"");
        }

    }


    private static void addLocationBody(StringBuffer jsonString, Message msg){
        jsonString.append("\"" + ATTR_TYPE + "\":\"loc\",");
        EMLocationMessageBody locationBody = (EMLocationMessageBody) msg.body();
        String address = locationBody.getAddress();
        if(address != null){
            jsonString.append("\"" + ATTR_ADDRESS + "\":" + "\"" + locationBody.getAddress() + "\",");
        }

        jsonString.append("\"" + ATTR_LATITUDE + "\":" + locationBody.getLatitude() + ",");
        jsonString.append("\"" + ATTR_LONGITUDE + "\":" + locationBody.getLongitude() + "");
    }

    public static Message getMsgFromJson(String jsonString){
        try {
            JSONObject jsonPayload = new JSONObject(jsonString);
            String from = jsonPayload.getString(ATTR_FROM);
            String to = jsonPayload.getString(ATTR_TO);

            JSONArray bodies = jsonPayload.getJSONArray("bodies");
            if (bodies.length() < 1) {
                EMLog.d(TAG, "wrong msg without body");
                return null;
            }

            // for now, we only deal with first body in bodies[]
            JSONObject bodyJson = bodies.getJSONObject(0);
            String type = bodyJson.getString(ATTR_TYPE);

            Message msg = null;
            if(type.equals(ATTR_TYPE_TXT)){
                msg = new Message(Message.Type.TXT);
                String bodyString = bodyJson.getString(ATTR_MSG);
                String decodeString = bodyString.replaceAll("%22", "\"");
                EMTextMessageBody body = new EMTextMessageBody(decodeString);
                msg.setBody(body);
            }else if(type.equals(ATTR_TYPE_IMG)){
                msg = new Message(Message.Type.IMAGE);
                String remoteUrl = null;
                String fileName = null;
                String localPath = "";
                String thumbnailLocalPath = "";

                if(bodyJson.has(ATTR_LOCALURL)){
                    localPath = bodyJson.getString(ATTR_LOCALURL);
                }
                if (bodyJson.has(ATTR_THUMB_LOCALURL)){
                    thumbnailLocalPath = bodyJson.getString(ATTR_THUMB_LOCALURL);
                }

                Uri localUri = UriUtils.getLocalUriFromString(localPath);
                if (localUri == null) {
                    localUri = Uri.parse("");
                }
                Uri thumnailLocalUri = UriUtils.getLocalUriFromString(thumbnailLocalPath);

                EMImageMessageBody body = new EMImageMessageBody(localUri, thumnailLocalUri);

                if(bodyJson.has(ATTR_URL)){
                    remoteUrl = bodyJson.getString(ATTR_URL);
                    body.setRemoteUrl(remoteUrl);
                }
                if(bodyJson.has(ATTR_FILENAME)){
                    fileName = bodyJson.getString(ATTR_FILENAME);
                    body.setFileName(fileName);
                }
                String thumbnailUrl = remoteUrl;
                if(bodyJson.has(ATTR_THUMBNAIL)){
                    thumbnailUrl = bodyJson.getString(ATTR_THUMBNAIL);
                    body.setThumbnailUrl(thumbnailUrl);
                }

                if(bodyJson.has(ATTR_SECRET)){
                    body.setSecret(bodyJson.getString(ATTR_SECRET));
                }
                if(bodyJson.has(ATTR_THUMBNAIL_SECRET)){
                    body.setThumbnailSecret(bodyJson.getString(ATTR_THUMBNAIL_SECRET));
                }

                if(bodyJson.has(ATTR_SIZE)){
                    JSONObject sizeJson = bodyJson.getJSONObject(ATTR_SIZE);
                    int width = 0, height = 0;
                    if(sizeJson.has(ATTR_IMG_WIDTH)){
                        width = sizeJson.getInt(ATTR_IMG_WIDTH);
                        height = sizeJson.getInt(ATTR_IMG_HEIGHT);
                    }
                    body.setSize(width, height);
                }
                msg.setBody(body);
            }else if(type.equals(ATTR_TYPE_file)){
                msg = new Message(Message.Type.FILE);
                String remoteUrl = null;
                String fileName = null;
                if(bodyJson.has(ATTR_URL)){
                    remoteUrl = bodyJson.getString(ATTR_URL);
                }
                if(bodyJson.has(ATTR_FILENAME)){
                    fileName = bodyJson.getString(ATTR_FILENAME);
                }
                EMNormalFileMessageBody body = new EMNormalFileMessageBody(remoteUrl, remoteUrl);
                if (fileName != null) {
                    body.setFileName(fileName);
                }
                long fileLength = 0;
                if(bodyJson.has(ATTR_FILE_LENGTH)){
                    fileLength = bodyJson.getLong(ATTR_FILE_LENGTH);
                }
                body.setFileLength(fileLength);
                String localUrl = null;
                String secret = null;
                if(bodyJson.has(ATTR_LOCALURL)){
                    localUrl = bodyJson.getString(ATTR_LOCALURL);
                }
                if(bodyJson.has(ATTR_SECRET)){
                    secret = bodyJson.getString(ATTR_SECRET);
                }
                body.setLocalUrl(localUrl);
                body.setSecret(secret);
                msg.setBody(body);
            }else if(type.equals(ATTR_TYPE_VIDEO)){
                msg = new Message(Message.Type.VIDEO);
                String remoteUrl = null;
                String fileName = null;
                String thumbnailUrl = null;
                int length = 0;
                if (bodyJson.has(ATTR_URL)) {
                    remoteUrl = bodyJson.getString(ATTR_URL);
                }
                if (bodyJson.has(ATTR_FILENAME)) {
                    fileName = bodyJson.getString(ATTR_FILENAME);
                }
                if (bodyJson.has(ATTR_THUMBNAIL)) {
                    thumbnailUrl = bodyJson.getString(ATTR_THUMBNAIL);
                }
                if (bodyJson.has(ATTR_LENGTH)) {
                    length = bodyJson.getInt(ATTR_LENGTH);
                }

                EMVideoMessageBody body = new EMVideoMessageBody(fileName, remoteUrl, thumbnailUrl, length);
                String localUrl = null;
                String localThumb = null;
                String secret = null;
                String thumbSecret = null;
                long fileLength = 0L;
                if(bodyJson.has(ATTR_LOCALURL)){
                    localUrl = bodyJson.getString(ATTR_LOCALURL);
                }
                if(bodyJson.has(ATTR_FILE_LENGTH)){
                    fileLength = bodyJson.getLong(ATTR_FILE_LENGTH);
                }
                if(bodyJson.has(ATTR_THUMB_LOCALURL)){
                    localThumb = bodyJson.getString(ATTR_THUMB_LOCALURL);
                }
                if(bodyJson.has(ATTR_SECRET)){
                    secret = bodyJson.getString(ATTR_SECRET);
                }
                if(bodyJson.has(ATTR_THUMBNAIL_SECRET)){
                    thumbSecret = bodyJson.getString(ATTR_THUMBNAIL_SECRET);
                }
                body.setLocalUrl(localUrl);
                body.setLocalThumb(localThumb);
                body.setFileLength(fileLength);
                body.setSecret(secret);
                body.setThumbnailSecret(thumbSecret);
                msg.setBody(body);
            }else if(type.equals(ATTR_TYPE_VOICE)){
                msg = new Message(Message.Type.VOICE);
                String remoteUrl = null;
                String fileName = null;
                int length = 0;

                if(bodyJson.has(ATTR_URL)){
                    remoteUrl = bodyJson.getString(ATTR_URL);
                }
                if(bodyJson.has(ATTR_FILENAME)){
                    fileName = bodyJson.getString(ATTR_FILENAME);
                }
                if(bodyJson.has(ATTR_LENGTH)){
                    length = bodyJson.getInt(ATTR_LENGTH);
                }
                EMVoiceMessageBody body = new EMVoiceMessageBody(fileName, remoteUrl, length);
                String localUrl = null;
                String secret = null;
                if(bodyJson.has(ATTR_LOCALURL)){
                    localUrl = bodyJson.getString(ATTR_LOCALURL);
                }
                if(bodyJson.has(ATTR_SECRET)){
                    secret = bodyJson.getString(ATTR_SECRET);
                }
                body.setLocalUrl(localUrl);
                body.setSecret(secret);
                msg.setBody(body);
            }else if(type.equals(ATTR_TYPE_LOCATION)){
                msg = new Message(Message.Type.LOCATION);
                String address = null;
                double latitude = 0.0;
                double longitude = 0.0;
                if(bodyJson.has(ATTR_ADDRESS)){
                    address = bodyJson.getString(ATTR_ADDRESS);
                }
                if(bodyJson.has(ATTR_LATITUDE)){
                    latitude = bodyJson.getDouble(ATTR_LATITUDE);
                }
                if(bodyJson.has(ATTR_LONGITUDE)){
                    longitude = bodyJson.getDouble(ATTR_LONGITUDE);
                }
                EMLocationMessageBody body = new EMLocationMessageBody(address, latitude, longitude);
                msg.setBody(body);

            }else if(type.equals(ATTR_TYPE_CMD)){
                msg = new Message(Message.Type.CMD);
                String action = "";
                if(bodyJson.has(ATTR_ACTION)){
                    action = bodyJson.getString(ATTR_ACTION);
                }
                EMCmdMessageBody body = new EMCmdMessageBody(action);
                msg.setBody(body);
            }
            if (msg == null){
                return null;
            }
            msg.setFrom(from);
            msg.setTo(to);
            if (jsonPayload.has(ATTR_EXT)) {
                JSONObject extAttrObj = jsonPayload.getJSONObject(ATTR_EXT);
                Iterator iter = extAttrObj.keys();
                while (iter.hasNext()) {
                    String key = (String) iter.next();
                    Object obj = extAttrObj.get(key);
                    if (obj instanceof String) {
                        msg.setAttribute(key, (String) obj);
                    } else if (obj instanceof Integer) {
                        msg.setAttribute(key, (Integer) obj);
                    } else if (obj instanceof Long) {
                        msg.setAttribute(key, ((Long) obj));
                    } else if (obj instanceof Boolean) {
                        msg.setAttribute(key, (Boolean) obj);
                    } else if (obj instanceof JSONObject) {
                        msg.setAttribute(key, (JSONObject) obj);
                    } else if (obj instanceof JSONArray) {
                        msg.setAttribute(key, (JSONArray) obj);
                    } else {
                        EMLog.e("msg", "unknow additonal msg attr:"
                                + obj.getClass().getName());
                    }
                }
            }
            return msg;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }



    }





}
