package com.hyphenate.chat;

import android.text.TextUtils;
import android.util.Base64;

import com.hyphenate.agora.FunctionIconItem;
import com.hyphenate.helpdesk.Error;
import com.hyphenate.helpdesk.callback.ValueCallBack;
import com.hyphenate.helpdesk.httpclient.HttpClient;
import com.hyphenate.helpdesk.httpclient.HttpRequestBuilder;
import com.hyphenate.helpdesk.httpclient.HttpResponse;
import com.hyphenate.helpdesk.util.Log;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.Utils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.FileNameMap;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

/**
 * Created by liyuzhao on 20/06/2017.
 */

final class MarketingHttpClient {

    private static final String TAG = "MarketingHttpClient";

    public static int DEFAULT_CONNECTION_TIMEOUT = 20 * 1000;
    public static int DEFAULT_READ_TIMEOUT = 2 * 60 * 1000;

    private static ExecutorService sendThreadPool = Executors
            .newFixedThreadPool(10);

    private static final Object lock = new Object();

    public static String currentConversationId;

    private MarketingHttpClient() {
        throw new IllegalAccessError("not support constructor new object");
    }

    private static void sendMarketingDelivered(String marketingTaskId, final String toUser, ValueCallBack<String> callBack) {
        String tenantId = ChatClient.getInstance().tenantId();
        if (TextUtils.isEmpty(tenantId)) {
            if (callBack != null) {
                callBack.onError(-1, "tenantid isnot nullable");
            }
            return;
        }

        String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/delivered", ChatClient.getInstance().kefuRestServer(), tenantId, marketingTaskId);
        sendMarketingStatusChanged(remoteUrl, toUser, callBack);
    }

    private static void sendMarketingOpened(String marketingTaskId, final String toUser, ValueCallBack<String> callBack) {
        String tenantId = ChatClient.getInstance().tenantId();
        if (TextUtils.isEmpty(tenantId)) {
            if (callBack != null) {
                callBack.onError(-1, "tenantid isnot nullable");
            }
            return;
        }
        String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/opened", ChatClient.getInstance().kefuRestServer(), tenantId, marketingTaskId);
        sendMarketingStatusChanged(remoteUrl, toUser, callBack);
    }

    private static void sendMarketingReplied(String marketingTaskId, final String toUser, ValueCallBack<String> callBack) {
        String tenantId = ChatClient.getInstance().tenantId();
        if (TextUtils.isEmpty(tenantId)) {
            if (callBack != null) {
                callBack.onError(-1, "tenantid isnot nullable");
            }
            return;
        }
        String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/replied", ChatClient.getInstance().kefuRestServer(), tenantId, marketingTaskId);
        sendMarketingStatusChanged(remoteUrl, toUser, callBack);
    }


    private static void sendMarketingStatusChanged(final String remoteUrl, final String toUser, final ValueCallBack<String> callBack) {
        if (!ChatClient.getInstance().isLoggedInBefore()) {
            if (callBack != null) {
                callBack.onError(-1, "please login first");
            }
            return;
        }

        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String visitorId;

                    synchronized (lock) {
                        visitorId = KefuDBManager.getInstance().getVisitorUserId(toUser);
                        if (TextUtils.isEmpty(visitorId)) {
                            visitorId = getVisitorIdFromRemote(toUser);
                        }
                    }
                    if (TextUtils.isEmpty(visitorId)) {
                        if (callBack != null) {
                            callBack.onError(-1, "visitorid is null");
                        }
                        return;
                    }

                    String currentUser = ChatClient.getInstance().currentUserName();
                    String token = ChatClient.getInstance().accessToken();
                    String appkey = ChatClient.getInstance().appKey();
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("visitor_id", visitorId);

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.put(remoteUrl);
                    assert appkey != null;
                    requestBuilder.param("orgName", appkey.split("#")[0]);
                    requestBuilder.param("appName", appkey.split("#")[1]);
                    requestBuilder.param("userName", currentUser);
                    requestBuilder.param("token", token);
                    requestBuilder.content(jsonObject.toString().getBytes(), "application/json");
                    HttpResponse httpResponse = requestBuilder.execute();
                    assert httpResponse != null;
                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "marketing status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    } else {
                        if (callBack != null) {
                            callBack.onError(-1, "send marketing status fail");
                        }
                    }

                } catch (Exception e) {
                    if (callBack != null) {
                        callBack.onError(-1, "error:" + e.getMessage());
                    }
                }
            }
        });

    }

    private static synchronized String getVisitorIdFromRemote(String toUser) {
        if (!ChatClient.getInstance().isLoggedInBefore()) {
            EMLog.e(TAG, "get visitor but not login");
            return "";
        }
        String tenantId = ChatClient.getInstance().tenantId();
        if (TextUtils.isEmpty(tenantId)) {
            return "";
        }
        try {
            String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors", ChatClient.getInstance().kefuRestServer(), tenantId);
            String appkey = ChatClient.getInstance().appKey();
            String token = ChatClient.getInstance().accessToken();
            String currentUser = ChatClient.getInstance().currentUserName();
            String techChannelinfo = appkey + "#" + toUser;
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
            HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
            assert appkey != null;
            requestBuilder.param("orgName", appkey.split("#")[0]);
            requestBuilder.param("appName", appkey.split("#")[1]);
            requestBuilder.param("userName", currentUser);
            requestBuilder.param("token", token);
            requestBuilder.param("techChannelInfo", techChannelinfo);
            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                return "";
            }
            int status_code = httpResponse.getStatusCode();
            EMLog.d(TAG, "getVisitorIdFromRemote status_code:" + status_code);
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                JSONObject resultJson = new JSONObject(result);
                if (resultJson.has("entity")) {
                    JSONObject entitiesJson = resultJson.getJSONObject("entity");
                    String userId = entitiesJson.getString("userId");
                    KefuDBManager.getInstance().saveVisitorInfo(toUser, userId);
                    return userId;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    static void asyncGetAgentState(final ConversationInfo info, final ValueCallBack<String> callBack) {

        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String remoteUrl = String.format("%1$s/v1/webimplugin/sessions/%2$s/agent-input-state",
                            ChatClient.getInstance().kefuRestServer(), info.session_id);
                    String appkey = ChatClient.getInstance().appKey();
                    String token = ChatClient.getInstance().accessToken();
                    String currentUser = ChatClient.getInstance().currentUserName();
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    assert appkey != null;
                    requestBuilder.param("orgName", appkey.split("#")[0]);
                    requestBuilder.param("appName", appkey.split("#")[1]);
                    requestBuilder.param("userName", currentUser);
                    requestBuilder.param("token", token);
                    requestBuilder.param("tenantId", ChatClient.getInstance().tenantId());
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "error: response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "asyncGetAgentState status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        JSONObject resultJson = new JSONObject(result);
                        JSONObject entitiesJson = resultJson.getJSONObject("entity");
                        String input_state_tips = entitiesJson.optString("input_state_tips");
                        if (callBack != null) {
                            callBack.onSuccess(input_state_tips);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error:" + e.getMessage());
                    }
                }

            }
        });
    }

    static void postMessagePredict(final String content, final ConversationInfo info, final ValueCallBack<String> callback) {

        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                if (!ChatClient.getInstance().isLoggedInBefore()) {
                    if (callback != null) {
                        callback.onError(-1, "no login");
                    }
                    return;
                }
                String tenantId = ChatClient.getInstance().tenantId();
                if (TextUtils.isEmpty(tenantId)) {
                    if (callback != null) {
                        callback.onError(-1, "tenantId is null");
                    }
                    return;
                }

                try {
                    String remoteUrl = String.format("%1$s/v1/webimplugin/servicesessions/%2$s/messagePredict", ChatClient.getInstance().kefuRestServer(), info.session_id);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Authorization", "Easemob IM " + ChatClient.getInstance().accessToken());
                    requestBuilder.header("Content-Type", "application/json");

                    JSONObject postContent = new JSONObject();
                    postContent.put("visitor_user_id", info.visitor_id);
                    postContent.put("content", content);
                    postContent.put("timestamp", System.currentTimeMillis());
                    requestBuilder.content(postContent.toString().getBytes(), "application/json");

                    HttpResponse httpResponse = requestBuilder.execute();
                    assert httpResponse != null;
                    int statusCode = httpResponse.getStatusCode();
                    if (statusCode == HttpURLConnection.HTTP_OK) {
                        StringBuffer sb = new StringBuffer();
                        httpResponse.read(sb);
                        if (callback != null) {
                            callback.onSuccess(sb.toString());
                        }
                    } else {
                        if (callback != null) {
                            callback.onError(-1, "statusCode:" + statusCode);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callback != null) {
                        callback.onError(-1, e.getMessage());
                    }
                }
            }
        });
    }

    static void asyncGetWaitCount(final ConversationInfo info, final ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String remoteUrl = String.format("%1$s/v1/visitors/waitings/data", ChatClient.getInstance().kefuRestServer());
                    String tenantId = ChatClient.getInstance().tenantId();
                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    requestBuilder.param("tenantId", tenantId);
                    requestBuilder.param("queueId", String.valueOf(info.skill_group_id));
                    requestBuilder.param("serviceSessionId", info.session_id);
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "asyncGetWaitCount status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });

    }

    public static synchronized OfficialAccount getSystemOfficialAccount(String conversationId) {
        String visitorId;
        String toUser = conversationId;
        if (toUser.contains(OfficialAccount.SEPARATOR)) {
            toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
        }
        synchronized (lock) {
            visitorId = KefuDBManager.getInstance().getVisitorUserId(conversationId);
            if (TextUtils.isEmpty(visitorId)) {
                visitorId = getVisitorIdFromRemote(toUser);
            }
        }
        if (TextUtils.isEmpty(visitorId)) {
            return null;
        }
        try {
            String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors/%3$s/official-accounts",
                    ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), visitorId);
            String appkey = ChatClient.getInstance().appKey();
            String token = ChatClient.getInstance().accessToken();
            String currentUser = ChatClient.getInstance().currentUserName();
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
            HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
            assert appkey != null;
            requestBuilder.param("orgName", appkey.split("#")[0]);
            requestBuilder.param("appName", appkey.split("#")[1]);
            requestBuilder.param("userName", currentUser);
            requestBuilder.param("token", token);
            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                return null;
            }
            int status_code = httpResponse.getStatusCode();
            EMLog.d(TAG, "getSystemOfficialAccount status_code:" + status_code);
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                JSONObject resultJson = new JSONObject(result);
                if (resultJson.has("entities") && !resultJson.isNull("entities")) {
                    JSONArray entitiesJson = resultJson.getJSONArray("entities");
                    for (int j = 0; j < entitiesJson.length(); j++) {
                        JSONObject itemJson = entitiesJson.getJSONObject(j);
                        String type = itemJson.getString("type");
                        if (type != null && type.equals("SYSTEM")) {
                            OfficialAccount officialAccount = new OfficialAccount();
                            officialAccount.setId(itemJson.getString("official_account_id"));
                            if (itemJson.has("img")) {
                                officialAccount.setImg(itemJson.getString("img"));
                            }
                            if (itemJson.has("name")) {
                                officialAccount.setName(itemJson.getString("name"));
                            }
                            officialAccount.setType("SYSTEM");
                            ChatClient.getInstance().chatManager().getConversation(conversationId).setOfficialAccount(officialAccount);
                            return officialAccount;
                        }
                    }
                }
            }
        } catch (Exception ignored) {
            ignored.printStackTrace();
        }
        return null;
    }

    static synchronized ConversationInfo getLastestSession(String officialAccountId, String conversationId) {
        String visitorId;
        String toUser = conversationId;
        if (toUser.contains(OfficialAccount.SEPARATOR)) {
            toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
        }
        synchronized (lock) {
            visitorId = KefuDBManager.getInstance().getVisitorUserId(conversationId);
            if (TextUtils.isEmpty(visitorId)) {
                visitorId = getVisitorIdFromRemote(toUser);
            }
        }
        if (TextUtils.isEmpty(visitorId)) {
            return null;
        }
        try {
            String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors/%3$s/official-accounts/%4$s/latest-session",
                    ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), visitorId, officialAccountId);
            String appkey = ChatClient.getInstance().appKey();
            String token = ChatClient.getInstance().accessToken();
            String currentUser = ChatClient.getInstance().currentUserName();
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
            HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
            assert appkey != null;
            requestBuilder.param("orgName", appkey.split("#")[0]);
            requestBuilder.param("appName", appkey.split("#")[1]);
            requestBuilder.param("userName", currentUser);
            requestBuilder.param("token", token);
            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                return null;
            }
            int status_code = httpResponse.getStatusCode();
            EMLog.d(TAG, "getLastestSession status_code:" + status_code);
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                JSONObject resultJson = new JSONObject(result);
                if (resultJson.has("entity") && !resultJson.isNull("entity")) {
                    JSONObject entitiesJson = resultJson.getJSONObject("entity");
                    ConversationInfo info = new ConversationInfo();
                    if (entitiesJson.has("session_id")) {
                        info.session_id = entitiesJson.getString("session_id");
                    }
                    if (entitiesJson.has("visitor_id")) {
                        info.visitor_id = entitiesJson.getString("visitor_id");
                    }
                    if (entitiesJson.has("skill_group_id")) {
                        info.skill_group_id = entitiesJson.getLong("skill_group_id");
                    }
                    if (entitiesJson.has("state")) {
                        info.state = entitiesJson.getString("state");
                    }
                    if (info.session_id != null) {
                        ChatClient.getInstance().chatManager().getConversation(conversationId).setConversationInfo(info);
                        currentConversationId = conversationId;
                    }
                    return info;
                }
            }
        } catch (Exception ignored) {
            ignored.printStackTrace();
        }
        return null;
    }


    static void asyncOpened(final String conversationId) {
        if (TextUtils.isEmpty(conversationId)) {
            return;
        }
        String toUser = conversationId;
        if (toUser.contains(OfficialAccount.SEPARATOR)) {
            toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
        }
        final String tempToUser = toUser;
        List<String> marketingIds = KefuDBManager.getInstance().getMarketingIdsByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_DELIVERED);
        if (marketingIds != null && !marketingIds.isEmpty()) {
            KefuDBManager.getInstance().updateMarketingInfoStatusByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_DELIVERED, Constants.MESSAGE_MARKETING_STATUS_OPENED);
            for (final String marketingId : marketingIds) {
                sendThreadPool.execute(new Runnable() {
                    @Override
                    public void run() {
                        MarketingHttpClient.sendMarketingOpened(marketingId, tempToUser, new ValueCallBack<String>() {
                            @Override
                            public void onSuccess(String value) {
                            }

                            @Override
                            public void onError(int error, String errorMsg) {
                                String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/opened", ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), marketingId);
                                RequestInfo info = new RequestInfo();
                                info.key = RequestInfo.REQUEST_KEY_MARKETING;
                                info.url = remoteUrl;
                                info.params = tempToUser;
                                KefuDBManager.getInstance().addRequestInfo(info);
                            }
                        });
                    }
                });
            }
        }

    }

    static void asyncDelivered(final String taskId, final String conversationId) {
        if (TextUtils.isEmpty(taskId) || TextUtils.isEmpty(conversationId)) {
            return;
        }
        KefuDBManager.getInstance().addMarketingInfo(taskId, conversationId, Constants.MESSAGE_MARKETING_STATUS_DELIVERED);
        String toUser = conversationId;
        if (toUser.contains(OfficialAccount.SEPARATOR)) {
            toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
        }
        final String tempToUser = toUser;
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                MarketingHttpClient.sendMarketingDelivered(taskId, tempToUser, new ValueCallBack<String>() {
                    @Override
                    public void onSuccess(String value) {
                    }

                    @Override
                    public void onError(int error, String errorMsg) {
                        String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/delivered", ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), taskId);
                        RequestInfo info = new RequestInfo();
                        info.key = RequestInfo.REQUEST_KEY_MARKETING;
                        info.url = remoteUrl;
                        info.params = tempToUser;
                        KefuDBManager.getInstance().addRequestInfo(info);
                    }
                });
            }
        });
    }

    static void asyncReplied(final String conversationId) {
        if (TextUtils.isEmpty(conversationId)) {
            return;
        }
        String toUser = conversationId;
        if (toUser.contains(OfficialAccount.SEPARATOR)) {
            toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
        }
        final String tempToUser = toUser;
        List<String> marketingIds = KefuDBManager.getInstance().getMarketingIdsByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_OPENED);
        if (marketingIds != null && !marketingIds.isEmpty()) {
            KefuDBManager.getInstance().deleteMarketingInfoByConversationId(conversationId, Constants.MESSAGE_MARKETING_STATUS_OPENED);
            for (final String marketingId : marketingIds) {
                sendThreadPool.execute(new Runnable() {
                    @Override
                    public void run() {
                        MarketingHttpClient.sendMarketingReplied(marketingId, tempToUser, new ValueCallBack<String>() {
                            @Override
                            public void onSuccess(String value) {
                            }

                            @Override
                            public void onError(int error, String errorMsg) {
                                String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/marketing-tasks/%3$s/replied", ChatClient.getInstance().kefuRestServer(), ChatClient.getInstance().tenantId(), marketingId);
                                RequestInfo info = new RequestInfo();
                                info.key = RequestInfo.REQUEST_KEY_MARKETING;
                                info.url = remoteUrl;
                                info.params = tempToUser;
                                KefuDBManager.getInstance().addRequestInfo(info);

                            }
                        });
                    }
                });
            }
        }

    }

    static void asyncRequest() {
        try {
            List<RequestInfo> requestInfos = KefuDBManager.getInstance().getRequestInfoByKey(RequestInfo.REQUEST_KEY_MARKETING);
            if (requestInfos != null && !requestInfos.isEmpty()) {
                for (final RequestInfo info : requestInfos) {
                    sendMarketingStatusChanged(info.url, info.params, new ValueCallBack<String>() {
                        @Override
                        public void onSuccess(String value) {
                            KefuDBManager.getInstance().deleteRequestInfoById(info.id);
                        }

                        @Override
                        public void onError(int error, String errorMsg) {

                        }
                    });
                }
            }
        } catch (Exception ignored) {
        }
    }


    static void asyncGetTenantIdFunctionIcons(final String tenantId, final ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    // /v1/grayscale/tenants
                    String remoteUrl = String.format("%1$s/v1/grayscale/tenants/" + tenantId, ChatClient.getInstance().kefuRestServer());
                    String tenantId = ChatClient.getInstance().tenantId();
                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "asyncGetTenantIdFunctionIcons status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });

    }

    // 加入互动白板房间
    static void asyncWhiteboardFileConversion(final String token, final String tenantId, final String callId, final ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String content = "{}";
                    //https://sandbox.kefu.easemob.com/v1/agorartc/tenant/121312/whiteboard/call/12345/conversion
                    String remoteUrl = String.format("%1$s/v1/agorartc/tenant/%2$s/whiteboard/call/%3$s/conversion",
                            ChatClient.getInstance().kefuRestServer(), tenantId, callId);
                    String tenantId = ChatClient.getInstance().tenantId();
                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Content-Type", "application/json");
                    requestBuilder.header("Authorization", token);
                    requestBuilder.content(content.getBytes(), "application/json");
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }

                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "asyncGetWhiteboardRoomId status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }


    private static String lineEnd = "\r\n";
    private static String twoHyphens = "--";
    private static String boundary = "*****";

    static void asyncUploadFile(final String filePath, final String token, final int callId, final String tenantId, final ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    //tenantId callId
                    // /v1/whiteboard/tenant/{%2$s}/session/{%3$s}/conversion/upload
                    // /v1/agorartc/tenant/{tenantId}/whiteboard/call/{callId}/conversion/upload
                    // ChatClient.getInstance().kefuRestServer();
                    // String urlRoot = "http://172.17.1.134:8599";
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/upload", urlRoot, tenantId, callId);
                    EMLog.d(TAG, "asyncUploadFile request url:" + remoteUrl);


                    File file = new File(filePath);
                    URL url = new URL(remoteUrl);
                    HttpURLConnection conn = (HttpURLConnection) url.openConnection();
                    //conn.setRequestProperty("User-Agent","Mozilla/5.0 ( compatible ) ");
                    //conn.setRequestProperty("Accept","*/*");
                    conn.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    conn.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    conn.setUseCaches(false);
                    conn.setRequestProperty("Connection", "Keep-Alive");
                    conn.setRequestMethod("POST");
                    conn.setRequestProperty("Authorization", /*"Bearer " + */token);
                    // conn.setRequestProperty("restrict-access", "true");
                    conn.setRequestProperty("Charset", "utf-8");
                    conn.setDoInput(true);

                    conn.setRequestProperty("Content-Type", "multipart/form-data;boundary=" + boundary);
                    conn.setRequestProperty("file", file.getAbsolutePath());

                    OutputStream outputStream = new DataOutputStream(conn.getOutputStream());
                    outputStream.write((twoHyphens + boundary + lineEnd).getBytes());
                    outputStream.write(("Content-Disposition: form-data; name=\"file\";filename=\""
                            + file.getName() + "\"" + lineEnd).getBytes());
                    outputStream.write(("Content-Type: " + /*guessMimeType(file.getPath())*/CommonUtils.getMIMEType(file) + lineEnd + lineEnd).getBytes());

                    DataInputStream in = new DataInputStream(new FileInputStream(file));
                    int length = -1;
                    byte[] bufferOut = new byte[1024];
                    //从文件读取数据到缓存区
                    while ((length = in.read(bufferOut)) != -1) {
                        //将资料写入 DataOutputStream中
                        outputStream.write(bufferOut, 0, length);
                    }

                    byte[] endData = (lineEnd + twoHyphens + boundary + twoHyphens + lineEnd).getBytes();
                    outputStream.write(endData);
                    outputStream.flush();
                    outputStream.close();
                    in.close();


                    InputStream inputStream = conn.getInputStream();
                    conn.connect();

                    int responseCode = conn.getResponseCode();

                    StringBuilder stringBuilder = new StringBuilder();
                    BufferedReader bf = new BufferedReader(new InputStreamReader(inputStream, "utf-8"));
                    String line = null;
                    while ((line = bf.readLine()) != null) {
                        stringBuilder.append(line);
                    }

                    bf.close();
                    inputStream.close();
                    if (callBack != null) {
                        callBack.onSuccess(stringBuilder.toString());
                        Log.e("","");
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + e.toString());
                    }
                }
            }
        });
    }

    static void asyncFileConverted(final String filePath, final String token, final int callId, final String tenantId, String type, final ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    //tenantId callId
                    // /v1/agorartc/tenant/{tenantId}/whiteboard/call/{callId}/conversion
                    // ChatClient.getInstance().kefuRestServer();
                    // String urlRoot = "http://172.17.1.134:8599";
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion", urlRoot, tenantId, callId);

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Content-Type", "application/json");
                    requestBuilder.header("Authorization", token);

                    JSONObject postContent = new JSONObject();
                    postContent.put("resource", filePath);
                    postContent.put("type", type); //"dynamic" static
                    // postContent.put("preview", "dynamic".equals(type));
                    //postContent.put("preview", false);
                    // 缩放比例，在静态时生效
                    //postContent.put("scale", 1);
                    //postContent.put("outputFormat", "png");
                    requestBuilder.content(postContent.toString().getBytes(), "application/json");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }

                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "asyncGetWhiteboardRoomId status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + e.toString());
                    }
                }
            }
        });
    }

    static void asyncFileConvertedProgress(final String tenantId, final String token, final int callId, String type, String taskId, final ValueCallBack<String> callBack) {

        try {
            // /v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/{taskId}
            //String urlRoot = "http://172.17.1.134:8599";
            String urlRoot = ChatClient.getInstance().kefuRestServer();
            final String remoteUrl = String.format("%1$s/v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/%4$s?type=%5$s", urlRoot, tenantId, callId, taskId, type);
            EMLog.d(TAG, "asyncFileConvertedProgress request url:" + remoteUrl);

            /*String tenantId = ChatClient.getInstance().tenantId();
            if (tenantId == null) {
                if (callBack != null) {
                    callBack.onError(-1, "tenantid is null");
                }
                return;
            }*/

            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
            httpClient.setReadTimeout(3 * 60 * 1000);
            HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
            requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");
            requestBuilder.header("Authorization", token);
            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                if (callBack != null) {
                    callBack.onError(-1, "response is null");
                }
                return;
            }
            int status_code = httpResponse.getStatusCode();
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                if (callBack != null) {
                    callBack.onSuccess(result);
                }
            }
        } catch (Exception ignored) {
            ignored.printStackTrace();
            if (callBack != null) {
                callBack.onError(-1, "error->" + ignored);
            }
        }


        /*sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    // /v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/{taskId}
                    //String urlRoot = "http://172.17.1.134:8599";
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/%4$s?type=%5$s", urlRoot, tenantId, callId, taskId, type);
                    Log.e("tttttttttt", "请求地址 remoteUrl = " + remoteUrl);

                    String tenantId = ChatClient.getInstance().tenantId();
                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(3 * 60 * 1000);
                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");
                    requestBuilder.header("Authorization", token);
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });*/

    }

    static void asyncFileSignName(final String tenantId, final String token, final int callId, String filePath,
                                  ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    String tenantId = ChatClient.getInstance().tenantId();
                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // /v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/sign/%4$s?type=%5$s
                    // String urlRoot = "http://172.17.1.134:8599";
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/agorartc/tenant/%2$s/whiteboard/call/%3$d/conversion/sign", urlRoot, tenantId, callId);
                    EMLog.d(TAG, "asyncFileSignName request url:" + remoteUrl);

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    requestBuilder.header("Content-Type", "application/json");
                    requestBuilder.header("Authorization", token);
                    JSONObject postContent = new JSONObject();
                    postContent.put("urls", filePath);
                    requestBuilder.content(postContent.toString().getBytes(), "application/json");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });

    }

    public static String getFileTypeByFilePath(File file) {
        String fileName = file.getName();
        String prefix = fileName.substring(fileName.lastIndexOf(".") + 1);
        String type = "application/octet-stream";
        try {
            if ("png".equals(prefix)) {
                type = "png";
            } else if ("jpg".equals(prefix)) {
                type = "jpg";
            } else if ("mp4".equals(prefix)) {
                type = "mp4";
            } else if ("mp3".equals(prefix)) {
                type = "mp3";
            } else if ("pptx".equals(prefix)) {
                type = "pptx";
            } else if ("ppt".equals(prefix)) {
                type = "ppt";
            } else if ("pdf".equals(prefix)) {
                type = "pdf";
            } else if ("doc".equals(prefix)) {
                type = "doc";
            } else if ("docx".equals(prefix)) {
                type = "docx";
            } else if ("xlsx".equals(prefix)) {
                type = "xlsx";
            } else if ("txt".equals(prefix)) {
                type = "txt";
            }else if ("webp".equals(prefix)) {
                type = "webp";
            }else {
                type = "";
            }
        }catch (Exception e){
            e.printStackTrace();
        }
        return type;
    }

    public static String doPostSubmitBody(String url, Map<String, String> map,
                                          String filePath, byte[] body_data, String charset) {
        // 设置三个常用字符串常量：换行、前缀、分界线（NEWLINE、PREFIX、BOUNDARY）；
        final String NEWLINE = "\r\n";
        final String PREFIX = "--";
        final String BOUNDARY = "#";
        HttpURLConnection httpConn = null;
        BufferedInputStream bis = null;
        DataOutputStream dos = null;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            // 实例化URL对象。调用URL有参构造方法，参数是一个url地址；
            URL urlObj = new URL(url);
            // 调用URL对象的openConnection()方法，创建HttpURLConnection对象；
            httpConn = (HttpURLConnection) urlObj.openConnection();
            // 调用HttpURLConnection对象setDoOutput(true)、setDoInput(true)、setRequestMethod("POST")；
            httpConn.setDoInput(true);
            httpConn.setDoOutput(true);
            httpConn.setRequestMethod("POST");
            // 设置Http请求头信息；（Accept、Connection、Accept-Encoding、Cache-Control、Content-Type、User-Agent）
            httpConn.setUseCaches(false);
            httpConn.setRequestProperty("Connection", "Keep-Alive");
            httpConn.setRequestProperty("Accept", "*/*");
            httpConn.setRequestProperty("Accept-Encoding", "gzip, deflate");
            httpConn.setRequestProperty("Cache-Control", "no-cache");
            httpConn.setRequestProperty("Content-Type",
                    "multipart/form-data; boundary=" + BOUNDARY);
            httpConn.setRequestProperty(
                    "User-Agent",
                    "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; .NET CLR 2.0.50727; .NET CLR 3.0.04506.30)");
            // 调用HttpURLConnection对象的connect()方法，建立与服务器的真实连接；
            httpConn.connect();

            // 调用HttpURLConnection对象的getOutputStream()方法构建输出流对象；
            dos = new DataOutputStream(httpConn.getOutputStream());
            // 获取表单中上传控件之外的控件数据，写入到输出流对象（根据HttpWatch提示的流信息拼凑字符串）；
            if (map != null && !map.isEmpty()) {
                for (Map.Entry<String, String> entry : map.entrySet()) {
                    String key = entry.getKey();
                    String value = map.get(key);
                    dos.writeBytes(PREFIX + BOUNDARY + NEWLINE);
                    dos.writeBytes("Content-Disposition: form-data; "
                            + "name=\"" + key + "\"" + NEWLINE);
                    dos.writeBytes(NEWLINE);
                    dos.writeBytes(URLEncoder.encode(value.toString(), charset));
                    // 或者写成：dos.write(value.toString().getBytes(charset));
                    dos.writeBytes(NEWLINE);
                }
            }

            // 获取表单中上传控件的数据，写入到输出流对象（根据HttpWatch提示的流信息拼凑字符串）；
            if (body_data != null && body_data.length > 0) {
                dos.writeBytes(PREFIX + BOUNDARY + NEWLINE);
                String fileName = filePath.substring(filePath
                        .lastIndexOf(File.separatorChar));
                dos.writeBytes("Content-Disposition: form-data; " + "name=\""
                        + "uploadFile" + "\"" + "; filename=\"" + fileName
                        + "\"" + NEWLINE);
                dos.writeBytes(NEWLINE);
                dos.write(body_data);
                dos.writeBytes(NEWLINE);
            }
            dos.writeBytes(PREFIX + BOUNDARY + PREFIX + NEWLINE);
            dos.flush();

            // 调用HttpURLConnection对象的getInputStream()方法构建输入流对象；
            byte[] buffer = new byte[8 * 1024];
            int c = 0;
            // 调用HttpURLConnection对象的getResponseCode()获取客户端与服务器端的连接状态码。如果是200，则执行以下操作，否则返回null；
            if (httpConn.getResponseCode() == 200) {
                bis = new BufferedInputStream(httpConn.getInputStream());
                while ((c = bis.read(buffer)) != -1) {
                    baos.write(buffer, 0, c);
                    baos.flush();
                }
            }
            // 将输入流转成字节数组，返回给客户端。
            return new String(baos.toByteArray(), charset);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                dos.close();
                bis.close();
                baos.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    private static String guessMimeType(String path) {
        FileNameMap fileNameMap = URLConnection.getFileNameMap();
        String contentTypeFor = null;
        try {
            contentTypeFor = fileNameMap.getContentTypeFor(path);
        } catch (Exception e) {
            return "application/octet-stream";
        }
        if (contentTypeFor == null) {
            contentTypeFor = "application/octet-stream";
        }
        return contentTypeFor;
    }


    public static void asyncInitLanguage(String tenantId, String language, String appKey, String visitorUserName, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    // v1/webimplugin/tenants/%2$s/visitors/init-language/set
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors/init-language/set", urlRoot, tenantId);
                    EMLog.d(TAG, "asyncInitLanguage request url:" + remoteUrl);

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Content-Type", "application/json");

                    JSONObject postContent = new JSONObject();
                    postContent.put("language", language);
                    postContent.put("appkey", appKey);
                    postContent.put("visitorUserName", visitorUserName);
                    requestBuilder.content(postContent.toString().getBytes(), "application/json");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }

                    int status_code = httpResponse.getStatusCode();
                    EMLog.d(TAG, "asyncInitLanguage status_code:" + status_code);
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + e.toString());
                    }
                }
            }
        });
    }

    public static void asyncInitStyle(String tenantId, String configId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            EMLog.e(TAG, "asyncInitStyle tenantId is null");
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // %1$s/v1/kefurtc/tenant/%2$s/plugin/webim/config/%3$s
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    // v1/kefurtc/visitor/plugin/webim/
                    // %1$s/v1/kefurtc/visitor/plugin/webim/config/%2$s
                    // final String remoteUrl = String.format("%1$s/v1/kefurtc/tenant/%2$s/plugin/webim/config/%3$s", urlRoot, tenantId, configId);
                    final String remoteUrl = String.format("%1$s/v1/kefurtc/visitor/plugin/webim/%2$s", urlRoot, configId);
                    EMLog.e(TAG, "asyncInitStyle request url:" + remoteUrl);

                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            EMLog.e(TAG, "asyncInitStyle httpResponse: requestBuilder.execute() result is null.");
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            EMLog.e(TAG, "asyncInitStyle result:" + result);
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    EMLog.e(TAG, "asyncInitStyle error:" + ignored.toString());
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncCloseVec(String tenantId, String rtcSessionId, String visitorId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // %1$s/v1/kefurtc/tenant/%2$s/session/%3$s/visitor/%4$s/close
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/kefurtc/tenant/%2$s/session/%3$s/visitor/%4$s/close", urlRoot, tenantId, rtcSessionId, visitorId);
                    EMLog.d(TAG, "asyncCloseVec request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void closeVec(String tenantId, String rtcSessionId, String visitorId, ValueCallBack<String> callBack) {
        try {

            if (tenantId == null) {
                if (callBack != null) {
                    callBack.onError(-1, "tenantid is null");
                }
                return;
            }

            // %1$s/v1/kefurtc/tenant/%2$s/session/%3$s/visitor/%4$s/close
            String urlRoot = ChatClient.getInstance().kefuRestServer();
            final String remoteUrl = String.format("%1$s/v1/kefurtc/tenant/%2$s/session/%3$s/visitor/%4$s/close", urlRoot, tenantId, rtcSessionId, visitorId);
            EMLog.d(TAG, "asyncCloseVec request url:" + remoteUrl);
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(8000);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
            HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
            requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");
            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                if (callBack != null) {
                    callBack.onError(-1, "response is null");
                }
                return;
            }
            int status_code = httpResponse.getStatusCode();
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                if (callBack != null) {
                    callBack.onSuccess(result);
                }
            }
        } catch (Exception ignored) {
            ignored.printStackTrace();
            if (callBack != null) {
                callBack.onError(-1, "error->" + ignored);
            }
        }
    }

    public static void asyncVisitorId(String toUser, ValueCallBack<String> callBack){
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                getVisitorIdAndVecSessionId(toUser, callBack);
            }
        });
    }

    public static void getVisitorId(String toUser, ValueCallBack<String> callBack){
        getVisitorIdAndVecSessionId(toUser, callBack);
    }

    public static void getVisitorIdAndVecSessionId(String conversationId, ValueCallBack<String> callBack) {
        String visitorId;
        String toUser = conversationId;
        if (toUser.contains(OfficialAccount.SEPARATOR)) {
            toUser = toUser.substring(0, toUser.indexOf(OfficialAccount.SEPARATOR));
        }
        /*synchronized (lock) {
            visitorId = KefuDBManager.getInstance().getVisitorUserId(conversationId);
            if (TextUtils.isEmpty(visitorId)) {
                visitorId = getVisitorId(toUser);
            }
        }*/

        visitorId = getVisitorId(toUser);
        if (TextUtils.isEmpty(visitorId)) {
            if (callBack != null){
                callBack.onError(-1, "visitorId = null");
            }
            return;
        }

        if (callBack != null){
            callBack.onSuccess(visitorId);
        }
    }

    private static String getVisitorId(String toUser) {
        if (!ChatClient.getInstance().isLoggedInBefore()) {
            EMLog.e(TAG, "get visitor but not login");
            return "";
        }
        String tenantId = ChatClient.getInstance().tenantId();
        if (TextUtils.isEmpty(tenantId)) {
            return "";
        }
        try {
            String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/visitors", ChatClient.getInstance().kefuRestServer(), tenantId);
            String appkey = ChatClient.getInstance().appKey();
            String token = ChatClient.getInstance().accessToken();
            String currentUser = ChatClient.getInstance().currentUserName();
            String techChannelinfo = appkey + "#" + toUser;
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
            HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
            assert appkey != null;
            requestBuilder.param("orgName", appkey.split("#")[0]);
            requestBuilder.param("appName", appkey.split("#")[1]);
            requestBuilder.param("userName", currentUser);
            requestBuilder.param("token", token);
            requestBuilder.param("techChannelInfo", techChannelinfo);
            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                return "";
            }
            int status_code = httpResponse.getStatusCode();
            EMLog.d(TAG, "getVisitorIdFromRemote status_code:" + status_code);
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                JSONObject resultJson = new JSONObject(result);
                if (resultJson.has("entity")) {
                    JSONObject entitiesJson = resultJson.getJSONObject("entity");
                    return entitiesJson.getString("userId");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }


    public static void getSkillGroupMenuWithVisitorUserName(String tenantId, String visitorUserName, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // %1$s/tenants/%2$s/skillgroup-menu
                    String urlRoot = ChatClient.getInstance().kefuRestServer();

                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/skillgroup-menu?visitorUserName=%3$s", urlRoot, tenantId, visitorUserName);
                    EMLog.d(TAG, "asyncInitStyle request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    //requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncResultReporting(String tenantId, String visitorId, String type, JSONObject resultObj, ValueCallBack<String> callBack) {

        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // %1$s/v1/kefurtc/visitor/tenant/%2$s/session/%3$s/business/report
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/kefurtc/visitor/%2$s/tenant/%3$s/business/report", urlRoot, visitorId, tenantId);
                    EMLog.d(TAG, "asyncResultReporting request url:" + remoteUrl);
                    Log.e("rrrrrrrr","asyncResultReporting remoteUrl = "+remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Content-Type", "application/json");

                    JSONObject postContent = new JSONObject();
                    postContent.put("type", type);


                    /*"result": {
                        "flowId":"11111111",  // 流程id，由坐席发过来的【开启信息推送】的消息中获取
                        "action":"infopush_end" // 操作动作：infopush_end 信息推送结束
                    }*/
                    /*JSONObject resultObj = new JSONObject();
                    resultObj.put("flowId",flowId);
                    resultObj.put("action",action);*/
                    postContent.put("result", resultObj); // 成功Success失败Failed

                    requestBuilder.content(postContent.toString().getBytes(), "application/json");



                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void resultReporting(String tenantId, String visitorId, String type, JSONObject resultObj,  ValueCallBack<String> callBack){
        try {

            if (tenantId == null) {
                if (callBack != null) {
                    callBack.onError(-1, "tenantid is null");
                }
                return;
            }

            // %1$s/v1/kefurtc/visitor/%2$s/tenant/%3$s/business/report
            String urlRoot = ChatClient.getInstance().kefuRestServer();
            final String remoteUrl = String.format("%1$s/v1/kefurtc/visitor/%2$s/tenant/%3$s/business/report", urlRoot, visitorId, tenantId);
            EMLog.d(TAG, "resultReporting request url:" + remoteUrl);
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(8000);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

            HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
            requestBuilder.header("Content-Type", "application/json");

            JSONObject postContent = new JSONObject();
            postContent.put("type", type);
            postContent.put("result", resultObj); // 成功Success失败Failed
            requestBuilder.content(postContent.toString().getBytes(), "application/json");

            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                if (callBack != null) {
                    callBack.onError(-1, "response is null");
                }
                return;
            }
            int status_code = httpResponse.getStatusCode();

            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                if (callBack != null) {
                    callBack.onSuccess(result);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            if (callBack != null) {
                callBack.onError(-1, "error->" + e);
            }
        }
    }

    public static void asyncUploadSignatureImage(File file, String token, String tenantId, String flowId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    // String urlRoot = "http://172.17.1.134:8599";
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/Service/vec/visitor/Tenant/%2$s/MediaFiles", urlRoot, tenantId);
                    EMLog.d(TAG, "asyncUploadSignatureImage request url:" + remoteUrl);

                    URL url = new URL(remoteUrl);
                    HttpURLConnection conn = (HttpURLConnection) url.openConnection();
                    conn.setConnectTimeout(DEFAULT_CONNECTION_TIMEOUT);
                    conn.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    conn.setUseCaches(false);
                    conn.setRequestProperty("Connection", "Keep-Alive");
                    conn.setRequestMethod("POST");
                    conn.setRequestProperty("Authorization", /*"Bearer " + */token);
                    // conn.setRequestProperty("restrict-access", "true");
                    conn.setRequestProperty("Charset", "utf-8");
                    conn.setDoInput(true);

                    conn.setRequestProperty("Content-Type", "multipart/form-data;boundary=" + boundary);
                    conn.setRequestProperty("file", file.getAbsolutePath());

                    OutputStream outputStream = new DataOutputStream(conn.getOutputStream());
                    outputStream.write((twoHyphens + boundary + lineEnd).getBytes());
                    outputStream.write(("Content-Disposition: form-data; name=\"file\";filename=\""
                            + file.getName() + "\"" + lineEnd).getBytes());
                    outputStream.write(("Content-Type: " + CommonUtils.getMIMEType(file) + lineEnd + lineEnd).getBytes());
                    DataInputStream in = new DataInputStream(new FileInputStream(file));
                    int length = -1;
                    byte[] bufferOut = new byte[1024];
                    //从文件读取数据到缓存区
                    while ((length = in.read(bufferOut)) != -1) {
                        //将资料写入 DataOutputStream中
                        outputStream.write(bufferOut, 0, length);
                    }

                    byte[] endData = (lineEnd + twoHyphens + boundary + twoHyphens + lineEnd).getBytes();
                    outputStream.write(endData);
                    outputStream.flush();
                    outputStream.close();
                    in.close();

                    conn.connect();
                    int responseCode = conn.getResponseCode();
                    if (responseCode != 200){
                        if (callBack != null){
                            callBack.onError(responseCode, "responseCode = "+responseCode);
                        }
                        return;
                    }

                    InputStream inputStream = conn.getInputStream();

                    StringBuilder stringBuilder = new StringBuilder();
                    BufferedReader bf = new BufferedReader(new InputStreamReader(inputStream, "utf-8"));
                    String line = null;
                    while ((line = bf.readLine()) != null) {
                        stringBuilder.append(line);
                    }

                    bf.close();
                    inputStream.close();

                    String content = stringBuilder.toString();
                    JSONObject jsonObject = new JSONObject(content);
                    String uuid = jsonObject.getString("uuid");
                    String contentType = jsonObject.getString("contentType");
                    String imageUrl = jsonObject.getString("url");
                    String fileName = jsonObject.getString("fileName");


                    String visitorId = VecConfig.newVecConfig().getVisitorId();
                    // String sUrl = new String(encrypt(imageUrl, visitorId),"utf-8");

                    // 上传图片id
                    JSONObject object = new JSONObject();
                    object.put("flowId",flowId);
                    object.put("action","elecsign_end");
                    object.put("url",imageUrl);

                    resultReporting(ChatClient.getInstance().tenantId(), VecConfig.newVecConfig().getVisitorId(), "elecsign", object, callBack);

                    /*getVisitorId(AgoraMessage.newAgoraMessage().getCurrentChatUsername(), new ValueCallBack<String>() {
                        @Override
                        public void onSuccess(String value) {
                            Log.e("rrrrrrrrrrrr","value = "+value);
                            resultReporting(ChatClient.getInstance().tenantId(), value, "elecsign", object, callBack);
                        }

                        @Override
                        public void onError(int error, String errorMsg) {
                            resultReporting(ChatClient.getInstance().tenantId(), "", "elecsign", object, callBack);
                        }
                    });*/

                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null){
                        callBack.onError(-1, e.toString());
                    }
                }
            }
        });
    }

    public static void getWaitNumber(String tenantId, String rtcSessionId, ValueCallBack<String> callBack){
        try {

            if (tenantId == null) {
                if (callBack != null) {
                    callBack.onError(-1, "tenantid is null");
                }
                return;
            }

            // %1$s/v1/kefurtc/tenant/%2$s/webim/session/%3$s/waiting
            String urlRoot = ChatClient.getInstance().kefuRestServer();
            final String remoteUrl = String.format("%1$s/v1/kefurtc/tenant/%2$s/webim/session/%3$s/waiting", urlRoot, tenantId, rtcSessionId);
            EMLog.d(TAG, "asyncGetWaitNumber request url:" + remoteUrl);
            HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
            httpClient.setConnectTimeout(8000);
            httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
            HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
            requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

            HttpResponse httpResponse = requestBuilder.execute();
            if (httpResponse == null) {
                if (callBack != null) {
                    callBack.onError(-1, "response is null");
                }
                return;
            }
            int status_code = httpResponse.getStatusCode();
            if (status_code >= 200 && status_code < 300) {
                StringBuffer buffer = new StringBuffer();
                httpResponse.read(buffer);
                String result = buffer.toString();
                if (callBack != null) {
                    callBack.onSuccess(result);
                }
            }
        } catch (Exception ignored) {
            ignored.printStackTrace();
            if (callBack != null) {
                callBack.onError(-1, "error->" + ignored);
            }
        }
    }

    private static byte[] getSecretKey(String pas, int keySize){
        try {
            String algorithmName = "AES";
            //密钥生成器
            KeyGenerator instance = KeyGenerator.getInstance(algorithmName);
            if(pas == null) {
                instance.init(keySize);
            }else {
                instance.init(keySize, new SecureRandom(pas.getBytes()));//初始化秘钥生成器
            }
            SecretKey secretKey = instance.generateKey();
            return secretKey.getEncoded();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static byte[] encrypt(String content, String secretKey) {
        try {
            String algorithmName = "AES";
            SecretKeySpec key = new SecretKeySpec(getSecretKey(secretKey, 128), algorithmName);//指定秘钥规则（秘钥规则生成器）
            Cipher cipher = Cipher.getInstance(algorithmName);
            byte[] byteContent = content.getBytes("utf-8");
            cipher.init(Cipher.ENCRYPT_MODE, key);// 初始化密码器，指定加密还是解密
            return cipher.doFinal(byteContent);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (NoSuchPaddingException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        } catch (BadPaddingException e) {
            e.printStackTrace();
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        } catch (IllegalBlockSizeException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static void asyncSubmitEvaluate(String tenantId, String jsonData, ValueCallBack<String> callBack) {

        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // POST %1$s/v1/kefurtc/tenant/%2$s/enquiry/visitor/submit
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/kefurtc/tenant/%2$s/enquiry/visitor/submit", urlRoot, tenantId);
                    EMLog.d(TAG, "asyncSubmitEvaluate request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    requestBuilder.header("Content-Type", "application/json");
                    requestBuilder.content(jsonData.getBytes(), "application/json");
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();

                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + e);
                    }
                }
            }
        });
    }

    public static void asyncGetInfo(String tenantId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // %1$s/tenants/%2$s/skillgroup-menu
                    //
                    String urlRoot = ChatClient.getInstance().kefuRestServer();

                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/info", urlRoot, tenantId);
                    EMLog.d(TAG, "asyncGetInfo request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    //requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncProblemSolvingOnServiceSessionResolved(String tenantId, final ValueCallBack<String> callBack){
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    // %1$s/v1/webimplugin/tenants/%2$s/options/problemSolvingOnServiceSessionResolved
                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/options/problemSolvingOnServiceSessionResolved", urlRoot, tenantId);
                    EMLog.d(TAG, "asyncProblemSolvingOnServiceSessionResolved request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    //requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncEvalSolveWord(String tenantId, String serviceSessionId, final ValueCallBack<String> callBack){
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    // %1$s/v1/webimplugin/tenants/%2$s/options/evaluteSolveWord?serviceSessionId=%3$s
                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/options/evaluteSolveWord?serviceSessionId=%3$s", urlRoot, tenantId, serviceSessionId);
                    EMLog.d(TAG, "asyncProblemSolvingOnServiceSessionResolved request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    //requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncResolutionParams(String tenantId, String serviceSessionId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    String appKey = ChatClient.getInstance().appKey();
                    if (appKey == null){
                        EMLog.e(TAG, "appkey is not set");
                        if (callBack != null){
                            callBack.onError(Error.INVALID_APP_KEY, "appkey is not set");
                        }
                        return;
                    }
                    final String orgName = appKey.split("#")[0];
                    final String appName = appKey.split("#")[1];
                    final String currentUserName = ChatClient.getInstance().currentUserName();
                    final String token = ChatClient.getInstance().accessToken();


                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    // %1$s/v1/webimplugin/tenants/%2$s/resolutionparams?orgName=%3$s&appName=%4$s&userName=%5$s&token=%6$s
                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/resolutionparams?serviceSessionId=%3$s&orgName=%4$s&appName=%5$s&userName=%6$s&token=%7$s",
                            urlRoot, tenantId, serviceSessionId, orgName, appName, currentUserName, token);
                    EMLog.d(TAG, "asyncResolutionParams request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    //requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncGreetingMsgEnquiryInvite(String tenantId, String serviceSessionId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    // %1$s/v1/webimplugin/tenants/%2$s/options/evaluteSolveWord?serviceSessionId=%3$s
                    // %1$s/v1/webimplugin/tenants/%2$s/options/GreetingMsgEnquiryInvite?serviceSessionId=%3$s
                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/options/GreetingMsgEnquiryInvite?serviceSessionId=%3$s", urlRoot, tenantId, serviceSessionId);
                    EMLog.d(TAG, "asyncProblemSolvingOnServiceSessionResolved request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);

                    HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
                    //requestBuilder.header("Content-Type", "application/x-www-form-urlencoded");

                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();
                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception ignored) {
                    ignored.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + ignored);
                    }
                }
            }
        });
    }

    public static void asyncMarkAllMessagesAsRead(String tenantId, String sessionId, ValueCallBack<String> callBack) {
        sendThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {

                    if (tenantId == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "tenantid is null");
                        }
                        return;
                    }

                    // %1$s/v1/webimplugin/tenants/%2$s/servicesessions/%3$s/read
                    String urlRoot = ChatClient.getInstance().kefuRestServer();
                    final String remoteUrl = String.format("%1$s/v1/webimplugin/tenants/%2$s/servicesessions/%3$s/read", urlRoot, tenantId, sessionId);
                    EMLog.d(TAG, "asyncMarkAllMessagesAsRead request url:" + remoteUrl);
                    HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
                    httpClient.setConnectTimeout(8000);
                    httpClient.setReadTimeout(DEFAULT_READ_TIMEOUT);
                    HttpRequestBuilder requestBuilder = httpClient.post(remoteUrl);
                    HttpResponse httpResponse = requestBuilder.execute();
                    if (httpResponse == null) {
                        if (callBack != null) {
                            callBack.onError(-1, "response is null");
                        }
                        return;
                    }
                    int status_code = httpResponse.getStatusCode();

                    if (status_code >= 200 && status_code < 300) {
                        StringBuffer buffer = new StringBuffer();
                        httpResponse.read(buffer);
                        String result = buffer.toString();
                        if (callBack != null) {
                            callBack.onSuccess(result);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    if (callBack != null) {
                        callBack.onError(-1, "error->" + e);
                    }
                }
            }
        });
    }
}
