package com.hyphenate.helpdesk.manager;

import android.text.TextUtils;
import android.util.Log;

import com.hyphenate.chat.ChatClient;
import com.hyphenate.chat.EMClient;
import com.hyphenate.helpdesk.Error;
import com.hyphenate.helpdesk.callback.ValueCallBack;
import com.hyphenate.helpdesk.emojicon.Emojicon;
import com.hyphenate.helpdesk.httpclient.HttpClient;
import com.hyphenate.helpdesk.httpclient.HttpRequestBuilder;
import com.hyphenate.helpdesk.httpclient.HttpResponse;
import com.hyphenate.util.PathUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 自定义表情管理类
 */

public class EmojiconManager {

	public final static String TAG = "EmojiconManager";
	private ExecutorService requestService = Executors.newCachedThreadPool();
	private List<EmojiconEntity> iconList = new ArrayList<>();
	private String iconsJson;
	private String iconsJsonNew;
	private List<EmojiconPackage> packageList = new ArrayList<>();
	private String packagesJson;
	private String packagesJsonNew;
	private Queue<Url> downloadQueue = new LinkedList<>();
	final private List<EmojiconManagerDelegate> delegates = Collections.synchronizedList(new ArrayList<EmojiconManagerDelegate>());
	private static EmojiconManager instance;
	private volatile boolean refleshing = false;

	public synchronized static EmojiconManager getInstance() {
		if (instance == null) {
			instance = new EmojiconManager();
		}
		return instance;
	}


	/**
	 * 获取表情包的自定义表情列表
	 * @param aPackage
	 * @return 自定义表情列表
	 */
	public List<Emojicon> getEmojiconList(EmojiconPackage aPackage) {
		List<Emojicon> list = new ArrayList<>();
//		for (int i = 1; i <= aPackage.fileNum; i++) {
			synchronized (iconList){
				for(EmojiconEntity entity: iconList) {
					if (entity.packageId == aPackage.id ) {
						Emojicon emojicon = new Emojicon();
						emojicon.setType(Emojicon.Type.BIG_EXPRESSION);
						if (!TextUtils.isEmpty(entity.origin.localUrl)) {
							emojicon.setBigIconPath(entity.origin.localUrl);
						}
						if (!TextUtils.isEmpty(entity.thumbnail.localUrl)) {
							emojicon.setIconPath(entity.thumbnail.localUrl);
						}

						if (!TextUtils.isEmpty(entity.origin.remoteUrl)) {
							emojicon.setBigIconRemotePath(entity.origin.remoteUrl);
						}

						if (!TextUtils.isEmpty(entity.thumbnail.remoteUrl)) {
							emojicon.setIconRemotePath(entity.thumbnail.remoteUrl);
						}
						if (!TextUtils.isEmpty(entity.fileName)) {
							emojicon.setName(entity.fileName);
						}

						list.add(emojicon);
					}
				}
			}
//		}
		return list;
	}

	/**
	 * 获取表情包列表
	 * @return 表情包列表
	 */
	public List<EmojiconPackage> getEmojiPackagesList() {
		reSortPackageList();
		return packageList;
	}

	/**
	 * 刷新自定义表情
	 */
	public void reflesh() {
		synchronized (this) {
			if (refleshing) {
				return;
			}
			String tenantId = ChatClient.getInstance().tenantId();
			iconsJson = ChatClient.getInstance().getEmojiconInfoIconsJson(tenantId);
			packagesJson = ChatClient.getInstance().getEmojiconInfoPackagesJson(tenantId);
			refleshing = true;
		}

		getAllEmojiiconPackages();
	}

	/**
	 * 情况缓存
	 */
	public void clear() {
		iconList.clear();
		packageList.clear();
		notifyDelegateEmojiconChanged();
	}

	private void getAllEmojiiconPackages() {
		getAllEmojiiconPackages(new ValueCallBack<String>() {
			@Override
			public void onSuccess(String value) {
				getPackagesFromJson(value);
				getAllEmojiiconFiles();
			}

			@Override
			public void onError(int error, String errorMsg) {
				getPackagesFromJson(packagesJson);
				getAllEmojiiconFiles();
			}
		});
	}

	private void getPackagesFromJson(String value) {
		packageList.clear();
		try {
			JSONObject response = new JSONObject(value);
			final JSONArray packagesEntities = response.getJSONArray("entities");
			JSONObject obj;
			EmojiconPackage emojiconPackage;
			if (packagesEntities != null) {
				for (int i = 0; i < packagesEntities.length(); i ++) {
					emojiconPackage = new EmojiconPackage();
					obj = packagesEntities.getJSONObject(i);
					emojiconPackage.id = obj.getInt("id");
					emojiconPackage.packageName = obj.getString("packageName");
					emojiconPackage.fileNum = obj.getInt("fileNum");
					emojiconPackage.orderSQ = obj.getInt("orderSQ");
					emojiconPackage.creatorId = obj.getString("creatorId");
					packageList.add(emojiconPackage);
				}
				packagesJsonNew = value;
			}
		} catch (JSONException e) {
			e.printStackTrace();
		}
	}

	private void getAllEmojiiconFiles() {
		getAllEmojiiconFiles(new ValueCallBack<String>() {
			@Override
			public void onSuccess(String value) {
				if (!getEmojiconsFromJson(value)) {
					getAllEmojiiconFiles(new ValueCallBack<String>() {
						@Override
						public void onSuccess(String value) {
							getEmojiconsFromJson(value);
						}

						@Override
						public void onError(int error, String errorMsg) {
							getEmojiconsFromJson(iconsJson);
						}
					});
				}
			}

			@Override
			public void onError(int error, String errorMsg) {
				iconList.clear();
				getAllEmojiiconFiles(new ValueCallBack<String>() {
					@Override
					public void onSuccess(String value) {
						getEmojiconsFromJson(value);
					}

					@Override
					public void onError(int error, String errorMsg) {
						getEmojiconsFromJson(iconsJson);
					}
				});
			}
		});
	}

	private boolean getEmojiconsFromJson(String value) {
		iconList.clear();
		try {
			JSONObject response = new JSONObject(value);
			JSONArray entities = response.getJSONArray("entities");
			JSONObject obj;
			EmojiconEntity emojicon;
			if (entities != null) {
				for (int i = 0; i < entities.length(); i ++) {
					emojicon = new EmojiconEntity();
					obj = entities.getJSONObject(i);
					emojicon.id = obj.getInt("id");
					emojicon.packageId = obj.getInt("packageId");
					emojicon.fileName = obj.getString("fileName");
//					emojicon.orderSQ = obj.getInt("orderSQ");

					emojicon.origin.remoteUrl = obj.getString("originUrl");
					if (emojicon.origin.remoteUrl.startsWith("/v1/Tenant/")) {
						emojicon.origin.remoteUrl = ChatClient.getInstance().kefuRestServer() + emojicon.origin.remoteUrl;
					}

					emojicon.origin.localUrl = new File(PathUtil.getInstance().getImagePath(), "origin_" + emojicon.packageId + "_" + emojicon.fileName).getPath();

					emojicon.thumbnail.remoteUrl = obj.getString("thumbnailUrl");
					if (emojicon.thumbnail.remoteUrl.startsWith("/v1/Tenant/")) {
						emojicon.thumbnail.remoteUrl = ChatClient.getInstance().kefuRestServer() + emojicon.thumbnail.remoteUrl;
					}

					emojicon.thumbnail.localUrl = new File(PathUtil.getInstance().getImagePath(), "thumbnail_" + "_" + emojicon.packageId + "_" + emojicon.id + ".jpg").getPath();
					iconList.add(emojicon);
				}
				iconsJsonNew = value;
				prepareDownloadAllIcons();
			}
			return true;
		} catch (JSONException e) {
			e.printStackTrace();
		}
		return false;
	}

	private void reSortPackageList() {
		List<EmojiconPackage> tempList = new ArrayList<>();
		for (int i = 1; tempList.size() < packageList.size(); i++) {
			for (EmojiconPackage tempPackage : packageList) {
				if (tempPackage.orderSQ == i) {
					tempList.add(tempPackage);
					break;
				}
			}
		}
		packageList = tempList;
	}

	/**
	 * 根据远端地址获取自定义表情实体
	 * @param remotePath 远端路径
	 * @return 自定义表情实体
	 */
	public EmojiconEntity getEmojicon(String remotePath) {
		if (TextUtils.isEmpty(remotePath)) {
			return null;
		}

		if (remotePath.startsWith("/v1/Tenant/")) {
			remotePath = ChatClient.getInstance().kefuRestServer() + remotePath;
		}

		for (EmojiconEntity entity: iconList) {
			if (entity.origin.remoteUrl.contains(remotePath)) {
				return entity;
			}
		}
		EmojiconEntity emojiconEntity = new EmojiconEntity();
		emojiconEntity.origin.remoteUrl = remotePath;
		return emojiconEntity;
	}

	/**
	 * 获取该租户下的所有表情文件
	 * @param callback
	 */
	private void getAllEmojiiconFiles(final ValueCallBack<String> callback) {
		requestService.submit(new Runnable() {
			@Override
			public void run() {
				if (!ChatClient.getInstance().isLoggedInBefore()) {
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, "no login");
					}
					return;
				}
				String tenantId = ChatClient.getInstance().tenantId();
				if(TextUtils.isEmpty(tenantId)){
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, "tenantId is null");
					}
					return;
				}
				try {
					StringBuilder urlStrBuilder = new StringBuilder();
					String host = ChatClient.getInstance().kefuRestServer();
					urlStrBuilder.append(host).append("/v1/webimplugin/emoj/tenants/").append(tenantId).append("/files");

					String remoteUrl = urlStrBuilder.toString();
					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
					requestBuilder.header("Authorization", "Easemob IM " + ChatClient.getInstance().accessToken());
					requestBuilder.header("Content-Type", "application/json");

					HttpResponse httpResponse = requestBuilder.execute();
					assert httpResponse != null;
					int statusCode = httpResponse.getStatusCode();
					if (statusCode == HttpURLConnection.HTTP_OK) {
						StringBuffer sb = new StringBuffer();
						httpResponse.read(sb);
						if (callback != null) {
							callback.onSuccess(sb.toString());
						}
					} else {
						if (callback != null) {
							callback.onError(Error.GENERAL_ERROR, "statusCode:" + statusCode);
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, e.getMessage());
					}
				}


			}
		});
	}

	/**
	 * 获取该租户下所有表情包列表
	 * @param callback
	 */
	private void getAllEmojiiconPackages(final ValueCallBack<String> callback) {
		requestService.submit(new Runnable() {
			@Override
			public void run() {
				if (!ChatClient.getInstance().isLoggedInBefore()) {
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, "no login");
					}
					return;
				}
				String tenantId = ChatClient.getInstance().tenantId();
				if(TextUtils.isEmpty(tenantId)){
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, "tenantId is null");
					}
					return;
				}
				try {
					StringBuilder urlStrBuilder = new StringBuilder();
					String host = ChatClient.getInstance().kefuRestServer();
					urlStrBuilder.append(host).append("/v1/webimplugin/emoj/tenants/").append(tenantId).append("/packages");

					String remoteUrl = urlStrBuilder.toString();
					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
					requestBuilder.header("Authorization", "Easemob IM " + ChatClient.getInstance().accessToken());
					requestBuilder.header("Content-Type", "application/json");

					HttpResponse httpResponse = requestBuilder.execute();
					assert httpResponse != null;
					int statusCode = httpResponse.getStatusCode();
					if (statusCode == HttpURLConnection.HTTP_OK) {
						StringBuffer sb = new StringBuffer();
						httpResponse.read(sb);
						if (callback != null) {
							callback.onSuccess(sb.toString());
						}
					} else {
						if (callback != null) {
							callback.onError(Error.GENERAL_ERROR, "statusCode:" + statusCode);
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, e.getMessage());
					}
				}


			}
		});
	}

	/**
	 * 获取该租户下某个表情包下的文件列表
	 * @param packageId
	 * @param callback
	 */
	private void getEmojiiconPackageFile(final int packageId, final ValueCallBack<String> callback) {
		requestService.submit(new Runnable() {
			@Override
			public void run() {
				if (!ChatClient.getInstance().isLoggedInBefore()) {
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, "no login");
					}
					return;
				}
				String tenantId = ChatClient.getInstance().tenantId();
				if(TextUtils.isEmpty(tenantId)){
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, "tenantId is null");
					}
					return;
				}
				try {
					StringBuilder urlStrBuilder = new StringBuilder();
					String host = ChatClient.getInstance().kefuRestServer();
					urlStrBuilder.append(host).append("/v1/webimplugin/emoj/tenants/").append(tenantId)
							.append("/packages/").append(packageId).append("/files");

					String remoteUrl = urlStrBuilder.toString();
					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					HttpRequestBuilder requestBuilder = httpClient.get(remoteUrl);
					requestBuilder.header("Authorization", "Easemob IM " + ChatClient.getInstance().accessToken());
					requestBuilder.header("Content-Type", "application/json");

					HttpResponse httpResponse = requestBuilder.execute();
					assert httpResponse != null;
					int statusCode = httpResponse.getStatusCode();
					if (statusCode == HttpURLConnection.HTTP_OK) {
						StringBuffer sb = new StringBuffer();
						httpResponse.read(sb);
						if (callback != null) {
							callback.onSuccess(sb.toString());
						}
					} else {
						if (callback != null) {
							callback.onError(Error.GENERAL_ERROR, "statusCode:" + statusCode);
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
					if (callback != null) {
						callback.onError(Error.GENERAL_ERROR, e.getMessage());
					}
				}

			}
		});
	}

	/**
	 * 下载所有icon
	 */
	private void prepareDownloadAllIcons() {
		if (iconsJsonNew != null && !iconsJsonNew.equals(iconsJson)) {
			downloadQueue.clear();
			if (iconList.size() != 0) {
				for (EmojiconEntity entity: iconList) {
					downloadQueue.add(entity.origin);
					downloadQueue.add(entity.thumbnail);
				}
				notifyDelegateEmojiconChanged();
			}
			downloadNextIcons();

		} else {
			if (packagesJsonNew != null && !packagesJsonNew.equals(packagesJson)) {
				notifyDelegateEmojiconChanged();
			}
			synchronized (this) {
				String tenantId = ChatClient.getInstance().tenantId();
				ChatClient.getInstance().deleteEmojiconInfo(tenantId);
				ChatClient.getInstance().addEmojiconInfo(tenantId, iconsJsonNew, packagesJsonNew);
				refleshing = false;
			}
		}
	}

	/**
	 * 递归下载所有的icon
	 */
	private void downloadNextIcons() {
		if (downloadQueue.size() == 0) {
			synchronized (this) {
				String tenantId = ChatClient.getInstance().tenantId();
				ChatClient.getInstance().deleteEmojiconInfo(tenantId);
				ChatClient.getInstance().addEmojiconInfo(tenantId, iconsJsonNew, packagesJsonNew);
				refleshing = false;
			}
			return;
		}
		final Url requestUrl = downloadQueue.poll();
		downloadFiles(requestUrl, new ValueCallBack<String>() {
			@Override
			public void onSuccess(String value) {
				downloadNextIcons();
			}

			@Override
			public void onError(int error, String errorMsg) {
				//下载失败重试一次
				downloadFiles(requestUrl, new ValueCallBack<String>() {
					@Override
					public void onSuccess(String value) {
						downloadNextIcons();
					}

					@Override
					public void onError(int error, String errorMsg) {
						Log.e(TAG, "onError: downloadIcons Failed requestUrl = " + requestUrl.remoteUrl + " errpr = " + error + " errorMsg = " + errorMsg);
						downloadNextIcons();
					}
				});
			}
		});
	}

	public void downloadFiles(final Url requestUrl, final ValueCallBack<String> callback) {
		if (TextUtils.isEmpty(requestUrl.remoteUrl) || TextUtils.isEmpty(requestUrl.localUrl)){
			return;
		}

		final File localFile = new File(requestUrl.localUrl);
		if (localFile.exists()) {
			if (callback != null){
				callback.onSuccess("");
			}
			return;
		}
		final File tempLocalFile = new File(localFile.getParent(), "tmp_" + localFile.getName());
		requestService.execute(new Runnable() {
				@Override
				public void run() {
					HttpClient httpClient = new HttpClient(EMClient.getInstance().getContext());
					HttpRequestBuilder requestBuilder = httpClient.get(requestUrl.remoteUrl);
					try {
						requestBuilder.to(tempLocalFile);
						HttpResponse httpResponse = requestBuilder.execute();
						if (httpResponse == null){
							if (callback != null){
								callback.onError(Error.FILE_DOWNLOAD_FAILED, "rfile download failed");
							}
							return;
						}
						int status = httpResponse.getStatusCode();
						if (status / 100 == 2){
							tempLocalFile.renameTo(localFile);
							if (callback != null){
								callback.onSuccess("");
							}
						}else{
							if (callback != null){
								callback.onError(Error.FILE_DOWNLOAD_FAILED, "file download failed");
							}
						}
					} catch (Exception e) {
						e.printStackTrace();
						if (callback != null){
							callback.onError(Error.FILE_DOWNLOAD_FAILED, "file download failed");
						}
					}

				}
			});

	}

	private void notifyDelegateEmojiconChanged() {
		synchronized (delegates) {
			for (EmojiconManagerDelegate item : delegates) {
				if (item != null) {
					item.onEmojiconChanged();
				}
			}
		}
	}

	public void addDelegate(EmojiconManagerDelegate listener) {
		if (!delegates.contains(listener)) {
			delegates.add(listener);
		}
	}

	public void removeDelegate(EmojiconManagerDelegate listener) {
		if (delegates.contains(listener)) {
			delegates.remove(listener);
		}
	}

	public interface EmojiconManagerDelegate {
		void onEmojiconChanged();
	}

	public static class EmojiconEntity {
		public int id;
		public int packageId;
		public String fileName;
//		public int orderSQ;
		public Url origin;
		public Url thumbnail;
		public EmojiconEntity() {
			origin = new Url();
			thumbnail = new Url();
			id = -1;
			packageId = -1;
			fileName = "";
//			orderSQ = -1;
		}
	}

	public static class EmojiconPackage {
		public int id;
		public String packageName;
		public int fileNum;
		public int orderSQ;
		public String creatorId;
		public EmojiconPackage() {
			id = -1;
			packageName = "";
			fileNum = 0;
			orderSQ = -1;
			creatorId = "";
		}
	}

	public static class Url {
		public String remoteUrl;
		public String localUrl;
		public Url() {
			remoteUrl = "";
			localUrl = "";
		}
	}
}
