/**
 * EasyPostResource.java
 * This file is a part of EasyPost API SDK.
 * (c) 2022 EasyPost
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
package com.easypost.net;

import com.easypost.EasyPost;
import com.easypost.exception.EasyPostException;
import com.easypost.model.Event;
import com.easypost.model.EventDeserializer;
import com.easypost.model.Fee;
import com.easypost.model.Rate;
import com.easypost.model.RateDeserializer;
import com.easypost.model.Shipment;
import com.easypost.model.SmartrateCollection;
import com.easypost.model.SmartrateCollectionDeserializer;
import com.easypost.model.TrackingDetail;
import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.net.URLStreamHandler;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

public abstract class EasyPostResource {
    public static final String EASYPOST_SUPPORT_EMAIL = "support@easypost.com";

    private static final int DEFAULT_CONNECT_TIMEOUT_MILLISECONDS = 30000;
    private static final int DEFAULT_READ_TIMEOUT_MILLISECONDS = 60000;
    private static final double APP_ENGINE_DEFAULT_TIMEOUT_SECONDS = 20.0;

    public static final Gson GSON =
            new GsonBuilder().setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
                    .registerTypeAdapter(Event.class, new EventDeserializer())
                    .registerTypeAdapter(Rate.class, new RateDeserializer())
                    .registerTypeAdapter(SmartrateCollection.class, new SmartrateCollectionDeserializer()).create();

    public static final Gson PRETTY_PRINT_GSON = new GsonBuilder().setPrettyPrinting().serializeNulls()
            .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
            .registerTypeAdapter(Event.class, new EventDeserializer()).create();

    /**
     * Returns a string representation of the object.
     */
    @Override
    public String toString() {

        return (String) this.getIdString();
    }

    /**
     * Pretty print the JSON representation of the object.
     *
     * @return the JSON representation of the object.
     */
    public String prettyPrint() {
        return String.format("<%s@%s id=%s> JSON: %s", this.getClass().getName(), System.identityHashCode(this),
                this.getIdString(), PRETTY_PRINT_GSON.toJson(this));
    }

    private Object getIdString() {
        try {
            Field idField = this.getClass().getDeclaredField("id");
            return idField.get(this);
        } catch (SecurityException e) {
            return "";
        } catch (NoSuchFieldException e) {
            return "";
        } catch (IllegalArgumentException e) {
            return "";
        } catch (IllegalAccessException e) {
            return "";
        }
    }

    private static String className(final Class<?> clazz) {
        return clazz.getSimpleName().replaceAll("([a-z])([A-Z])", "$1_$2").toLowerCase().replace("$", "");

    }

    protected static String singleClassURL(final Class<?> clazz) {
        return String.format("%s/%s", EasyPost.API_BASE, className(clazz));
    }

    protected static String classURL(final Class<?> clazz) {
        String singleURL = singleClassURL(clazz);
        if (singleURL.charAt(singleURL.length() - 1) == 's' || singleURL.charAt(singleURL.length() - 1) == 'h') {
            return String.format("%ses", singleClassURL(clazz));
        } else {
            return String.format("%ss", singleClassURL(clazz));
        }
    }

    protected static String instanceURL(final Class<?> clazz, final String id) {
        return String.format("%s/%s", classURL(clazz), id);
    }

    /**
     * Merge two EasyPostResource objects.
     *
     * @param obj    the base object
     * @param update the object to merge
     */
    public void merge(final EasyPostResource obj, final EasyPostResource update) {
        if (!obj.getClass().isAssignableFrom(update.getClass())) {
            return;
        }

        Method[] methods = obj.getClass().getMethods();

        for (Method fromMethod : methods) {
            if ((fromMethod.getDeclaringClass().equals(obj.getClass()) && fromMethod.getName().startsWith("get")) ||
                    GLOBAL_FIELD_ACCESSORS.contains(fromMethod.getName())) {

                String fromName = fromMethod.getName();
                String toName = fromName.replace("get", "set");

                try {
                    Object value = fromMethod.invoke(update, (Object[]) null);
                    if (value != null) {
                        Method toMethod = obj.getClass().getMethod(toName, fromMethod.getReturnType());
                        toMethod.invoke(obj, value);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public static final String CHARSET = "UTF-8";

    private static final String DNS_CACHE_TTL_PROPERTY_NAME = "networkaddress.cache.ttl";


    // Set this property to override your environment's default URLStreamHandler.
    private static final String CUSTOM_URL_STREAM_HANDLER_PROPERTY_NAME = "com.easypost.net.customURLStreamHandler";

    protected enum RequestMethod {
        GET, POST, DELETE, PUT
    }

    private static String urlEncodePair(final String key, final String value) throws UnsupportedEncodingException {
        return String.format("%s=%s", URLEncoder.encode(key, CHARSET), URLEncoder.encode(value, CHARSET));
    }

    static Map<String, String> getHeaders(String apiKey) {
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Accept-Charset", CHARSET);
        headers.put("User-Agent", String.format("EasyPost/v2 JavaClient/%s Java/%s"
                                , EasyPost.VERSION, System.getProperty("java.version")));

        if (apiKey == null) {
            apiKey = EasyPost.apiKey;
        }

        headers.put("Authorization", String.format("Bearer %s", apiKey));

        // debug headers
        String[] propertyNames = { "os.name", "os.version", "os.arch", "java.version", "java.vendor", "java.vm.version",
                "java.vm.vendor" };
        Map<String, String> propertyMap = new HashMap<String, String>();
        for (String propertyName : propertyNames) {
            propertyMap.put(propertyName, System.getProperty(propertyName));
        }
        propertyMap.put("lang", "Java");
        propertyMap.put("publisher", "EasyPost");
        headers.put("X-Client-User-Agent", GSON.toJson(propertyMap));

        return headers;
    }

    private static javax.net.ssl.HttpsURLConnection createEasyPostConnection(final String url, final String apiKey)
            throws IOException {
        URL easypostURL = null;
        String customURLStreamHandlerClassName = System.getProperty(CUSTOM_URL_STREAM_HANDLER_PROPERTY_NAME, null);
        if (customURLStreamHandlerClassName != null) {
            // instantiate the custom handler provided
            try {
                Class<URLStreamHandler> clazz =
                        (Class<URLStreamHandler>) Class.forName(customURLStreamHandlerClassName);
                Constructor<URLStreamHandler> constructor = clazz.getConstructor();
                URLStreamHandler customHandler = constructor.newInstance();
                easypostURL = new URL(null, url, customHandler);
            } catch (ClassNotFoundException e) {
                throw new IOException(e);
            } catch (SecurityException e) {
                throw new IOException(e);
            } catch (NoSuchMethodException e) {
                throw new IOException(e);
            } catch (IllegalArgumentException e) {
                throw new IOException(e);
            } catch (InstantiationException e) {
                throw new IOException(e);
            } catch (IllegalAccessException e) {
                throw new IOException(e);
            } catch (InvocationTargetException e) {
                throw new IOException(e);
            }
        } else {
            easypostURL = new URL(url);
        }
        javax.net.ssl.HttpsURLConnection conn = (javax.net.ssl.HttpsURLConnection) easypostURL.openConnection();
        conn.setConnectTimeout(DEFAULT_CONNECT_TIMEOUT_MILLISECONDS);

        int readTimeout;
        if (EasyPost.readTimeout != 0) {
            readTimeout = EasyPost.readTimeout;
        } else {
            readTimeout = DEFAULT_READ_TIMEOUT_MILLISECONDS;
        }
        conn.setReadTimeout(readTimeout);

        conn.setUseCaches(false);
        for (Map.Entry<String, String> header : getHeaders(apiKey).entrySet()) {
            conn.setRequestProperty(header.getKey(), header.getValue());
        }

        return conn;
    }

    private static javax.net.ssl.HttpsURLConnection writeBody(final javax.net.ssl.HttpsURLConnection conn,
                                                              final JsonObject body) throws IOException {
        if (body != null) {
            conn.setDoOutput(true);
            conn.setRequestProperty("Content-Type", "application/json");
            OutputStream output = null;
            try {
                output = conn.getOutputStream();
                String jsonString = body.toString();
                output.write(jsonString.getBytes(CHARSET));
            } finally {
                if (output != null) {
                    output.close();
                }
            }
        }
        return conn;
    }

    private static javax.net.ssl.HttpsURLConnection createGetConnection(final String url, final String query,
                                                                        final String apiKey) throws IOException {
        String getURL = String.format("%s?%s", url, query);
        javax.net.ssl.HttpsURLConnection conn = createEasyPostConnection(getURL, apiKey);
        conn.setRequestMethod("GET");
        return conn;
    }

    private static javax.net.ssl.HttpsURLConnection createPostConnection(final String url, final JsonObject body,
                                                                         final String apiKey) throws IOException {
        javax.net.ssl.HttpsURLConnection conn = createEasyPostConnection(url, apiKey);
        conn.setRequestMethod("POST");
        conn = writeBody(conn, body);
        return conn;
    }

    private static javax.net.ssl.HttpsURLConnection createDeleteConnection(final String url, final String query,
                                                                           final String apiKey) throws IOException {
        String deleteUrl = String.format("%s?%s", url, query);
        javax.net.ssl.HttpsURLConnection conn = createEasyPostConnection(deleteUrl, apiKey);
        conn.setRequestMethod("DELETE");
        return conn;
    }

    private static javax.net.ssl.HttpsURLConnection createPutConnection(final String url, final JsonObject body,
                                                                        final String apiKey) throws IOException {
        javax.net.ssl.HttpsURLConnection conn = createEasyPostConnection(url, apiKey);
        conn.setRequestMethod("PUT");
        writeBody(conn, body);
        return conn;
    }

    private static JsonObject createBody(final Map<String, Object> params) {
        Gson gson = new Gson();
        return gson.toJsonTree(params).getAsJsonObject();
    }

    private static String createQuery(final Map<String, Object> params) throws UnsupportedEncodingException {
        Map<String, String> flatParams = flattenParams(params);
        StringBuilder queryStringBuilder = new StringBuilder();
        for (Map.Entry<String, String> entry : flatParams.entrySet()) {
            queryStringBuilder.append("&");
            queryStringBuilder.append(urlEncodePair(entry.getKey(), entry.getValue()));
        }
        if (queryStringBuilder.length() > 0) {
            queryStringBuilder.deleteCharAt(0);
        }
        return queryStringBuilder.toString();
    }

    private static Map<String, String> flattenParams(final Map<String, Object> params) {
        if (params == null) {
            return new HashMap<String, String>();
        }
        Map<String, String> flatParams = new HashMap<String, String>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            if (value instanceof Map<?, ?>) {
                Map<String, Object> flatNestedMap = new HashMap<String, Object>();
                Map<?, ?> nestedMap = (Map<?, ?>) value;
                for (Map.Entry<?, ?> nestedEntry : nestedMap.entrySet()) {
                    flatNestedMap.put(String.format("%s[%s]", key, nestedEntry.getKey()), nestedEntry.getValue());
                }
                flatParams.putAll(flattenParams(flatNestedMap));
            } else if (value instanceof List) {
                Map<String, Object> flatNestedMap = new HashMap<String, Object>();
                List<?> nestedList = (List<?>) value;
                for (int i = 0; i < nestedList.size(); i++) {
                    flatNestedMap.put(String.format("%s[%s]", key, i), nestedList.get(i));
                    flatParams.putAll(flattenParams(flatNestedMap));
                }
            } else if (value instanceof EasyPostResource) {
                flatParams.put(String.format("%s[%s]", key, "id"), value.toString());

            } else if (value != null) {
                flatParams.put(key, value.toString());
            }
        }

        return flatParams;
    }

    // represents Errors returned as JSON
    private static class ErrorContainer {
        private EasyPostResource.Error error;
    }

    private static class Error {
        @SuppressWarnings ("unused")
        private String type;
        private String message;
        private String code;
        private String param;
        private String error;

        public String getType() {
            return type;
        }

        public String getMessage() {
            return message;
        }

        public String getCode() {
            return code;
        }

        public String getParam() {
            return param;
        }

        public String getError() {
            return error;
        }
    }

    private static String getResponseBody(final InputStream responseStream) throws IOException {
        String rBody = new Scanner(responseStream, CHARSET).useDelimiter("\\A").next();
        responseStream.close();
        return rBody;
    }

    private static EasyPostResponse makeURLConnectionRequest(final EasyPostResource.RequestMethod method,
                                                             final String url, final String query,
                                                             final JsonObject body, final String apiKey)
            throws EasyPostException {
        javax.net.ssl.HttpsURLConnection conn = null;
        try {
            switch (method) {
                case GET:
                    conn = createGetConnection(url, query, apiKey);
                    break;
                case POST:
                    conn = createPostConnection(url, body, apiKey);
                    break;
                case PUT:
                    conn = createPutConnection(url, body, apiKey);
                    break;
                case DELETE:
                    conn = createDeleteConnection(url, query, apiKey);
                    break;
                default:
                    throw new EasyPostException(
                            String.format("Unrecognized HTTP method %s. Please contact EasyPost at %s.", method,
                                    EasyPostResource.EASYPOST_SUPPORT_EMAIL));
            }
            int rCode = conn.getResponseCode(); // sends the request
            String rBody = null;
            if (rCode == HttpURLConnection.HTTP_NO_CONTENT) {
                rBody = "";
            } else if (rCode >= HttpURLConnection.HTTP_OK && rCode < HttpURLConnection.HTTP_MULT_CHOICE) {
                rBody = getResponseBody(conn.getInputStream());
            } else {
                rBody = getResponseBody(conn.getErrorStream());
            }
            return new EasyPostResponse(rCode, rBody);
        } catch (IOException e) {
            throw new EasyPostException(String.format("Could not connect to EasyPost (%s). " +
                    "Please check your internet connection and try again. If this problem persists," +
                    "please contact us at %s.", EasyPost.API_BASE, EasyPostResource.EASYPOST_SUPPORT_EMAIL), e);
        } finally {
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    protected static <T> T request(final EasyPostResource.RequestMethod method, final String url,
                                   final Map<String, Object> params, final Class<T> clazz, final String apiKey)
            throws EasyPostException {
        return request(method, url, params, clazz, apiKey, true);
    }

    protected static <T> T request(final EasyPostResource.RequestMethod method, final String url,
                                   final Map<String, Object> params, final Class<T> clazz, final String apiKey,
                                   final boolean apiKeyRequired) throws EasyPostException {
        String originalDNSCacheTTL = null;
        boolean allowedToSetTTL = true;
        try {
            originalDNSCacheTTL = java.security.Security.getProperty(DNS_CACHE_TTL_PROPERTY_NAME);
            // disable DNS cache
            java.security.Security.setProperty(DNS_CACHE_TTL_PROPERTY_NAME, "0");
        } catch (SecurityException se) {
            allowedToSetTTL = false;
        }

        try {
            return _request(method, url, params, clazz, apiKey, apiKeyRequired);
        } finally {
            if (allowedToSetTTL) {
                if (originalDNSCacheTTL == null) {
                    // value unspecified by implementation
                    java.security.Security.setProperty(DNS_CACHE_TTL_PROPERTY_NAME, "-1"); // cache forever
                } else {
                    java.security.Security.setProperty(DNS_CACHE_TTL_PROPERTY_NAME, originalDNSCacheTTL);
                }
            }
        }
    }

    @SuppressWarnings("checkstyle:methodname")
    protected static <T> T _request(final EasyPostResource.RequestMethod method, final String url,
                                       final Map<String, Object> params, final Class<T> clazz, String apiKey,
                                       final boolean apiKeyRequired) throws EasyPostException {
        if ((EasyPost.apiKey == null || EasyPost.apiKey.length() == 0) && (apiKey == null || apiKey.length() == 0)) {
            if (apiKeyRequired) {
                throw new EasyPostException(String.format(
                        "No API key provided. (set your API key using 'EasyPost.apiKey = {KEY}'. " +
                                "Your API key can be found in your EasyPost dashboard, " +
                                "or you can email us at %s for assistance.", EasyPostResource.EASYPOST_SUPPORT_EMAIL));

            }
        }

        if (apiKey == null) {
            apiKey = EasyPost.apiKey;
        }

        String query = null;
        JsonObject body = null;
        if (params != null) {
            switch (method) {
                case GET:
                case DELETE:
                    try {
                        query = createQuery(params);
                    } catch (UnsupportedEncodingException e) {
                        throw new EasyPostException(
                                String.format("Unable to encode parameters to %s. Please email %s for assistance.",
                                        CHARSET, EasyPostResource.EASYPOST_SUPPORT_EMAIL), e);
                    }
                    break;
                case POST:
                case PUT:
                    try {
                        body = createBody(params);
                    } catch (Exception e) {
                        throw new EasyPostException(String.format(
                                "Unable to create JSON body from parameters. Please email %s for assistance.",
                                EasyPostResource.EASYPOST_SUPPORT_EMAIL), e);
                    }
                    break;
                default:
                    break;
            }
        }


        EasyPostResponse response;
        try {
            // HTTPSURLConnection verifies SSL cert by default
            response = makeURLConnectionRequest(method, url, query, body, apiKey);
        } catch (ClassCastException ce) {
            // appengine
            String appEngineEnv = System.getProperty("com.google.appengine.runtime.environment", null);
            if (appEngineEnv != null) {
                response = makeAppEngineRequest(method, url, query, body, apiKey);
            } else {
                throw ce;
            }
        }
        int rCode = response.getResponseCode();
        String rBody = response.getResponseBody();
        if (rCode < HttpURLConnection.HTTP_OK || rCode >= HttpURLConnection.HTTP_MULT_CHOICE) {
            handleAPIError(rBody, rCode);
        }

        return GSON.fromJson(rBody, clazz);
    }

    private static void handleAPIError(final String rBody, final int rCode) throws EasyPostException {
        try {
            EasyPostResource.Error error = GSON.fromJson(rBody, EasyPostResource.Error.class);

            if (error.getError().length() > 0) {
                throw new EasyPostException(error.getError());
            }

            throw new EasyPostException(error.getMessage(), error.getParam(), null);
        } catch (Exception e) {
            throw new EasyPostException(
                    String.format("An error occurred. Response code: %s Response body: %s", rCode, rBody));
        }
    }

    private static EasyPostResponse makeAppEngineRequest(final RequestMethod method, String url, final String query,
                                                         final JsonObject body, final String apiKey)
            throws EasyPostException {
        String unknownErrorMessage = String.format(
                "Sorry, an unknown error occurred while trying to use the Google App Engine runtime." +
                        "Please email %s for assistance.", EasyPostResource.EASYPOST_SUPPORT_EMAIL);
        try {
            if ((method == RequestMethod.GET || method == RequestMethod.DELETE) && query != null) {
                url = String.format("%s?%s", url, query);
            }
            URL fetchURL = new URL(url);

            Class<?> requestMethodClass = Class.forName("com.google.appengine.api.urlfetch.HTTPMethod");
            Object httpMethod = requestMethodClass.getDeclaredField(method.name()).get(null);

            Class<?> fetchOptionsBuilderClass = Class.forName("com.google.appengine.api.urlfetch.FetchOptions$Builder");
            Object fetchOptions = null;
            try {
                fetchOptions = fetchOptionsBuilderClass.getDeclaredMethod("validateCertificate").invoke(null);
            } catch (NoSuchMethodException e) {
                System.err.printf(
                        "Warning: this App Engine SDK version does not allow verification of SSL certificates;" +
                                "this exposes you to a MITM attack. Please upgrade your App Engine SDK to >=1.5.0. " +
                                "If you have questions, email %s.%n", EasyPostResource.EASYPOST_SUPPORT_EMAIL);
                fetchOptions = fetchOptionsBuilderClass.getDeclaredMethod("withDefaults").invoke(null);
            }

            Class<?> fetchOptionsClass = Class.forName("com.google.appengine.api.urlfetch.FetchOptions");

            // Heroku times out after 30s, so leave some time for the API to return a response
            fetchOptionsClass.getDeclaredMethod("setDeadline", java.lang.Double.class)
                    .invoke(fetchOptions, APP_ENGINE_DEFAULT_TIMEOUT_SECONDS);

            Class<?> requestClass = Class.forName("com.google.appengine.api.urlfetch.HTTPRequest");

            Object request = requestClass.getDeclaredConstructor(URL.class, requestMethodClass, fetchOptionsClass)
                    .newInstance(fetchURL, httpMethod, fetchOptions);

            if ((method == RequestMethod.POST || method == RequestMethod.PUT) && body != null) {
                String bodyString = body.toString();
                requestClass.getDeclaredMethod("setPayload", byte[].class)
                        .invoke(request, (Object) bodyString.getBytes());
            }

            for (Map.Entry<String, String> header : getHeaders(apiKey).entrySet()) {
                Class<?> httpHeaderClass = Class.forName("com.google.appengine.api.urlfetch.HTTPHeader");
                Object reqHeader = httpHeaderClass.getDeclaredConstructor(String.class, String.class)
                        .newInstance(header.getKey(), header.getValue());
                requestClass.getDeclaredMethod("setHeader", httpHeaderClass).invoke(request, reqHeader);
            }

            Class<?> urlFetchFactoryClass = Class.forName("com.google.appengine.api.urlfetch.URLFetchServiceFactory");
            Object urlFetchService = urlFetchFactoryClass.getDeclaredMethod("getURLFetchService").invoke(null);

            Method fetchMethod = urlFetchService.getClass().getDeclaredMethod("fetch", requestClass);
            fetchMethod.setAccessible(true);
            Object response = fetchMethod.invoke(urlFetchService, request);

            int responseCode = (Integer) response.getClass().getDeclaredMethod("getResponseCode").invoke(response);
            String responseBody =
                    new String((byte[]) response.getClass().getDeclaredMethod("getContent").invoke(response), CHARSET);

            return new EasyPostResponse(responseCode, responseBody);

        } catch (InvocationTargetException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (MalformedURLException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (NoSuchFieldException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (SecurityException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (NoSuchMethodException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (ClassNotFoundException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (IllegalArgumentException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (IllegalAccessException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (InstantiationException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        } catch (UnsupportedEncodingException e) {
            throw new EasyPostException(unknownErrorMessage, e);
        }
    }

    public static final ArrayList<String> GLOBAL_FIELD_ACCESSORS =
            new ArrayList<>(Arrays.asList("getCreatedAt", "getUpdatedAt", "getFees"));
    private Date createdAt;
    private Date updatedAt;
    private ArrayList<Fee> fees;

    /**
     * @return the Date this object was created
     */
    public Date getCreatedAt() {
        return createdAt;
    }

    /**
     * Set the Date this object was created.
     *
     * @param createdAt the Date this object was created
     */
    public void setCreatedAt(final Date createdAt) {
        this.createdAt = createdAt;
    }

    /**
     * @return the Date this object was last updated
     */
    public Date getUpdatedAt() {
        return updatedAt;
    }

    /**
     * Set the Date this object was last updated.
     *
     * @param updatedAt the Date this object was last updated
     */
    public void setUpdatedAt(final Date updatedAt) {
        this.updatedAt = updatedAt;
    }

    /**
     * @return the Fees associated with this object
     */
    public ArrayList<Fee> getFees() {
        return fees;
    }

    /**
     * Set the Fees associated with this object.
     *
     * @param fees the Fees associated with this object
     */
    public void setFees(final ArrayList<Fee> fees) {
        this.fees = fees;
    }

    /**
     * @return the ID of this object
     */
    public String getId() {
        return "";
    }

    /**
     * @return the API mode used to create this object
     */
    public String getMode() {
        return "";
    }

    // Batch

    /**
     * @return the list of shipments in this batch
     */
    public List<Shipment> getShipments() {
        return new ArrayList<Shipment>();
    }

    /**
     * @return the URL of the label for this object
     */
    public String getLabelUrl() {
        return "";
    }

    // Tracker

    /**
     * @return the ID of this shipment
     */
    public String getShipmentId() {
        return "";
    }

    /**
     * @return the tracking code of this shipment
     */
    public String getTrackingCode() {
        return "";
    }

    /**
     * @return the status of this object
     */
    public String getStatus() {
        return "";
    }

    /**
     * @return the tracking details of this shipment
     */
    public List<TrackingDetail> getTrackingDetails() {
        return new ArrayList<TrackingDetail>();
    }

}
