/*
 * Copyright (c) 2020 Jon Chambers
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package com.eatthepath.pushy.apns.server;

import java.time.Instant;
import java.util.Objects;

/**
 * An exception thrown by {@link PushNotificationHandler} instances to indicate that a push notification should be
 * rejected by the server because its destination device token is no longer valid (presumably because the receiving app
 * has been removed from the destination device).
 *
 * @since 0.12
 */
public class UnregisteredDeviceTokenException extends RejectedNotificationException {
    private final Instant deviceTokenExpirationTimestamp;

    /**
     * Constructs a new unregistered device token exception indicating that the device token expired at the given time.
     *
     * @param deviceTokenExpirationTimestamp the time at which the destination device token expired
     */
    public UnregisteredDeviceTokenException(final Instant deviceTokenExpirationTimestamp) {
        super(RejectionReason.UNREGISTERED);

        Objects.requireNonNull(deviceTokenExpirationTimestamp, "Device token expiration timestamp must not be null.");

        this.deviceTokenExpirationTimestamp = deviceTokenExpirationTimestamp;
    }

    Instant getDeviceTokenExpirationTimestamp() {
        return deviceTokenExpirationTimestamp;
    }
}
