package sbtbuildinfo

case class Scala3CaseClassRenderer(options: Seq[BuildInfoOption], pkg: String, obj: String) extends Scala3Renderer {
  override def fileType = BuildInfoType.Source
  override def extension = "scala"

  val traitNames = options.collect{case BuildInfoOption.Traits(ts @ _*) => ts}.flatten
  val objTraits = if (traitNames.isEmpty) "" else "extends " ++ traitNames.mkString(" with ")

  // It is safe to add `import scala.Predef` even though we need to keep `-Ywarn-unused-import` in mind
  // because we always generate code that has a reference to `String`. If the "base" generated code were to be
  // changed and no longer contain a reference to `String`, we would need to remove `import scala.Predef` and
  // fully qualify every reference. Note it is NOT safe to use `import scala._` because of the possibility of
  // the project using `-Ywarn-unused-import` because we do not always generated references that are part of
  // `scala` such as `scala.Option`.
  val importScalaPredef = options.contains(BuildInfoOption.ImportScalaPredef)
  def header = List(
    "// $COVERAGE-OFF$",
    s"package $pkg",
    ""
  )
  val imports = if (importScalaPredef) List(
    "import scala.Predef.*",
    ""
  ) else Nil
  val generateComment = List(
    s"/** This file was generated by sbt-buildinfo. */"
  )

  def footer = List("// $COVERAGE-ON$")

  override def renderKeys(buildInfoResults: Seq[BuildInfoResult]) =
    header ++ imports ++ generateComment ++
    caseClassDefinitionBegin ++
      buildInfoResults.flatMap(caseClassParameter).mkString(",\n").split("\n") ++
      caseClassDefinitionEnd ++
      toMapLines(buildInfoResults) ++
      toJsonLines ++
      caseClassEnd ++
      List("") ++
      caseObjectLine(buildInfoResults) ++
      footer

  private def caseClassDefinitionBegin = List(
    withPkgPriv(s"case class $obj(")
  )

  private def caseClassParameter(r: BuildInfoResult): Seq[String] = {
    val typeDecl = getType(r.typeExpr) getOrElse "scala.Any"

    List(
      s"  ${r.identifier}: $typeDecl"
    )
  }

  private def caseClassDefinitionEnd = List(s") $objTraits {", "")
  private def caseClassEnd = List("}")

  private def caseObjectLine(buildInfoResults: Seq[BuildInfoResult]) = List(
    withPkgPriv(s"case object $obj {"),
    s"  def apply(): $obj = new $obj(${buildInfoResults.map(r => s"\n    ${r.identifier} = ${quote(r.value)}").mkString(",")})",
    s"  val get = apply()",
    s"  val value = apply()",
    s"}"
  )
}
