package sbtbuildinfo

case class Scala3CaseObjectRenderer(options: Seq[BuildInfoOption], pkg: String, obj: String) extends Scala3Renderer {

  override def fileType = BuildInfoType.Source
  override def extension = "scala"
  val traitNames = options.collect{case BuildInfoOption.Traits(ts @ _*) => ts}.flatten
  val objTraits = if (traitNames.isEmpty) "" else " extends " ++ traitNames.mkString(" with ")
  val constantValue = options.contains(BuildInfoOption.ConstantValue)

  // It is safe to add `import scala.Predef` even though we need to keep `-Ywarn-unused-import` in mind
  // because we always generate code that has a reference to `String`. If the "base" generated code were to be
  // changed and no longer contain a reference to `String`, we would need to remove `import scala.Predef` and
  // fully qualify every reference. Note it is NOT safe to use `import scala._` because of the possibility of
  // the project using `-Ywarn-unused-import` because we do not always generated references that are part of
  // `scala` such as `scala.Option`.
  val importScalaPredef = options.contains(BuildInfoOption.ImportScalaPredef)
  def header = List(
    "// $COVERAGE-OFF$",
    s"package $pkg",
    ""
  )
  val imports = if (importScalaPredef) List(
    "import scala.Predef.*",
    ""
  ) else Nil
  val objectHeader = List(
    s"/** This object was generated by sbt-buildinfo. */",
    withPkgPriv(s"case object $obj$objTraits {")
  )

  def footer = List("}", "// $COVERAGE-ON$")

  override def renderKeys(buildInfoResults: Seq[BuildInfoResult]) =
    header ++ imports ++ objectHeader ++
    buildInfoResults.flatMap(line) ++
    Seq(toStringLines(buildInfoResults)) ++
    toMapLines(buildInfoResults) ++
    toJsonLines ++
    footer

  private val constantTypes = Set("scala.Int", "scala.Long", "scala.Double", "scala.Boolean", "scala.Symbol", "String")

  private def line(result: BuildInfoResult): Seq[String] = {
    import result._
    val (typeDecl, modifier) =
      getType(result.manifest) match {
        case Some(tp) if !constantValue || !constantTypes(tp) =>
          (s": $tp", "")
        case _ if constantValue =>
          ("", "final ")
        case _ =>
          ("", "")
      }
    List(
      s"  /** The value is ${quote(value)}. */",
      s"  ${modifier}val $identifier$typeDecl = ${quote(value)}"
    )
  }

  def toStringLines(results: Seq[BuildInfoResult]): String = {
    val idents = results.map(_.identifier)
    val fmt = idents.map("%s: %%s" format _).mkString(", ")
    val vars = idents.mkString(", ")
    s"""  override val toString: String = {
         |    "$fmt".format(
         |      $vars
         |    )
         |  }""".stripMargin
  }
}
