/*
 * Copyright 2021 Elvis Hew
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.elvishew.xlog.printer.file.backup;

/**
 * Decide whether and how the log file should be backed up and use a new file for next logging.
 *
 * @since 1.9.0
 */
public interface BackupStrategy2 extends BackupStrategy {

  /**
   * Don't limit the backup index, which means there is no limit to the number of backup files.
   */
  int NO_LIMIT = 0;

  /**
   * Get the max index of backup.
   * <p>
   * Generally, the max index should be greater than 1, and recommended to be less than 10.
   * <p>
   * Imagine the normal log file name is 'log', and max backup index is 'n', as the log grows,
   * a log record would go from 'log' to 'log.bak.1', then to 'log.bak.2', 'log.bak.3', and finally
   * to 'log.bak.n', the greater index, the older log.
   * <br>After that, the log record will no longer exist in the file system.
   * <p>
   * If you don't want to limit the max index, then return {@link #NO_LIMIT}.
   * <br>With returning {@link #NO_LIMIT}, When you backing up, the oldest log would be saved to
   * 'log.bak.1', and then 'log.bak.2'...'log.bak.n', the greater index, the newer log.
   * <p>
   * Don't return {@link Integer#MAX_VALUE} or any value less than 0, otherwise an exception will
   * be thrown.
   *
   * @return the max index of backup
   */
  int getMaxBackupIndex();

  /**
   * Get the backup file name for specific index.
   * <p>
   * Generally, a backup file with normal file name 'log' and index 'n' could simply be 'log.bak.n',
   * you can specify your own naming rules, by overriding this method.
   * <p>
   * Make sure to return different backup file name with different backup index, and same backup
   * file name with same index. Otherwise, it will lead to unexpected behavior.
   *
   * @param fileName    the normal file name, generated by
   *                    {@link com.elvishew.xlog.printer.file.naming.FileNameGenerator#generateFileName(int, long)}
   * @param backupIndex the backup index, which will increase from 1 to {@link #getMaxBackupIndex()}
   * @return the backup file name
   */
  String getBackupFileName(String fileName, int backupIndex);
}
