package com.emarsys.logger.loggable

import java.time._

import cats.syntax.all._
import cats.{Contravariant, Show, Traverse}

import scala.annotation.implicitNotFound

@implicitNotFound("""
  Cannot use a value of type ${A} as a log parameter, as no implicit
  LoggableEncoder[${A}] instance is in scope.

  If you wish to create a LoggableEncoder instance for your class, in case ${A}
  is a case class or a sealed trait hierarchy, you may use the
  LoggableEncoder.deriveLoggableEncoder method to automatically generate it.
  """)
trait LoggableEncoder[A] {
  def toLoggable(a: A): LoggableValue
}

object LoggableEncoder
    extends LoggableEncoderStdlib1
    with LoggableEncoderTypeClassInterface
    with LoggableEncoderJdk8DateTime
    with LoggableEncoderScalaDuration
    with LoggableEncoderStdlib2
    with GenericLoggableEncoder {

  implicit val contravariantLoggableEncoder: Contravariant[LoggableEncoder] = new Contravariant[LoggableEncoder] {
    override def contramap[A, B](fa: LoggableEncoder[A])(f: B => A): LoggableEncoder[B] = b => fa.toLoggable(f(b))
  }

  implicit lazy val loggableValue: LoggableEncoder[LoggableValue] = identity[LoggableValue](_)
  implicit lazy val long: LoggableEncoder[Long]                   = LoggableIntegral(_)
  implicit lazy val double: LoggableEncoder[Double]               = LoggableFloating(_)
  implicit lazy val boolean: LoggableEncoder[Boolean]             = LoggableBoolean(_)
  implicit lazy val string: LoggableEncoder[String]               = LoggableString(_)

  def fromToString[A]: LoggableEncoder[A]   = string.contramap[A](_.toString)
  def fromShow[A: Show]: LoggableEncoder[A] = string.contramap[A](_.show)

  implicit lazy val int: LoggableEncoder[Int]       = long.contramap(_.toLong)
  implicit lazy val short: LoggableEncoder[Short]   = long.contramap(_.toLong)
  implicit lazy val byte: LoggableEncoder[Byte]     = long.contramap(_.toLong)
  implicit lazy val unit: LoggableEncoder[Unit]     = long.contramap(_ => 1)
  implicit lazy val float: LoggableEncoder[Float]   = double.contramap(_.toDouble)
  implicit lazy val char: LoggableEncoder[Char]     = fromToString
  implicit lazy val symbol: LoggableEncoder[Symbol] = string.contramap(_.name)
}

private[loggable] trait LoggableEncoderStdlib1 {
  self: LoggableEncoder.type =>
  import com.emarsys.logger.syntax._

  implicit def option[A: LoggableEncoder]: LoggableEncoder[Option[A]] = {
    case Some(value) => value.toLoggable
    case None        => LoggableNil
  }

  implicit def either[A: LoggableEncoder, B: LoggableEncoder]: LoggableEncoder[Either[A, B]] = {
    case Left(value)  => value.toLoggable
    case Right(value) => value.toLoggable
  }

  implicit def list[A: LoggableEncoder]: LoggableEncoder[List[A]] = l => LoggableList(l.map(_.toLoggable))

  implicit def set[A: LoggableEncoder]: LoggableEncoder[Set[A]] = list[A].contramap(_.toList)

  implicit def dict[A: LoggableEncoder]: LoggableEncoder[Map[String, A]] =
    m => LoggableObject(m.map { case (k, v) => (k, v.toLoggable) })
}

private[loggable] trait LoggableEncoderJdk8DateTime {
  self: LoggableEncoder.type =>

  implicit lazy val instant: LoggableEncoder[Instant]                = fromToString
  implicit lazy val localDate: LoggableEncoder[LocalDate]            = fromToString
  implicit lazy val localTime: LoggableEncoder[LocalTime]            = fromToString
  implicit lazy val localDateTime: LoggableEncoder[LocalDateTime]    = fromToString
  implicit lazy val zonedDateTime: LoggableEncoder[ZonedDateTime]    = fromToString
  implicit lazy val offsetTime: LoggableEncoder[OffsetTime]          = fromToString
  implicit lazy val offsetDateTime: LoggableEncoder[OffsetDateTime]  = fromToString
  implicit lazy val jdkduration: LoggableEncoder[java.time.Duration] = fromToString
}

private[loggable] trait LoggableEncoderScalaDuration {
  self: LoggableEncoder.type =>

  implicit lazy val scalaFiniteDuration: LoggableEncoder[scala.concurrent.duration.FiniteDuration] = fromToString
  implicit lazy val scalaDuration: LoggableEncoder[scala.concurrent.duration.Duration]             = fromToString
}

private[loggable] trait LoggableEncoderStdlib2 {
  self: LoggableEncoder.type with LoggableEncoderStdlib1 =>

  implicit def traversable[T[_]: Traverse, A: LoggableEncoder]: LoggableEncoder[T[A]] = list[A].contramap(_.toList)
}

private[loggable] trait LoggableEncoderTypeClassInterface {
  self: LoggableEncoder.type =>

  def apply[A](implicit instance: LoggableEncoder[A]): LoggableEncoder[A] = instance

  trait Ops[A] {
    def typeClassInstance: LoggableEncoder[A]
    def self: A
    def toLoggable: LoggableValue = typeClassInstance.toLoggable(self)
  }

  trait ToLoggableEncoderOps {
    import scala.language.implicitConversions
    implicit def toLoggableEncoderOps[A](target: A)(implicit tc: LoggableEncoder[A]): Ops[A] = new Ops[A] {
      override def self: A = target
      override def typeClassInstance: LoggableEncoder[A] = tc
    }
  }

  @deprecated("Use com.emarsys.logger.syntax._ import", since = "0.8.0")
  object nonInheritedOps extends ToLoggableEncoderOps

  @deprecated("Use LoggableEncoder.Ops[A] instead", since = "0.8.0")
  trait AllOps[A] extends Ops[A] {
    def typeClassInstance: LoggableEncoder[A]
  }

  @deprecated("Use com.emarsys.logger.syntax._ import", since = "0.8.0")
  object ops {
    import scala.language.implicitConversions
    implicit def toLoggableEncoderOps[A](target: A)(implicit tc: LoggableEncoder[A]): AllOps[A] = new AllOps[A] {
      override def self: A = target
      override def typeClassInstance: LoggableEncoder[A] = tc
    }
  }
}
