/*
 * Decompiled with CFR 0.152.
 */
package com.enonic.lib.license;

import com.enonic.lib.license.FormatHelper;
import com.enonic.lib.license.KeyPair;
import com.enonic.lib.license.LicenseDetails;
import com.enonic.lib.license.LicenseDetailsJSONConverter;
import com.enonic.lib.license.LicenseManager;
import com.enonic.lib.license.PrivateKey;
import com.enonic.lib.license.PublicKey;
import com.enonic.xp.app.ApplicationKey;
import com.enonic.xp.branch.Branch;
import com.enonic.xp.context.Context;
import com.enonic.xp.context.ContextAccessor;
import com.enonic.xp.context.ContextBuilder;
import com.enonic.xp.data.PropertyTree;
import com.enonic.xp.home.HomeDir;
import com.enonic.xp.node.CreateNodeParams;
import com.enonic.xp.node.Node;
import com.enonic.xp.node.NodePath;
import com.enonic.xp.node.NodeService;
import com.enonic.xp.node.UpdateNodeParams;
import com.enonic.xp.repository.CreateRepositoryParams;
import com.enonic.xp.repository.RepositoryId;
import com.enonic.xp.repository.RepositoryService;
import com.enonic.xp.resource.Resource;
import com.enonic.xp.resource.ResourceKey;
import com.enonic.xp.resource.ResourceService;
import com.enonic.xp.security.PrincipalKey;
import com.enonic.xp.security.RoleKeys;
import com.enonic.xp.security.User;
import com.enonic.xp.security.acl.AccessControlEntry;
import com.enonic.xp.security.acl.AccessControlList;
import com.enonic.xp.security.auth.AuthenticationInfo;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(immediate=true, service={LicenseManager.class})
public final class LicenseManagerImpl
implements LicenseManager {
    private static final Logger LOG = LoggerFactory.getLogger(LicenseManagerImpl.class);
    static final int KEY_SIZE = 2048;
    private static final String LICENSE_HEADER = "LICENSE";
    private static final String LIC_FILE_EXT = ".lic";
    private static final String LICENSE_DIR = "license";
    public static final RepositoryId REPO_ID = RepositoryId.from((String)"system-repo");
    public static final PrincipalKey MANAGER_ROLE = PrincipalKey.ofRole((String)"com.enonic.app.licensemanager");
    private static final String INSTALLED_LICENSES = "licenses";
    public static final NodePath INSTALLED_LICENSES_PATH = NodePath.create((NodePath)NodePath.ROOT, (String)"licenses").build();
    public static final String NODE_LICENSE_PROPERTY = "license";
    private NodeService nodeService;
    private RepositoryService repositoryService;
    private ResourceService resourceService;
    private ApplicationKey currentApp;
    private final Cache<String, Optional<LicenseDetails>> appValidationCache;

    public LicenseManagerImpl() {
        try {
            this.currentApp = ApplicationKey.from(this.getClass());
        }
        catch (NullPointerException e) {
            this.currentApp = null;
        }
        this.appValidationCache = CacheBuilder.newBuilder().maximumSize(10L).expireAfterWrite(20L, TimeUnit.SECONDS).build();
    }

    @Override
    public KeyPair generateKeyPair() {
        KeyPairGenerator keyGen = this.getRSAKeyPairGenerator();
        keyGen.initialize(2048);
        java.security.KeyPair rsaKeyPair = keyGen.genKeyPair();
        return new KeyPair(rsaKeyPair);
    }

    @Override
    public String generateLicense(PrivateKey privateKey, LicenseDetails license) {
        String licenseData = LicenseDetailsJSONConverter.serialize(license);
        licenseData = Base64.getEncoder().withoutPadding().encodeToString(licenseData.getBytes(StandardCharsets.UTF_8));
        String signature = this.sign(licenseData, privateKey.getRsaKey());
        return FormatHelper.asPEM(licenseData + "." + signature, LICENSE_HEADER);
    }

    @Override
    public LicenseDetails validateLicense(PublicKey publicKey, String license) {
        String signature;
        String licenseStr = this.unwrapLicense(license);
        int p = licenseStr.indexOf(46);
        if (p == -1) {
            return null;
        }
        String licenseData = licenseStr.substring(0, p);
        boolean valid = this.verify(licenseData, signature = licenseStr.substring(p + 1), publicKey.getRsaKey());
        if (!valid) {
            return null;
        }
        try {
            byte[] licenseDataBytes = Base64.getUrlDecoder().decode(licenseData);
            String licenseDataJson = new String(licenseDataBytes, StandardCharsets.UTF_8);
            return LicenseDetailsJSONConverter.parse(licenseDataJson);
        }
        catch (Exception e) {
            return null;
        }
    }

    @Override
    public LicenseDetails validateLicense(String appKey) {
        try {
            Optional licenseValue = (Optional)this.appValidationCache.get((Object)appKey, () -> {
                LicenseDetails license = this.doValidateLicense(appKey);
                return license == null ? Optional.empty() : Optional.of(license);
            });
            return licenseValue.orElse(null);
        }
        catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    private LicenseDetails doValidateLicense(String appKey) {
        String license;
        ApplicationKey app = this.currentApp;
        Resource pubKeyRes = this.resourceService.getResource(ResourceKey.from((ApplicationKey)app, (String)"/app.pub"));
        if (!pubKeyRes.exists()) {
            return null;
        }
        String publicKeyStr = pubKeyRes.readString();
        PublicKey publicKey = PublicKey.from(publicKeyStr);
        if (publicKey == null) {
            return null;
        }
        if (appKey == null) {
            appKey = app.toString();
        }
        if ((license = this.readLicenseFile(appKey)) == null) {
            license = this.readLicenseFromRepo(appKey);
        }
        if (license == null) {
            return null;
        }
        return this.validateLicense(publicKey, license);
    }

    @Override
    public boolean installLicense(String license, PublicKey publicKey, String appKey) {
        boolean installed = this.doInstallLicense(license, publicKey, appKey);
        this.appValidationCache.invalidate((Object)appKey);
        return installed;
    }

    private boolean doInstallLicense(String license, PublicKey publicKey, String appKey) {
        LicenseDetails licDetails = this.validateLicense(publicKey, license);
        if (licDetails == null) {
            return false;
        }
        Context currentCtx = ContextAccessor.current();
        if (!currentCtx.getAuthInfo().hasRole(RoleKeys.AUTHENTICATED)) {
            LOG.warn("License could not be installed, user not authenticated");
            return false;
        }
        Context ctxSudo = ContextBuilder.from((Context)currentCtx).repositoryId(REPO_ID).branch(Branch.from((String)"master")).authInfo(AuthenticationInfo.create().principals(new PrincipalKey[]{RoleKeys.ADMIN}).user(User.ANONYMOUS).build()).build();
        if (!((Boolean)ctxSudo.callWith(this::initializeRepo)).booleanValue()) {
            return false;
        }
        Context ctxRepo = ContextBuilder.from((Context)currentCtx).repositoryId(REPO_ID).branch(Branch.from((String)"master")).build();
        ctxRepo.runWith(() -> this.storeLicense(license, appKey));
        return true;
    }

    @Override
    public boolean uninstallLicense(String appKey) {
        boolean uninstalled = this.doUninstallLicense(appKey);
        this.appValidationCache.invalidate((Object)appKey);
        return uninstalled;
    }

    private boolean doUninstallLicense(String appKey) {
        Context currentCtx = ContextAccessor.current();
        if (!currentCtx.getAuthInfo().hasRole(RoleKeys.AUTHENTICATED)) {
            LOG.warn("License could not be uninstalled, user not authenticated");
            return false;
        }
        Context ctx = ContextBuilder.from((Context)currentCtx).repositoryId(REPO_ID).branch(Branch.from((String)"master")).authInfo(AuthenticationInfo.create().principals(new PrincipalKey[]{RoleKeys.ADMIN}).user(User.ANONYMOUS).build()).build();
        ctx.runWith(() -> {
            if (this.repositoryService.isInitialized(REPO_ID)) {
                this.deleteLicense(appKey);
            }
        });
        return true;
    }

    private void deleteLicense(String appKey) {
        NodePath path = NodePath.create((NodePath)INSTALLED_LICENSES_PATH, (String)appKey).build();
        this.nodeService.deleteByPath(path);
    }

    private void storeLicense(String license, String appKey) {
        PropertyTree data = new PropertyTree();
        data.setString("license", license);
        NodePath path = NodePath.create((NodePath)INSTALLED_LICENSES_PATH, (String)appKey).build();
        if (this.nodeService.nodeExists(path)) {
            UpdateNodeParams updateNode = UpdateNodeParams.create().path(path).editor(node -> {
                node.data = data;
            }).build();
            this.nodeService.update(updateNode);
            return;
        }
        CreateNodeParams createNode = CreateNodeParams.create().parent(INSTALLED_LICENSES_PATH).name(appKey).data(data).inheritPermissions(true).build();
        this.nodeService.create(createNode);
    }

    private boolean initializeRepo() {
        if (!this.repositoryService.isInitialized(REPO_ID)) {
            AccessControlList acl = AccessControlList.create().add(AccessControlEntry.create().principal(RoleKeys.ADMIN).allowAll().build()).add(AccessControlEntry.create().principal(MANAGER_ROLE).allowAll().build()).build();
            CreateRepositoryParams createRepo = CreateRepositoryParams.create().repositoryId(REPO_ID).rootPermissions(acl).build();
            this.repositoryService.createRepository(createRepo);
        }
        try {
            if (this.nodeService.nodeExists(INSTALLED_LICENSES_PATH)) {
                return true;
            }
            CreateNodeParams createNode = CreateNodeParams.create().name(INSTALLED_LICENSES_PATH.getName()).inheritPermissions(true).parent(NodePath.ROOT).build();
            this.nodeService.create(createNode);
            return true;
        }
        catch (Exception e) {
            LOG.warn("Could not initialize license repo", (Throwable)e);
            return false;
        }
    }

    private String unwrapLicense(String license) {
        if (license == null) {
            return "";
        }
        String[] allLines = license.split("\\r?\\n");
        List lines = Arrays.stream(allLines).map(String::trim).filter(s -> !s.isEmpty()).collect(Collectors.toList());
        if (lines.size() < 3) {
            return "";
        }
        String header = (String)lines.get(0);
        String footer = (String)lines.get(lines.size() - 1);
        lines.remove(0);
        lines.remove(lines.size() - 1);
        if (!FormatHelper.isPEMHeader(header, LICENSE_HEADER) || !FormatHelper.isPEMFooter(footer, LICENSE_HEADER)) {
            return "";
        }
        return lines.stream().collect(Collectors.joining(""));
    }

    private String sign(String plainText, java.security.PrivateKey privateKey) {
        try {
            Signature privateSignature = Signature.getInstance("SHA256withRSA");
            privateSignature.initSign(privateKey);
            privateSignature.update(plainText.getBytes(StandardCharsets.UTF_8));
            byte[] signature = privateSignature.sign();
            return Base64.getEncoder().withoutPadding().encodeToString(signature);
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    private boolean verify(String plainText, String signature, java.security.PublicKey publicKey) {
        try {
            Signature publicSignature = Signature.getInstance("SHA256withRSA");
            publicSignature.initVerify(publicKey);
            publicSignature.update(plainText.getBytes(StandardCharsets.UTF_8));
            byte[] signatureBytes = Base64.getDecoder().decode(signature);
            return publicSignature.verify(signatureBytes);
        }
        catch (Exception e) {
            return false;
        }
    }

    private KeyPairGenerator getRSAKeyPairGenerator() {
        try {
            return KeyPairGenerator.getInstance("RSA");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    private String readLicenseFile(String appKey) {
        String fileName = appKey + LIC_FILE_EXT;
        Path xpHome = HomeDir.get().toFile().toPath();
        Path licenseDir = xpHome.resolve("license");
        Path licensePath = licenseDir.resolve(fileName);
        if (!licensePath.toFile().isFile()) {
            return null;
        }
        try {
            return new String(Files.readAllBytes(licensePath), StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            return null;
        }
    }

    private String readLicenseFromRepo(String appKey) {
        Context ctxRepo = ContextBuilder.from((Context)ContextAccessor.current()).repositoryId(REPO_ID).branch(Branch.from((String)"master")).authInfo(AuthenticationInfo.create().principals(new PrincipalKey[]{RoleKeys.ADMIN}).user(User.ANONYMOUS).build()).build();
        return (String)ctxRepo.callWith(() -> this.loadLicense(appKey));
    }

    private String loadLicense(String appKey) {
        try {
            NodePath path = NodePath.create((NodePath)INSTALLED_LICENSES_PATH, (String)appKey).build();
            Node licenseNode = this.nodeService.getByPath(path);
            if (licenseNode == null) {
                return null;
            }
            return licenseNode.data().getString("license");
        }
        catch (Exception e) {
            return null;
        }
    }

    public void setCurrentApp(ApplicationKey currentApp) {
        this.currentApp = currentApp;
    }

    @Reference
    public void setNodeService(NodeService nodeService) {
        this.nodeService = nodeService;
    }

    @Reference
    public void setResourceService(ResourceService resourceService) {
        this.resourceService = resourceService;
    }

    @Reference
    public void setRepositoryService(RepositoryService repositoryService) {
        this.repositoryService = repositoryService;
    }
}

