package com.enterprisemath.utils.cache;

import com.enterprisemath.utils.DomainUtils;
import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Implementation of object cache which stores everything in memory.
 *
 * @author radek.hecl
 */
public class InMemoryObjectCache implements ObjectCache {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public InMemoryObjectCache build() {
            return new InMemoryObjectCache(this);
        }
    }

    /**
     * Buffer map.
     */
    private Map<String, Object> buffer;

    /**
     * Object for locking.
     */
    private final Object lock = new Object();

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public InMemoryObjectCache(Builder builder) {
        guardInvariants();
        buffer = new HashMap<String, Object>();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
    }

    @Override
    public void put(String key, Object object) {
        synchronized (lock) {
            buffer.put(key, object);
        }
    }

    @Override
    public void remove(String key) {
        synchronized (lock) {
            buffer.remove(key);
        }
    }

    @Override
    public Object get(String key) {
        synchronized (lock) {
            if (!buffer.containsKey(key)) {
                throw new NoSuchElementException("element doesn't exists: key = " + key);
            }
            return buffer.get(key);
        }
    }

    @Override
    public Object get(String key, Object def) {
        synchronized (lock) {
            if (!buffer.containsKey(key)) {
                return def;
            }
            return buffer.get(key);
        }
    }

    @Override
    public Set<String> getKeys() {
        synchronized (lock) {
            return DomainUtils.softCopySet(buffer.keySet());
        }
    }

    @Override
    public void clear() {
        synchronized (lock) {
            buffer = new HashMap<String, Object>();
        }
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     *
     * @return created instance
     */
    public static InMemoryObjectCache create() {
        return new InMemoryObjectCache.Builder().
                build();
    }
}
