package com.enterprisemath.utils.engine;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Task service provider which stores available services in the plain map.
 *
 * @author radek.hecl
 */
public class MapTaskServiceProvider implements TaskServiceProvider {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Services.
         */
        private Map<String, Object> services = new HashMap<String, Object>();

        /**
         * Sets services.
         *
         * @param services services
         * @return this instance
         */
        public Builder setServices(Map<String, Object> services) {
            this.services = DomainUtils.softCopyMap(services);
            return this;
        }

        /**
         * Adds service.
         *
         * @param name name
         * @param service service
         * @return this instance
         */
        public Builder addService(String name, Object service) {
            this.services.put(name, service);
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public MapTaskServiceProvider build() {
            return new MapTaskServiceProvider(this);
        }
    }

    /**
     * Services.
     */
    private Map<String, Object> services;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public MapTaskServiceProvider(Builder builder) {
        services = DomainUtils.softCopyUnmodifiableMap(builder.services);
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotEmptyNullMap(services, "services cannot have empty key or null value");
    }

    @Override
    public <T> T getService(String name, Class<T> clazz) {
        return (T) services.get(name);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
