package com.enterprisemath.utils.messaging;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.PostConstruct;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.velocity.app.VelocityEngine;

import com.enterprisemath.utils.Dates;
import com.enterprisemath.utils.DatesProvider;
import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.io.StringWriter;
import org.apache.commons.io.FileUtils;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.exception.VelocityException;

/**
 * Messenger implementation which stores messages as files.
 *
 * @author radek.hecl
 *
 */
public class FileMessenger implements Messenger {

    /**
     * Path to the output directory.
     */
    private String outputDirectoryPath;

    /**
     * Provider for templates.
     */
    private MessageTemplateProvider templateProvider;

    /**
     * Model provided by default to the
     */
    private Map<String, Object> defaultModel;

    /**
     * Velocity engine for formatting messages.
     */
    private VelocityEngine velocityEngine;

    /**
     * Provider for dates.
     */
    private DatesProvider datesProvider;

    /**
     * Creates new instance.
     */
    private FileMessenger() {
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    @PostConstruct
    private void guardInvariants() {
        ValidationUtils.guardNotEmpty(outputDirectoryPath, "outputDirectoryPath cannot be empty");
        ValidationUtils.guardNotNull(templateProvider, "templateProvider cannot be null");
        ValidationUtils.guardNotEmptyNullMap(defaultModel, "defaultModel cannot have empty key or null value");
        ValidationUtils.guardNotNull(velocityEngine, "velocityEngine cannot be null");
        ValidationUtils.guardNotNull(datesProvider, "datesProvider cannot be null");
    }

    @Override
    public synchronized void send(String target, Message message) {
        // format
        Map<String, Object> model = new HashMap<String, Object>();
        model.putAll(defaultModel);
        model.putAll(DomainUtils.convertPropertyMapIntoVelocityModel(message.getParameters()));
        String subject = applyTemplate(templateProvider.getTemplate(message.getType(), "subject"), model);
        String text = applyTemplate(templateProvider.getTemplate(message.getType(), "message"), model);

        // send
        String dateStr = Dates.format(datesProvider.now(), "yyyy-MM-dd-HH-mm-ss");
        File headerFile = new File(outputDirectoryPath + "/" + dateStr + "-header.txt");
        File messageFile = new File(outputDirectoryPath + "/" + dateStr + "-message.html");
        File directory = headerFile.getParentFile();
        if (!directory.exists() && !directory.mkdirs()) {
            throw new RuntimeException("error during directory creation: path = " + directory.getAbsolutePath());
        }
        try {
            String headerTxt = target + "\n" + subject;
            FileUtils.writeStringToFile(headerFile, headerTxt, "utf-8");
            FileUtils.writeStringToFile(messageFile, text, "utf-8");
            for (Attachment att : message.getAttachments()) {
                File attFile = new File(outputDirectoryPath + "/" + dateStr + "-" + att.getName());
                FileUtils.writeByteArrayToFile(attFile, att.getBuf());
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        try {
            Thread.sleep(2000);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Applies template to the model and returns result
     *
     * @param templateLocation template location
     * @param model model
     * @return result after operation is done
     */
    public String applyTemplate(String templateLocation, Map<?, ?> model) {
        StringWriter result = new StringWriter();
        try {
            VelocityContext velocityContext = new VelocityContext(model);
            velocityEngine.mergeTemplate(templateLocation, "utf-8", velocityContext, result);
        } catch (VelocityException e) {
            throw new RuntimeException(e);
        }
        return result.toString();
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     *
     * @param outputDirectoryPath path for the output directory
     * @param templateProvider provider for message templates
     * @param defaultModel model accessible always in all templates
     * @param velocityEngine velocity engine
     * @param datesProvider provider for dates
     * @return created instance
     */
    public static FileMessenger create(String outputDirectoryPath, MessageTemplateProvider templateProvider,
            Map<String, Object> defaultModel, VelocityEngine velocityEngine, DatesProvider datesProvider) {
        FileMessenger res = new FileMessenger();
        res.outputDirectoryPath = outputDirectoryPath;
        res.templateProvider = templateProvider;
        res.defaultModel = DomainUtils.softCopyUnmodifiableMap(defaultModel);
        res.velocityEngine = velocityEngine;
        res.datesProvider = datesProvider;
        res.guardInvariants();
        return res;
    }

}
