package com.enterprisemath.utils.messaging;

import com.enterprisemath.utils.ValidationUtils;
import java.util.Map;
import javax.annotation.PostConstruct;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.exception.ExceptionUtils;

/**
 * Implementation of system messenger which proxies messages to the normal messenger.
 *
 * @author radek.hecl
 *
 */
public class ProxySystemMessenger implements SystemMessenger {

    /**
     * Target for the system messages.
     */
    private String target;

    /**
     * Underlined messenger.
     */
    private Messenger messenger;

    /**
     * Creates new instance.
     */
    private ProxySystemMessenger() {
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    @PostConstruct
    private void guardInvariants() {
        ValidationUtils.guardNotEmpty(target, "target cannot be empty");
        ValidationUtils.guardNotNull(messenger, "messenger cannot be null");
    }

    @Override
    public void send(Message message) {
        messenger.send(target, message);
    }

    @Override
    public void sendError(String caption, String explanation) {
        Message message = new Message.Builder().
                setType("error").
                addParameter("caption", caption).
                addParameter("explanation", explanation).
                addParameter("exception.class", "").
                addParameter("exception.message", "").
                addParameter("exception.stackTrace", "").
                build();
        messenger.send(target, message);
    }

    @Override
    public void sendError(String caption, String explanation, Throwable e) {
        Message message = new Message.Builder().
                setType("error").
                addParameter("caption", caption).
                addParameter("explanation", explanation).
                addParameter("exception.class", e.getClass().toString()).
                addParameter("exception.message", e.getMessage()).
                addParameter("exception.stackTrace", ExceptionUtils.getStackTrace(e)).
                build();
        messenger.send(target, message);
    }

    @Override
    public void sendError(String caption, String explanation, Map<String, String> exceptionDetails) {
        Message message = new Message.Builder().
                setType("error").
                addParameter("caption", caption).
                addParameter("explanation", explanation).
                addParameter("exception.class", StringUtils.defaultString(exceptionDetails.get("class"))).
                addParameter("exception.message", StringUtils.defaultString(exceptionDetails.get("message"))).
                addParameter("exception.stackTrace", StringUtils.defaultString(exceptionDetails.get("stackTrace"))).
                build();
        messenger.send(target, message);

    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     *
     * @param target default target
     * @param messenger underlined messenger
     * @return created instance
     */
    public static ProxySystemMessenger create(String target, Messenger messenger) {
        ProxySystemMessenger res = new ProxySystemMessenger();
        res.target = target;
        res.messenger = messenger;
        res.guardInvariants();
        return res;

    }

}
