/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.db;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import com.entitystream.monster.db.Document;
import com.google.gson.Gson;
import com.entitystream.identiza.entity.resolve.metadata.Purpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumn;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.ITableColumn;
import com.entitystream.identiza.entity.resolve.metadata.Table;
import com.entitystream.identiza.entity.resolve.metadata.TableColumn;
import com.entitystream.identiza.entity.resolve.processing.IdentizaException;
import com.entitystream.identiza.entity.resolve.storage.Record;



public class Node implements Serializable, INode{
	IDBContainer db;
	String id;
	Date when;
	Document baseDoc;
	private Logger logger = Logger.getLogger("Node");
	private String tableName;
	public Node(IDBContainer db, Document document) throws Exception {
		this.baseDoc=document;
		this.db=db;
		if (baseDoc!=null&&baseDoc.containsKey("_id"))
			this.id=baseDoc.getString("_id");
		else
			this.id=null;
	}

	public Node(Document document) throws Exception {
		this.baseDoc=document;
		if (baseDoc.containsKey("_id"))
			this.id=baseDoc.getString("_id");
		else
			this.id=null;
	}

	public Node(IDBContainer db, String id, Date when) throws Exception {
		this.baseDoc=db.getNode(id);
		this.db=db;
		this.id=id;
		this.when=when;
	}
	public Node(IDBContainer db, String id, Document baseDoc) throws Exception {
		this.baseDoc=baseDoc;
		this.db=db;
		this.id=id;
	}

	public Node(IDBContainer db, String id, Date when,Document baseDoc) throws Exception {
		this.baseDoc=baseDoc;
		this.db=db;
		this.id=id;
		this.when=when;
	}


	public Node(IDBContainer db, String id) {
		this.id=id;
		this.db=db;
		this.baseDoc=db.getNode(id);

	}

	public Node(ITable table, Map values) {
		//creates a node from some fake values
		String incPath="";
		if (baseDoc==null)
			baseDoc = new Document();

		for (Object col : values.keySet()){
			Object parent=baseDoc;
			String colname="";
			if (col instanceof TableColumn)
				colname=((TableColumn)col).getColName();
			else if (col instanceof String)
				colname=(String)col;
			String[] path = colname.split("\\.");
			for (int p=0; p<path.length-1; p++){
				String property=path[p];
				int inspos=property.indexOf("[");
				int instance=0;
				if (inspos>-1)
				{
					String instanceS=property.substring(inspos).replace("[", "").replace("]", "");
					instance=Integer.parseInt(instanceS);
					property=property.substring(0,inspos);
				}

				incPath+=property;
				ITableColumn pathCol = table.getColumn(incPath);
				if (parent instanceof ArrayList){
					if (pathCol.getDisplayType().equalsIgnoreCase("Structure")){
						((ArrayList)parent).add(new Document());
					} else if(pathCol.getDisplayType().equalsIgnoreCase("List")){
						((ArrayList)parent).add(instance, new ArrayList());
					}
				} else if (parent instanceof Document){
					if (pathCol.getDisplayType().equalsIgnoreCase("Structure")){
						((Document)parent).append(property,new Document());
					} else if(pathCol.getDisplayType().equalsIgnoreCase("List")){
						((Document)parent).append(property,new ArrayList());
					}
				}
				incPath+=".";
			}
			//parent now represents the path to where the value should be put - and it will be a documnet
			if (parent instanceof Document)
				((Document)parent).append(path[path.length-1], values.get(col));

		}

	}


	@Override
	public boolean hasProperty(String arg0)
	{
		if (baseDoc!=null)
			return baseDoc.containsKey(arg0);
		else
			return false;
	}

	

	public Object setProperty(String arg0, Object arg1){
		if (baseDoc==null)
			baseDoc=new Document();


		Object parent=baseDoc;
		String incPath="";
		if (arg0==null)
			return null;
		String[] paths= arg0.split("\\.");
		for (int p=0;p<paths.length-1; p++){
			String pathItem=paths[p];
			String sPathItem=pathItem;
			if (sPathItem.indexOf("[")!=-1)
				sPathItem.substring(0,sPathItem.indexOf("["));
			Object newParent=((Document)parent).get(sPathItem);
			if (newParent==null) {
				if (pathItem.contains("["))
					newParent=new ArrayList<>();
				else 
					newParent=new Document();
			}
			if (parent instanceof Document){
				if (!((Document)parent).containsKey(pathItem)){
					((Document) parent).append(pathItem, newParent);
				}
			} else if (parent instanceof ArrayList){
				String[] spl =pathItem.split("[");
				pathItem=spl[0];
				int instance=Integer.parseInt(spl[1].replace("]", ""));
				((ArrayList) parent).add(instance,new Document(pathItem, newParent));
			}
			parent=newParent;
		}
		arg0=arg0.substring(arg0.lastIndexOf(".")+1);
		if (parent instanceof Document){
			((Document)parent).append(arg0,arg1);
		} 

		return baseDoc;
	}

	

	public String getId(){
		return id;
	}

	public Iterator<IRelationship> getRelationships(String arg0, Date when) {
		ArrayList<IRelationship> rels = new ArrayList<IRelationship>();
		try{
		    Iterator<String> iterator = db.getNodeRelationships(id, arg0, when).iterator();
		
			while (iterator.hasNext()){
			    String lng = iterator.next();
			    Document reldoc = db.getRelProperties(lng, when);
			    Relationship rel = new Relationship(db, reldoc, when);
			    rels.add(rel);
			}
		}catch (Exception oe){
			oe.printStackTrace();
		}
		return rels.iterator();
	}

	public Iterator<IRelationship> getRelationships(Date when) {
		ArrayList<IRelationship> rels = new ArrayList<IRelationship>();
		try{
		    Iterator <String> iterator = db.getNodeRelationships(id, when).iterator();
		
			while (iterator.hasNext()){
			    String lng = iterator.next();
			    Document reldoc = db.getRelProperties(lng, when);
				Relationship rel = new Relationship(db, reldoc, when);
				rels.add(rel);
			}
		}catch (Exception oe){
			oe.printStackTrace();
		}
		return rels.iterator();
	}

	public Iterator<IRelationship> getRelationships(String reltype, String dir, Date when) throws Exception {
		ArrayList<IRelationship> rels = new ArrayList<IRelationship>();
		if (id!=null){
			ArrayList<String> frels = db.getNodeRelationships(id, reltype, dir, when);
			if (frels!=null){
				Iterator <String> iterator = frels.iterator();
				try{
					while (iterator.hasNext()){
					    	String lng = iterator.next();
					    	Document reldoc = db.getRelProperties(lng, when);
						Relationship rel = new Relationship(db, reldoc, when);
						rels.add(rel);
					}
				}catch (IdentizaException oe){
					oe.printStackTrace();
				}
			}
		}
		return rels.iterator();
	}

	

	public Object getProperty(String arg0) {
		return getProperty(baseDoc,arg0);
	}

	private Object getProperty(Document doc, String arg){
		Object obj = getPropertyPath(doc, arg+"");
		if (obj!=null){
			arg=arg.substring(arg.lastIndexOf(".")+1);
			if (obj instanceof Document){
				return ((Document)obj).get(arg);
			} else if (obj instanceof ArrayList) {
				ArrayList<Object> ret = new ArrayList();
				for (Object arrayItem : ((ArrayList)obj)){
					//base has to be a document?
					if (arrayItem instanceof Document)
						ret.add(((Document) arrayItem).get(arg));
					else if ((arrayItem instanceof ArrayList))
						ret.addAll((ArrayList)arrayItem);
					else
						ret.add(arrayItem);
				}
				return ret;
			} else return obj;
		} else return null;
	}

	private Object getPropertyPath(Document doc, String arg){
		if (arg==null)
			return null;
		//paths look like this
		//LEI.Entity.Addresses[0].Address.Lines[0].LineText - yields one item - the parent of LineText
		//LEI.Entity.Addresses.Address.Lines.LineText - yields a cartesian product to Lines (array of Lines)
		int pos =arg.indexOf(".");
		if (pos>-1){
			Object parent=doc;
			String[] propList=arg.split("\\.");
			for (int p=0; p<propList.length-1; p++){
				String property=propList[p];
				int instance=0;
				int inspos=property.indexOf("[");
				if (inspos>-1)
				{
					String instanceS=property.substring(inspos).replace("[", "").replace("]", "");
					instance=Integer.parseInt(instanceS);
					property=property.substring(0,inspos);
				}

				if (parent instanceof Document)
					parent=((Document)parent).get(propList[p]);
				else if (parent instanceof ArrayList){
					ArrayList newparent=new ArrayList();
					for (Object arrayItem : ((ArrayList)parent)){
						if (arrayItem instanceof Document)
							newparent.add(((Document)arrayItem).get(propList[p]));
					}
					parent=newparent;
				}
			}
			return parent;
		} else
			return doc;
	}

	public IRelationship createRelationshipTo(INode arg0, String arg1, Date when, Map<String, Object> values) throws Exception{
		try {
			Document rel =db.createNodeRelationshipTo(id, this.getProperty("Table").toString(), arg0.getId(), arg0.getProperty("Table").toString(), arg1, values, when);
			return new Relationship(db,rel, when);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return null;
		}
	}

	public Map<String, Object> getPropertyValues() throws Exception {
		//flatten the structure into fake record format
		Map<String, Object> ret = new HashMap<String, Object>();
		for (String k : baseDoc.keyString())
			flattenDoc(k,baseDoc.get(k), ret,"");
		return ret;
	}

	
	public static Document expandDoc(Map<String, Object> flatdoc, ITable iTable) {
		Document temp = new Document();


		for (String key : flatdoc.keySet()){
			Object value =  flatdoc.get(key);
			if (key!=null && value!=null && !(value instanceof String && ((String)value).length()==0)){
				//key==LEI.Entity.Registration.regDate
				String[] parts = key.split("\\.");
				Object parent=temp;
				String partInc="";
				for (String part : parts){
					partInc+=part;

					ITableColumn col = null;
					
					String colName = partInc.replaceAll("\\[[0-9]*?\\]", "");
					
					if (iTable!=null)
					   col=iTable.getColumn(colName);
					String type=null;
					if (col!=null)
						type=col.getDisplayType();
					if (type==null || type.length()==0)
						type="text";
					int levelinstance=0;
					if (part.contains("[")){
						String[] partSplit= part.split("\\[");
						part=partSplit[0];
						levelinstance=Integer.parseInt(partSplit[1].replace("]", ""));
					}
					if (type.equalsIgnoreCase("Structure")){
						Object newParent=new Document();
						if (parent instanceof Document){
							if (!((Document)parent).containsKey(part))
								((Document)parent).append(part, newParent);
							else newParent=((Document)parent).get(part);
						}
						else if (parent instanceof ArrayList){
							//expand the array to enable us to place an item at the level
							if (levelinstance>((ArrayList)parent).size()-1)
								for (int expand=((ArrayList)parent).size()-1; expand<levelinstance; expand++)
									((ArrayList)parent).add(new Document());
							//get the current document?
							Document listparent = ((Document) ((ArrayList)parent).get(levelinstance));
							Document listitem = (Document) listparent.get(part);
							if (listitem==null)
								listitem=new Document();
							if (!(listparent).containsKey(part))
								listparent.append(part, newParent);
							else 
								newParent=listitem;
							//((ArrayList)parent).set(levelinstance,listparent);
						}
						parent=newParent;

					} else if (type.equalsIgnoreCase("List")){
						Object newParent=new ArrayList();
						if (parent instanceof Document)
							if (!((Document)parent).containsKey(part))
								((Document)parent).append(part, newParent);
							else newParent=((Document)parent).get(part);
						else if (parent instanceof ArrayList){
							if (levelinstance>((ArrayList)parent).size()-1)
								for (int expand=((ArrayList)parent).size()-1; expand<levelinstance-1; expand++)
									((ArrayList)parent).add(new Document());
							((ArrayList)parent).set(levelinstance, new Document(part,newParent));
						}
						parent=newParent;
					} else {///text-date-etc
						if (parent instanceof Document)
							((Document)parent).append(part, value);
						else if (parent instanceof ArrayList){
							//get the position in the array defined by instance - it will be a document
							//add this value to it.
							Object instancearray = null;
							if (((ArrayList)parent).size()>levelinstance)
								instancearray = ((ArrayList)parent).get(levelinstance);
							if (instancearray==null)
								instancearray=new Document();
							if (instancearray instanceof Document)
								((Document)instancearray).append(part, value);
						}
					}
					partInc+=".";
				}
			}
		}
		return temp;
	}

	public static void flattenDoc(String path, Object obj, Map<String, Object> ret, String instanceText){
		if (obj instanceof Document){
			for (String k : ((Document)obj).keyString())
			    if (path.length()>0)
				flattenDoc(path+"."+k+instanceText,((Document)obj).get(k), ret,"");
			    else 
				flattenDoc(k+instanceText,((Document)obj).get(k), ret,"");
		} else if (obj instanceof ArrayList){
			int instance=0;
			for (Object innerobj : ((ArrayList)obj)){
				flattenDoc(path,innerobj, ret, "["+instance+"]");
				instance++;
			}
		} else if (obj!=null)
			ret.put(path+instanceText, obj);
	}

	public static void flattenDocToString(String path, Object obj, StringBuilder ret, String instanceText){
		if (obj instanceof Document){
			for (String k : ((Document)obj).keyString())
			    if (path.length()>0)
				flattenDocToString(path+"."+k+instanceText,((Document)obj).get(k), ret,"");
			    else
				flattenDocToString(k+instanceText,((Document)obj).get(k), ret,"");
		} else if (obj instanceof ArrayList){
			int instance=0;
			for (Object innerobj : ((ArrayList)obj)){
				flattenDocToString(path,innerobj, ret,"["+instance+"]");
				instance++;
			}
		} else ret.append(obj + " ");
	}

	public void setDB(IDBContainer db2) {
		db=db2;

	}

	public void setProperties(Document vals) {
		baseDoc.putAll(vals);
	}

	public Document getDocument() {
		return baseDoc;
	}

	public ArrayList<String> getProperties(List<String> cols) {

		ArrayList<StringBuilder> fullvals = new ArrayList<StringBuilder>();
		for (String colname : cols){
			String lastParent="";
			String parent="";
			if (colname.indexOf(".")>-1)
				parent=colname.substring(0,colname.lastIndexOf("."));
			else 
				parent=colname;
			Object value=getProperty(colname);
			if (parent.equalsIgnoreCase(lastParent) /*same parent*/ || lastParent.length()==0 /*first*/){
				//append next value in sequence
				if (value!=null){
					if (value instanceof ArrayList){
						for (int p=0; p<((ArrayList)value).size(); p++){
							String s=Document.objectToString(((ArrayList)value).get(p))+" ";
							if (fullvals.size()-1<p || fullvals.get(p)==null)
								fullvals.add(p, new StringBuilder(s));
							else
								fullvals.get(p).append(s);
						}
					} else {
						String s=Document.objectToString(value)+" ";
						if (fullvals.size()-1 <0 || fullvals.get(0)==null)
							fullvals.add(0, new StringBuilder(s));
						else 
							fullvals.get(0).append(s);
					}
				}
			} else {
				//append each value to each existing value
				for (StringBuilder fullval : fullvals){
					if (value instanceof ArrayList){
						for (int p=0; p<((ArrayList)value).size(); p++){
							String s=Document.objectToString(((ArrayList)value).get(p))+" ";
							fullval.append(s);
						}
					} else {
						String s=Document.objectToString(value)+" ";
						fullval.append(s);
					}
				}
			}
			lastParent=parent;
		}
		ArrayList<String> ret = new ArrayList<String>();
		for (StringBuilder fullval : fullvals)
			ret.add(fullval.toString().trim());
		return ret;
	}

	public static boolean applyInheritanceFrom(INode iNode2, INode iNode, HashSet<IPurpose> hashSet, ISchemaMeta schDoc) {
		boolean hasChanged=false;
		for (IPurpose purpose : hashSet){
			int targetAlgo=purpose.getTargetAlgo();
			if (targetAlgo != -1){

				Date tlu=iNode.getDocument().getDate("lastUpdated");
				Date lu=iNode2.getDocument().getDate("lastUpdated");

				for (PurposeColumn pc : schDoc.getPurposeColumns(purpose.getPurposeName(),iNode2.getTableName()))
					//loop through the maps
					for (PurposeColumnMap pcm : schDoc.getPurposeColumnMaps(purpose.getPurposeName(), pc.getColumn(),iNode2.getTableName())){
						//copy the value from the source field to the target 
						boolean done=false;
						for (PurposeColumnMap pcmExt : schDoc.getPurposeColumnMaps(purpose.getPurposeName(), pc.getColumn(),iNode.getTableName())){
							//copy the value from the source field to the target 
							
							if (pcmExt.getPurposeColumn().equalsIgnoreCase(pcm.getPurposeColumn()) &&
								pcmExt.getPurposeName().equalsIgnoreCase(pcm.getPurposeName()) &&
								pcm.getTableName().equalsIgnoreCase(iNode2.getTableName())
							   ){
								try {
									Object targetValue=iNode.getProperty(pcmExt.getTableColumn());
									Object currentValue=iNode2.getProperty(pcm.getTableColumn());
									if ((lu==null || tlu==null || tlu.after(lu)) || targetAlgo==1){
										//System.out.println(pcmExt.getPurposeName()+":"+pcmExt.getPurposeColumn()+":"+pcmExt.getTableName()+":"+pcmExt.getTableColumn()+"="+currentValue);
										iNode.setProperty(pcmExt.getTableColumn(), currentValue);
										hasChanged=true;
										done=true;
										break;
									}
									else if (targetAlgo==2 &&
											targetValue!=null &&
											Document.objectToString(currentValue).length()>Document.objectToString(targetValue).length()){
										//System.out.println(pcmExt.getPurposeName()+":"+pcmExt.getPurposeColumn()+":"+pcmExt.getTableName()+":"+pcmExt.getTableColumn()+"="+currentValue);
										iNode.setProperty(pcmExt.getTableColumn(), currentValue);
										hasChanged=true;
										done=true;
										break;
									}
									else if (targetAlgo==0){
										iNode.setProperty(pcmExt.getTableColumn(), currentValue);
										//System.out.println(pcmExt.getPurposeName()+":"+pcmExt.getPurposeColumn()+":"+pcmExt.getTableName()+":"+pcmExt.getTableColumn()+"="+currentValue);
										hasChanged=true;
										done=true;
										break;
									}
									

								} catch (Exception e) {
									e.printStackTrace();
								}
							}
						}
						if (done)
							break;
					}
			}

		}
		try{
			if (hasChanged)
				iNode.setProperty("lastUpdated", new Date());
		} catch (Exception e){
			e.printStackTrace();
		}
		return hasChanged;
	}

	public String getTableName() {
		if (tableName==null && baseDoc.containsKey("Table"))
			this.tableName=baseDoc.getString("Table");
		return tableName;
	}

	public void setID(String id) {
		this.id=id;
		if (baseDoc!=null)
			baseDoc.put("ROWID", id);
	}

	public String toJSON() throws Exception {			
		return baseDoc.toJson();

	}

	



}