/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    aString with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.db.path;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.entitystream.identiza.db.DBWorker;

public class Dijkstra {

	  private DBWorker db;
	  private Set<String> settledNodes;
	  private Set<String> unSettledNodes;
	  private Map<String, String> predecessors;
	  private Map<String, Integer> distance;
	  private List<String> validTables;

	  public Dijkstra(DBWorker db, List<String> validTables) {
		  this.db=db;
		  this.validTables=validTables;
	  }

	  public void execute(String source, Date when) {
	    settledNodes = new HashSet<String>();
	    unSettledNodes = new HashSet<String>();
	    distance = new HashMap<String, Integer>();
	    predecessors = new HashMap<String, String>();
	    distance.put(source, 0);
	    unSettledNodes.add(source);
	    while (unSettledNodes.size() > 0) {
	      String node = getMinimum(unSettledNodes);
	      settledNodes.add(node);
	      unSettledNodes.remove(node);
	      findMinimalDistances(node, when);
	    }
	  }

	  private void findMinimalDistances(String node, Date when) {
	    List<String> adjacentNodes = getNeighbors(node, when);
	    for (String target : adjacentNodes) {
	      if (getShortestDistance(target) > getShortestDistance(node)
	          + getDistance(node, target, when)) {
	        distance.put(target, getShortestDistance(node)
	            + getDistance(node, target, when));
	        predecessors.put(target, node);
	        unSettledNodes.add(target);
	      }
	    }

	  }

	  private int getDistance(String node, String target, Date when) {
	    for (String edge : db.getNodeRelationships(node, when)) {
	      if (db.getRelOtherNode(edge, node, new Date())==target) {
	        return 0;
	      }
	    }
	    throw new RuntimeException("Should not happen");
	  }

	  private List<String> getNeighbors(String node, Date when) {
	    List<String> neighbors = new ArrayList<String>();
	    for (String edge : db.getNodeRelationships(node, when)) {
		String dest=db.getRelOtherNode(edge, node, new Date());
		  if (db.hasNodeProperty(dest, "Table" )){
				String table = (String) db.getNodeProperty( dest, "Table" );
				if (validTables.contains(table)){
				      if (!isSettled(dest)) {
				        neighbors.add(dest);
				      }
				}
			} 
	    }
	    return neighbors;
	  }

	  private String getMinimum(Set<String> Stringes) {
	    String minimum = null;
	    for (String String : Stringes) {
	      if (minimum == null) {
	        minimum = String;
	      } else {
	        if (getShortestDistance(String) < getShortestDistance(minimum)) {
	          minimum = String;
	        }
	      }
	    }
	    return minimum;
	  }

	  private boolean isSettled(String String) {
	    return settledNodes.contains(String);
	  }

	  private int getShortestDistance(String destination) {
	    Integer d = distance.get(destination);
	    if (d == null) {
	      return Integer.MAX_VALUE;
	    } else {
	      return d;
	    }
	  }

	  /*
	   * This method returns the path from the source to the selected target and
	   * NULL if no path exists
	   */
	  public LinkedList<String> getPath(String target) {
	    LinkedList<String> path = new LinkedList<String>();
	    String step = target;
	    // Check if a path exists
	    if (predecessors.get(step) == null) {
	      return null;
	    }
	    path.add(step);
	    while (predecessors.get(step) != null) {
	      step = predecessors.get(step);
	      path.add(step);
	    }
	    // Put it into the correct order
	    Collections.reverse(path);
	    return path;
	  }

	} 