/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.match;

import java.lang.reflect.Type;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation.Builder;

import com.entitystream.identiza.db.BaseNode;
import com.entitystream.identiza.db.BaseRelationship;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.WorkTypes;
import com.entitystream.identiza.entity.resolve.match.MatchRecordInterface;
import com.entitystream.identiza.entity.resolve.match.MatchRule;
import com.entitystream.identiza.entity.resolve.match.MatchRuleInterface;
import com.entitystream.identiza.entity.resolve.metadata.IIndex;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.ITableColumn;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumn;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap;
import com.entitystream.identiza.entity.resolve.metadata.SchemaMeta;
import com.entitystream.identiza.entity.resolve.types.MatchProcDefinition;
import com.entitystream.identiza.entity.resolve.types.MatchProcDefinitionInterface;
import com.entitystream.identiza.entity.resolve.types.MatchProcInterface;
import com.entitystream.monster.db.Document;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

/**
 * @author roberthaynes
 *
 */
public class Indexable implements Matchable{
    protected Map<String, List<MatchProcDefinition>> indexProcDefs = new HashMap<String, List<MatchProcDefinition>>();
    protected Map<String, Map<String, MatchProcInterface>> matchProcs = new HashMap<String, Map<String, MatchProcInterface>>();
    protected List<IRule> rules;
    protected ConcurrentHashMap<String, MatchRuleInterface> cacheRules = new ConcurrentHashMap<String, MatchRuleInterface> ();
    protected Map<String, java.util.Collection<String>> purposeTableColumns = new HashMap<String, java.util.Collection<String>>();
    protected ISchemaMeta schDoc;

    protected Map<String, Document> indexCatalogCache=new ConcurrentHashMap<String, Document>();
    protected LinkedBlockingQueue<Document> publishQueue = new LinkedBlockingQueue<Document>();
    public final List<IIndex> matchIndexes = new ArrayList<IIndex>();

    public com.entitystream.identiza.entity.resolve.match.MatchRecordInterface calculateScore(
	    Document base, Document comparitor,
	    boolean forSearch) throws Exception {
	MatchRecordInterface result=null;

	for (int i = 0; i < rules.size(); i++) {
	    IRule rule = rules.get(new Integer(i));
	    MatchRuleInterface mr = getMatchRule(rule.getHashKey(), rule.getOrder());

	    if (forSearch || mr.canRun(base, comparitor)) {
		MatchRecordInterface resultnew = mr.score(base, comparitor, forSearch, false, !forSearch);


		if (result==null || (resultnew!=null && resultnew.getScore()>result.getScore()))
		    result=resultnew;
		if (result.getScore() >= rule.getHighScore() && !forSearch)
		    break;
	    }
	}
	return result;
    }




    public static Document addXRef(Document survivedRec, Document deletedRec) throws Exception {
	if (survivedRec.get("_id")!= deletedRec.get("_id")){
	    List<Document> deletedXref = getXref(deletedRec);
	    List<Document> survivedXref = getXref(survivedRec);
	    survivedXref.add(deletedRec);
	    for (Document xd : deletedXref) {
		xd.remove("XREF");
		survivedXref.add(xd);
	    }
	    survivedRec.append("XREF", survivedXref);
	}
	return survivedRec;
    }

    public static List<Document> getXref(Document baseNode) throws Exception{
	List<Document> xref = (List<Document>) baseNode.get("XREF");
	if (xref==null)
	    xref = new ArrayList<Document>();

	return xref;
    }


    public static Document addHistoryRecord(Document existingDoc, Document newValues, String action, String username) throws Exception {
	if (existingDoc==null)
	    return newValues;

	Document newDoc = new Document();
	newDoc.putAll(newValues);
	Document currentDoc = new Document();
	currentDoc.putAll(existingDoc);
	currentDoc.remove("tokens");
	currentDoc.remove("standardised");
	//get current history - if any
	List histValues=(List) currentDoc.get("history");
	if (histValues==null)
	    histValues= new ArrayList<Document>();
	else
	    currentDoc.remove("history");
	currentDoc.append("action", action);
	currentDoc.append("updatedBy", username);
	histValues.add(0,currentDoc);
	//if the new doc has a history - we should merge them in too - for merges
	if (newDoc.containsKey("history")){
	    histValues.addAll((List)newDoc.get("history"));
	    newDoc.remove("history");
	}
	newDoc.append("history", histValues);
	newDoc.put("action", "CURRENT");
	newDoc.put("updatedBy", username);

	return newDoc;
    }

    public  Document calcGoldenRec(
	    Document original,
	    Document mergedrec, 
	    String tableName) {
	return Indexable.calcGoldenRec(original, mergedrec, tableName, schDoc);
    }

    public static Document calcGoldenRec(
	    Document original,
	    Document mergedrec, 
	    String tableName, ISchemaMeta schDoc) {

	Document res = new Document();
	try {
	    Date recLUD = original.getDate("lastUpdated");
	    // searchLUD is never null as its added just before this, except
	    // when called by the UI
	    if (!mergedrec.containsKey("lastUpdated"))
		mergedrec.put(
			"lastUpdated",
			schDoc.getTable(tableName).createDateFormatter().format(LocalDate.now()));
	    Date searchLUD = mergedrec.getDate("lastUpdated");
	    boolean hasLastUpd = recLUD != null;
	    // recLast tells us if the record was updated last
	    boolean recLast = false;

	    if (hasLastUpd)
		recLast = recLUD.after(searchLUD);

	    if (recLast)
		res.putAll(original);
	    else
		res.putAll(mergedrec);

	    for (ITableColumn col : schDoc.getColumns(tableName)) {
		// rec1 always survives, rec overrites
		Object exist = original.get(col.getColName());
		if (exist != null) {
		    res.put(col.getColName(), exist);
		    if (col.isPrimaryKey())
			continue;
		}
		String oldValue = Document.objectToString(original.get(col.getColName()));
		String newValue = Document.objectToString(mergedrec.get(col.getColName()));
		if (col.getAction().equals("latest") && newValue != null
			&& newValue.trim().length() > 0) {
		    // there is a last update date to compare
		    if (!recLast)
			res.put(col.getColName(), newValue);

		} else if (col.getAction().equals("overrite")
			&& newValue != null && newValue.trim().length() > 0) {
		    res.put(col.getColName(), newValue);
		} else if (col.getAction().equals("longest")
			&& newValue != null
			&& newValue.trim().length() > oldValue.length()) {
		    res.put(col.getColName(), newValue);
		} else if (col.getAction().equals("shortest")
			&& newValue != null
			&& newValue.trim().length() < oldValue.length()) {
		    res.put(col.getColName(), newValue);
		} else if (col.getAction().equals("sum")) {
		    // are both numbers?
		    try {
			double d1 = Double.parseDouble(oldValue);
			double d2 = Double.parseDouble(newValue);
			res.put(col.getColName(), "" + (d1 + d2));

		    } catch (Exception e) {
			if (newValue != null) {
			    if (oldValue == null || oldValue.isEmpty())
				res.put(col.getColName(), newValue);
			    else
				res.put(col.getColName(), oldValue.trim() + " "
					+ newValue);
			}
		    }

		} else if (col.getAction().equals("first")) {
		    // do nothing unless the oldvalue is null
		    if (oldValue == null && newValue != null
			    && newValue.trim().length() > 0)
			res.put(col.getColName(), newValue);
		}

	    }

	} catch (Exception e) {
	    e.printStackTrace();
	}

	return res;
    }


    protected MatchRuleInterface getMatchRule(String hashKey, int index) {		
	MatchRuleInterface rule  = cacheRules.get(hashKey);
	if (rule==null) {
	    rule = new MatchRule(schDoc, rules.get(index));
	    cacheRules.put(hashKey, rule);
	}
	return rule;
    }

    /**
     * 
     */
    public void initialize(ISchemaMeta iSchemaMeta) {
	this.schDoc=iSchemaMeta;
	rules = iSchemaMeta.getRules(null);
	matchProcs = new HashMap<String, Map<String, MatchProcInterface>> ();
	purposeTableColumns = new HashMap<String, java.util.Collection<String>>();
	indexProcDefs = new HashMap<String, List<MatchProcDefinition>>();
	matchIndexes.clear();
	for (IIndex ind : schDoc.getIndexes())
	    if(ind.isMatch()) {
		matchIndexes.add(ind);
		break;
	    }

    }



    protected boolean isPrefixPattern(Object input) {
	if (input instanceof Pattern) {
	    if (((Pattern)input).pattern().startsWith("^") || //startswith prefix
		    ((Pattern)input).pattern().startsWith("\\A") || //alternative for ^
		    ((Pattern)input).pattern().matches("^\\w.*")) //starts with word character
		return true;
	}
	return false;
    }

    protected boolean isPattern(Object input) {
	try {
	    if (input instanceof Pattern)
		return true;
	    if (input instanceof String) {
		String s = (String)input;
		if (Pattern.compile("\\/|\\^|\\?|\\*").matcher(s).find())
		    return true;

		return false;
	    } else return false;
	} catch (PatternSyntaxException e) {
	    return false;
	}
    }

    protected LinkedHashSet<Object> processKey(ArrayList<java.util.Collection> alternates, int startpos, Object keySoFar, String indexMethod){		
	LinkedHashSet<Object> allKeys = new LinkedHashSet<Object>();


	if (alternates.size()>0){
	    for (Object comb1 : alternates.get(startpos)){
		Object newKey="";
		boolean mustStop=false;
		if (indexMethod.equalsIgnoreCase("IndexLookup")) {
		    if (((String)keySoFar).length()>0)
			newKey = keySoFar+"|"+Document.objectToString(comb1);
		    else
			newKey = Document.objectToString(comb1);
		} else if (indexMethod.equalsIgnoreCase("IndexRangeScan")) {
		    //TODO - return full value, pattern prefix is last and pattern prefix becomes text
		    if (isPrefixPattern(comb1)) {
			String prefix = ((Pattern)comb1).pattern();
			String[] prefixElements = prefix.split("\\A|^|\\/|\\.|\\?|\\*");
			for (String pfe : prefixElements) {
			    if (!pfe.equalsIgnoreCase("\\A") && !pfe.equalsIgnoreCase("^")) {
				prefix=pfe;
				break;
			    }
			}

			comb1=prefix;
			mustStop=true;
		    }
		    if (((String)keySoFar).length()>0)
			newKey = keySoFar+"|"+Document.objectToString(comb1);
		    else
			newKey = Document.objectToString(comb1);



		} else if (indexMethod.equalsIgnoreCase("IndexScan")) {
		    //TODO - return full value including full pattern, pattern must stay as pattern, process key should convert pattern to include all values so far
		    if (isPattern(comb1)) {
			comb1 = Pattern.compile(keySoFar + "|" + ((Pattern)comb1).pattern());
			newKey = comb1;
			mustStop=true;
		    } else {
			if (((String)keySoFar).length()>0)
			    newKey = keySoFar+"|"+Document.objectToString(comb1);
			else
			    newKey = Document.objectToString(comb1);
		    }
		}
		//recurse??
		if (newKey instanceof String && ((String)newKey).length()==0) {

		} else {
		    if (!mustStop && startpos<alternates.size()-1)
			allKeys.addAll(processKey(alternates, startpos+1, newKey, indexMethod));
		    else {
			allKeys.add(newKey);
		    }
		}
		if (mustStop)
		    break;
	    }
	}



	return allKeys;
    }

    protected  java.util.Collection<String> getMatchKeyCombinations(
	    ArrayList<HashSet<String>> proctokens, int startpos, String keySoFar, int deep) {


	Set<String> allKeys = new HashSet<String>();
	if (deep>10){
	    return allKeys;
	}

	if (proctokens.size() > 0) {
	    for (String comb1 : proctokens.get(startpos)) {
		if (comb1 == null)
		    comb1 = "";
		String newKey = "";
		if (keySoFar.length() > 0)
		    newKey = keySoFar + comb1;
		else
		    newKey = comb1;
		if (startpos < proctokens.size() - 1)
		    allKeys.addAll(getMatchKeyCombinations(proctokens, startpos + 1,
			    newKey, deep+1));
		else
		    allKeys.add(newKey);

	    }
	}
	return allKeys;
    }


    protected List<MatchProcDefinition> getCacheDefs(String indexName) {
	if (!indexProcDefs.containsKey(indexName)) {
	    if (indexCatalogCache.get(indexName)!=null) {
		Document options = indexCatalogCache.get(indexName).getAsDocument("Options");
		String matchProcsJson = options.getString("matchProcs");
		Type listType = new TypeToken<List<MatchProcDefinition>>() {}.getType();
		List<MatchProcDefinition> defs = new Gson().fromJson(matchProcsJson, listType);
		indexProcDefs.put(indexName,defs);
	    } else return Collections.EMPTY_LIST;
	}
	return indexProcDefs.get(indexName);
    }



    protected java.util.Collection<String> getCachePurposeColumns(String purposeName, String tableName){
	String key=purposeName+":"+tableName;
	if (!purposeTableColumns.containsKey(key)) {
	    IPurpose purp = schDoc.getPurpose(purposeName);
	    if (purp!=null)
		purposeTableColumns.put(key, purp.getTableColumns(tableName));
	}
	return purposeTableColumns.get(key);
    }
    public java.util.Collection<String> getMatchKeys(String value, IIndex index) {
	//Multiple procs ie Company_Name and Address
	//each has multiple fields - mapped into a single value
	//each field could be a list or atomic value and BOTH ie Country Code 1* and Name 2*
	if (index== null)
	    return Collections.EMPTY_LIST;
	ArrayList<HashSet<String>> proctokens = new ArrayList<HashSet<String>>();

	List<MatchProcDefinition> defs = getCacheDefs(index.getIndexName());



	for (MatchProcDefinitionInterface procn : defs){
	    MatchProcInterface proc = getProc(index, procn);
	    if (proc!=null ){

		HashSet<String> keys = new HashSet<String>();

		java.util.Collection<String> keySet = proc.getKeys(value, true, true);
		if (keySet!=null && keySet.size()>0)
		    keys.addAll(keySet);

		proctokens.add(keys);


	    }
	}
	//get the product
	java.util.Collection<String> newkeys = new ArrayList<String>();
	newkeys=getMatchKeyCombinations(proctokens, 0, "",0);
	return newkeys;
    }



    public java.util.Collection<String> getMatchKeys(Document values, ITable table, IIndex index, boolean isSearch) {
	//Multiple procs ie Company_Name and Address
	//each has multiple fields - mapped into a single value
	//each field could be a list or atomic value and BOTH ie Country Code 1* and Name 2*
	if (table==null || index== null)
	    return Collections.EMPTY_LIST;
	ArrayList<HashSet<String>> proctokens = new ArrayList<HashSet<String>>();

	List<MatchProcDefinition> defs = getCacheDefs(index.getIndexName());



	for (MatchProcDefinitionInterface procn : defs){

	    MatchProcInterface proc = getProc(index, procn);

	    if (proc!=null ){
		java.util.Collection<String> cols = getCachePurposeColumns(procn.getPurposeName(), table.getTableName());

		if (cols!=null){
		    List<String> thisvals = values.getProperties(cols);
		    HashSet<String> keys = new HashSet<String>();
		    for (String fullval : thisvals){

			java.util.Collection<String> keySet = proc.getKeys(fullval, true, isSearch);
			if (keySet!=null && keySet.size()>0)
			    keys.addAll(keySet);

			proctokens.add(keys);
		    }
		}



	    }

	}
	//get the product
	java.util.Collection<String> newkeys = new ArrayList<String>();
	newkeys=getMatchKeyCombinations(proctokens, 0, "",0);
	return newkeys;
    }






    /**
     * @param procn
     * @return
     */
    private MatchProcInterface getProc(IIndex index, MatchProcDefinitionInterface procn) {
	if (matchProcs.get(index.getIndexName()) ==null) {
	    matchProcs.put(index.getIndexName(), new HashMap<String, MatchProcInterface>());
	}
	if (matchProcs.get(index.getIndexName()).get(procn.getProcName()) ==null) {
	    try {
		MatchProcInterface proc = procn.build();
		matchProcs.get(index.getIndexName()).put(procn.getProcName(), proc );
	    } catch (Exception e) {
		e.printStackTrace();
	    }
	}
	return matchProcs.get(index.getIndexName()).get(procn.getProcName());

    }




    protected Document applyInheritanceFrom(Document sourceRec, Document targetRec) {
	boolean hasChanged=false;
	for (IPurpose purpose : schDoc.getPurposes()){
	    int targetAlgo=purpose.getTargetAlgo();
	    if (targetAlgo != -1){

		Date tlu=targetRec.getDate("lastUpdated");
		Date lu=sourceRec.getDate("lastUpdated");

		for (PurposeColumn pc : purpose.getPurposeColumns())
		    //loop through the maps
		    for (PurposeColumnMap pcm : schDoc.getPurposeColumnMaps(purpose.getPurposeName() ,pc.getColumn(), sourceRec.getString("Table"))){
			//copy the value from the source field to the target 
			for (PurposeColumnMap pcmExt : schDoc.getPurposeColumnMaps(purpose.getPurposeName() ,pc.getColumn(), targetRec.getString("Table"))){
			    //copy the value from the source field to the target 
			    if (pcmExt.getColumnOrder()==pcm.getColumnOrder()){
				try {
				    Object targetValue=targetRec.get(pcmExt.getTableColumn());
				    Object currentValue=sourceRec.get(pcm.getTableColumn());
				    if ((lu==null || tlu==null || tlu.after(lu)) || targetAlgo==1){
					targetRec.append(pcm.getTableColumn(), targetValue);
					hasChanged=true;
				    }
				    else if (targetAlgo==2 &&
					    targetValue!=null &&
					    Document.objectToString(targetValue).length()>Document.objectToString(currentValue).length()){
					targetRec.append(pcm.getTableColumn(), targetValue);
					hasChanged=true;
				    }
				    else if (targetAlgo==0){
					targetRec.append(pcm.getTableColumn(), targetValue);
					hasChanged=true;
				    }


				} catch (Exception e) {
				    e.printStackTrace();
				}
			    }
			}
		    }
	    }

	}
	try{
	    if (hasChanged)
		targetRec.append("lastUpdated", new Date());
	} catch (Exception e){
	    e.printStackTrace();
	}
	return targetRec;
    }

    public class MatchItem  /*extends RecursiveAction*/ implements WorkTypes{

	private static final long serialVersionUID = 1L;

	protected Document baseNode = null;

	public int action;

	public MatchItem(Document mr, int type) {

	    action=type;
	    this.baseNode=mr;

	    //System.out.println("AutoMatching: " + mr.toJson() + " action: " + WorkTypes.toString(action));

	}

	//
	public void compute() {
	    if (action==WorkTypes.MATCH)
		match(baseNode);
	    else if (action==WorkTypes.SORT)
		mergeSort(baseNode);
	    else if (action==WorkTypes.TASK)
		addTask(baseNode);
	    else if (action==WorkTypes.LINK)
		link(baseNode);
	    else if (action==WorkTypes.EID)
		eid(baseNode);

	}



	public Document getMatchRecord() {
	    return baseNode;
	}

    }





    protected static javax.ws.rs.core.Response doREST(String url, String method, Entity payload, Document replacements, Document headers){
	Client client = ClientBuilder.newClient();

	javax.ws.rs.core.Response res =null;
	try{
	    if (replacements!=null)
		for (String ID : replacements.keyString()){
		    if (replacements.get(ID) instanceof String)
			url=url.replaceAll("\\{"+ID+"\\}", replacements.getString(ID));
		    else if (replacements.get(ID) instanceof Long )
			url=url.replaceAll("\\{"+ID+"\\}", ""+replacements.getLong(ID));
		    else if (replacements.get(ID) instanceof Integer )
			url=url.replaceAll("\\{"+ID+"\\}", ""+replacements.getInteger(ID));
		}
	    Logger.getGlobal().log(Level.INFO, url);
	    Builder call = client.target(url).request("application/json");
	    if (headers!=null)
		for (Object hk: headers.keySet())
		    call.header((String)hk, headers.get(hk));

	    if (method!=null && method.equalsIgnoreCase("GET"))
		res = call.get();
	    if (method!=null && method.equalsIgnoreCase("POST"))
		res = call.post(payload);
	    if (method!=null && method.equalsIgnoreCase("PUT"))
		res = call.put(payload);
	    if (method!=null && method.equalsIgnoreCase("DELETE"))
		res = call.delete(); 
	    if (res!=null)
		Logger.getLogger("MatchSchema").info(method+":"+url + "="+res.getStatus());
	    else
		Logger.getLogger("MatchSchema").info(method+":"+url + " returned null");
	} catch (Exception e){
	    Logger.getLogger("MatchSchema").info(method+":"+url + " returned error: " + e.getClass().getSimpleName());
	    Logger.getLogger("MatchSchema").log(Level.SEVERE, e.toString());
	    return null;
	}
	return res;
    }

    protected void send(Document mr) throws Exception {
	ITable table = schDoc.getTable(mr.getString("Table"));
	if (table.isInternal()){
	    new MatchItem(mr, WorkTypes.PUBLISH).compute();
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#match(com.entitystream.monster.db.Document)
     */
    @Override
    public boolean match(Document node) {
	// TODO Auto-generated method stub
	return false;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#mergeSort(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override
    public boolean mergeSort(Document mr) {
	// TODO Auto-generated method stub
	return false;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#eid(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override
    public boolean eid(Document mr) {
	// TODO Auto-generated method stub
	return false;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#link(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override
    public boolean link(Document mr) {
	// TODO Auto-generated method stub
	return false;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#merge(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override
    public boolean merge(Document mr) {
	// TODO Auto-generated method stub
	return false;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#addTask(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override
    public boolean addTask(Document mr) {
	// TODO Auto-generated method stub
	return false;
    }




    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#resolveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public void resolveTask(Document mr) {

    }




    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#invoke(java.lang.String, com.entitystream.monster.db.Document)
     */
    @Override
    public void invoke(String type, Document data) {
	// TODO Auto-generated method stub

    }

}
