/*******************************************************************************
 * Copyright notice
 * 
 * This source code is copyright of Robert James Haynes - (c) 2010, 2011. All rights reserved.
 * 
 * Any redistribution, reproduction or decompilation of part or all of the code in any form is prohibited 
 * 
 * You may not, except with our express written permission, distribute or commercially exploit the content. Nor may you transmit it or store it in or display it on any website or other form of electronic retrieval system.
 ******************************************************************************/
/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.match;

import java.io.Serializable;
import java.util.Date;

import com.entitystream.monster.db.Document;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.entitystream.identiza.db.IDBContainer;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.Node;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.processing.IdentizaException;
import com.entitystream.identiza.entity.resolve.storage.Record;
import com.entitystream.identiza.entity.resolve.storage.RecordInterface;
import com.entitystream.identiza.entity.resolve.storage.Result;
import com.entitystream.identiza.entity.resolve.storage.Task;

public class MatchRecord implements Serializable, MatchRecordInterface{
	private String record=null;
	private String baserecord=null;
	private String recordTableName;
	private String baserecordTableName;
	private String recordkey;
	private String baserecordkey;
	private double score;
	private IRule rule;
	private String action;
	private String actionText;
	private String baserecordsystem;
	private String recordsystem;
	private Date when;
	private Document recordDoc;
	private Document baseRecordDoc;
	private long baserecordEID;
	private long recordEID;
	private int type;
	private String taskID;

	public static MatchRecordInterface fromTask(MatchSchemaInterface schema, Task task) {
		try {
			Result[] nodes = task.getNodes();
			MatchRecord matchRecord = new MatchRecord();
			int count=0;
			if (nodes!=null)
			for (Result node: nodes) {
				String rowid = node.getID();
				RecordInterface r = schema.getMatchTable(node.getTable()).getRecord(rowid);
				if (count==0)
					matchRecord.setRecord(r);
				else 
					matchRecord.setBaseRecord(r);
				count++;
			}
			matchRecord.setAction(task.getAction());
			matchRecord.setActionText(task.getType());
			matchRecord.setScore(task.getScore());
			matchRecord.setWhen(task.getCreated());
			matchRecord.taskID=task.getId();
			return matchRecord;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;

	}


	public MatchRecord(RecordInterface record, RecordInterface baserecord, double score, IRule rule, String action, String actionText) {
		if (record!=null){
			if (record.getBaseNode()!=null){
				this.record = record.getBaseNode().getId();
				this.recordDoc=record.getBaseNode().getDocument();
			}
			if (record.getPkey()==null){
				System.out.println("Base node primary key null");
			}
			try {
				this.recordkey=record.getPkey();
				this.recordTableName=record.getTableName();
				this.setRecordsystem(record.getSystemName());
				String eids=record.getValue(record.getEIDField());
				if (eids!=null)
					this.setRecordEID(Long.parseLong(eids));

			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		if (baserecord!=null){
			if ( baserecord.getBaseNode()!=null){
				this.baseRecordDoc=baserecord.getBaseNode().getDocument();
				this.baserecord = baserecord.getBaseNode().getId();
			}
			else 
				this.baserecord=null;
			try{
				this.baserecordkey=baserecord.getPkey();
				this.baserecordTableName=baserecord.getTableName();
				this.setBaserecordsystem(baserecord.getSystemName());
				String eids=baserecord.getValue(baserecord.getEIDField());
				if (eids!=null)
					this.setBaseRecordEID(Long.parseLong(eids));

			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		this.score=Math.min(score,100);
		this.rule=rule;
		this.action=action;
		this.actionText=actionText;
		this.setWhen(new Date());
		this.type=type;
	}


	public MatchRecord(Document record, Document baserecord, double score, IRule rule, String action, String actionText, String key, String baseKey) {
		if (record!=null){
			if (record!=null){
				this.record = record.getString("_id");
				this.recordDoc=record;
			}

			try {
				this.recordkey=key;
				this.recordTableName=record.getString("Table");
				this.setRecordEID(record.getLong("EID"));

			} catch (Exception e) {
				e.printStackTrace();
			}
		}


		if (baserecord!=null){
			if (baserecord!=null){
				this.baserecord = baserecord.getString("_id");
				this.baseRecordDoc=baserecord;
			}

			try {
				this.baserecordkey=baseKey;
				this.baserecordTableName=baserecord.getString("Table");
				this.setBaseRecordEID(baserecord.getLong("EID"));

			} catch (Exception e) {
				e.printStackTrace();
			}
		}

		this.score=Math.max(score,0);
		this.rule=rule;
		this.action=action;
		this.actionText=actionText;
		this.setWhen(new Date());
	}


	private MatchRecord() {
		// TODO Auto-generated constructor stub
	}


	

	
	private void setRecordEID(long value) {
		this.recordEID=value;

	}


	private void setBaseRecordEID(long value) {
		this.baserecordEID=value;
	}


	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRecordID()
	 */
	@Override
	public String getRecordID() {
		return record;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaseNode()
	 */
	@Override
	public Node getBaseNode() throws Exception{
		return new Node(baseRecordDoc);
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getNode()
	 */
	@Override
	public INode getNode() throws Exception{
		return new Node(recordDoc);
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRecord(com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface)
	 */
	@Override
	public RecordInterface getRecord(MatchSchemaInterface schema) throws Exception {
		try {
			RecordInterface ret = RecordInterface.build(schema).assignNode(new Node(recordDoc));
			ret.load();
			return ret;
		} catch (IdentizaException e) {
			e.printStackTrace();
			return null;
		}
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setRecord(com.entitystream.identiza.entity.resolve.storage.RecordInterface)
	 */
	@Override
	public void setRecord(RecordInterface record) {
		this.record = record.getBaseNode().getId();
		this.recordkey=record.getPkey();
		this.recordTableName=record.getTableName();
		this.recordDoc=record.getBaseNode().getDocument();
		try {
			String eids=record.getValue(record.getEIDField());
			if (eids!=null)
				this.recordEID=Long.parseLong(eids);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaseRecordID()
	 */
	@Override
	public String getBaseRecordID() {
		return baserecord;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaseRecord(com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface)
	 */
	@Override
	public RecordInterface getBaseRecord(MatchSchemaInterface schema) throws Exception {
		try {
			RecordInterface ret = RecordInterface.build(schema).assignNode(new Node(baseRecordDoc));
			ret.load();
			return ret;
		} catch (IdentizaException e) {
			e.printStackTrace();
			return null;
		}
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setBaseRecord(com.entitystream.identiza.entity.resolve.storage.RecordInterface)
	 */
	@Override
	public void setBaseRecord(RecordInterface record) {
		this.baserecord = record.getBaseNode().getId();
		this.baserecordkey=record.getPkey();
		this.baserecordTableName=record.getTableName();
		this.baseRecordDoc=record.getBaseNode().getDocument();
		try {
			String eids=record.getValue(record.getEIDField());
			if (eids!=null)
				this.baserecordEID=Long.parseLong(eids);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getScore()
	 */
	@Override
	public double getScore() {
		return score;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setScore(double)
	 */
	@Override
	public void setScore(double score) {
		this.score = score;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRule()
	 */
	@Override
	public IRule getRule() {
		return rule;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getAction()
	 */
	@Override
	public String getAction(){
		return action;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getActionText()
	 */
	@Override
	public String getActionText(){
		if (actionText!=null && actionText.length()>0)
			return actionText;
		else
			return "Link";
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setActionText(java.lang.String)
	 */
	@Override
	public void setActionText(String actionText){
		this.actionText=actionText;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setAction(java.lang.String)
	 */
	@Override
	public void setAction(String action){
		this.action=action;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaseRecordTableName()
	 */
	@Override
	public String getBaseRecordTableName(){
		return baserecordTableName;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRecordTableName()
	 */
	@Override
	public String getRecordTableName(){
		return recordTableName;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaseRecordKey()
	 */
	@Override
	public String getBaseRecordKey(){
		return baserecordkey;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRecordKey()
	 */
	@Override
	public String getRecordKey(){
		return recordkey;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getAcceptance()
	 */
	@Override
	public String getAcceptance(){
		if (rule!=null && score>=rule.getLowScore()) {
			if (rule == null && score == 100.0)
				return "Auto-Match";
			else {
				if (score >= rule.getHighScore())
					return "Auto-Match";
				else if (score >= rule.getLowScore())
					return "User-Review";
			}
		}
		return "Reject";
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaserecordsystem()
	 */
	@Override
	public String getBaserecordsystem() {
		return baserecordsystem;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setBaserecordsystem(java.lang.String)
	 */
	@Override
	public void setBaserecordsystem(String baserecordsystem) {
		this.baserecordsystem = baserecordsystem;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRecordsystem()
	 */
	@Override
	public String getRecordsystem() {
		return recordsystem;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setRecordsystem(java.lang.String)
	 */
	@Override
	public void setRecordsystem(String recordsystem) {
		this.recordsystem = recordsystem;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getWhen()
	 */
	@Override
	public Date getWhen() {
		return when;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#setWhen(java.util.Date)
	 */
	@Override
	public void setWhen(Date when) {
		this.when = when;
	}


	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getBaseRecordEID()
	 */
	@Override
	public long getBaseRecordEID() {
		return baserecordEID;
	}


	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getRecordEID()
	 */
	@Override
	public long getRecordEID() {
		return recordEID;
	}


	@Override
	public JsonObject toJSON() {
		JsonParser parser = new JsonParser();
		JsonObject json = new JsonObject();
		json.addProperty("score", getScore());
		json.addProperty("acceptance", getAcceptance());
		json.addProperty("action", getAction());
		json.addProperty("actionText", getActionText());
		if (rule!=null)
	           json.add("rule", getRule().toJSON());
		JsonArray array=new JsonArray();
		array.add(parser.parse(recordDoc.toJson()));
		array.add(parser.parse(baseRecordDoc.toJson()));
		json.add("records", array);
		return json;

	}


	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#toDocument()
	 */
	@Override
	public Document toDocument() {
		Document json = new Document();
		json.append("score", getScore());
		json.append("acceptance", getAcceptance());
		json.append("action", getAction());
		json.append("actionText", getActionText());
		if (getRule()!=null)
		   json.append("rule", getRule().toJSON());
		return json;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getType()
	 */
	@Override
	public int getType() {
		return type;
	}


	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.match.MatchRecordInterface#getTaskId()
	 */
	@Override
	public String getTaskId() {
		
		return taskID;
	}
}
