/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.match;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import javax.ejb.Lock;
import javax.ejb.LockType;


import com.entitystream.identiza.db.IDBContainer;
import com.entitystream.identiza.entity.resolve.metadata.IIndex;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.RulePurpose;
import com.entitystream.identiza.entity.resolve.metadata.Table;
import com.entitystream.identiza.entity.resolve.storage.RecordInterface;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.monster.db.Document;

/**
 * @author roberthaynes
 *
 */
public class MatchSchemaSimpleImpl implements MatchSchemaInterface{



    private String nodeName = null;
    /**
     * 
     */
    private ArrayList<MatchIndexInterface> matchIndexes;
  
    private String projectName;
    private Logger logger = Logger.getLogger("com.entitystream.identiza");
    List<IRule> rules = new ArrayList<IRule>();
    HashMap<String, IPurpose> ps = new HashMap<String, IPurpose>();
    private ISchemaMeta schMeta;

    private IDBContainer db;
    private ConcurrentHashMap<String, MatchRule> cacheRules = new ConcurrentHashMap<String, MatchRule> ();
    private String groupName;
    private int threads;
    private int commitSize;




    public static MatchSchemaInterface build(ISchemaMeta meta) throws Exception{
	MatchSchemaSimpleImpl schema = new MatchSchemaSimpleImpl(meta);
	//schema.initialise();
	return schema;
    }


    private MatchSchemaSimpleImpl(ISchemaMeta meta)
	    throws Exception {
	String schemaName=meta.getName();
	this.schMeta=meta;
	this.projectName = schemaName;
	logger.fine("Starting " + schemaName);
	create();
    }

    public MatchSchemaSimpleImpl(String schemaName, String projectName, String groupName)
	    throws Exception {
	
	this.projectName = projectName;
	logger.fine("Starting " + schemaName + ":" + projectName);
	create();
    }

    private void create(){
	ISchemaMeta schDoc = getSchDoc();
	//tables = new HashMap<String, MatchTableInterface>();
	if (!projectName.equalsIgnoreCase("INTERNAL")) {
	    if (schDoc!=null) {
		logger.fine("Started Project : " + projectName);

		rules = getSchDoc().getRules(null);
		Collection<IPurpose> _ps = getSchDoc().getPurposes();
		for (IPurpose purpose : _ps){
		    ps.put(purpose.getPurposeName(), purpose);	
		}

		//for (Table tname : getSchDoc().getTables()){
		//	tableInternal.put(tname.getTableName(), tname);
		//}

		matchIndexes = new ArrayList<MatchIndexInterface>();
		Collection<IIndex> temp = getSchDoc().getIndexes();

		if (temp.size() > 0) {
		    java.util.Iterator i = temp.iterator();
		    while (i.hasNext()) {
			IIndex index = (IIndex) i.next();
			MatchIndex matI = new MatchIndex(index, this);
			if (matI.getDomainName()!=null){
			    //if (matI.getTableName().equals(tableName)) {

			    matchIndexes.add(matI);
			    logger.fine("Recalled index " + matI.getDomainName() + "/"
				    + matI.getInstance() 
				    + " with " + matI.getMatchProcs().size()
				    + " procs : " + matI.getMatchProcs().toString());

			    //}
			}
		    }
		}

	    } else {
		logger.severe("Not Started " + projectName);
	    }
	}



    }
    @Override

    public MatchRecordInterface calculateScore(
	    RecordInterface base, RecordInterface comparitor,
	    boolean forSearch,
	    boolean googleSearch, boolean asContent, boolean matchScoring) throws Exception {
	MatchRecordInterface result=null;
	for (int i = 0; i < rules.size(); i++) {
	    IRule rule = rules.get(new Integer(i));
	    MatchRuleInterface mr = getMatchRule(rule.getHashKey(), rule.getOrder());

	    if (forSearch || mr.canRun(base, comparitor)) {
		MatchRecordInterface resultnew = mr.score(base.getBaseNode().getDocument(), comparitor.getBaseNode().getDocument(), forSearch, asContent, matchScoring);
		if (result==null || resultnew.getScore()>result.getScore())
		    result=resultnew;
		if (result.getScore() >= rule.getHighScore() && !forSearch)
		    break;
	    }
	}
	return result;
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getTable(java.lang.String)
     */
    @Override
    public ITable getTable(String tableName) {
	// TODO Auto-generated method stub
	return schMeta.getTable(tableName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * com.entitystream.identiza.entity.resolve.match.MatchSchemaRemote#getMatchTable(java
     * .lang.String)
     */

    public MatchTableInterface getMatchTable(String name) throws Exception {
	if (name!=null)
	    name=name.trim();
	MatchTableInterface tab = new MatchTable();
	ITable mtable=getSchDoc().getTable(name);
	if (mtable!=null){
	    tab.initialise(mtable, this);
	    return tab;
	} else return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaRemote#getPurposes()
     */

    public Collection<IPurpose> getPurposes() {

	return getSchDoc().getPurposes();
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getPurpose(java.lang.String, java.lang.String)
     */
    @Override
    public IPurpose getPurpose(String purposeName, String groupName) {
	return getSchDoc().getPurpose(purposeName);
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getSchDoc()
     */
    @Override
    public ISchemaMeta getSchDoc() {
	return schMeta;
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getTableNames(java.lang.String)
     */
    @Override
    public List<String> getTableNames(String prefix) {
	List<String> list = new ArrayList<String>();
	for (ITable t : getSchDoc().getTables())
	    list.add(t.getTableName());
	return list;
    }




    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getProjectName()
     */
    @Override
    public String getProjectName() {

	return this.projectName;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getGroupName()
     */
    @Override
    public String getGroupName() {

	return this.groupName;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getThreads()
     */
    @Override
    public int getThreads() {
	// TODO Auto-generated method stub
	return 0;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getCommitSize()
     */
    @Override
    public int getCommitSize() {
	// TODO Auto-generated method stub
	return 0;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getThreads()
     */
    @Override
    public void setThreads(int t) {
	// TODO Auto-generated method stub
	this.threads=t;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getCommitSize()
     */
    @Override
    public void setCommitSize(int cs) {
	this.commitSize=cs;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#equals(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public boolean equals(String projectName2, String groupName2) {
	// TODO Auto-generated method stub
	return this.projectName.equalsIgnoreCase(projectName2) && this.groupName.equalsIgnoreCase(groupName2);
    }




    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getMatchIndexes()
     */
    @Override
    public ArrayList<MatchIndexInterface> getMatchIndexes() {
	return this.matchIndexes;
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#setSchDoc(com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta)
     */
    @Override
    public void setSchDoc(ISchemaMeta schDoc) {
	this.schMeta=schDoc;

    }

    public String updateRule(Document doc) {
	try {
	    RuleFactory.invalidate();
	    return this.getDb().updateRule(doc);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return "[]";
    }


    public String findRule(String rulePurpose, String query) {
	try {

	    return Document.stringify(this.getDb().findRule(rulePurpose, query));
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return "[]";
    }

    public List<Document> getMessages(Date since){
	try {
	    return getDb().getMessages(since);
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return new ArrayList<Document>();
    }


    @Lock(LockType.READ)

    public MatchRuleInterface getMatchRule(String hashKey, int index) {		
	MatchRule rule  = cacheRules.get(hashKey);
	if (rule==null) {
	    rule = new MatchRule(getSchDoc(), rules.get(index));
	    cacheRules.put(hashKey, rule);
	}
	return rule;
    }

    @Lock(LockType.READ)

    public void clearRules() {
	logger.info("Match Rules are cleared");
	cacheRules.clear();
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#addMessages(java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean)
     */
    @Override
    public void addMessages(String jobID, String action, String taskTitle,
	    String messageText, boolean status) {
	try {
	    Document d;
	    if (action.indexOf("{")>-1){
		d=Document.parse(action);
	    } else 
		d= new Document("action", action);
	    d.append("ROWID", jobID)
	    .append("status", status)
	    .append("lastUpdated", new Date())
	    .append("title", taskTitle)
	    .append("content", messageText);

	    getDb().addMessage(d);
	} catch (Exception e) {
	    e.printStackTrace();
	}
    }



    @Override public void setDb(IDBContainer db) {
	this.db=db;
    }


    public IDBContainer getDb() throws Exception {

	return db;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getNodeName()
     */
    @Override
    public String getNodeName() {
	return this.nodeName;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface#getNodeName()
     */
    @Override
    public void setNodeName(String nodeName) {
	this.nodeName=nodeName;
    }

    public RecordInterface standardise(RecordInterface RecordInterface) {

	for (int i = 0; i < rules.size(); i++) {
	    IRule rule = rules.get(new Integer(i));

	    if (rule == null) {
		logger.severe("Rule "
			+ i						
			+ " is not valid or the rules are not contiguous for this table and starting at 1");
		continue;
	    }


	    if (rule.isActive() ) {
		ArrayList<RulePurpose> rulePurposes = rule.getRulePurpose();

		for (RulePurpose rp : rulePurposes) {
		    String purposeName=rp.getPurposeName();
		    try {
			RecordInterface=getSchDoc().getPurpose(purposeName).standardize(RecordInterface);
		    } catch (Exception e) {
			e.printStackTrace();
		    }
		}
	    }
	}
	return RecordInterface;

    }



}
