/*******************************************************************************
 * Copyright notice
 * 
 * This source code is copyright of Robert James Haynes - © 2010,2011. All rights reserved.
 * 
 * Any redistribution, reproduction or decompilation of part or all of the code in any form is prohibited 
 * 
 * You may not, except with our express written permission, distribute or commercially exploit the content. Nor may you transmit it or store it in or display it on any website or other form of electronic retrieval system.
 ******************************************************************************/
package com.entitystream.identiza.entity.resolve.match;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


import java.util.logging.Logger;


import com.entitystream.monster.db.Document;
import com.entitystream.identiza.db.IDBContainer;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.IRelationship;
import com.entitystream.identiza.db.Node;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.Relationship;
import com.entitystream.identiza.db.WorkTypes;
import com.entitystream.identiza.entity.resolve.metadata.Purpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumn;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.metadata.RulePurpose;
import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.ITableColumn;
import com.entitystream.identiza.entity.resolve.metadata.TableColumn;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.TableColumn;
import com.entitystream.identiza.entity.resolve.processing.Candidate;
import com.entitystream.identiza.entity.resolve.storage.ContextMap;
import com.entitystream.identiza.entity.resolve.storage.Record;
import com.entitystream.identiza.entity.resolve.storage.RecordInterface;
import com.entitystream.identiza.entity.resolve.storage.Result;
import com.entitystream.identiza.entity.resolve.storage.Task;
import com.entitystream.identiza.userexits.MergeObject;
import com.entitystream.identiza.userexits.PreTaskObject;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.identiza.wordlist.WordObject;


public class MatchTable implements MatchTableInterface, Serializable {
	/**
	 * 
	 */

	private static final long serialVersionUID = -1033487478846380426L;
	private String tableName;
	private List<IRule> rules = null;
	private ConcurrentHashMap<String, Object> defaultVals = null;
	private static Logger logger = Logger.getLogger("com.entitystream.identiza");
	private MatchSchemaInterface matchSchema;
	private ITable table;
	public String REF_RELATED_BY_VERB = null;
	private boolean isHistory = false;
	private String keyField;
	private List<String> labelFields;
	private List<String> tags;
	private transient ExecutorService threadPool;
	private ObjectOutputStream matchStream;
	private int matchFileSize;
	private ObjectOutputStream putStream;
	private int putFileSize;
	private HashSet<String> sequenceCols;
	private String iconField;
	private String EIDField;
	private String systemField;
	private String sensitivityField;
	private boolean isInternal;
	private String lastUpdatedCol;
	private Collection<MatchTableInterface> targetTables;

	
	@Override
	public boolean isHistory() {
		return isHistory;
	}

	public MatchTable() {

	}

	
	@Override
	public void initialise(ITable table, MatchSchemaInterface msc)
			throws Exception {
		//logger.info("Starting table " + tablename);
		this.matchSchema = msc;
		this.tableName = table.getTableName();
		this.table = table;
		if (table == null)
			return;
		this.isHistory = table.isHistory();

		this.tags = table.getTags();
		this.isInternal=table.isInternal();
		this.lastUpdatedCol=table.getLastUpdatedCol();

		defaultVals = new ConcurrentHashMap<String, Object>();
		for (ITableColumn col : table.getColumns()) {
			if (col.getDefaultValue() != null
					&& !col.getDefaultValue().equalsIgnoreCase("NULL"))
				defaultVals.put(col.getColName(), col.getDefaultValue());
		}

		sequenceCols = new HashSet<String>();
		for (ITableColumn col : table.getColumns()) {
			if (col.isSequence() || col.getColName().equalsIgnoreCase(getEIDField()))
				sequenceCols.add(col.getColName());
		}

		rules=msc.getSchDoc().getRules(tableName);
	}

	
	@Override
	public void initialise(String tablename, MatchSchemaInterface msc)
			throws Exception {
		//logger.info("Starting table " + tablename);
		this.matchSchema = msc;
		this.tableName = tablename;
		this.table = msc.getTable(tablename);
		if (table == null)
			return;
		this.isHistory = table.isHistory();

		this.tags = table.getTags();
		this.isInternal=table.isInternal();
		this.lastUpdatedCol=table.getLastUpdatedCol();


		defaultVals = new ConcurrentHashMap<String, Object>();
		for (ITableColumn col : table.getColumns()) {
			if (col.getDefaultValue() != null
					&& !col.getDefaultValue().equalsIgnoreCase("NULL"))
				defaultVals.put(col.getColName(), col.getDefaultValue());
		}

		sequenceCols = new HashSet<String>();
		for (ITableColumn col : table.getColumns()) {
			if (col.isSequence() || col.getColName().equalsIgnoreCase(getEIDField()))
				sequenceCols.add(col.getColName());
		}


	}

	
	@Override
	public String getTableName() {
		return tableName;
	}

	
	@Override
	public void setTableName(String tableName) {
		this.tableName = tableName;
	}

	
	@Override
	public Collection<MatchTableInterface> getTargetTables() throws Exception{
		targetTables=null;
		if (targetTables == null) {
			targetTables = new HashSet<MatchTableInterface>();
			if (!this.getMetaTable().isTarget()) {
				for (String tableName : matchSchema.getTableNames("")) {
					if (!tableName.equalsIgnoreCase(this.tableName)) {
						MatchTableInterface mt = matchSchema.getMatchTable(tableName);
						ITable t = matchSchema.getTable(tableName);
						if (t.isTarget()) {
							//are there any shared purposes?
							for (IPurpose pt : this.getPurposesForTable())
								for (IPurpose pot : mt.getPurposesForTable())
									if (pt.getPurposeName().equalsIgnoreCase(pot.getPurposeName())){
										//they share a purpose - but the ourposes has to cover A label field ie FirstName, Company Name etc
										for (PurposeColumn potc : pot.getPurposeColumns()) {
											for (PurposeColumnMap potcm : matchSchema.getSchDoc().getPurposeColumnMaps(potc)){
												//does this cover a label field?
												if (potcm.getTableName().equalsIgnoreCase(tableName))
												   if (t.getColumn(potcm.getTableColumn()).getLabelPos()>0) {
													  targetTables.add(mt);
													  break;
												   }


											}
										}

									}
						}
					}
				}
			}
		}	
		return targetTables;
	}


	
	@Override
	public RecordInterface putRecord(RecordInterface rec, boolean keyChanged,
			boolean updateMergeCount, String username) throws Exception {


		rec=rec.save(keyChanged, username);
		//if this table is not a target table, and we have a target table in play - we should create a target record

		for (MatchTableInterface mt : getTargetTables()) {
			RecordInterface newRecord = mt.newRecord(rec.getPkey());
			newRecord.setValue("Table", mt.getTableName());
			newRecord.setValue(mt.getKeyField(), rec.getPkey());
			newRecord.setValue(newRecord.getEIDField(), rec.getValues().get(rec.getEIDField()));
			newRecord.applyInheritanceFrom(rec);
			//will the target have a label?
			String label = newRecord.display();
			if (label!=null && label.length()>0)
				newRecord=newRecord.save(true, username);

		}


		return rec;
	}

	
	@Override
	public RecordInterface getRecord(String id) {
		try {
			INode rec = new Node(matchSchema.getDb(), id);
			String recTable=(String) rec.getProperty("Table");
			ITable _table=table;
			if (recTable!=null && !recTable.equalsIgnoreCase(tableName))
				_table=matchSchema.getTable(recTable);

			if (rec.hasProperty(_table.getKeyField())) {
				String pkey = (String) rec.getProperty(_table.getKeyField());
				RecordInterface res = Record.build(matchSchema).assignNode(new Node(getDB(), id));
				if (res.isValid())
					return res;
				else
					return null;
			}
		} catch (Exception e) {
			logger.severe(e.getMessage());
			e.printStackTrace();
		}
		return null;
	}

	
	@Override
	public RecordInterface getRecord(INode node) throws Exception {
		RecordInterface rec = Record.build(matchSchema).assignNode(node);
		rec.load();
		return rec;
	}

	
	@Override
	public Record getRecord(String table, String pkey) {
		try {
			
			if (pkey.indexOf(":")>1){
				table=pkey.substring(0,pkey.lastIndexOf(":"));
				pkey=pkey.substring(pkey.lastIndexOf(":")+1);
			}
			if (table==null)
			    table=this.tableName;

			Record res = new Record(table, pkey, matchSchema);
			res.load();
			if (res.isValid())
				return res;
			else
				return null;
		} catch (Exception e) {
			logger.severe(e.getMessage());
			//e.printStackTrace();
		}
		return null;
	}


	
		
	
	@Override
	public String getKeyField() {
		if (table!=null){
			if (keyField == null)
				keyField = table.getKeyField();
			if (keyField == null && table.getColumns().size()>0) {
				logger.severe("TABLE (" + table.getTableName()
				+ ") MUST HAVE AT LEAST ONE PRIMARY KEY!");				
				return table.getColumns().get(0).getColName();
			}
		}
		return keyField;
	}

	
	@Override
	public String getSystemField() {
		if (systemField == null && table!=null)
			systemField = table.getSystemField();
		return systemField;
	}

	
	@Override
	public String getSensitivityField() {
		if (sensitivityField == null && table!=null)
			sensitivityField = table.getSensitivityField();
		return sensitivityField;
	}




	
	@Override
	public List<IPurpose> getPurposesForTable() {
		return matchSchema.getSchDoc().getPurposes(this.getTableName());
	}



	



	@Override
	public IDBContainer getDB() throws Exception {
		if (matchSchema!=null)
			return matchSchema.getDb();
		else
			return null;
	}

	
	@Override
	public ArrayList<String> getFields() {
		HashMap<String, ITableColumn> cols = new HashMap<String, ITableColumn>();
		List<TableColumn> fqr = matchSchema.getSchDoc().getColumns(
				tableName);
		// populate values?
		if (fqr.size() > 0) {
			java.util.Iterator<TableColumn> i = fqr.iterator();
			while (i.hasNext()) {
				ITableColumn def = ((TableColumn) i.next());
				cols.put(def.getColName(), def);
			}
		}

		ArrayList<String> list = new ArrayList<String>(cols.keySet());
		Collections.sort(list);
		return list;
	}

	
	
	
	@Override
	public List<String> getLabelFields() {
		if (labelFields==null)
			labelFields=table.findLabelFields();
		return labelFields;
	}

	
	@Override
	public Record newRecord(String pkey) throws Exception {
		return new Record(tableName, pkey, matchSchema);
	}





	
	

	
	@Override
	public boolean hasRecord(String pkey) throws Exception {
		return matchSchema.getDb()
				.getSingleNode(tableName, getKeyField(), pkey).getString("_id") != null;
	}

	
	



	


	
	@Override
	public ConcurrentHashMap<String, Object> getDefaults() {
		if(defaultVals==null && table!=null)
			defaultVals=table.createDefaults();
		return defaultVals;
	}

	
	@Override
	public ITable getMetaTable() {
		// TODO Auto-generated method stub
		return table;
	}

	
	@Override
	public List<IRule> getRules() {
		return rules;
	}

	
	@Override
	public void setRules(ArrayList<IRule> rules) {
		this.rules = rules;
	}

	
	@Override
	public ArrayList<Result> getMergedRecords() throws Exception {
		HashMap<String, String> matchC = new HashMap<String, String>();
		matchC.put("XREF", "*");
		Iterator<Document> it = getDB().search(matchC, tableName).iterator();
		ArrayList<Result> ret = new ArrayList<Result>();
		try {
			while (it.hasNext()) {
				INode node = new Node(getDB(), it.next());
				Result result = new Result(Record.build(matchSchema).assignNode(node));
				if (result.getPkey() != null)
					ret.add(result);
			}
		} catch (Exception oe) {
			oe.printStackTrace();
		}
		return ret;
	}


	
	@Override
	public List<String> getTags() {
		return tags;
	}

	private ExecutorService getThreadPool() {
		if (threadPool == null)
			threadPool = Executors.newFixedThreadPool(matchSchema.getThreads());
		return threadPool;
	}

	@Override
	public Collection<String> getSequenceColumns() {
		return sequenceCols;

	}

	@Override
	public List<TableColumn> getColumns() {
		if (table!=null)
			return table.getColumns();
		else return new ArrayList<TableColumn>();
	}

	@Override
	public String getIconField() {	
		if (iconField != null) {
			for (ITableColumn col : table.getColumns())
				if (col.isIcon()) {
					iconField = col.getColName();
					break;
				}
		}
		return iconField;
	}

	@Override
	public String getEIDField() {
		EIDField="EID";

		return EIDField;
	}
	@Override
	public String getDefaultIcon() {
		if (table!=null)
			return table.getIcon();
		return "Document.png";
	}


	
	@Override
	public ArrayList<ITableColumn> getIndexColumns() {

		ArrayList<ITableColumn> ret = new ArrayList<ITableColumn>();
		if (matchSchema!=null)
			for (MatchIndexInterface index : matchSchema.getMatchIndexes()){
				for (String col: index.getMappedColumnNames(tableName)){
					TableColumn tc = (TableColumn) table.getColumn(col);
					if (tc != null && !ret.contains(tc))
						ret.add(tc);
				}
			}
		return ret;
	}

	
	

	
	

	@Override
	public boolean isInternal() {
		return isInternal;
	}

	@Override
	public String getLastUpdatedCol() {
		return lastUpdatedCol;
	}

}
