/*******************************************************************************
 * Copyright notice
 * 
 * This source code is copyright of Robert James Haynes - (c) 2010, 2011. All rights reserved.
 * 
 * Any redistribution, reproduction or decompilation of part or all of the code in any form is prohibited 
 * 
 * You may not, except with our express written permission, distribute or commercially exploit the content. Nor may you transmit it or store it in or display it on any website or other form of electronic retrieval system.
 ******************************************************************************/
/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.metadata;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Logger;

import com.entitystream.monster.db.Document;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.entity.resolve.storage.Record;
import com.entitystream.identiza.entity.resolve.storage.RecordInterface;
import com.entitystream.identiza.entity.resolve.types.MatchProcInterface;
import com.entitystream.identiza.entity.resolve.types.Standardized;
import com.entitystream.identiza.metadata.IdentizaSettings;

public class Purpose implements Serializable, IPurpose {
	private String purposeName;
	//Logger.getLogger("com.identiza") Logger.getLogger("com.identiza") = Logger.getLogger("com.identiza").getLogger.getLogger("com.identiza")("com.identiza")Logger.getLogger("com.identiza").getLogger.getLogger("com.identiza")("com.identiza");
	private List<PurposeColumn> purposeColumns = new ArrayList<PurposeColumn>();
	private int targetAlgo;
	private String purposeType;
	private String schemaName;
	
	private transient ISchemaMeta schDoc;

	
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getSchDoc()
	 */
	@Override
	public ISchemaMeta getSchDoc() {
		
		return schDoc;
	}
	
	public Purpose(){

	}

	public Purpose(String schemaName, String purposeName, List<PurposeColumn> purposeColumns) {
		this.purposeName=purposeName;
		this.schemaName=schemaName;
		this.purposeColumns=purposeColumns;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#calculateScore(com.entitystream.identiza.entity.resolve.storage.RecordInterface, com.entitystream.identiza.entity.resolve.storage.RecordInterface, boolean, boolean, boolean)
	 */
	@Override
	public double calculateScore(RecordInterface basein, RecordInterface comparitorin,			
			boolean forSearch, boolean asContent, boolean matchScoring) throws Exception {

		double bestscore = -1;
		double weightsum = 0;

		if (purposeColumns.size() > 0) {
			double totalscore=0.0d;
			int numcols=0;
			for (PurposeColumn pc : purposeColumns){
				List<PurposeColumnMap> map1 = getSchDoc().getPurposeColumnMaps(pc.getPurposeName(), pc.getColumn(), basein.getTableName());
				List<PurposeColumnMap> map2 = getSchDoc().getPurposeColumnMaps(pc.getPurposeName(), pc.getColumn(), comparitorin.getTableName() );
				// we want all the maps that relate to the purpose name and table- there
				// might be multiple columns for th
				// the same purpose ie Address_billing and Address_home for
				// purpose Location

				// for each column name - standardize
				if (map1!=null && map2!=null && map1.size()>0 && map2.size()>0){
					MatchProcInterface proc = pc.createMatchProc();
					List<Standardized> baseStdList = basein.getStandardised(
							pc.getColumn(), proc, map1, pc.popRuleSet().getRuleAnon());
					List<Standardized> compStdList = comparitorin.getStandardised(
							pc.getColumn(), proc, map2, pc.popRuleSet().getRuleAnon());
					for (Standardized baseStd : baseStdList) {
						for (Standardized compStd : compStdList) {
							if (baseStd.getComparitorWords().length>0 && compStd.getComparitorWords().length>0){
								double score = proc.calculateComparisonScore(baseStd,
										compStd, forSearch, asContent);
								if (score > bestscore) {
									bestscore = score;
								}
							}
						}
					}

				}
				totalscore+=bestscore;

				numcols++;

			}
			if (numcols>0)
				bestscore=totalscore/numcols;
			else
				bestscore=0.0d;
			Logger.getLogger("com.identiza").fine(purposeName + " score: " + bestscore);
		} else Logger.getLogger("com.identiza").fine(purposeName + " was ignored");
		return bestscore;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#standardize(com.entitystream.identiza.entity.resolve.storage.RecordInterface)
	 */
	@Override
	public RecordInterface standardize(RecordInterface basein) throws Exception {
		//if (this.getPurposeType().equalsIgnoreCase("None")) {
			Document std = (Document) basein.getBaseNode().getDocument().get("standardized");
			if (std==null) {
				std=new Document();
				basein.getBaseNode().getDocument().append("standardized",std);
			}
			if (purposeColumns.size() > 0) {
				for (PurposeColumn pc : purposeColumns){

					List<PurposeColumnMap> map1 = getSchDoc().getPurposeColumnMaps(pc.getPurposeName(), pc.getColumn(), basein.getTableName());

					if (map1!=null  && map1.size()>0){
						MatchProcInterface proc = pc.createMatchProc();
						List<Standardized> baseStdList = basein.getStandardised(
								pc.getColumn(), proc, map1, pc.popRuleSet().getRuleAnon());

						if (baseStdList.size()>0)
							std.append(pc.getColumn(),INode.serialize(baseStdList));
					}


				}
				//basein.getBaseNode().setProperty("standardized",std);
			} else Logger.getLogger("com.identiza").fine(purposeName + " was ignored");
		//}
		return basein;

	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getPurposeName()
	 */
	@Override
	public String getPurposeName() {
		return purposeName;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#setPurposeName(java.lang.String)
	 */
	@Override
	public void setPurposeName(String purposeName) {
		this.purposeName = purposeName;
	}




	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#addPurposeColumn(com.entitystream.identiza.entity.resolve.metadata.PurposeColumn)
	 */
	@Override
	public void addPurposeColumn(PurposeColumn pc){
		int pos=-1;
		for (int p=0; p<purposeColumns.size(); p++){
			PurposeColumn _pc = purposeColumns.get(p);
			if (_pc.getColumn().equalsIgnoreCase(pc.getColumn()) && _pc.getPurposeName().equalsIgnoreCase(pc.getPurposeName())){
				pos=p;
				break;
			}
		}
		if (pos==-1)
			purposeColumns.add(pc);
		else 
			purposeColumns.set(pos, pc);

	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getPurposeColumns()
	 */
	@Override
	public List<PurposeColumn> getPurposeColumns() {
		return purposeColumns;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#setPurposeColumns(java.util.List)
	 */
	@Override
	public void setPurposeColumns(List<PurposeColumn> pc) {
		purposeColumns=pc;
	}



	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getTargetAlgo()
	 */
	@Override
	public int getTargetAlgo(){
		return targetAlgo;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#setTargetAlgo(int)
	 */
	@Override
	public void setTargetAlgo(int algo){
		targetAlgo=algo;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getPurposeType()
	 */
	@Override
	public String getPurposeType(){
		if (purposeType==null || purposeType.length()==0)
			purposeType="None";
		return purposeType;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#setPurposeType(java.lang.String)
	 */
	@Override
	public void setPurposeType(String purposeType){
		if (purposeType==null || purposeType.length()==0)
			purposeType="None";
		this.purposeType=purposeType;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#convertTargetAlgo(java.lang.String)
	 */
	@Override
	public void convertTargetAlgo(String algo){
		if (algo==null || algo.equalsIgnoreCase("dont"))
			targetAlgo=-1;
		else if (algo.equalsIgnoreCase("latest"))
			targetAlgo=0;
		else if (algo.equalsIgnoreCase("any"))
			targetAlgo=1;
		else if (algo.equalsIgnoreCase("longest"))
			targetAlgo=2;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getTableColumns(java.lang.String)
	 */
	@Override
	public Collection<String> getTableColumns(String tableName) {
		Set<String> ret = new HashSet<String>();
		for (PurposeColumnMap pcm : getSchDoc().getPurposeColumnMaps(this.purposeName, null, tableName))
			ret.add(pcm.getTableColumn());
		return ret;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#setSchemaName(java.lang.String)
	 */
	@Override
	public void setSchemaName(String schemaName) {
		this.schemaName=schemaName;
	}
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#getSchemaName()
	 */
	@Override
	public String getSchemaName() {
		return this.schemaName;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#deletePurposeColumn(java.lang.String)
	 */
	@Override
	public void deletePurposeColumn(String purposeColumnName) {
		List<PurposeColumn> npc = new ArrayList<PurposeColumn>();
		for (PurposeColumn pc : purposeColumns) {
			if (!pc.getColumn().equalsIgnoreCase(purposeColumnName))
				npc.add(pc);
		}
		purposeColumns=npc;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.metadata.IPurpose#setSchDoc(com.entitystream.identiza.entity.resolve.metadata.SchemaMeta)
	 */
	@Override
	public void setSchDoc(SchemaMeta schemaMeta) {
		schDoc=schemaMeta;
		
	}
}
