/**
 *
	Identiza - Fuzzy matching Libraries

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.metadata;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.apache.commons.collections.CollectionUtils;

import com.entitystream.monster.db.Document;
import com.google.gson.Gson;
import com.entitystream.identiza.entity.resolve.types.MatchProcDefinition;
import com.entitystream.identiza.entity.resolve.types.MatchProcDefinitionInterface;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.identiza.wordlist.RuleSetMap;

public class SchemaMeta implements Serializable, ISchemaMeta{

    int commitSize=10000;
    Logger logger = Logger.getLogger("ISchemaMeta");

    private String DBStore;
    private String defaultDBAddress;
    private int cacheSize;
    private Integer threads;
    private String projectName;
    private boolean autoMatch=false;


    private Map<String, ITable> tables=new HashMap<String, ITable>();
    private Map<String,List<PurposeColumnMap>> pcms = new HashMap<String,List<PurposeColumnMap>>();
    Map<String, IPurpose> purposes = new HashMap<String, IPurpose>();
    Map<String, IIndex> indexes = new HashMap<String,IIndex>();
    List<IRule> rules = new ArrayList<IRule>();


    //used externallly
    public static SchemaMeta createSchemaMeta(Document connection){
	try{
	    SchemaMeta sch=new SchemaMeta(connection);
	    if(connection.containsKey("projectName"))
		sch.projectName=connection.getString("ProjectName");
	    if(connection.containsKey("dbStore"))
		sch.DBStore=connection.getString("dbStore");
	    if(connection.containsKey("commitSize"))
		sch.commitSize=connection.getInteger("commitSize");
	    if(connection.containsKey("cacheSize"))
		sch.cacheSize=connection.getInteger("cacheSize");
	    if(connection.containsKey("threads"))
		sch.threads=connection.getInteger("threads");
	    if(connection.containsKey("defaultDBAddress"))
		sch.defaultDBAddress=connection.getString("defaultDBAddress");
	    if(connection.containsKey("autoMatch"))
		sch.autoMatch=connection.getBoolean("autoMatch", false);
	    sch.getTables();
	    sch.getRules(null);
	    sch.getPurposes();
	    sch.getIndexes();
	    return sch;
	} catch(Exception e){
	    e.printStackTrace();
	}
	return null;
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#toDocument()
     */
    @Override
    public Document toDocument(){
	Document doc = new Document();
	doc.append("ProjectName",projectName);
	doc.append("commitSize",this.commitSize);
	doc.append("dbStore",this.DBStore);
	doc.append("threads",this.threads);
	doc.append("cacheSize",this.cacheSize);
	doc.append("defaultDBAddress",this.defaultDBAddress);
	doc.append("Rules", getRules(null));	
	doc.append("Purposes",getPurposesDocument());
	doc.append("Indexes", getIndexes());
	doc.append("Tables", getTablesDocument());
	doc.append("autoMatch", isAutoMatch());
	return doc;
    }


    /**
     * @return
     */
    @Override
    public boolean isAutoMatch() {
	// TODO Auto-generated method stub
	return autoMatch;
    }
    @Override
    public void setAutoMatch(boolean autoM) {
	this.autoMatch=autoM;
    }

    public List<Document> getTablesDocument() {
	List<Document> ret = new ArrayList<Document>();
	for (ITable t : getTables()) {
	    ret.add(t.toDocument());
	}
	return ret;
    }
    
  public List<Document> getPurposesDocument() {
	Gson gson = new Gson();
	ArrayList<Document> ret= new ArrayList<Document>();
	for (IPurpose p: purposes.values()) {
	    Document pdoc = Document.parse(gson.toJson(p, Purpose.class));
	    pdoc.remove("purposeColumns");//add them manually
	    ArrayList<Document> myCols = new ArrayList<Document>();
	    for (PurposeColumn pc : p.getPurposeColumns()) {
		  Document pcdoc = Document.parse(gson.toJson(pc, PurposeColumn.class));
		  ArrayList<Document> myMaps = new ArrayList<Document>();
		  pcdoc.append("purposeColumnMaps", myMaps);
		  //find the pc and add the pcms.
		  for (List<PurposeColumnMap> pcml : pcms.values()) {
		    
		      for (PurposeColumnMap pcm : pcml) {
			    if (pcm.getPurposeColumn()!=null && pcm.getPurposeName()!=null && 
				   pcm.getPurposeName().equalsIgnoreCase(p.getPurposeName()) && pcm.getPurposeColumn().equalsIgnoreCase(pc.getColumn())) {
			       myMaps.add(Document.parse(gson.toJson(pcm, PurposeColumnMap.class)));

			    }
		      }
		   
		  }
		  myCols.add(pcdoc);
	    }
	    pdoc.append("purposeColumns", myCols);
	    ret.add(pdoc);
	}
	
	return ret;
  }


  private SchemaMeta(Document connection) throws Exception{


	commitSize=connection.getInteger("commitSize",1000);
	threads=connection.getInteger("threads",1);
	cacheSize=connection.getInteger("cacheSize",10000);
	projectName=connection.getString("SchemaName");
	autoMatch=connection.getBoolean("autoMatch", false);
	if (connection.get("Tables") !=null && connection.get("Tables") instanceof List) {
	    List<Document> tabs = (List<Document>) connection.get("Tables");
	    if (tabs!=null)
		for (Document d : tabs){
		    Table table = (Table) Document.toObject(d, Table.class);
		    table.init();
		    tables.put(table.getTableName(), table);
		    this.tables.put(table.getTableName(), table);
		}
	}
	if (connection.get("Rules")!=null && connection.get("Rules") instanceof List){

	    List rules = (List) connection.get("Rules");
	    for (Object o : rules){
		Rule rule = null;
		if (o instanceof Rule)
		    rule = (Rule)o;
		else 
		    rule = (Rule) Document.toObject((Document)o, Rule.class);
		this.rules.add(rule);
	    }
	}



	if (connection.get("Purposes")!=null && connection.get("Purposes") instanceof List){

	    List<Document> purps = (List<Document>) connection.get("Purposes");
	    for (Document p: purps){
		Purpose purp = (Purpose) Document.toObject(p, Purpose.class);
		purp.setSchemaName(projectName);
		this.purposes.put(purp.getPurposeName(), purp);
	    }
	}



	if (connection.get("Indexes")!=null && connection.get("Indexes") instanceof List){
	    List indexes = (List) connection.get("Indexes");
	    for (Object o: indexes){
		if (o instanceof Document) {
		    Document i = (Document)o;
		    if (i.containsKey("indexName")){
			Index index = (Index) Document.toObject(i, Index.class);
			this.indexes.put(index.getIndexName(), index);
		    }
		} else {
		    Index index = (Index)o;
		    this.indexes.put(index.getIndexName(), index);
		}

	    }
	}

	if (connection.containsKey("Purposes") && connection.get("Purposes") instanceof List){
	    //iterate the purposeColumns to pick out the matchClass
	    for (Document purpose : (List<Document>) connection.get("Purposes")){
		for (Document purposeColumn : (List<Document>) purpose.get("purposeColumns")){
		    String matchClass=purposeColumn.getString("matchClass");
		    if (purposeColumn.get("purposeColumnMaps")!=null)
			for (Document purposeColumnMap : (List<Document>) purposeColumn.get("purposeColumnMaps")){
			    PurposeColumnMap pcm = PurposeColumnMap.fromDocument(purposeColumnMap);
			    String key = pcm.getPurposeName()+":"+pcm.getPurposeColumn()+":"+pcm.getTableName();
			    if (!pcms.containsKey(key))
				pcms.put(key, new ArrayList<PurposeColumnMap>());
			    pcms.get(key).add(pcm);
			}
		}
	    }


	}

    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getCommitSize()
     */
    @Override
    public int getCommitSize() {

	return commitSize;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getTables()
     */
    @Override
    public Collection<ITable> getTables() {	
	return tables.values();
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getTable(java.lang.String)
     */
    @Override
    public ITable getTable(String tablename) {
	if (tables.containsKey(tablename))
	    return tables.get(tablename);

	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getPurposes()
     */
    @Override
    public List<IPurpose> getPurposes() {
	ArrayList<IPurpose> ret= new ArrayList<IPurpose>();
	for (IPurpose p: purposes.values()) {
	    p.setSchDoc(this);
	    ret.add(p);
	}
	return ret;
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getRules(java.lang.String)
     */
    @Override
    public List<IRule> getRules(String tableName) {

	if (tableName==null)
	    return rules;
	else {
	    List<IRule> ret = new ArrayList<IRule>();
	    for (IRule r : rules){
		for (RulePurpose rp : r.getRulePurpose()){
		    if (getPurposeColumns(rp.getPurposeName(),tableName) != null) {

			ret.add(r);
		    }
		}
	    }
	    return ret;
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getIndexes()
     */
    @Override
    public List<IIndex> getIndexes() {

	return new ArrayList<IIndex>(indexes.values());
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getDBStore()
     */
    @Override
    public String getDBStore() {

	return DBStore;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getDefaultDBAddress()
     */
    @Override
    public String getDefaultDBAddress() {

	return defaultDBAddress;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getCacheSize()
     */
    @Override
    public int getCacheSize() {

	return cacheSize;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getColumns(java.lang.String)
     */
    @Override
    public List<TableColumn> getColumns(String tableName) {
	ITable t = getTable(tableName);
	if (t!=null)
	    return t.getColumns();
	else return  new ArrayList<TableColumn>();

    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#updatePurpose(com.entitystream.identiza.entity.resolve.metadata.Purpose)
     */
    @Override
    public void updatePurpose(IPurpose p){

	if (!purposes.containsKey(p.getPurposeName()))
	    purposes.put(p.getPurposeName(), p);
	else {
	    //bring over the old PCMS
	    IPurpose oldp = purposes.get(p.getPurposeName());
	    for (PurposeColumn newpc : p.getPurposeColumns()){
		//does this pc exist in the new purpose?
		boolean foundpc=false;
		for (PurposeColumn oldpc : oldp.getPurposeColumns()){
		    if (oldpc.getColumn().equalsIgnoreCase(newpc.getColumn())){
			foundpc=true;
			for (PurposeColumnMap pcm : getPurposeColumnMaps(newpc)){
				 boolean found=false;
			     if (pcm.getTableColumn()!=null){
			      for (PurposeColumnMap oldpcm : getPurposeColumnMaps(oldpc)){
			    	if (oldpcm.getTableColumn()!=null){
				      if (oldpcm.getTableColumn().equalsIgnoreCase(pcm.getTableColumn()) &&
					    oldpcm.getTableName().equalsIgnoreCase(pcm.getTableName()) )
				        found=true;
			          }
			        }
				 }
			     if (!found)
				    addPurposeColumnMap(pcm);
			}	
		    }
		}
		if (!foundpc)
		    oldp.addPurposeColumn(newpc); //and all its pcms
	    }
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addPurpose(com.entitystream.identiza.entity.resolve.metadata.Purpose)
     */
    @Override
    public void addPurpose(IPurpose p) {

	purposes.put(p.getPurposeName(), p);
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addPurposeColumn(com.entitystream.identiza.entity.resolve.metadata.PurposeColumn)
     */
    @Override
    public void addPurposeColumn(PurposeColumn pc) {

	if (purposes.get(pc.getPurposeName())!=null)
	    purposes.get(pc.getPurposeName()).addPurposeColumn(pc);

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addTable(com.entitystream.identiza.entity.resolve.metadata.Table)
     */
    @Override
    public void addTable(ITable table) {
	this.tables.put(table.getTableName(),table);


    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addColumn(com.entitystream.identiza.entity.resolve.metadata.TableColumn)
     */
    @Override
    public void addColumn(TableColumn column) {
	ITable table = getTable(column.getTableName());
	if (table!=null){
	    table.addColumn(column);
	    addTable(table);
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#findConcepts(java.lang.String)
     */
    @Override
    public Document findConcepts(String concept) {
	Document ret = new Document();
	for (String pname : purposes.keySet()){
	    IPurpose p = purposes.get(pname);
	    Document pcDoc=new Document();
	    ret.append(p.getPurposeName(), pcDoc);
	    for (PurposeColumn pc : p.getPurposeColumns()){
		if (concept==null || pc.getColumn().equalsIgnoreCase(concept)){
		    pcDoc.append(pc.getColumn(), pc.getMatchClass());
		}
	    }
	}
	return ret;
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#findConceptsForProcType(java.lang.String)
     */
    @Override
    public Document findConceptsForProcType(String procType){
	Document ret = new Document();
	for (String pname : purposes.keySet()){
	    IPurpose p = purposes.get(pname);
	    Document pcDoc=new Document();
	    ret.append(p.getPurposeName(), pcDoc);
	    for (PurposeColumn pc : p.getPurposeColumns()){
		if (procType==null || pc.getMatchClass().equalsIgnoreCase(procType)){
		    pcDoc.append(pc.getColumn(), pc.getMatchClass());
		}
	    }
	}
	return ret;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getPurpose(java.lang.String)
     */
    @Override
    public IPurpose getPurpose(String purposeName) {

	return purposes.get(purposeName);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getThreads()
     */
    @Override
    public int getThreads() {

	return threads;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getColumn(java.lang.String, java.lang.String)
     */
    @Override
    public TableColumn getColumn(String tablename, String columnName) {
	ITable table = getTable(tablename);
	if (table!=null)
	    return table.getColumn(columnName);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addIndex(com.entitystream.identiza.entity.resolve.metadata.Index)
     */
    @Override
    public void addIndex(IIndex index) {

	indexes.put(index.getIndexName(),index);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getIndex(java.lang.String)
     */
    @Override
    public IIndex getIndex(String indexname) {

	return indexes.get(indexname);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addRule(com.entitystream.identiza.entity.resolve.metadata.Rule)
     */
    @Override
    public void addRule(IRule rule) {
	if (rules.size()==0)
	    getRules(null);
	int foundpos=-1;
	for (int pos=0; pos< rules.size(); pos++){
	    IRule exrule = rules.get(pos);
	    if (exrule.getHashKey().equalsIgnoreCase(rule.getHashKey())){
		foundpos=pos;
		break;
	    }
	}
	if (foundpos==-1)
	    rules.add(rule);
	else
	    rules.set(foundpos, rule);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getMatchProcs(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public Collection<MatchProcDefinitionInterface> getMatchProcs(String domainName, String instance, String groupName) {
	Collection<MatchProcDefinitionInterface> mpds = new ArrayList<MatchProcDefinitionInterface>(); 

	//found index - return match procs
	IIndex indexNode = getIndex(domainName);
	String purposeName = indexNode.getPurposeName();
	IPurpose purposeNode = getPurpose(purposeName);
	if (purposeNode!=null){
	    //if there a match between the purpose on the index and the name on the purpose?
	    for (PurposeColumn pc : purposeNode.getPurposeColumns()){
		MatchProcDefinition mpd = new MatchProcDefinition();
		mpd.setPurposeName(purposeName);
		//find the puposeColumns that relates to the purposeName
		mpd.setClassName(pc.getMatchClass());
		mpd.setDomainName(domainName);
		mpd.setProcName(purposeName + "_" + pc.getColumn());
		mpd.setRuleSet(RuleFactory.getInstance(pc.getMatchClass()));
		mpd.setGroupName(groupName);
		mpd.setMinWidth(pc.getMinWidth());
		mpd.setMaxWidth(pc.getMaxWidth());
		mpd.setIsMandatory(pc.isMandatory());
		mpd.setInstance(instance);
		mpds.add(mpd);		
	    }
	}


	return mpds;


    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getMatchTypes()
     */
    @Override
    public Map<String,Document> getMatchTypes(){
	Map<String,Document> retval = new HashMap<String,Document>();
	//add the default system ones first
	retval.put("MatchAddress", new Document("BaseClass", "MatchAddress").append("Description","Address"));
	retval.put("MatchFormula", new Document("BaseClass", "MatchFormula").append("Description","Formula"));
	retval.put("MatchAge", new Document("BaseClass", "MatchAge").append("Description","Person Age"));
	retval.put("MatchAgeRange", new Document("BaseClass", "MatchAgeRange").append("Description","Age Range"));
	retval.put("MatchBeatCode", new Document("BaseClass", "MatchBeatCode").append("Description","UK Police Beat Code"));
	retval.put("MatchChinesePersonName", new Document("BaseClass", "MatchChinesePersonName").append("Description","Chinese First Name"));
	retval.put("MatchCode", new Document("BaseClass", "MatchCode").append("Description","Code"));
	retval.put("MatchCountry", new Document("BaseClass", "MatchCountry").append("Description","Country"));
	retval.put("MatchColor", new Document("BaseClass", "MatchColor").append("Description","Color"));
	retval.put("MatchCompanyName", new Document("BaseClass", "MatchCompanyName").append("Description","Company Name"));
	retval.put("MatchEditDistanceDate", new Document("BaseClass", "MatchEditDistanceDate").append("Description","Fuzzy Date"));
	retval.put("MatchEditDistanceString", new Document("BaseClass", "MatchEditDistanceString").append("Description","Fuzzy String"));
	retval.put("MatchEmail", new Document("BaseClass", "MatchEmail").append("Description","Email Address"));
	retval.put("MatchEndAddress", new Document("BaseClass", "MatchEndAddress").append("Description","End Address"));
	retval.put("MatchHeight", new Document("BaseClass", "MatchHeight").append("Description","Height"));
	retval.put("MatchPersonName", new Document("BaseClass", "MatchPersonName").append("Description","Person Name"));
	retval.put("MatchPersonNameSyllables", new Document("BaseClass", "MatchPersonNameSyllables").append("Description","Person Name Using Syllable Coding"));
	retval.put("MatchPhonetic", new Document("BaseClass", "MatchPhonetic").append("Description","Phonetic String"));
	retval.put("MatchPostCode", new Document("BaseClass", "MatchPostCode").append("Description","Postal Code"));
	retval.put("MatchSortedEditDistanceString", new Document("BaseClass", "MatchSortedEditDistanceString").append("Description","Sorted Fuzzy String"));
	retval.put("MatchString", new Document("BaseClass", "MatchString").append("Description","Simple String"));
	retval.put("MatchText", new Document("BaseClass", "MatchText").append("Description","Text"));
	retval.put("MatchColumn", new Document("BaseClass", "MatchColumn").append("Description","Column To Concept"));
	retval.put("MatchTicker", new Document("BaseClass", "MatchTicker").append("Description","Stock Ticker"));

	return retval;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addMatchTypes(com.entitystream.monster.db.Document)
     */
    @Override
    public void addMatchTypes(Document mt){

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#clearPurposeColumnMaps(java.lang.String, java.lang.String)
     */
    @Override
    public void clearPurposeColumnMaps(String purposeName){
	Map<String, List<PurposeColumnMap>> newpcms = new HashMap<String, List<PurposeColumnMap>>();
	for (String key : pcms.keySet()) {
	    if (!key.startsWith(purposeName+":"))
		newpcms.put(key, pcms.get(key));
	}
	pcms=newpcms;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#addPurposeColumnMap(com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap)
     */
    @Override
    public void addPurposeColumnMap(PurposeColumnMap pcm) {
	String key = pcm.getPurposeName()+":"+pcm.getPurposeColumn()+":"+pcm.getTableName();
	if (!pcms.containsKey(key))
	    pcms.put(key, new ArrayList<PurposeColumnMap>());
	pcms.get(key).add(pcm);
    }

    @Override
    public List<PurposeColumnMap> getPurposeColumnMaps(PurposeColumn pc) {

	return getPurposeColumnMaps(pc.getPurposeName(), pc.getColumn(), null);
    }

    @Override
    public List<PurposeColumnMap> getPurposeColumnMaps(String purposeName, String purposeColumnName, String tableName) {
	if (pcms.containsKey(purposeName+":"+purposeColumnName+":"+tableName))
	    return pcms.get(purposeName+":"+purposeColumnName+":"+tableName);
	else{
	    //table could be null
	    ArrayList<PurposeColumnMap> purposeColumnMaps = new ArrayList<PurposeColumnMap>();
	    if (purposeColumnName==null && purposeName==null&& tableName==null) {
		for (String k : pcms.keySet()) {
		    purposeColumnMaps.addAll(pcms.get(k));
		}
	    } else
		if (purposeColumnName==null && purposeName==null) {
		    for (String k : pcms.keySet()) {
			if (k.endsWith(":"+tableName))
			    purposeColumnMaps.addAll(pcms.get(k));
		    }
		} else
		    if (purposeColumnName==null && tableName==null) {
			for (String k : pcms.keySet()) {
			    if (k.startsWith(purposeName+":"))
				purposeColumnMaps.addAll(pcms.get(k));
			}
		    } else if (tableName==null) {
			for (String k : pcms.keySet()) {
			    if (k.startsWith(purposeName+":"+purposeColumnName))
				purposeColumnMaps.addAll(pcms.get(k));
			}
		    }
		    else if (purposeColumnName==null) {
			for (String k : pcms.keySet()) {
			    if (k.startsWith(purposeName+":") && k.endsWith(":"+tableName))
				purposeColumnMaps.addAll(pcms.get(k));
			}
		    }
	    return purposeColumnMaps;
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getPurposeColumns(java.lang.String, java.lang.String)
     */
    @Override
    public List<PurposeColumn> getPurposeColumns(String purposeName, String tableName) {
	List<PurposeColumn> ret = new ArrayList<PurposeColumn>();
	if (tableName==null){
	    IPurpose p = getPurpose(purposeName);
	    if (p!=null)
		ret=p.getPurposeColumns();
	} else {
	    IPurpose p=getPurpose(purposeName);
	    if (p!=null)
		for (PurposeColumn pc : p.getPurposeColumns()){
		    if (getPurposeColumnMaps(pc.getPurposeName(), pc.getColumn(), tableName).size()>0)
			ret.add(pc);
		}
	}
	return ret;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#deleteColumn(com.entitystream.identiza.entity.resolve.metadata.TableColumn)
     */
    @Override
    public void deleteColumn(ITableColumn col) {
	ITable table = getTable(col.getTableName());
	if (table !=null)
	{
	    table.removeColumn(col.getColName());
	    addTable(table);
	}

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getName()
     */
    @Override
    public String getName() {
	return this.projectName;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#deletePurpose(java.lang.String)
     */
    @Override
    public void deletePurpose(String id) {

	purposes.remove(id);

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#deletePurposeColumn(java.lang.String, java.lang.String)
     */
    @Override
    public void deletePurposeColumn(String purposeName, String purposeColumnName) {
	IPurpose p = purposes.get(purposeName);
	if (p!=null)
	    p.deletePurposeColumn(purposeColumnName);

    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#deleteTable(java.lang.String)
     */
    @Override
    public void deleteTable(String tablename) {
	tables.remove(tablename);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#clearRules()
     */
    @Override
    public void clearRules(){
	rules.clear();
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#clearIndexes()
     */
    @Override
    public void clearIndexes() {
	indexes.clear();
    }


    //helper methods
    @Override
    public List<PurposeColumnMap> getPurposeColumnMaps(String optPurposeName) {
	PurposeColumn pc = new PurposeColumn();
	pc.setPurposeName(optPurposeName);
	return getPurposeColumnMaps(pc);
    }

    @Override
    public List<String> getTableColumnMaps(String purposeName, String tableName) {
	List<String> ret = new ArrayList<String>();
	for (PurposeColumnMap pcm : getPurposeColumnMaps(purposeName, null,tableName)){
	    ret.add(pcm.getTableColumn());
	}
	return ret;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getPurposes(java.lang.String)
     */
    @Override
    public List<IPurpose> getPurposes(String tableName) {
	List<IPurpose> ret = new ArrayList<IPurpose>();
	for (PurposeColumnMap pcm : getTablePurposeColumnMaps(tableName)){
	    ret.add(getPurpose(pcm.getPurposeName()));
	}
	return ret;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getTablePurposeColumnMaps(java.lang.String, java.lang.String)
     */
    @Override
    public List<PurposeColumnMap> getTablePurposeColumnMaps(String tableName) {
	return getPurposeColumnMaps(null, null, tableName);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta#getTargetTables()
     */
    @Override
    public List<ITable> getTargetTables() {
	List<ITable> targetTables = new ArrayList<ITable>();
	for (ITable t : this.getTables())
	    if (t.isTarget())
		targetTables.add(t);

	return targetTables;
    }

    @Override
    public void deletePurposeColumnMap(PurposeColumnMap pcm) {
	String key = pcm.getPurposeName()+":"+pcm.getPurposeColumn()+":"+pcm.getTableName();
	if (!pcms.containsKey(key))
	    pcms.remove(key);
    }


    @Override
    public void deleteMatchRule(long order) {
	IRule found = null;
	for (IRule r : rules) {
	    if (r.getOrder()==order)
		found=r;
	}
	if (found!=null)
	    rules.remove(found);
    }

    @Override
    public void deleteIndex(String def) {
	indexes.remove(def);

    }



}
