/*******************************************************************************
 * Copyright notice
 * 
 * This source code is copyright of Robert James Haynes - (c) 2010, 2011. All rights reserved.
 * 
 * Any redistribution, reproduction or decompilation of part or all of the code in any form is prohibited 
 * 
 * You may not, except with our express written permission, distribute or commercially exploit the content. Nor may you transmit it or store it in or display it on any website or other form of electronic retrieval system.
 ******************************************************************************/
/**
 *
	Identiza - Fuzzy matching Libraries

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.metadata;

import java.io.Serializable;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.entitystream.monster.db.Document;
public class Table implements Serializable, ITable{

    private ConcurrentHashMap<String, Object> defaultVals;
    private boolean useInPaths;
    private String tableName;
    private int lowThreshold;
    private int keyThreshold;
    private double maxProbs=0.05;
    private boolean document=false;
    private boolean history;
    private String _keyField;
    private Map<String,TableColumn> columnMap;
    private List<TableColumn> columns;
    private List<String> tags;
    private String icon;
    private List<String> labelFields;
    private String systemField;
    private String sensitivityField;
    private boolean internal;
    public String dateFormat;
    private String displayName;
    private String lastUpdatedCol;
    private String baseType="NODES"; //OR RELS
    private String schemaName;
    private String transmitFormat;
    private String transmitDestination;
    private boolean target;
    private String targetAuthToken;
    private boolean relTable=false;
    private boolean restrictedRelationships=true;
    private String _eidField;

    public Table(){
	defaultVals = new ConcurrentHashMap<String, Object>();
	columns = new ArrayList<TableColumn>();
	columnMap = new HashMap<String, TableColumn>();
    }


    public Table(String name, ISchemaMeta schemaDoc) {
	setTableName(name);
	columns = new ArrayList<TableColumn>();
	columnMap = new HashMap<String, TableColumn>();
	init();
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#init()
     */
    @Override
    public void init() {
	for (TableColumn c : columns){
	    c.setTableName(getTableName());
	    columnMap.put(c.getColName(), c);
	}
	defaultVals = new ConcurrentHashMap<String, Object>();
	for (ITableColumn col : getColumns()) {
	    if (col.getDefaultValue() != null
		    && !col.getDefaultValue().equalsIgnoreCase("NULL"))
		defaultVals.put(col.getColName(), col.getDefaultValue());
	}
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTableName(java.lang.String)
     */
    @Override
    public void setTableName(String tableName) {
	this.tableName = tableName;
    }
    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getTableName()
     */
    @Override
    public String getTableName() {
	return tableName;
    }


    @Override
    public TableColumn getColumn(String colname) {
	return columnMap.get(colname.replaceAll("\\[[0-9]*\\]", ""));
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setMaxProb(double)
     */
    @Override
    public void setMaxProb(double val){
	this.maxProbs=val;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getSensitivityField()
     */
    @Override
    public String getSensitivityField(){
	if (sensitivityField==null){
	    List<TableColumn> cols = getColumns();
	    for(TableColumn col : cols){
		if (col.isSensitivityField()){
		    sensitivityField= col.getColName();
		    break;
		}
	    }
	}
	return sensitivityField;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getSystemField()
     */
    @Override
    public String getSystemField() {
	if (systemField==null){

	    List<TableColumn> cols = getColumns();
	    for(TableColumn col : cols){
		if (col.isSystem()){
		    systemField= col.getColName();
		    break;
		}
	    }
	    if (systemField==null)
		systemField="systemName";
	}
	return systemField;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getKeyField()
     */
    @Override
    public String getKeyField() {
	if (_keyField==null){

	    List<TableColumn> cols = getColumns();
	    for(TableColumn col : cols){
		if (col.isPrimaryKey()){
		    _keyField= col.getColName();
		    break;
		}
	    }
	}
	return _keyField;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getColumns()
     */
    @Override
    public List<TableColumn> getColumns() {
	if (columnMap.size()==0 && columns.size()!=0) 	
	    return columns;
	else 	
	    return new ArrayList<TableColumn>(columnMap.values());
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setColumns(java.util.List)
     */
    @Override
    public void setColumns(List<TableColumn> cols) {
	for (ITableColumn c : cols){
	    c.setTableName(tableName); //tablename is not in the table JSON in the db
	    columnMap.put(c.getColName(), (TableColumn)c);
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setCandThreshold(int)
     */
    @Override
    public void setCandThreshold(int lowThreshold) {
	this.lowThreshold = lowThreshold;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getCandThreshold()
     */
    @Override
    public int getCandThreshold() {
	return lowThreshold;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setUseInPaths(boolean)
     */
    @Override
    public void setUseInPaths(boolean boolean1) {
	useInPaths = boolean1;

    }
    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getUseInPaths()
     */
    @Override
    public Boolean getUseInPaths() {

	return useInPaths;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getKeyThreshold()
     */
    @Override
    public int getKeyThreshold() {
	return keyThreshold;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setKeyThreshold(int)
     */
    @Override
    public void setKeyThreshold(int val){
	keyThreshold=val;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getMaxProb()
     */
    @Override
    public double getMaxProb() {

	return maxProbs;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#isDocument()
     */
    @Override
    public boolean isDocument() {

	return document;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setDocument(boolean)
     */
    @Override
    public void setDocument(boolean isDoc){
	document = isDoc;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#isHistory()
     */
    @Override
    public boolean isHistory() {

	return history;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setHistory(boolean)
     */
    @Override
    public void setHistory(boolean ish){
	history=ish;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getTags()
     */
    @Override
    public List<String> getTags() {
	return tags;

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTags(java.util.List)
     */
    @Override
    public void setTags(List<String> tagsin) {
	tags=tagsin;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setIcon(java.lang.String)
     */
    @Override
    public void setIcon(String iconName) {
	icon=iconName;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getIcon()
     */
    @Override
    public String getIcon() {
	if (icon!=null)
	    return icon;
	else return "";
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#createDefaults()
     */
    @Override
    public ConcurrentHashMap<String, Object> createDefaults() {
	return defaultVals;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#findLabelFields()
     */
    @Override
    public List<String> findLabelFields() {
	if (labelFields == null) {
	    HashMap<Integer, String> cols = new HashMap<Integer, String>();
	    Collection<TableColumn> fqr = getColumns();
	    // populate values?
	    if (fqr.size() > 0) {
	  	  java.util.Iterator<TableColumn> i = fqr.iterator();
		  while (i.hasNext()) {
		    TableColumn def = ((TableColumn) i.next());
		    if (def.getLabelPos() > 0)
			   cols.put(def.getLabelPos(), def.getColName());
		  }
	    }

	    ArrayList<String> res = new ArrayList<String>();
	    for (int i = 1; i <= cols.size(); i++) {
		   res.add(i-1,cols.get(new Integer(i)));
	    }
	    labelFields = res;
	}
	return labelFields;
    }

    @Override
    public void addColumn(TableColumn col) {
	columnMap.put(col.getColName(),col);
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#removeColumn(java.lang.String)
     */
    @Override
    public void removeColumn(String name) {
	columnMap.remove(name);

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#isInternal()
     */
    @Override
    public boolean isInternal() {
	return internal;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setInternal(boolean)
     */
    @Override
    public void setInternal(boolean intFlag) {
	this.internal=intFlag;

    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getDateFormat()
     */
    @Override
    public String getDateFormat() {
	return dateFormat;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#createDateFormatter()
     */
    @Override
    public DateTimeFormatter createDateFormatter() {
	if (dateFormat==null || dateFormat.equalsIgnoreCase("NULL"))
	    return DateTimeFormatter.ISO_DATE_TIME.withZone(ZoneId.systemDefault());
	return DateTimeFormatter.ofPattern(dateFormat).withZone(ZoneId.systemDefault());
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setDateFormat(java.lang.String)
     */
    @Override
    public void setDateFormat(String dateFormat) {
	if (dateFormat!=null && !dateFormat.equalsIgnoreCase("NULL"))
	    this.dateFormat = dateFormat;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTableDisplayName(java.lang.String)
     */
    @Override
    public void setTableDisplayName(String string) {
	this.displayName=string;
    }
    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getTableDisplayName()
     */
    @Override
    public String getTableDisplayName() {
	if (this.displayName!=null)
	    return this.displayName;
	else
	    return this.tableName;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getLastUpdatedCol()
     */
    @Override
    public String getLastUpdatedCol() {
	if (lastUpdatedCol==null){

	    List<TableColumn> cols = getColumns();
	    for(ITableColumn col : cols){
		if (col.isLastUpdated()){
		    lastUpdatedCol= col.getColName();
		    break;
		}
	    }
	}
	return lastUpdatedCol;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getBaseType()
     */
    @Override
    public String getBaseType() {
	return baseType;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setBaseType(java.lang.String)
     */
    @Override
    public void setBaseType(String baseType) {
	this.baseType = baseType;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getSequenceFields()
     */
    @Override
    public ArrayList<String> getSequenceFields() {
	ArrayList<String> seqf = new ArrayList<String>();
	for (ITableColumn col : columns){
	    if (col.isSequence())
		seqf.add(col.getColName());

	}
	return seqf;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getSchemaName()
     */
    @Override
    public String getSchemaName() {
	return schemaName;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setSchemaName(java.lang.String)
     */
    @Override
    public void setSchemaName(String schemaName) {
	this.schemaName = schemaName;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getTransmitFormat()
     */
    @Override
    public String getTransmitFormat() {
	return transmitFormat;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTransmitFormat(java.lang.String)
     */
    @Override
    public void setTransmitFormat(String tmf){
	transmitFormat=tmf;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getTransmitDestination()
     */
    @Override
    public String getTransmitDestination() {
	return transmitDestination;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTransmitDestination(java.lang.String)
     */
    @Override
    public void setTransmitDestination(String td){
	transmitDestination=td;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#isTarget()
     */
    @Override
    public boolean isTarget() {
	return target;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTarget(boolean)
     */
    @Override
    public void setTarget(boolean ist){
	target=ist;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getTargetAuthToken()
     */
    @Override
    public String getTargetAuthToken() {
	//80984bac20dd812352a54d4438a51b6d  EntityStream UHS3QccXCujR
	return targetAuthToken;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setTargetAuthToken(java.lang.String)
     */
    @Override
    public void setTargetAuthToken(String targetAuthToken){
	this.targetAuthToken=targetAuthToken;
    }


    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#isRelTable()
     */
    @Override
    public boolean isRelTable() {
	// TODO Auto-generated method stub
	return relTable;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setRelTable(boolean)
     */
    @Override
    public void setRelTable(boolean tf) {
	this.relTable=tf;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#setRestrictedRelationships(boolean)
     */
    @Override
    public void setRestrictedRelationships(boolean set) {

	restrictedRelationships=set;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getRestrictedRelationships()
     */
    @Override
    public boolean getRestrictedRelationships() {

	return restrictedRelationships;
    }

    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.metadata.ITable#getEIDField()
     */
    @Override
    public String getEIDField() {
	if (_eidField==null){

	    List<TableColumn> cols = getColumns();
	    for(TableColumn col : cols){
		if (col.isEID()){
		    _eidField= col.getColName();
		    break;
		}
	    }
	}
	return _keyField;

    }

    public Document toDocument() {
	Document ret = new Document();
	ret.append("columns", columns);
	ret.append("internal", internal);
	ret.append("document", document);
	ret.append("baseType", baseType);
	ret.append("schemaName", "p6");
	ret.append("target", target);
	ret.append("relTable", relTable);
	ret.append("restrictedRelationships", restrictedRelationships);
	ret.append("defaultVals", defaultVals);
	ret.append("useInPaths", useInPaths);
	ret.append("tableName", tableName);
	ret.append("lowThreshold",lowThreshold);
	ret.append("keyThreshold", keyThreshold);
	ret.append("maxProbs", maxProbs);
	ret.append("history", history);
	return ret;
    }


   
    @Override
    public List<String> getLabelFields() {
	return labelFields;
    }


}

