/*******************************************************************************
 * Copyright notice
 * 
 * This source code is copyright of Robert James Haynes - (c) 2010, 2011. All rights reserved.
 * 
 * Any redistribution, reproduction or decompilation of part or all of the code in any form is prohibited 
 * 
 * You may not, except with our express written permission, distribute or commercially exploit the content. Nor may you transmit it or store it in or display it on any website or other form of electronic retrieval system.
 ******************************************************************************/
/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.storage;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import java.util.logging.Logger;

import com.entitystream.monster.db.Document;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.IRelationship;
import com.entitystream.identiza.db.Node;
import com.entitystream.identiza.entity.resolve.match.MatchCriteria;
import com.entitystream.identiza.entity.resolve.match.MatchIndexInterface;
import com.entitystream.identiza.entity.resolve.match.MatchRecordInterface;
import com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface;
import com.entitystream.identiza.entity.resolve.match.MatchTableInterface;
import com.entitystream.identiza.entity.resolve.match.Token;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.ITableColumn;
import com.entitystream.identiza.entity.resolve.metadata.Purpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumn;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap;
import com.entitystream.identiza.entity.resolve.metadata.TableColumn;
import com.entitystream.identiza.entity.resolve.processing.IdentizaException;

import com.entitystream.identiza.entity.resolve.types.MatchProcInterface;
import com.entitystream.identiza.entity.resolve.types.Standardized;
import com.entitystream.identiza.entity.resolve.types.StandardizedSerialized;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.identiza.wordlist.RuleSet;
import com.entitystream.identiza.wordlist.WordList;
import com.entitystream.identiza.wordlist.WordObject;


public class Record implements Serializable, Comparable, RecordInterface{	
	//private ConcurrentHashMap<String, Object> vals=null;
	private String pkey = null;
	private String keyField = null;
	private String tableName;
	private String systemName=null;
	private List<String> labelFields;
	private Logger logger = Logger.getLogger("com.identiza");
	private boolean isNew=true;
	private HashMap<String, List<Standardized>> standards = new HashMap<String,List<Standardized>>();
	
	private String project;
	private String groupName;
	private String nodeName="";
	private List<TableColumn> cols;

	private volatile boolean valid = true;
	private volatile boolean persisted = false;
	private volatile INode baseNode = null;
	private volatile boolean loaded = false;

	private MatchSchemaInterface matchSchema;
	private MatchTableInterface matchTable = null;
	private String systemField=null;
	private int linkcount=0;
	private int taskcount=-1;
	private ArrayList<Object> hist=null;
	private String updatedBy=""; 
	private String sensitivityValue=null;
	private String sensitivityField=null;
	private boolean isInternal;
	private String EIDField="EID";
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getBaseNode()
	 */
	@Override
	public INode getBaseNode() {
		if (baseNode==null)
			load();
		return baseNode;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getPkey()
	 */
	@Override
	public String getPkey() {
		return pkey;	
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#neverLoad()
	 */
	@Override
	public void neverLoad() throws Exception{
		loaded=true;
		valid=true;
		if (baseNode==null)
			baseNode = new Node(new Document());
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getTableName()
	 */
	@Override
	public String getTableName(){
		return tableName;		
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setPkey(java.lang.String)
	 */
	@Override
	public void setPkey(String pkey) {
		try {
		
			this.pkey = pkey;
			if (this.pkey==null || this.pkey.length()==0)
				this.pkey = ""+getSchema().getDb().getNextId(getKeyField());
			if (getBaseNode()!=null)
				getBaseNode().setProperty(getKeyField(), this.pkey);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();

		} 

	}




	@Override
	public RecordInterface assignNode(INode n) throws Exception{
		this.baseNode=n;
		if (n.hasProperty("Table"))
			tableName = (String) n.getProperty("Table");

		if (tableName !=null && tableName.length()>0){		
			ITable table=matchSchema.getTable(tableName);
			if (table!=null){
				isInternal=table.isInternal();
				this.EIDField="EID";
				if (n.hasProperty(table.getKeyField())){
					if (n.getProperty(table.getKeyField()) instanceof String)
						this.pkey = (String)n.getProperty(table.getKeyField());
					else 
						this.pkey = n.getProperty(table.getKeyField()).toString();
				}
				cols = table.getColumns();
				keyField = table.getKeyField();
				labelFields = table.findLabelFields();		
				systemField=table.getSystemField();
			}

			loaded = false;
			valid=true;
		} else this.valid=false;

		if (n.hasProperty(getKeyField()))
			this.pkey = (String)n.getProperty(getKeyField());
		
		if (this.pkey==null || this.pkey.length()==0)
			this.pkey = n.getId();
		
		if (getBaseNode()!=null && getKeyField()!=null && this.pkey!=null)
			getBaseNode().setProperty(getKeyField(), this.pkey);
		loaded=true;

		return this;
	}

	@Override
	public RecordInterface assignTable(MatchTableInterface table) throws Exception{
		this.matchTable=table;
		if (table!=null)
			cols = table.getColumns();
		else {
			valid=false;
			return this;
		}
		isNew=true;
		isInternal=table.isInternal();
		EIDField=table.getEIDField();
		tableName = table.getTableName();
		if (tableName !=null && tableName.length()>0){						
			keyField = table.getKeyField();
			systemField=getMatchTable().getSystemField();
			labelFields = table.getLabelFields();		
			loaded = false;
			valid=true;
		} else this.valid=false;
		neverLoad();
		return this;
	}


	public String getKeyField() {
		return keyField;
	}

	public Record(MatchSchemaInterface matchSchema1) throws Exception{
		if (matchSchema1!=null){
			this.nodeName=matchSchema1.getNodeName();
			this.groupName=matchSchema1.getGroupName();
			this.project=matchSchema1.getProjectName();
			
			this.matchSchema=matchSchema1;
		}
		neverLoad();
		valid=false;
	}

	private Record(MatchSchemaInterface matchSchema, MatchTableInterface table, Node node) throws Exception{
		if (matchSchema!=null){
			this.nodeName=matchSchema.getNodeName();
			this.groupName=matchSchema.getGroupName();
			this.project=matchSchema.getProjectName();
			
			this.matchSchema=matchSchema;
		}
		this.matchTable=table;
		cols = table.getColumns();
		this.baseNode=node;
		isNew=true;
		isInternal=table.isInternal();
		EIDField=table.getEIDField();
		tableName = table.getTableName();
		if (tableName !=null && tableName.length()>0){						
			if (node.hasProperty(table.getKeyField()))
				this.pkey = (String)node.getProperty(table.getKeyField());
			keyField = table.getKeyField();
			systemField=getMatchTable().getSystemField();
			labelFields = table.getLabelFields();		
			loaded = false;
			valid=true;
		} else this.valid=false;
		
		if (this.pkey==null || this.pkey.length()==0)
			this.pkey = ""+getSchema().getDb().getNextId(getKeyField());
		if (getBaseNode()!=null)
			getBaseNode().setProperty(getKeyField(), this.pkey);
	}

	public Record(MatchSchemaInterface matchSchema, INode node) throws Exception{
		if (matchSchema!=null){
			this.nodeName=matchSchema.getNodeName();
			this.groupName=matchSchema.getGroupName();
			this.project=matchSchema.getProjectName();
			
			this.matchSchema=matchSchema;
		}
		this.baseNode=node;

		isNew=true;	
		if (node.hasProperty("Table"))
			tableName = (String) node.getProperty("Table");

		if (tableName !=null && tableName.length()>0){		
			ITable table=matchSchema.getTable(tableName);
			if (table!=null){
				isInternal=table.isInternal();
				this.EIDField="EID";
				if (node.hasProperty(table.getKeyField())){
					if (node.getProperty(table.getKeyField()) instanceof String)
						this.pkey = (String)node.getProperty(table.getKeyField());
					else 
						this.pkey = node.getProperty(table.getKeyField()).toString();
				}
				cols = table.getColumns();
				keyField = table.getKeyField();
				labelFields = table.findLabelFields();		
				systemField=table.getSystemField();
			}

			loaded = false;
			valid=true;
		} else this.valid=false;
		if (this.pkey==null || this.pkey.length()==0)
			this.pkey = ""+getSchema().getDb().getNextId(getKeyField());
		if (getBaseNode()!=null)
			getBaseNode().setProperty(getKeyField(), this.pkey);
	}

	

	private Record(MatchSchemaInterface matchSchema, MatchTableInterface mtable) throws Exception{
		this.nodeName=matchSchema.getNodeName();
		this.groupName=matchSchema.getGroupName();
		this.project=matchSchema.getProjectName();
		
		this.matchSchema=matchSchema;
		this.matchTable=mtable;
		this.isInternal=mtable.isInternal();
		this.EIDField=mtable.getEIDField();
		cols = mtable.getColumns();

		this.tableName=mtable.getTableName();
		this.labelFields=mtable.getLabelFields();
		this.keyField=mtable.getKeyField();
		persisted=false;		
		isNew=true;
		loaded = true;
		valid=true;

		neverLoad();
	}


	private Record(Record original){

		persisted=false;		
		isNew=true;
		loaded = true;
		valid=true;
		try {neverLoad();} catch (Exception e){e.printStackTrace();};
		EIDField=original.getEIDField();
		this.baseNode=original.baseNode;
		this.cols=original.cols;
		this.groupName=original.groupName;
		this.isNew=false;
		this.isInternal=original.isInternal;
		this.labelFields=original.labelFields;
		this.matchSchema=original.matchSchema;
		this.matchTable=original.matchTable;
		this.nodeName=original.nodeName;
		this.hist=null;
		this.pkey=original.pkey;
		this.project=original.project;
		
		this.standards=original.standards;
		this.systemField=original.systemField;
		this.systemName=original.systemName;
		this.tableName=original.tableName;
		this.updatedBy=original.updatedBy;

	}




	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getEIDField()
	 */
	@Override
	public String getEIDField() {
		return EIDField;
	}

	public MatchTableInterface getMatchTable() {
		try{
			if (matchTable==null)
				matchTable=getSchema().getMatchTable(tableName);
		} catch (Exception e){
			e.printStackTrace();
		}
		return matchTable;
	}

	public Record(String tableName, String pkey, MatchSchemaInterface matchSchema) throws Exception{
		this.nodeName=matchSchema.getNodeName();
		this.groupName=matchSchema.getGroupName();
		this.project=matchSchema.getProjectName();
		
		this.matchSchema=matchSchema;
		this.tableName = tableName;
		
		
		
		persisted=false;
		isNew=true;	
		if (getMatchTable()!=null){
			if (getMatchTable().getSystemField()!=null)
				systemField=getMatchTable().getSystemField();
			EIDField=getMatchTable().getEIDField();
			cols=getMatchTable().getColumns();
			if (getMatchTable().getKeyField()!=null){
				labelFields = getMatchTable().getLabelFields();
				keyField = getMatchTable().getKeyField();	
				isInternal = getMatchTable().isInternal();
			}
		}
		
		if (pkey==null || pkey.length()==0)
			pkey = ""+getSchema().getDb().getNextId(getKeyField());;
		this.pkey=pkey;
		if (getBaseNode()!=null)
			getBaseNode().setProperty(getKeyField(), pkey);
		
		
		loaded = false;
	}

	public Record(MatchCriteria matchcrits, MatchSchemaInterface matchSchema) throws Exception{
		this.nodeName=matchSchema.getNodeName();
		this.groupName=matchSchema.getGroupName();
		this.project=matchSchema.getProjectName();
		
		this.matchSchema=matchSchema;
		systemField=getMatchTable().getSystemField();
		EIDField=getMatchTable().getEIDField();
		persisted=false;
		isNew=true;
		this.tableName = matchcrits.getTableName();

		if (tableName!=null){
			this.pkey=matchcrits.getValue(getMatchTable().getKeyField());
			this.isInternal=getMatchTable().isInternal();
			cols = getMatchTable().getColumns();

			labelFields = getMatchTable().getLabelFields();
			keyField = getMatchTable().getKeyField();	
		}
		//this is a fake record.
		baseNode = new Node(getMatchTable().getMetaTable(), matchcrits.getValues());

		if (this.pkey==null || this.pkey.length()==0)
			this.pkey = ""+getSchema().getDb().getNextId(getKeyField());;
		if (getBaseNode()!=null)
			getBaseNode().setProperty(getKeyField(), this.pkey);
		loaded = false;		
	}



	public Record(ITable table, Map<Object, Object> newvals, MatchSchemaInterface schema) throws Exception{
		persisted=false;
		isNew=false;
		this.tableName = table.getTableName();			
		this.keyField =  table.getKeyField();	
		this.isInternal=table.isInternal();
		systemField=getMatchTable().getSystemField();
		this.matchSchema=schema;
		cols = table.getColumns();
		EIDField=getMatchTable().getEIDField();
		baseNode = new Node(table, newvals);


		if (this.pkey==null || this.pkey.length()==0){
			this.pkey = ""+getSchema().getDb().getNextId(getKeyField());
		}
		if (getBaseNode()!=null)
			getBaseNode().setProperty(getKeyField(), this.pkey);
		labelFields = table.findLabelFields();		
		loaded = true;
	}



	public void load(){
		if (!loaded){
			try{
				if (valid){
					if (baseNode ==null || baseNode.getId()==null){
						Document node = getSchema().getDb().getSingleNode( tableName, keyField, getPkey());
						
						baseNode = new Node (getSchema().getDb(), node);					
					} 

					if (baseNode !=null && baseNode.getId()!=null){
						persisted=true;
						isNew=false;
						loaded = true;

						hist=(ArrayList<Object>) getBaseNode().getProperty("history");
						setValue("lastUpdated", Document.fromISO8601UTC(getBaseNode().getProperty("lastUpdated")));
						setValue("action", (String)getBaseNode().getProperty("action"));
						updatedBy=(String)getBaseNode().getProperty("updatedBy");
						if (getBaseNode().getProperty(keyField)==null)
							getBaseNode().setProperty(keyField, pkey);
						pkey = (String) getBaseNode().getProperty(keyField);
						
						systemName = (String) getBaseNode().getProperty(systemField);
						if (systemName==null)
							systemName = tableName;
						//get the standards
						baseNode.setDB(this.matchSchema.getDb());
					} else {
						persisted=false;
						isNew=true;
						loaded = true;
					}

				}
			} catch (Exception e){
				logger.severe("Could not load record : " + e.getMessage());
				e.printStackTrace();
				loaded=false;
			}
		}

	}

	

	
	public ConcurrentHashMap<INode, ContextMap> getRelatedNodes(String reltype,String direction, Date when) throws Exception{
		ConcurrentHashMap<INode, ContextMap> res = new ConcurrentHashMap<INode, ContextMap>(); 
		load();
		if (baseNode!=null){
			Iterator it=null;
			if (reltype!=null && direction!=null)
				it = baseNode.getRelationships(reltype, direction, when);
			else if (reltype!=null)
				it = baseNode.getRelationships(reltype, when);
			else  it = baseNode.getRelationships(when);

			while (it.hasNext()){
				IRelationship rel = (IRelationship) it.next();
				String relname = rel.getType();
				if (relname!=null && ((reltype!=null && reltype.equalsIgnoreCase("ANTI-MATCH")) || !relname.equalsIgnoreCase("ANTI-MATCH"))){
					Map<String, Object> props = rel.getProperties();
					props.put("$id", rel.getId());
					ContextMap contextMap=null;

					if (props!=null && props.containsKey("history"))
					{
						ArrayList<Map<String, Object>> strContextMap = (ArrayList<Map<String, Object>>) props.get("history");
						contextMap= new ContextMap(relname,strContextMap);						
					} else {
						contextMap=new ContextMap(props);
						contextMap.setRelName(relname);						
					}
					//add all the other properties
					//if (props!=null)
					//	contextMap.pushProperties(props);

					INode otherNode = rel.getOtherNode(baseNode);
					if (otherNode !=null) {
					    	
						res.put(otherNode, contextMap);
					}
				}
			}
		}
		return res;		
	}

	public HashMap<String, String> getRelatedRecords(String relTypeName, Date when) throws Exception{

		HashMap<String, String> res = new HashMap<String, String>(); 
		Iterator it = baseNode.getRelationships(relTypeName, when);
		while (it.hasNext()){
			IRelationship rel = (IRelationship) it.next();
			Map<String, Object> otherVals = rel.getOtherNode(baseNode).getPropertyValues();
			if (otherVals.containsKey("Table")){
				String keyfield=this.getSchema().getTable(otherVals.get("Table").toString()).getKeyField();
				if (otherVals.get(keyfield)!=null){
					String _pkey=otherVals.get(keyfield).toString();
					String relname = rel.getType();
					res.put(_pkey, relname);
				}
			}

		}
		return res;		
	}

	public List<RecordInterface> getRelatedRecords(String relTypeName, String direction, Date when) throws Exception{
		List<RecordInterface> res = new ArrayList<RecordInterface>(); 
		Iterator it = baseNode.getRelationships(relTypeName, direction, when);
		while (it.hasNext()){
			IRelationship rel = (IRelationship) it.next();
			INode otherNode = rel.getOtherNode(baseNode);
			res.add(new Record(this.getSchema(),otherNode));
		}
		return res;				
	}

	private MatchSchemaInterface getSchema() throws Exception{
	
		return matchSchema;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getFieldNames()
	 */
	@Override
	public ArrayList<String> getFieldNames() throws Exception{		
		ArrayList<String> res = new ArrayList<String>();
		if (cols!=null)
			for(ITableColumn col: cols)
				res.add(col.getColName());
		return res;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setValue(java.lang.String, java.lang.Object)
	 */
	@Override
	public void setValue(String colname, Object value){
		try{
			if (colname!=null && colname.equals("updatedBy"))
				updatedBy=value.toString();
			if (colname!=null && value!=null)
				getBaseNode().setProperty(colname, value);
		} catch (Exception e){
			e.printStackTrace();
		}
	}

	public void saveNoIndexes(String username) throws Exception{
		String action="updat";
		setLastUpdated(new Date());
		if (getLastUpdated()==null)
			setLastUpdated(new Date());
		if (getAction()==null)
			setAction("NEW");
		try{
			baseNode=new Node(getSchema().getDb(),getSchema().getDb().addInternalNode(keyField, getPkey(), getBaseNode().getDocument(), getTableName(), false, isHistory(), username));
		} catch (Exception e){
			e.printStackTrace();
		}
		persisted=true;
		isNew=false;
		loaded=true;			
	}



	public RecordInterface save(boolean hasKeyChanged, String username) throws Exception{

		try
		{	
			String action="updat";

		
			
			StringBuilder error=new StringBuilder();
			for (ITableColumn col : this.cols){
				
				//lookups
				String lu=col.getLookup();
				if (lu!=null && lu.length()!=0){

					RuleSet rules = RuleFactory.getInstance(col.getLookup());
					StringBuilder valtolookup = new StringBuilder();
					if (col.getLookupColumns()!=null)
						for (String column : col.getLookupColumns().split(",")){
							valtolookup.append(getBaseNode().getProperty(column.trim()));
							valtolookup.append(" ");
						}
					else valtolookup.append(getBaseNode().getProperty(col.getColName()));

					String wordSearch=valtolookup.toString().trim();
					WordObject word = rules.getRuleLookup().getWord(wordSearch);
					String valueForward=wordSearch;				

					if (word!=null && valueForward!=null){											
						valueForward = word.getParent();
					}

					if (valueForward!=null && valueForward.trim().length()>0)
						getBaseNode().setProperty(col.getColName(), valueForward.trim());
				}
				if (col.getLookupFallback()!=null){
					if (getBaseNode().getProperty(col.getColName()) == null){
						Object fallback = getBaseNode().getProperty(col.getLookupFallback());
						if (fallback!=null)
							getBaseNode().setProperty(col.getColName(), fallback.toString());
					}
				}
			}

			
			setLastUpdated(new Date());
			if (getAction()==null)
				setAction("CURRENT");
			setUpdatedBy(username);
			getBaseNode().getProperty("lastUpdated");
			if (getBaseNode().getProperty(keyField)==null)
			    getBaseNode().setProperty(keyField, getPkey());
			action="add";

			
			//index EID
			String EIDField=getMatchTable().getEIDField();
			String eid=getValue(EIDField);
			if (eid==null) {
				long eidl=getSchema().getDb().getNextId(EIDField);
				getBaseNode().setProperty(EIDField, eidl);
				eid=""+eidl;
			}
			


			//add relations to foreign key records
			baseNode=new Node(getSchema().getDb(),getSchema().getDb().addInternalNode(keyField, getPkey(), getBaseNode().getDocument(), getTableName(), hasKeyChanged, isHistory(), username));

			
			//create a new target record
			

			persisted=true;
			isNew=false;
			loaded=true;			
			getSchema().addMessages(""+(tableName+pkey).hashCode(), "{image: \"fa fa-edit\", actionName: \"Edit\", action: \"doEditNode(\'"+tableName+':'+pkey+"\');\"}" ,tableName + " Updated", username + " recently updated " + display(),true);

		}
		catch (Exception e){
			e.printStackTrace();
		}
		return this;

	}

	public void setAction(String action) throws Exception {
		getBaseNode().setProperty("action", action);
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getValue(java.lang.String)
	 */
	@Override
	public String getValue(String colname) throws Exception{
		if (!loaded) load();
		Field f;
		try {
			if (colname!=null){
				if (getBaseNode().getProperty(colname)!=null)
					return Document.objectToString(getBaseNode().getProperty(colname));
				else
					return null;
			} else return null;
		} catch (Exception e) {
			e.printStackTrace();
			throw new IdentizaException(e.getMessage());
		}

	}

	public boolean isValid() {
		return valid;
	}

	public boolean isNew() {
		return isNew;
	}

	public boolean isPersisted() {
		return persisted;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#toHashMap()
	 */
	@Override
	public Map<String, Object> toHashMap() throws Exception {
		if (!loaded) load();
		return getBaseNode().getPropertyValues();
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#display()
	 */
	@Override
	public String display() throws Exception {
		return Record.buildLabel(this.baseNode.getDocument(), labelFields);
	}

	public String displayAll() throws Exception {
		String res = "";
		int l=-1;
		for(String label : labelFields){
			String val = Document.objectToString(getBaseNode().getProperty(label));
			if (val!=null && !val.equals("null") && val.length()>0){
				res = res + val;
				if (++l<labelFields.size()-1)
					res = res + ", ";
			}
		}
		res = res + ":";
		for(String value : getMatchTable().getFields()){
			if (getBaseNode().getProperty(value) instanceof String){
				String val = (String) getBaseNode().getProperty(value);
				if (val!=null && !val.equals("null") && val.length()>0){
					res = res + value +":" + val + ',';
				}
			}
		}
		return res.substring(0,res.length()-1).trim();
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setLabelFields(java.util.List)
	 */
	@Override
	public void setLabelFields(List<String> labelFields) {
		this.labelFields = labelFields;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getLabelFields()
	 */
	@Override
	public List<String> getLabelFields() {
		return labelFields;
	}

	public void putRecord(boolean keychanged, String username) throws Exception{
		getSchema().getMatchTable(tableName).putRecord(this, keychanged, false, username);
	}

	@Override
	public void addRelatedRec(RecordInterface targetRec, java.lang.String reasonVerb,  Date dateOf, String actionThatCausedThis, String user) throws Exception {
		addRelatedRec(targetRec, reasonVerb, dateOf, new Document(), actionThatCausedThis, user);
	}

	public void addRelatedRec(RecordInterface targetRec, java.lang.String reasonVerb, Date dateOf, Document values, String actionThatCausedThis, String user) throws Exception {

		//reasonVerb is not always passed by the automatcher
		if (reasonVerb==null || reasonVerb.length()==0)
			if (targetRec.getTableName().equalsIgnoreCase(this.getTableName()))
				reasonVerb="Link";
			else
				reasonVerb="External Link";

		logger.fine("Relating " + this.display() + " to " + targetRec.display() + " with a verb " + reasonVerb);

		if (!targetRec.equals(this) && this.getBaseNode()!=null){
			//chekc to so if this has already been done
			Iterator<?> rels = this.getBaseNode().getRelationships(reasonVerb, "BOTH", null);
			boolean found=false;
			IRelationship rel=null;
			while (rels.hasNext())
			{
				rel = (IRelationship) rels.next();
				INode on = rel.getOtherNode(this.getBaseNode());
				if (on!= null && on.getId()==targetRec.getBaseNode().getId()){
					found = true;
					break;
				}
			}
			if (!found){
				rel = this.getBaseNode().createRelationshipTo(targetRec.getBaseNode(), reasonVerb,dateOf, values);
				//assess the inheritance impact
				boolean changed=false;
				if (getMatchTable().isInternal() && !targetRec.isInternal()){
					changed=applyInheritanceFrom(targetRec);
					if (changed)
						this.saveNoIndexes(user);

				}
				else if (targetRec.isInternal() && !getMatchTable().isInternal()){
					changed=targetRec.applyInheritanceFrom(this);
					if (changed)
						targetRec.saveNoIndexes(user);
				}

			}
			//need to remove the task relating to this as a result of the crossmatch
			else{
				if (!values.containsKey("status"))
					values.put("status", "ACTIVE");
				rel.setLastUpdate(new Date());
				rel.setProperties(values);
			}


		}
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#applyInheritanceFrom(com.entitystream.identiza.entity.resolve.storage.RecordInterface)
	 */
	@Override
	public boolean applyInheritanceFrom(RecordInterface targetRec) {
		return Node.applyInheritanceFrom(this.getBaseNode(), targetRec.getBaseNode(), new HashSet<IPurpose>(getMatchTable().getPurposesForTable()), matchSchema.getSchDoc());
	}

	public boolean equals(RecordInterface that){
		if (this!=null && that!=null && this.getBaseNode()!=null && that.getBaseNode()!=null)
			return (this.getBaseNode().getId() == that.getBaseNode().getId());
		else return false;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#toString()
	 */
	@Override
	public String toString(){
		return baseNode.getDocument().toJson();
	}



	public boolean isRelatedTo(RecordInterface record) throws Exception {
		boolean found = false;
		Iterator rels = this.getBaseNode().getRelationships(new Date());

		IRelationship rel=null;
		while (rels.hasNext())
		{
			rel = (IRelationship) rels.next();
			if (rel.getEndNode().getId()==record.getBaseNode().getId()){
				//logger.info("Found direct relationship between nodes");
				found = true;
				break;
			}
		}

		return found;
	}


	public boolean isRelatedTo(RecordInterface record, String reltype, Date when) throws Exception {
		boolean found = false;
		Iterator rels = this.getBaseNode().getRelationships(reltype, "BOTH",new Date());

		IRelationship rel=null;
		while (rels.hasNext())
		{
			rel = (IRelationship) rels.next();
			if (rel.getEndNode().getId()==record.getBaseNode().getId()){
				//logger.info("Found direct relationship between nodes");
				found = true;
				break;
			}
		}

		return found;
	}




	public IRelationship getRelatedTo(RecordInterface otherRec, String relTypeName, Date when) throws Exception {

		if (relTypeName!=null){
			Iterator rels = this.getBaseNode().getRelationships(relTypeName, when);
			IRelationship rel=null;
			while (rels.hasNext())
			{
				rel = (IRelationship) rels.next();
				if (rel.getOtherNode(getBaseNode()).getId()==otherRec.getBaseNode().getId()){
					//logger.info("Found direct relationship between nodes");
					return rel;
				}
			}
		}
		return null;
	}

	@Override
	public boolean setValues(Document vals) {
		//boolean isEffected=false;
		//ConcurrentHashMap<String, Object> oldValues = getValues();
		baseNode.setProperties(vals);
		return true;
	}


	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getHistoryRecords()
	 */
	@Override
	public Collection<RecordInterface> getHistoryRecords(){
		List<RecordInterface> hrecs = new ArrayList<RecordInterface>();
		if (baseNode!=null && hist!= null){
			synchronized(hist){
				try {
					for (Object hrec : hist){
						Document histr=(Document)hrec;
						if (histr.getString("_id")!=null) {
							RecordInterface rec=new Record(getSchema(), getMatchTable());
							rec.setBaseNode(baseNode);
							rec.setPkey(histr.get(keyField).toString());
							rec.setValues(histr);

							hrecs.add(rec);
						}
					}

				} catch (Exception e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}

		Collections.sort(hrecs);
		return hrecs;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#addHistoryRecord(com.entitystream.identiza.db.Node, java.lang.String, java.lang.String)
	 */
	@Override
	public void addHistoryRecord(INode newValues, String action, String username) throws Exception {

		Document newDoc = new Document();
		newDoc.putAll(newValues.getDocument());
		Document currentDoc = new Document();
		currentDoc.putAll(this.baseNode.getDocument());
		currentDoc.remove("tokens");
		currentDoc.remove("standardised");
		//get current history - if any
		List histValues=(List) currentDoc.get("history");
		if (histValues==null)
			histValues= new ArrayList<Document>();
		else
			currentDoc.remove("history");
		currentDoc.append("action", action);
		currentDoc.append("updatedBy", username);
		histValues.add(0,currentDoc);
		//if the new doc has a history - we should merge them in too - for merges
		if (newDoc.containsKey("history")){
			histValues.addAll((List)newDoc.get("history"));
			newDoc.remove("history");
		}
		newDoc.append("history", histValues);
		newDoc.put("action", "CURRENT");
		newDoc.put("updatedBy", username);

		this.getBaseNode().setProperties(newDoc);
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getValues()
	 */
	@Override
	public ConcurrentHashMap<String, Object> getValues() throws Exception {
		ConcurrentHashMap<String, Object> ret = new ConcurrentHashMap<String, Object>();
		if (getBaseNode().getPropertyValues()!=null)
			ret.putAll(getBaseNode().getPropertyValues());
		return ret;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setBaseNode(com.entitystream.identiza.db.Node)
	 */
	@Override
	public INode setBaseNode(INode baseNode2) {

		return baseNode=baseNode2;
	}

	public boolean isComplete() throws Exception {
		if (getBaseNode()!=null)
			return getBaseNode().hasProperty("Table");
		else return false;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getLastUpdated()
	 */
	@Override
	public Date getLastUpdated() {
		// TODO Auto-generated method stub
		try {
			if (getBaseNode()!=null){
				Object o = getBaseNode().getProperty("lastUpdated");
				if (o!=null)
					return (Date)o;
				else 
					return null;
			} else return null;
		} catch (Exception e) {
			// TODO Auto-generated catch block
			return null;
		}
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setLastUpdated(java.util.Date)
	 */
	@Override
	public void setLastUpdated(Date date) throws Exception {
		getBaseNode().setProperty("lastUpdated", date);		
	}

	public String getAction() throws Exception {
		// TODO Auto-generated method stub
		if (getBaseNode()!=null)
			return (String) getBaseNode().getProperty("action");
		else return null;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getStandardised(java.lang.String, com.entitystream.identiza.entity.resolve.types.MatchProcInterface, java.util.List, com.entitystream.identiza.wordlist.WordList)
	 */
	@Override
	public List<Standardized> getStandardised(String purposeColumnName,
			MatchProcInterface proc, List<PurposeColumnMap> map, WordList anon) throws Exception {
		List<Standardized> standardsarray=null;
		try {
			//is it cached?
			if (standards.get(purposeColumnName)==null || standards.get(purposeColumnName).size()==0){
				//look for it in the database

				if (baseNode!=null && baseNode.hasProperty("standardized")){
					Document stds = baseNode.getDocument().getAsDocument("standardized");
					List<Document> stdvals = (List<Document>) stds.get(purposeColumnName);
					if (stdvals!=null)
						standardsarray= (List<Standardized>) INode.deserialise(stdvals, proc.getStandardClass());
				}
				if (standardsarray ==null || 
						(
								standardsarray.size()>0 && 
								((String[])standardsarray.get(0).getComparitorWords()).length==0
								)
						){
					boolean needtorebuild=false;
					for (PurposeColumnMap mapitem:map)
						if (mapitem.getTableName().equalsIgnoreCase(this.getTableName())){
							needtorebuild=true;
							break;
						}

					if (needtorebuild){
						//for each record including history
						standardsarray = standardise(purposeColumnName, proc, map, anon);
						//save it
						if (baseNode!=null){
							if (baseNode.getDocument().getAsDocument("standardized")==null)
								baseNode.getDocument().append("standardized", new Document());
							(baseNode.getDocument().getAsDocument("standardized"))
							.append(purposeColumnName, INode.serialize(standardsarray));
						}
					}
				}
				standards.put(purposeColumnName, standardsarray);
			} else standardsarray=standards.get(purposeColumnName);

		} catch (IdentizaException e) {
			e.printStackTrace();
		}
		return standardsarray;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#standardise(java.lang.String, com.entitystream.identiza.entity.resolve.types.MatchProcInterface, java.util.List, com.entitystream.identiza.wordlist.WordList)
	 */
	@Override
	public List<Standardized> standardise(String purposeName,
			MatchProcInterface proc, List<PurposeColumnMap> pcm, WordList anon) throws Exception {
		return Record.standardise(this, getSchema(), purposeName, proc, pcm, anon);
	}
	

	public static List<Standardized> standardise(RecordInterface rec, MatchSchemaInterface schema, String purposeName, MatchProcInterface proc, List<PurposeColumnMap> pcm, WordList anon) throws Exception {
		ArrayList<Standardized> standardsarray=null;
		try {
			standardsarray = new ArrayList<Standardized>();
			for (RecordInterface record : rec.getAllRecords()){
				Object[] objects= INode.getWords(record.getBaseNode(), pcm,  anon, new Date());
				ArrayList<ArrayList<String>> baseTokensALL = (ArrayList<ArrayList<String>>) objects[0];
				String originalText = (String) objects[1];
				for (ArrayList<String> baseTokens : baseTokensALL){
					Standardized stdItem = proc.standardise(originalText,baseTokens.toArray(new String[baseTokens.size()]));
					if (stdItem.getCalculatedWords()!=null)
						standardsarray.add(stdItem);
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return standardsarray;
	}
	

	

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getAllRecords()
	 */
	@Override
	public Collection<RecordInterface> getAllRecords() {
		ArrayList<RecordInterface> recs = new ArrayList<RecordInterface>();
		recs.add(this);
		recs.addAll( getHistoryRecords() );
		return recs;
	}

	
	public static Integer calcMergeCount(Object o, int mergeIncrement){
		Integer mc=null;
		try {
			if (o==null)
				mc=0;
			else {
				if (o instanceof Integer)
					mc = (Integer) o;
				else mc = Integer.parseInt((String)o);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (mc==null) 
			return mergeIncrement;
		else 
			return mc+mergeIncrement;
	}

	public void incrementMergeCount(int mergeIncrement) throws NumberFormatException, Exception {
		Object o = baseNode.getProperty("MergeCount");
		baseNode.setProperty("MergeCount", calcMergeCount(o, mergeIncrement));
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#calculateScore(com.entitystream.identiza.entity.resolve.storage.RecordInterface, boolean, boolean, boolean, boolean)
	 */
	@Override
	public MatchRecordInterface calculateScore(RecordInterface comparitor, boolean forSearch,
			boolean googleSearch, boolean asContent, boolean matchScoring) throws Exception {		
		return getSchema().calculateScore(this, comparitor, forSearch, googleSearch, asContent,matchScoring);

	}

	public boolean isHistory(){
		if (getMatchTable()!=null)
			return getMatchTable().isHistory();
		else return false;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getIcon()
	 */
	@Override
	public String getIcon() {
		//if this record has a icon field then use it else use the table default
		String iconValue=null;
		if (getMatchTable()!=null){
			String iconField = getMatchTable().getIconField();

			try{
				if (iconField!=null){
					iconValue = this.getValue(iconField);			
				}
			} catch (Exception e){}
			if (iconField==null || iconValue==null)		
				iconValue=getMatchTable().getDefaultIcon();
		}
		return iconValue;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getSystemName()
	 */
	@Override
	public String getSystemName() throws Exception {
		if (systemName==null && getMatchTable()!=null){
			if (systemField==null)
				systemField=getMatchTable().getSystemField();
			if (systemField!=null)
				systemName=(String) getBaseNode().getProperty(systemField);
			if (systemName==null)
				systemName=tableName;
		}
		return systemName;
	}

	public String getSensitivityValue() throws Exception {
		if (sensitivityField==null){
			if (sensitivityField==null)
				sensitivityField=getMatchTable().getSensitivityField();
			if (sensitivityField!=null)
				sensitivityValue=(String) getBaseNode().getProperty(sensitivityField);
		}
		return sensitivityValue;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setSystemName(java.lang.String)
	 */
	@Override
	public void setSystemName(String systemName) throws Exception {
		this.systemName = systemName;
		if (systemField==null)
			systemField=getMatchTable().getSystemField();			
		getBaseNode().setProperty(systemField, systemName);
	}

	
	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#setUpdatedBy(java.lang.String)
	 */
	@Override
	public void setUpdatedBy(String updatedBy){
		this.updatedBy=updatedBy;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getUpdatedBy()
	 */
	@Override
	public String getUpdatedBy() {
		return this.updatedBy;
	}

	@Override
	public int compareTo(Object o) {
		RecordInterface other = (RecordInterface)o;
		if (other.getLastUpdated()!=null && this.getLastUpdated()!=null)
			return other.getLastUpdated().compareTo(this.getLastUpdated());
		else return 0;
	}

	

	@Override
	public boolean isSensitive() throws Exception {
		String v=getSensitivityValue();
		if (v==null)
			return false;
		else if (v.equalsIgnoreCase("false"))
			return false;
		else if (v.length()==0)
			return false;
		else if (v.equalsIgnoreCase("null"))
			return false;
		else return true;
	}

	
	public boolean isInternal() {
		return isInternal;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getDocument()
	 */
	@Override
	public RecordInterface getDocument() {
		// TODO Auto-generated method stub
		return null;
	}

	/* (non-Javadoc)
	 * @see com.entitystream.identiza.entity.resolve.storage.RecordInterface#getXref()
	 */
	@Override
	public List<Document> getXref() {
	    // TODO Auto-generated method stub
	    return null;
	}

	
	public static RecordInterface build(MatchSchemaInterface matchSchema2) throws Exception {
	   return new Record(matchSchema2);
	}

	
	
	public static String buildLabel(Document record, List<String> labelFields) {
	    String res = "";
		int l=-1;
		if (labelFields!=null ){
			for(String label : labelFields){
				if (record.getProjection(label)!=null){
					String val = (String) Document.objectToString(record.getProjection(label));
					if (val!=null && !val.equals("null") && val.length()>0){
						res = res + val;
						if (++l<labelFields.size()-1)
							res = res + ", ";
					}
				}
			}
		}
		res=res.trim();
		if (res.endsWith(","))
			res =res.substring(0,res.length()-1);
		return res;
	}



	

}