/*******************************************************************************
 * Copyright notice
 * 
 * This source code is copyright of Robert James Haynes - (c) 2010, 2011. All rights reserved.
 * 
 * Any redistribution, reproduction or decompilation of part or all of the code in any form is prohibited 
 * 
 * You may not, except with our express written permission, distribute or commercially exploit the content. Nor may you transmit it or store it in or display it on any website or other form of electronic retrieval system.
 ******************************************************************************/
/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.storage;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringEscapeUtils;

import com.entitystream.monster.db.Document;
import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.Node;
import com.entitystream.identiza.entity.resolve.match.MatchRecordInterface;
import com.entitystream.identiza.entity.resolve.match.MatchSchemaInterface;




public class Result implements Serializable, Comparable{
	private static final long serialVersionUID = 7728756772762055259L;
	String pkey;
	String tablename;
	String label;
	boolean isInternal=true;
	Document values;
	double score=0d;
	private List<Document>  xref;
	private Date lastUpdated;
	private String action;
	private String icon;
	private String id;
	
	private String lastUpdater;
	private List<Map<String, Object>> history;
	private String sensitivityValue;
	private Map<Result, ContextMap> children;
	private Document document;
	private List<Document> histDoc=new ArrayList<Document>();
	private long eid;

	public Result(String pkey, MatchRecordInterface matchRecord, Record rec, boolean isDirect) throws Exception {
		rec.load();
		setPkey(pkey);
		setTable(rec.getTableName());
		setScore(matchRecord.getScore());
		setValues(rec.toHashMap());
		setLabel(rec.display());
		setXREF(rec.getXref());
		setLastUpdated(rec.getLastUpdated());
		setUpdatedBy(rec.getUpdatedBy());
		setAction(rec.getAction());
		this.icon=rec.getIcon();
		
		this.isInternal=rec.isInternal();
		this.eid=rec.getBaseNode().getDocument().getLong("EID");
		setSensitivityValue(rec.getSensitivityValue());
		history= new ArrayList<Map<String, Object>>();
		for (RecordInterface rh : rec.getAllRecords()){
			Map<String, Object> hm = rh.toHashMap();
			hm.put("label", rec.display());
			history.add(hm);
		}


	}

	@Override
	public int hashCode(){
		return (getTable() +":"+ getPkey()).hashCode();
	}

	@Override
	public String toString(){
		return getTable() +":"+ getPkey();
	}

	@Override
	public boolean equals(Object other){
		return other.toString().equalsIgnoreCase(toString());
	}

	private void setAction(String action2) {
		this.action=action2;
	}


	public Result(String pkey, MatchRecordInterface matchRecord, RecordInterface recordInterface) throws Exception {
		recordInterface.load();
		if (recordInterface.isValid()){
			setPkey(pkey);
			setTable(recordInterface.getTableName());
			setScore(matchRecord.getScore());
			setValues(recordInterface.toHashMap());
			setDocument(recordInterface.getBaseNode().getDocument());
			setLabel(recordInterface.display());
			this.isInternal=recordInterface.isInternal();
			
			setXREF(recordInterface.getXref());
			setLastUpdated(recordInterface.getLastUpdated());
			setAction(recordInterface.getAction());
			setUpdatedBy(recordInterface.getUpdatedBy());
			this.icon=recordInterface.getIcon();
			this.eid=recordInterface.getBaseNode().getDocument().getLong("EID");
			
			setSensitivityValue(recordInterface.getSensitivityValue());
			history= new ArrayList<Map<String, Object>>();
			for (RecordInterface rh : recordInterface.getAllRecords()){
				history.add(rh.toHashMap());
			}
		}
	}

	public void setXREF(List<Document> xref) {
		this.xref=xref;

	}

	public List<Document>  getXREF(){
		return xref;
	}

	
	/**
	 * @param record
	 */
	public Result(Document record, String key, String label, String icon) {
	    this.values=record;
	    this.document=record;
	    setPkey(key);
	    setNodeID(record.getString("_id"));
	    setTable(values.getString("Table"));
	    setLabel(label);
	    setIcon(icon);
	    setLastUpdated(values.getDate("lastUpdated"));
	    setUpdatedBy(values.getString("updatedBy"));
	    this.eid=record.getLong("EID");
	}

	public Result(RecordInterface record) throws Exception {
		if (record!=null && record.isValid()){
			record.load();
			setPkey(record.getPkey());
			setTable(record.getTableName());
			setScore(100);
			Document doc = new Document(record.getBaseNode().getDocument());
			doc.remove("standardized");
			doc.remove("tokens");
			setDocument(doc);
			setValues(record.getValues());
			setLabel(record.display());
			setXREF(record.getXref());
			setLastUpdated(record.getLastUpdated());
			setUpdatedBy(record.getUpdatedBy());
			setAction(record.getAction());
			this.isInternal=record.isInternal();
			
			setSensitivityValue(record.getSensitivityValue());
			
			if (record !=null){
			    this.eid=record.getBaseNode().getDocument().getLong("EID");
				this.icon=record.getIcon();
				if (record.getBaseNode()!=null)
					this.id=record.getBaseNode().getId();
				history= new ArrayList<Map<String, Object>>();
				for (RecordInterface rh : record.getAllRecords()){
					Document doc2 = new Document(rh.getBaseNode().getDocument());
					doc2.remove("standardized");
					doc2.remove("tokens");
					history.add(rh.toHashMap());
					histDoc.add(doc2);
				}
			}
		}
	}

	
	
	
	private void setSensitivityValue(String sensitivityValue) {
		this.sensitivityValue=sensitivityValue;

	}

	public String getSensitivityValue() {
		return this.sensitivityValue;

	}

	

	public Result(MatchSchemaInterface schema, Node node) throws Exception {
		this(RecordInterface.build(schema).assignNode(node));
	}


	public Result() {
		values = new Document();
	}

	

	

	public String getPkey() {
		return pkey;
	}
	public void setPkey(String pkey) {
		this.pkey = pkey;
	}
	public Map<String, Object> getValues() {
		return values;
	}
	public void setValues(Map<String, Object> values) {
		this.values = new Document(values);
	}
	public double getScore() {
		return score;
	}
	public void setScore(double score) {
		this.score = score;
	}
	public void setTable(String tableName) {
		this.tablename = tableName;		
	}

	public void setLabel(String label) {
		this.label = label;		
	}
	public String getTable() {
		return this.tablename;		
	}
	public String getLabel() {
		//auto generated label
		return label;
	}



	public void setDocument(Document doc) {
		this.document = doc;
	}

	
	public Date getLastUpdated() {
		// TODO Auto-generated method stub
		return lastUpdated;
	}

	public void setLastUpdated(Date date) {
		// TODO Auto-generated method stub
		lastUpdated=date;
	}


	public String getAction() {
		// TODO Auto-generated method stub
		return action;
	}


	@Override
	public int compareTo(Object arg0) {
		// TODO Auto-generated method stub
		if (((Result)arg0).score<this.score) return -1;
		else if (((Result)arg0).score>this.score) return 1; 
		else return 0;
	}


	public String getIcon() {
		return icon;
	}

	public void setIcon(String icon){
		this.icon=icon;
	}


	public String toJSON(boolean isHistory) {
		return toJSON(null, isHistory);
	}

	
	public Document toDocument(ContextMap ctx){
		Document ret = new Document(document);
		ret.append("label",getLabel());
		ret.append("id", getPkey());
		if(ctx!=null)
		   ret.append("context", Document.parse("{list:"+ctx.valuesToJSONString()+"}").get("list"));
		
		if (children!=null && children.size()>0){
			ArrayList<Document> childrenList = new ArrayList<Document>();
			ret.append("children", childrenList);
			for (Result rel : children.keySet()){
				if (rel.id!=id){
					Document childNode = new Document();
					ctx = children.get(rel);
					childNode.append("node", rel.toDocument(ctx));
					childrenList.add(childNode);
				}	
			}
		}
		return ret;
	}
	
	
	
	public Document toDocument() {
	    Document result = new Document();
	    if (getPkey()!=null)
	        result.append("node",getPkey());
	    if (getLabel()!=null)
	        result.append("label", getLabel());
	    if (getIcon()!=null)
		result.append("icon",getIcon());
	    result.append("values", values);
	   
	    return result;
	}
	

	public String toJSON(ContextMap map, boolean isHistory) {
		StringBuilder result = new StringBuilder();
		boolean first=true;
		result.append("{");
		if (values!=null){
			for (Object key: values.keySet()){
				if ( !((String)key).equalsIgnoreCase("TaskIds")){
					String value=Document.objectToString(values.get(key));

					if (value==null)
						value="";
					if (!first)
						result.append(",");
					first=false;
					result.append("\""+key+"\": \"" + value + "\"");
				}	
			}

			if (!first)
				result.append(",");
			result.append("\"Label\": \"" + getLabel() + "\"");
			result.append(",\"Score\": \"" + getScore() + "\"");
			result.append(",\"Icon\": \"" + getIcon() + "\"");
			result.append(",\"id\": \"" + getPkey() + "\"");
			result.append(",\"_id\": \"" + getID()  + "\"");
			if (!values.containsKey("Table"))
				result.append(",\"Table\": \"" + this.getTable() + "\"");
			//now add the map
			if (map!=null){
				result.append(",");
				result.append(map.valuesToJSONString());
			}
			
			result.append(", \n\"children\": [");	
			
			if (children!=null && children.size()>0){
				//Collections.sort(children);
				boolean cfirst=true;
				for (Result rel : children.keySet()){
					ContextMap ctx = children.get(rel);
					if (!cfirst)
						result.append(",");
					if (rel.id!=id)
						result.append("\n"+rel.toTreeString(ctx));
					cfirst=false;
				}
			}
			result.append( "]" ) ;
			
			
		}
		result.append("}");
		return result.toString();
	}


	public void setNodeID(String id) {
		this.id=id;

	}

	public String getID(){
		return id;
	}

	
	public String getUpdatedBy() {

		return this.lastUpdater;
	}

	public void setUpdatedBy(String updatedBy) {
		this.lastUpdater=updatedBy;

	}

	public List<Map<String, Object>> getHistoryValues() {
		return history;
	}

	public String getJsonValues() {
		// TODO Auto-generated method stub
		return null;
	}

	public Map<Result, ContextMap> getChildren() {
		return children;
	}

	public void setChildren(Map<Result, ContextMap> children) {
		this.children = children;
	}

	public String toTreeString(ContextMap ctxm) {
		String ret = "";
		if (icon!=null)
			ret+="{\"id\": \"" + getPkey() + "\", \"name\": \"" + label + "\", \"type\": \"\", \"data\": {\"entity\": \"" + this.tablename + "\", \"icon\": \"" + icon + "\", \"$datatype\": \"star\", \"sensitivityValue\": \""+sensitivityValue+"\"}";
		else
			ret+="{\"id\": \"" + getPkey() + "\", \"name\": \"" + label + "\", \"type\": \"\", \"data\": {\"entity\": \"" + this.tablename + "\", \"$datatype\": \"star\", \"sensitivityValue\": \""+sensitivityValue+"\"}";

		if (ctxm!=null)
			ret = ret+ ", \"contextMap\": " + ctxm.valuesToJSONString();

		ret = ret + ", \"values\": [";
		if (values!=null && values.size()!=0){
			for (Object key: values.keySet()){
				//					ret = ret + "{\"key\": \"" + key + "\", \"value\": \"" + (values.get(key).toString().replaceAll("[^0-9|^a-z|^A-Z^|^ |^-|^/^\\\\|^:|^;|^\\.|^@|^\\(|^\\)]", "")) + "\"},";
				ret = ret + "{\"key\": \"" + key + "\", \"value\": \"" + StringEscapeUtils.escapeJava(values.get(key).toString()) + "\"},";
			}
			ret = ret.substring(0,ret.length()-1);
		}

		ret = ret + "], \n\"children\": [";	
		
		if (children!=null && children.size()>0){
			//Collections.sort(children);

			for (Result rel : children.keySet()){
				ContextMap ctx = children.get(rel);
				if (rel.id!=id)
					ret = ret + "\n"+rel.toTreeString(ctx)+",";
			}
			ret = ret.substring(0,ret.length()-1);//removes last ,

		}
		ret = ret + "]";

		ret = ret + ", \n\"history\": [";
		
		if (history!=null && history.size()!=0){
			for (Map<String, Object> histrec : history){
				ret+="{";
				for (String key: histrec.keySet())
					ret = ret + "\""+key+"\": \"" + StringEscapeUtils.escapeJava(histrec.get(key).toString()) + "\",";
				ret = ret.substring(0,ret.length()-1)+"},";
			}
			ret = ret.substring(0,ret.length()-1);
		}
		ret = ret + "]}";
		return ret ;

	}

	public boolean isInternal() {

		return isInternal;
	}

	/**
	 * @return
	 */
	public long getRecordEID() {
	    // TODO Auto-generated method stub
	    return eid;
	}
}
