/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.entity.resolve.types;

import java.io.Serializable;

public class MatchHeight extends MatchBase implements Serializable{

	public MatchHeight(String name, int minwidth, int maxwidth) throws Exception{
		super(name, minwidth, maxwidth);
	}

	public MatchHeight(String name, int minwidth, int maxwidth, int start, int end,int gradient, String keyField)  throws Exception{
		super(name, minwidth, maxwidth, "HEIGHT");
	}

	public MatchHeight(String name){
		super(name);	
	}

	@Override
	public double calculateComparisonScore(Standardized stdBase, Standardized stdComp, boolean isSearch, boolean asContent){   
		return stdBase.compare(stdComp, getRuleAnon(), getRuleCv(), isSearch, asContent);
	}
	
	@Override
	public Standardized standardise(String originalText, String[] words){
		return new StandardHeight(words, this.getRuleAnon(), this.getRuleLookup(), gradient,ruleFunction);
	}
	
	private String padNumber(double num, int noChars){
		String pad="";
		for (int i=0; i<noChars; i++)
			pad =pad +"0";			
		String intt = pad+(new Double(num)).intValue();		
		return intt.substring(intt.length()-noChars-1, intt.length()-1);
	}

	@Override
	public String getKey(String word) {
		try{
			//convert height to a decimal for cm
			if (word.contains("'") || word.contains("f")){
				//convert to 6.16 format - replace f's after foot and ft as it might take precedence 
				word = word.replaceAll("foot||ft||\'", ".").replaceAll("f", ".");
				word = word.replaceAll("\"", "");
				int dotPos = word.indexOf(".");
				double feet=0;
				double inches=0;
				if (dotPos!=-1){
					feet = Double.parseDouble(word.substring(dotPos-1));
					inches = Double.parseDouble(word.substring(dotPos+1,999));
				}
				return padNumber((2.54*(inches+(feet*12))),5);
			}
			if (word.contains("cm")){
				return (word.substring(0,word.indexOf("cm")-1));
			} else if (word.contains("mm")){
				double height = Double.parseDouble(word.substring(0,word.indexOf("mm")-1));
				return padNumber(height*10,5);
			} else if (word.contains("m")){
				double height = Double.parseDouble(word.substring(0,word.indexOf("m")-1));
				return padNumber(height*100,5);
			}
			//no markers so guess based on size of number
			double height = Double.parseDouble(word);
			if (height>2 && height <= 7){
				//likely feet
				height=height*39.37;
			} else if (height <= 2)
			{
				//likely meters
				height = height * 100;
			} else if (height >7 && height <= 200){
				//likely cm
			} else {
				//likely mm
				height = height/10;
			}
			height = (new Double(height / 10)).intValue()*10;
			return padNumber(height,5);
		} catch (Exception e){
			return "";
		}
	}


}
