/**
 *
	Identiza - Fuzzy matching Libraries
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.identiza.wordlist;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.entitystream.monster.db.Document;

public class RuleSet implements Serializable{

	/**
	 * 
	 */
	private static final long serialVersionUID = -7626868195796103299L;
	ControlledVocabularyWordList keyCV;
	AnonymousWordList keyAnon;
	ControlledVocabularyWordList ruleCV;
	AnonymousWordList ruleAnon;
	ControlledVocabularyWordList keyLookup;
	ControlledVocabularyWordList ruleLookup;
	private RuleSet(){
		try{
			keyCV=new ControlledVocabularyWordList();
			keyCV.setChildisParent(false);
			ruleCV=new ControlledVocabularyWordList();
			ruleCV.setChildisParent(false);
			keyAnon=new AnonymousWordList();
			keyAnon.setChildisParent(false);
			ruleAnon=new AnonymousWordList();
			ruleAnon.setChildisParent(false);
			keyLookup=new ControlledVocabularyWordList();
			keyLookup.setChildisParent(true);
			ruleLookup=new ControlledVocabularyWordList();
			ruleLookup.setChildisParent(true);
		} catch (Exception e){
			Logger.getLogger("RuleSet").log(Level.SEVERE, "Failed to create ruleset with an error");
			e.printStackTrace();
		}
	}


	public List<Document> toDocuments(String purpose, boolean custom){
		List<Document> ret = new ArrayList<Document>();
		ret.addAll(toDocuments(keyCV, "EK",purpose, custom));
		ret.addAll(toDocuments(ruleCV, "ER",purpose, custom));
		ret.addAll(toDocuments(keyCV, "NK",purpose, custom));
		ret.addAll(toDocuments(ruleCV, "NR",purpose, custom));
		ret.addAll(toDocuments(keyAnon, "IK",purpose, custom));
		ret.addAll(toDocuments(ruleAnon, "IR",purpose, custom));
		ret.addAll(toDocuments(keyLookup, "TK",purpose, custom));
		ret.addAll(toDocuments(ruleLookup, "TR",purpose, custom));
		return ret;
	}

	private List<Document> toDocuments(WordList wl, String ty, String purpose, boolean custom){
		List<Document> ret = new ArrayList<Document>();
		for (WordObject wo : wl.getList().values()){
			if (ty.endsWith(wo.getRuleType()) && (wo.isCustom()==custom)){
				ret.add(wo.toDocument(purpose));
			} 
		}
		return ret;
	}

	private WordList getForPrefix(String prefix){
		if (prefix.startsWith("IK")){
			return keyAnon;
		} else if (prefix.startsWith("IR")){
			return ruleAnon;
		} else if (prefix.startsWith("ER")){
			return ruleCV;
		} else if (prefix.startsWith("EK")){
			return keyCV;
		} else if (prefix.startsWith("NR")){
			return ruleCV;
		} else if (prefix.startsWith("NK")){
			return keyCV;
		} else if (prefix.startsWith("TK")){
			return keyLookup;
		} else if (prefix.startsWith("TR")){
			return ruleLookup;
		} else return null;
	}

	public static RuleSet createRuleSet(Iterator<com.entitystream.monster.db.Document> docs){
		RuleSet rules = new RuleSet();
		return updateRuleSet(rules, docs);
	}

	public static RuleSet updateRuleSet(RuleSet rules, Iterator<com.entitystream.monster.db.Document> docs){
		//process file
		/*Basis of prefixes = IK, IR, ER, EK, NR, NK, TK, TR
		 * I = Ignore (anon)
		 * E = Equal (cv)
		 * N = Not equal (cv)
		 * T = Lookup parent ie Translate
		 * for either index keys or compariason rules:
		 * R = Rule 
		 * K = Key
		 */
		try {

			String line = null;
			String parentkey = null;
			while (docs.hasNext()){

				Object ruleo= docs.next();
				System.out.println(ruleo);
				Document rule = null;
				if (ruleo instanceof Map)
					rule = new Document((Map)ruleo);
				if (ruleo instanceof Document)
					rule =(Document)ruleo;


				String ruleType=rule.getString("type");
				if (ruleType!=null && ruleType.length()>0) {
					WordList wl = rules.getForPrefix(ruleType);
					boolean same=true;
					boolean ignore=false;
					String parent = rule.getString("parent");
					boolean isCustom = rule.getBoolean("custom", false);
					boolean childIsParent=false;
					if (parent==null || parent.length()==0)
						childIsParent=true;
					if (ruleType.charAt(0)=='N')
						same=false;
					if (ruleType.charAt(0)=='E' || ruleType.charAt(0)=='T')
						same=true;
					if (ruleType.charAt(0)=='I')
						ignore=true;
					if (ruleType.charAt(0)=='T')
						childIsParent=true;

					List<String> items=null;
					if (rule.get("items") instanceof List)
						items = (List<String>)rule.get("items");
					else {
						Logger.getLogger("RuleSet").log(Level.SEVERE, parent + " has invalid items");
					}
					if (items!=null && items.size()>0){
						if (ignore){
							for (String word : items)
								wl.addWord(word, ruleType, isCustom);
						} else {

							if (ruleType.charAt(0)=='T') {
								wl.addWord(parent, parent, false, ruleType, isCustom);

								for (String word: items)
								{
									if (parent!=null && word!=null)
										if (!word.equals(parent)){
											wl.addWord(word, parent, same, ruleType, isCustom);
											if (childIsParent) wl.addWord(parent, word, same, ruleType, isCustom);
										}

								}
							} else {
								for (String word: items)
								
									wl.addWord(word, items.get(0), same, ruleType, isCustom);
							}
						}
					}
				}
			}

		}
		catch (Exception io) {
			io.printStackTrace();
			Logger.getLogger("RuleSet").log(Level.SEVERE, "Rules could not be loaded from DB for the ruleset to load " + io.toString());
		}
		return rules;
	}
	public ControlledVocabularyWordList getKeyControlledVocabulary(){
		return keyCV;
	}
	public ControlledVocabularyWordList getRuleControlledVocabulary(){
		return ruleCV;
	}
	public ControlledVocabularyWordList getKeyLookup(){
		return keyLookup;
	}
	public ControlledVocabularyWordList getRuleLookup(){
		return ruleLookup;
	}
	public AnonymousWordList getKeyAnon(){
		return keyAnon;
	}
	public AnonymousWordList getRuleAnon(){
		return ruleAnon;
	}

	public static RuleSet emptyRuleSet() {
		return new RuleSet();
	}


	public static RuleSet createRuleSet(Document def) {
		RuleSet ruleSet = new RuleSet();

		ruleSet.keyAnon=def.getAsDocument("keyAnon").toObject(AnonymousWordList.class);
		ruleSet.keyCV=def.getAsDocument("keyCV").toObject(ControlledVocabularyWordList.class);
		ruleSet.keyLookup=def.getAsDocument("keyLookup").toObject(ControlledVocabularyWordList.class);
		ruleSet.ruleAnon=def.getAsDocument("ruleAnon").toObject(AnonymousWordList.class);
		ruleSet.ruleCV=def.getAsDocument("ruleCV").toObject(ControlledVocabularyWordList.class);
		ruleSet.ruleLookup=def.getAsDocument("ruleLookup").toObject(ControlledVocabularyWordList.class);

		return ruleSet;
	}
}
