package com.entitystream.monster.geo;
import com.entitystream.monster.db.*;
import java.util.List;
import java.util.Set;
import java.util.LinkedList;
import java.util.Collections;
import java.util.Comparator;
import com.github.davidmoten.geo.*;
public interface GeoType{
     public boolean checkInside(GeoType geo);
     public Point[] getPoints();
     public static GeoType fromDoc(Document document){
            if (document.containsKey("$geoWithin"))
                document=document.getAsDocument("$geoWithin");
                
       
            if (document.containsKey("type")){
              if (document.getString("type").equalsIgnoreCase("Polygon")){
                for (Object coords : document.getList("coordinates")){
                    if (coords instanceof List){
                       List coordinner = (List)coords;
                       Point[] points = new Point[coordinner.size()];
                       int posCo=-1;
                       for (Object point : coordinner ){
                           
                           
                           if (point instanceof List){
                               int pos=-1;
                               double x=0;
                               double y=0;
                               for (Object xy : (List)point ){
                                 if (++pos==0)
                                     x=((Number)xy).doubleValue();
                                 else
                                     y=((Number)xy).doubleValue();
                                 
                               }
                               points[++posCo]=new Point(x,y);
                           }
                       }
                       return new Polygon(points);
                    }
                }
            } else if (document.getString("type").equalsIgnoreCase("LineString")){
               
                Object coords = document.getList("coordinates");
                if (coords instanceof List){
                    List coordinner = (List)coords;
                    Point[] points = new Point[coordinner.size()];
                     int posCo=-1;
                     for (Object point : coordinner ){
                           if (point instanceof List){
                               int pos=-1;
                               double x=0;
                               double y=0;
                               for (Object xy : (List)point ){
                                
                                   if (++pos==0)
                                     x=((Number)xy).doubleValue();
                                   else
                                     y=((Number)xy).doubleValue();
                                 }
                              
                               points[++posCo]=new Point(x,y);
                           }
                      }
                      return new LineString(points);
                }
               
            } else if (document.getString("type").equalsIgnoreCase("Line")){
               
                Object coords = document.getList("coordinates");
                if (coords instanceof List){
                    List coordinner = (List)coords;
                    Point[] points = new Point[coordinner.size()];
                     int posCo=-1;
                     for (Object point : coordinner ){
                           if (point instanceof List){
                               int pos=-1;
                               double x=0;
                               double y=0;
                               for (Object xy : (List)point ){
                                
                                   if (++pos==0)
                                     x=((Number)xy).doubleValue();
                                   else
                                     y=((Number)xy).doubleValue();
                               
                               }
                               points[++posCo]=new Point(x,y);
                           }
                      }
                      return new LineString(points).toLine();
                }
               
            } else if (document.getString("type").equalsIgnoreCase("Point")){
                Object point=document.getList("coordinates");
                if (point instanceof List){
                    int pos=-1;
                    double x=0;
                    double y=0;
                    for (Object xy : (List)point ){
                        
                            if (++pos==0)
                                 x=((Number)xy).doubleValue();
                            else
                                 y=((Number)xy).doubleValue();
                        
                    }
                    return new Point(x,y);
                }
            }
        }
        
        else if (document.containsKey("$center")){
            //two items in a list, one is a list, the other a number
            Point centre=null;
            double radius=0.0d;
            for (Object item : document.getList("$center")){
                if (item instanceof List){
                    int pos=-1;
                    double x=0; double y=0;
                    for (Object xy : (List)item ){
                            if (++pos==0)
                                 x=((Number)xy).doubleValue();
                            else
                                 y=((Number)xy).doubleValue();
                        
                    }
                    centre=new Point(x,y);
                } else {
                    radius=((Number)item).doubleValue();
                }
            }
            return new Centre(centre, radius);
            
        }
        
        return null;
    }
    
    
    public static List<Line> filterIntersectingLines(List<Line> lines, double y) {
    List<Line> results = new LinkedList<Line>();
    for (Line line : lines) {
        if (GeoType.isLineIntersectingAtY(line, y)) {
            results.add(line);
        }
    }
    return results;
}

public static boolean isLineIntersectingAtY(Line line, double y) {
    double minY = Math.min(
        line.getFrom().getY(), line.getTo().getY()
    );
    double maxY = Math.max(
        line.getFrom().getY(), line.getTo().getY()
    );
    return y >= minY && y <= maxY;
}

public static List<Point> calculateIntersectionPoints(List<Line> lines, double y) {
    List<Point> results = new LinkedList<Point>();
    for (Line line : lines) {
        Point result = GeoType.calculateIntersectionPoint(line, y);
        if (result!=null)
            results.add(result);
    }
    return results;
}

public static Point calculateIntersectionPoint(Line line, double y){
    double x = GeoType.calculateLineXAtY(line, y);
    if (x>=0.0d)
        return new Point(x, y);
    return null;
}

public static double calculateLineXAtY(Line line, double y) {
    Point from = line.getFrom();
    double slope = GeoType.calculateSlope(line);
    return from.getX() + (y - from.getY()) / slope;
}

public static double calculateSlope(Line line) {
    Point from = line.getFrom();
    Point to = line.getTo();
    return (to.getY() - from.getY()) / (to.getX() - from.getX());
}

public static void sortPointsByX(List<Point> points) {
    Collections.sort(points, new Comparator<Point>() {
        public int compare(Point p1, Point p2) {
            return Double.compare(p1.getX(), p2.getX());
        }
    });
}

public static boolean calculateInside(List<Point> sortedPoints, Point p) {
    boolean inside = false;
    for (Point point : sortedPoints) {
        if (point.equals(p))
           return true;
        if (p.getX() < point.getX()) {
            break;
        }
        inside = !inside;
    }
    return inside;
}


public static Point calculateLineIntersection(Line l1, Line l2){
        double a1 = l1.getTo().getY() - l1.getFrom().getY();
        double b1 = l1.getFrom().getX() - l1.getTo().getX();
        double c1 = a1 * l1.getFrom().getX() + b1 * l1.getFrom().getY();
 
        double a2 = l2.getTo().getY() - l2.getFrom().getY();
        double b2 = l2.getFrom().getX() - l2.getTo().getX();
        double c2 = a2 * l2.getFrom().getX() + b2 * l2.getFrom().getY();
 
        double delta = a1 * b2 - a2 * b1;
        return new Point((b2 * c1 - b1 * c2) / delta, (a1 * c2 - a2 * c1) / delta);
}

public static Set<String> encodeGeohash(GeoType point, int bits) {
    
    if (point instanceof Point)
        return Collections.singleton(encodeGeohash((Point)point, bits));
    if (point instanceof LineString)
        return encodeGeohash((LineString)point, bits);
    if (point instanceof Polygon)
        return encodeGeohash((Polygon)point, bits);
    if (point instanceof Line)
        return encodeGeohash((Line)point, bits);
    if (point instanceof Centre)
        return encodeGeohash((Centre)point, bits);
    return null;
}


public static Set<String> encodeGeohash(Line point, int bits) {
    //define a box around the geotype/Line and calculate the height and width as 
    double west = Math.min(point.getFrom().getX(), point.getTo().getX());
    double north = Math.max(point.getFrom().getY(), point.getTo().getY());
    double east = Math.max(point.getFrom().getX(), point.getTo().getX());
    double south = Math.min(point.getFrom().getY(), point.getTo().getY());
    int maxL = GeoHash.hashLengthToCoverBoundingBox(north,west, south, east);
    //maxL determines the maximum length of the hash that would cover this box, we can override the index selection because the index will be scanned instead.
    Coverage coverage = GeoHash.coverBoundingBox(north,west, south, east,Math.min(bits, maxL));
    return coverage.getHashes();
}

public static  Set<String> encodeGeohash(LineString linestring, int bits) {
    //define a box around the geotype/Line and calculate the height and width as 
    boolean start=true;
    double west=0,east=0,south=0,north=0;
    for (Point point : linestring.getPoints()){
       if (start){
          west=point.getX();
          east=point.getX();
          north=point.getY();
          south=point.getY();
       } else {
          west = Math.min(west, point.getX());
          north = Math.max(north, point.getY());
          east = Math.max(east, point.getX());
          south = Math.min(south, point.getY());
       }
       start=false;
    }
    int maxL = GeoHash.hashLengthToCoverBoundingBox(north,west, south, east);
    //maxL determines the maximum length of the hash that would cover this box, we can override the index selection because the index will be scanned instead.
    Coverage coverage = GeoHash.coverBoundingBox(north,west, south, east,Math.min(bits, maxL));
    return coverage.getHashes();
}

public static  Set<String> encodeGeohash(Polygon poly, int bits) {
    boolean start=true;
    double west=0,east=0,south=0,north=0;
    for (Point point : poly.getPoints()){
       if (start){
          west=point.getX();
          east=point.getX();
          north=point.getY();
          south=point.getY();
       } else {
          west = Math.min(west, point.getX());
          north = Math.max(north, point.getY());
          east = Math.max(east, point.getX());
          south = Math.min(south, point.getY());
       }
       start=false;
    }
    int maxL = GeoHash.hashLengthToCoverBoundingBox(north,west, south, east);
    //maxL determines the maximum length of the hash that would cover this box, we can override the index selection because the index will be scanned instead.
    Coverage coverage = GeoHash.coverBoundingBox(north,west, south, east,Math.min(bits, maxL));
    return coverage.getHashes();
}

public static  Set<String> encodeGeohash(Centre center, int bits) {
    double west=center.getFrom().getX()-center.getRadius();
    double east=center.getFrom().getX()+center.getRadius();
    double north=center.getFrom().getY()+center.getRadius();
    double south=center.getFrom().getY()-center.getRadius();
    int maxL = GeoHash.hashLengthToCoverBoundingBox(north,west, south, east);
    //maxL determines the maximum length of the hash that would cover this box, we can override the index selection because the index will be scanned instead.
    Coverage coverage = GeoHash.coverBoundingBox(north,west, south, east,Math.min(bits, maxL));
    return coverage.getHashes();
}

public static  String encodeGeohash(Point point, int bits) {
    return GeoHash.encodeHash(point.getX(), point.getY(), bits);
}


}