/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.File;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.SimpleBindings;

import org.apache.commons.lang3.text.StrLookup;
import org.apache.commons.lang3.text.StrSubstitutor;

import com.entitystream.identiza.metadata.IdentizaSettings;
import com.entitystream.monster.external.ExternalData;
import com.entitystream.monster.external.RestAPI;

import com.google.gson.Gson;
import com.google.gson.JsonPrimitive;

public class AggregateIterable implements Iterable<Document> {
    private List<Document> pipeline;
    private Document options;
    private DBCursor lazy = null;
    private ICollection collection;
    private List<Document> in;
    private Map<String, PrintWriter> peekers = new HashMap<String, PrintWriter>();
    private boolean allowDiskUse = false;
    private Document context;

    public AggregateIterable(ICollection c, List<Document> pipeline, Document options) {
	this.pipeline = pipeline;
	this.options = options;
	this.collection = c;
    }

    public AggregateIterable(ICollection c, List<Document> in, List<Document> pipeline, Document options) {
	this.pipeline = pipeline;
	this.options = options;
	this.collection = c;
	this.in = in;
    }

    private Stream aggregateStage(Document pipeStage, Stream<Document> in, Document options) {
	ICollection collection = this.collection;
	Document stageoptions = options;
	if (pipeStage.containsKey("options")) {
	    stageoptions = pipeStage.getAsDocument("options");
	    pipeStage.remove("options");
	}
	if (stageoptions.containsKey("node")) {
	    int nodeid = 0;
	    if (stageoptions.get("node") instanceof String
		    && !stageoptions.getString("node").equalsIgnoreCase("default"))
		nodeid = Integer.parseInt(stageoptions.getString("node"));
	    if (stageoptions.get("node") instanceof Number)
		nodeid = stageoptions.getInteger("node");
	    if (nodeid != DBServer.getNodeNum()) // really fing important!
		collection = this.collection.getReplicaConnections().get(nodeid);
	}

	if (pipeStage.containsKey("$match")) {
	    return collection.match(processContext(pipeStage.getAsDocument("$match")), in, stageoptions);

	}
	if (pipeStage.containsKey("$find")) {
	    return collection.match(processContext(pipeStage.getAsDocument("$find")), in, stageoptions);
	} else if (pipeStage.containsKey("$findTasks")) {
	    return collection.findTasks(processContext(pipeStage.getAsDocument("$findTasks"))).stream();
	} else if (pipeStage.containsKey("$analyze")) {
	    return collection.analyse(processContext(pipeStage.getAsDocument("$analyze")), in, stageoptions);
	}  else if (pipeStage.containsKey("$deltas")) {
	    return collection.getDeltas();

	} else {

	    if (in == null)
		in = collection.findStream(new Document());
	    if (pipeStage.containsKey("$limit")) {
		return collection.limit(pipeStage.getLong("$limit"), in, stageoptions);
	    }  else if (pipeStage.containsKey("$validate")) {
		return collection.validate(pipeStage.getString("$validate"), in, stageoptions);
	    }  else if (pipeStage.containsKey("$task")) {
		return collection.task(processContext(pipeStage.getAsDocument("$task")), in, stageoptions);
	    }  else if (pipeStage.containsKey("$applyType")) {
		return applyType(processContext(pipeStage.getAsDocument("$applyType")), mappings(in, pipeStage.getAsDocument("$applyType")));
	    }  else if (pipeStage.containsKey("$lookup")) {
		return collection.lookup(processContext(pipeStage.getAsDocument("$lookup")), in, stageoptions);
	    } else if (pipeStage.containsKey("$saveContext")) {
		if (pipeStage.get("$saveContext") instanceof Map) {
		    context=pipeStage.getAsDocument("$saveContext");
		} else {
		    Optional<Document> newcontext=in.findFirst();
		    if (newcontext.isPresent())
		       context=newcontext.get();
		}
		return Stream.empty();
	    } else if (pipeStage.containsKey("$join")) {
		return collection.join(processContext(pipeStage.getAsDocument("$join")), in, stageoptions);
	    } else if (pipeStage.containsKey("$minus")) {
		return collection.minus(processContext(pipeStage.getAsDocument("$minus")), in, stageoptions);
	    } else if (pipeStage.containsKey("$skip")) {
		return collection.skip(pipeStage.getLong("$skip"), in, stageoptions);
	    } else if (pipeStage.containsKey("$group")) {
		return collection.group(processContext(pipeStage.getAsDocument("$group")), in, stageoptions);
	    } else if (pipeStage.containsKey("$out")) {
		return collection.out(pipeStage.getString("$out"), in, stageoptions);
	    } else if (pipeStage.containsKey("$writeRel")) {
		return collection.writeRel(processContext(pipeStage.getAsDocument("$writeRel")), in, stageoptions);
	    } else if (pipeStage.containsKey("$sort")) {
		return collection.sort(processContext(pipeStage.getAsDocument("$sort")), in, stageoptions);
	    } else if (pipeStage.containsKey("$bucket")) {
		return collection.bucket(processContext(pipeStage.getAsDocument("$bucket")), in, stageoptions);
	    } else if (pipeStage.containsKey("$fuzzySearch")) {
		return collection.fuzzySearch(pipeStage.getString("$fuzzySearch"), in, stageoptions);
	    } else if (pipeStage.containsKey("$fuzzy")) {
		return collection.fuzzyMatch(processContext(pipeStage.getAsDocument("$fuzzy")), in, stageoptions);
	    } else if (pipeStage.containsKey("$unwind")) {
		return collection.unwind(processContext(pipeStage.getAsDocument("$unwind")), in, stageoptions);
	    } else if (pipeStage.containsKey("$filter")) {
		return ICollection.filter(processContext(pipeStage.getAsDocument("$filter")), in, stageoptions);
	    } else if (pipeStage.containsKey("$spinOut")) {
		return collection.spinOut(processContext(pipeStage.getAsDocument("$spinOut")), in, stageoptions);
	    } else if (pipeStage.containsKey("$getRelated")) {
		return collection.getRelated(processContext(pipeStage.getAsDocument("$getRelated")), in, stageoptions);
	    } else if (pipeStage.containsKey("$rematch")) {
		return collection.rematch(processContext(pipeStage.getAsDocument("$rematch")), in, stageoptions);
	    } else if (pipeStage.containsKey("$classifierBuild")) {
		return collection.classifierBuild(processContext(pipeStage.getAsDocument("$classifierBuild")), in, stageoptions);
	    } else if (pipeStage.containsKey("$classifierPredict")) {
		return collection.classifierPredict(processContext(pipeStage.getAsDocument("$classifierPredict")), in, stageoptions);
	    } else if (pipeStage.containsKey("$arrf")) {
		return collection.arrf(processContext(pipeStage.getAsDocument("$arrf")), in, stageoptions);
	    } else if (pipeStage.containsKey("$classifierTree")) {
		return collection.classifierTree(processContext(pipeStage.getAsDocument("$classifierTree")), in, stageoptions);
	    } else if (pipeStage.containsKey("$coerce")) {
		return collection.coerce(processContext(pipeStage.getAsDocument("$coerce")), in, stageoptions);
	    } else if (pipeStage.containsKey("$compare")) {
		return collection.compare(processContext(pipeStage.getAsDocument("$compare")), in, stageoptions);
	    } else if (pipeStage.containsKey("$count")) {
		return collection.count(processContext(pipeStage.getAsDocument("$count")), in, stageoptions);
	    } else if (pipeStage.containsKey("$first")) {
		return collection.first(processContext(pipeStage.getAsDocument("$first")), in, stageoptions);
	    } else if (pipeStage.containsKey("$last")) {
		return collection.last(processContext(pipeStage.getAsDocument("$last")), in, stageoptions);
	    } else if (pipeStage.containsKey("$between")) {
		return collection.between(processContext(pipeStage.getAsDocument("$between")), in, stageoptions);
	    } else if (pipeStage.containsKey("$cluster")) {
		return collection.cluster(processContext(pipeStage.getAsDocument("$cluster")), in, stageoptions);
	    } else if (pipeStage.containsKey("$project")) {
		return collection.project(processContext(pipeStage.getAsDocument("$project")), in, stageoptions);
	    } else if (pipeStage.containsKey("$restGet")) {
		return RestAPI.executeGet(processContext(pipeStage.getAsDocument("$restGet")), in, stageoptions);
	    } else if (pipeStage.containsKey("$readFile")) {
		return ExternalData.readFile(processContext(pipeStage.getAsDocument("$readFile")), in, stageoptions);
	    } else if (pipeStage.containsKey("$writeFile")) {
		return ExternalData.writeFile(processContext(pipeStage.getAsDocument("$writeFile")), in, stageoptions);
	    } else if (pipeStage.containsKey("$log")) {
		return in.peek(doc -> System.out.println(doc.toJson()));
	    } else if (pipeStage.containsKey("$peek")) {
		return in.peek(doc -> peekTo(pipeStage.getString("$peek"), doc));
	    } else if (pipeStage.containsKey("$readJDBC")) {
		return ExternalData.readJDBC(processContext(pipeStage.getAsDocument("$readJDBC")), in, stageoptions);
	    } else if (pipeStage.containsKey("$writeActiveMQ")) {
		return ExternalData.writeActiveMQ(processContext(pipeStage.getAsDocument("$writeActiveMQ")), in, stageoptions);
	    } else if (pipeStage.containsKey("$readActiveMQ")) {
		return ExternalData.readActiveMQ(processContext(pipeStage.getAsDocument("$readActiveMQ")), in, stageoptions);
	    } else if (pipeStage.containsKey("$distinct")) {
		return in.collect(Collectors.toSet()).stream();
	    } else if (pipeStage.containsKey("$empty")) {
		return Stream.empty();
	    } else if (pipeStage.containsKey("$new")) {
		return Stream.of(processContext(pipeStage.getAsDocument("$new")));
	    } else if (pipeStage.containsKey("$evaluate")) {
		return collection.evaluate(processContext(pipeStage.getAsDocument("$evaluate")), in, stageoptions);
	    } else if (pipeStage.containsKey("$union")) {
		Document innerStage = processContext(pipeStage.getAsDocument("$union"));
		Document filter = new Document();
		if (innerStage.containsKey("filter"))
		    filter = innerStage.getAsDocument("filter");
		if (innerStage.containsKey("with"))
		    return Stream.concat(in,
			    collection.getDatabase().getCollection(innerStage.getString("with")).find(filter).stream());
		else
		    return in;
	    } else if (pipeStage.containsKey("$switch")) {
		Document innerStage = processContext(pipeStage.getAsDocument("$switch"));
		if (innerStage.containsKey("to"))
		    collection = (Collection) collection.getDatabase().getCollection(innerStage.getString("to"));
		if (innerStage.containsKey("filter")) {
		    if (innerStage.get("filter") instanceof Map)
		       return collection.find(innerStage.getAsDocument("filter")).stream();
		    else if (innerStage.get("filter") instanceof String)
			return collection.find(Document.parse(innerStage.getString("filter"))).stream();
		}
		else
		    return collection.find(new Document()).stream();
	    } else if (pipeStage.containsKey("$split")) {
		try {
		    Object innerPipelineText = pipeStage.get("$split");
		    List<Document> list = null;
		    if (innerPipelineText instanceof String) {
			list = (List<Document>) Document.parseListOrDocument((String) innerPipelineText);
			if (list.size() > 0) {

			    List<Document> olist = new ArrayList();
			    for (Object o : list) {
				if (o instanceof String || o instanceof JsonPrimitive) {
				    // then it needs parsing
				    o = o.toString();
				    List<Document> pipeline = new ArrayList<Document>();
				    String[] aggString = ((String) o).split("\\|");
				    for (String argString : aggString) {
					argString = argString.replaceAll("&pipe;", "|");
					if (argString.indexOf("(") != -1) {
					    String command = argString.substring(0, argString.indexOf("("));
					    if (command.startsWith("\""))
						command = command.substring(1);
					    argString = argString.substring(argString.indexOf("(") + 1);
					    if (argString.indexOf(")") != -1) {
						argString = argString.substring(0, argString.lastIndexOf(")"));
					    }
					    if (argString.startsWith("{") && argString.indexOf("\\") > -1)
						argString = argString.replaceAll(Matcher.quoteReplacement("\\"), "");
					    // decode argString to Documents
					    List args = null;
					    if (argString.length() != 0 && (argString.trim().startsWith("{"))) {
						Document argso = Document.parse(argString);
						pipeline.add(new Document(
							"$" + command.trim().replaceAll("\\\"|'|\\\\|", ""), argso));
					    } else {
						// string
						pipeline.add(new Document(
							"$" + command.trim().replaceAll("\\\"|'|\\\\|", ""),
							argString.trim().replaceAll("\\\"|'|\\\\|", "")));
					    }

					}
				    }
				    olist.add(new Document("pipeline", pipeline));
				} else if (o instanceof Document)
				    olist.add((Document) o);
			    }
			    list = olist;
			}
		    } else
			list = pipeStage.getList("$split");
		    Map<Document, List<Document>> ret = collection.split(list, in, stageoptions);
		    Stream<Document> out = null;
		    for (Document path : ret.keySet()) {
			List<Document> innerPipeline = path.getList("pipeline");
			List<Document> innerIn = ret.get(path);
			Document innerOptions = path.getAsDocument("options");
			AggregateIterable aggInner = new AggregateIterable(collection, innerIn, innerPipeline,
				innerOptions);
			if (out == null)
			    out = aggInner.evaluateStream();
			else
			    out = Stream.concat(out, aggInner.evaluateStream());
		    }
		    return out;
		} catch (Exception e) {
		    e.printStackTrace();
		}
	    }
	}
	return in;

    }

    /**
     * @param in2
     * @return
     */
    private Stream<Document> mappings(Stream<Document> in, Document stage) {
	// evaluate mappings
	if (stage.containsKey("mappings")) {
	   // get the metadata and pass it to the collection to preprocess
	   return collection.script(stage.getAsDocument("mappings"), in, stage.getAsDocument("metadata"));
	} else return in;
    }

    /**
     * @param processContext
     * @param in2
     * @return
     */
    
    ///each document type should have an entry in the actions to convert to that type - each entry holds metadata to mapp to....
    private Stream<Document> applyType(Document definition, Stream<Document> in) {
	return in.map(doc -> {
	   return doc;
	});
	
    }

    /**
     * @param asDocument
     * @return
     */
    private Document processContext(Document indoc) {
	if (context==null && indoc==null)
	    return indoc; //quick exit
	
	
	StrSubstitutor sub = new StrSubstitutor(new DocumentStrLookup(context) );
	
	//System.out.println("in:"+indoc.toJson());
	for (Object key : indoc.keySet()) {
	    Object value=indoc.get((String)key);
	    if (value instanceof String) {
		if (((String)value).contains("${")) {
		    value=sub.replace((String)value);
		    indoc.append((String) key, value);
		}
	    } else {
		if (value instanceof Map) {
		    value=processContext(new Document(value));
		    indoc.append((String) key, value);
		} else if (value instanceof List) {
		    List newList = new ArrayList();
		    for (Object listVal : (List)value) {
			if (listVal instanceof String) {
			    if (((String)listVal).contains("${")) {
				listVal=sub.replace((String)listVal,context);
			    }
			} else if (listVal instanceof Map) {
			    listVal=processContext(new Document(listVal));
			} 
			newList.add(listVal);    
		    }
		    indoc.append((String) key, newList);
		}
	    }
	    
	}
	//System.out.println("out:"+indoc.toJson());
	return indoc;
    }

    public Document getMetadata() {
	Stream<Document> output = null;
	if (in != null)
	    output = in.stream();

	for (Document pipeStage : pipeline) {
	    if (    !pipeStage.containsKey("$out") 
		 && !pipeStage.containsKey("$writeRel")
		 && !pipeStage.containsKey("$writeFile") 
		 && !pipeStage.containsKey("$writeActiveMQ")
		 && !pipeStage.containsKey("$snowFlakeStage") 
		 && !pipeStage.containsKey("$soap"))
		output = aggregateStage(pipeStage, output, options);

	}
	Document outputD = new Document();
	if (output!=null)
	output.limit(1000).forEach(doc -> {
	    outputD.putAll(doc);
	});
	return outputD;
    }

    /**
     * @param string
     * @param doc
     * @return
     */
    private synchronized Document peekTo(String where, Document doc) {

	boolean first = false;
	where = where.replaceAll("\\\"|'", "");
	try {

	    if (!peekers.containsKey(where)) {

		File dest = new File(IdentizaSettings.getApplicationRootPath("tmp") + File.separator + where);

		File fpath = new File(
			dest.getAbsolutePath().substring(0, dest.getAbsolutePath().lastIndexOf(File.separator)));
		if (!fpath.exists())
		    fpath.mkdirs();

		peekers.put(where, new PrintWriter(dest));
		first = true;
	    }
	} catch (Exception e) {
	    e.printStackTrace();
	}
	PrintWriter peek = peekers.get(where);
	if (!first)
	    peek.println(",");
	peek.print(doc.toJson());
	peek.flush();

	return doc;
    }


    
   
    
    private DBCursor evaluate() {
	if (lazy == null) {

	    Stream<Document> output = null;
	    if (in != null)
		output = in.stream();
	    for (Document pipeStage : pipeline) {
		output = aggregateStage(pipeStage, output, options);

	    }
	    if (output != null)
		lazy = new DBCursor((List<Document>) output.collect(Collectors.toList()));
	}
	return lazy;
    }

    protected Stream<Document> evaluateStream() {

	Stream<Document> output = null;
	if (in != null)
	    output = in.stream();

	for (Document pipeStage : pipeline) {

	    output = aggregateStage(pipeStage, output, options);

	}
	return output;
    }

    @Override
    public Iterator<Document> iterator() {
	return evaluate();
    }

    public AggregateIterable allowDiskUse(boolean allow) {
	this.allowDiskUse = allow;
	return this;
    }

    public Document first() {
	return evaluate().first();
    }
    
    
    private class DocumentStrLookup extends StrLookup {
	private Document doc;
	public DocumentStrLookup(Document doc) {
	    this.doc=doc;
	}
	@Override
	public String lookup(String key) {
	    return (String) doc.getProjection(key);
	}
	
    }
}
