/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.script.Bindings;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.script.SimpleBindings;


import org.infinispan.commons.util.concurrent.ConcurrentHashSet;
import org.mapdb.Serializer;

import com.entitystream.identiza.db.Node;
import com.entitystream.identiza.entity.resolve.match.Indexable;
import com.entitystream.identiza.entity.resolve.match.MatchRecordInterface;
import com.entitystream.identiza.entity.resolve.match.MatchRule;
import com.entitystream.identiza.entity.resolve.match.Matchable;
import com.entitystream.identiza.entity.resolve.metadata.IIndex;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumn;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.ITableColumn;
import com.entitystream.identiza.entity.resolve.metadata.Index;
import com.entitystream.identiza.entity.resolve.metadata.Purpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap;
import com.entitystream.identiza.entity.resolve.metadata.Rule;
import com.entitystream.identiza.entity.resolve.metadata.SchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.Table;
import com.entitystream.identiza.entity.resolve.types.Standardized;
import com.entitystream.identiza.metadata.IdentizaSettings;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.identiza.wordlist.RuleSet;
import com.entitystream.identiza.wordlist.WordObject;
import com.entitystream.monster.db.CollectionLocal.BucketRange;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonIOException;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;

import weka.classifiers.AbstractClassifier;
import weka.classifiers.Classifier;
import weka.classifiers.evaluation.Evaluation;
import weka.classifiers.evaluation.NominalPrediction;
import weka.core.Drawable;
import weka.core.FastVector;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.SerializedObject;
import weka.core.converters.JSONLoader;
import weka.core.json.JSONInstances;
import weka.core.json.JSONNode;

public class Collection extends Indexable implements Serializable, ICollection, Matchable{


    private static final long serialVersionUID = -4586850884783086753L;
    private Map<Integer, ICollection> replicaConnections = new HashMap<Integer, ICollection>();
    private List<String> replicaSet;

    private Database parent;
    //private ISchemaMeta schDoc;
    private Document definition;
    private CollectionLocal localCollection;

    private boolean explaining=false;
    private transient Logger logger;
    private int nodeNum;
    public Collection() {
	logger = Logger.getAnonymousLogger();
    }

    public Collection(Database db, Document collDoc, boolean initialised) {
	logger = Logger.getAnonymousLogger();
	this.parent=db;
	definition=collDoc;
	localCollection = new CollectionLocal(this, collDoc, false);
	this.nodeNum=parent.getNodeNum();
	replicaConnections.put(nodeNum, localCollection);



	if (collDoc.containsKey("Definition")) {

	    this.schDoc = SchemaMeta.createSchemaMeta(collDoc.getAsDocument("Definition"));
	    initialiseCollection();
	}

	if (parent.getReplicaSet()!=null) {
	    this.replicaSet=parent.getReplicaSet();
	    if (replicaSet.size()>0) {
		for (String r : replicaSet) {
		    try {
			MonsterClient client = new MonsterClient(r);
			CollectionRemote collection = new CollectionRemote(client, parent.getName(), collDoc, initialised);
			if (client.isConnected()) {
			    if (!replicaConnections.containsKey(client.getNodeNum())) {
				replicaConnections.put(client.getNodeNum(), collection);
				System.out.println("Node #"+client.getNodeNum() + " is up!");
			    }
			    else
				System.err.println("Node #"+client.getNodeNum() + " is already allocated, you need to change the -n settings on the node");
			}  else
			    System.err.println("Node #"+client.getNodeNum() + " is not available, restart it and this node afterwards");
		    } catch (Exception e) {
			e.printStackTrace();
		    }
		}
	    }

	} 
    }


    public Collection(Database parent, Document collDoc) {
	this(parent, collDoc, true);

    }


    private ISchemaMeta getSchDoc() {
	if (schDoc==null) {
	    if (definition.containsKey("Definition"))	
		schDoc= SchemaMeta.createSchemaMeta(definition.getAsDocument("Definition"));
	    else {
		schDoc=SchemaMeta.createSchemaMeta(new Document());
		definition.append("Definition", schDoc.toDocument());
	    }
	}
	return schDoc;
    }


    @Override
    public void createIndex(Document fields, Document options) {
	for (ICollection collection : replicaConnections.values()) {
	    collection.createIndex(fields, options);
	}

    }

    @Override
    public void createUniqueIndex(Document fields) {
	for (ICollection collection : replicaConnections.values()) {
	    collection.createUniqueIndex(fields);
	}

    }

    @Override
    public void createIndex(Document document) {
	for (ICollection collection : replicaConnections.values()) {
	    collection.createIndex(document);
	}

    }

    @Override
    public DBCursor find(Document filter) {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.find(filter).stream().forEach(doc -> {
		if (doc!=null)
		    set.put(doc.getString("_id"),doc);
	    });
	});
	return new DBCursor(set.values());
    }


    @Override
    public Stream<Document> findStream(Document filter) {
	return replicaConnections
		.values()
		.stream()
		.map(collection -> collection.findStream(filter))
		.flatMap(s -> s);
    }




    @Override
    public DBCursor find() {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.find().stream().forEach(doc -> {
		if (doc!=null)
		    set.put(doc.getString("_id"),doc);
	    });
	});
	return new DBCursor(set.values());
    }



    @Override
    public Document save(Document doc) {
	if (!doc.containsKey("_id")) {
	    String _id = UUID.randomUUID().toString();
	    doc.append("_id", _id);
	}
	Document retdoc = null;
	for (Integer key : replicaConnections.keySet()) {
	    ICollection collection = replicaConnections.get(key);
	    //System.out.println("Sending to Connection: "+key);
	    retdoc=collection.save(doc);

	}


	return retdoc;
    }


    @Override
    public Document insertOne(Document doc) {
	if (!doc.containsKey("_id")) {
	    String _id = UUID.randomUUID().toString();
	    doc.append("_id", _id);
	}
	Document retdoc = null;
	for (ICollection collection : replicaConnections.values()) {
	    retdoc=collection.insertOne(doc);
	}
	return retdoc;
    }

    @Override
    public long count(Document query) {

	if (replicaConnections.size()==1) {
	    AtomicLong count = new AtomicLong(0);
	    replicaConnections.values().parallelStream().forEach(collection -> {
		count.addAndGet(collection.count(query));
	    });
	    return count.get();
	} else {
	    return find(query).count();
	}


    }

    @Override
    public Document findOneAndReplace(Document filter, Document replacement, Document options) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	if (options.getBoolean("upsert", false))
	    options.append("_id", UUID.randomUUID().toString());
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.findOneAndReplace(filter, replacement, options);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;

    }

    @Override
    public Document findOneAndReplace(Document filter, Document replacement) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.findOneAndReplace(filter, replacement);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public Document findOneAndUpdate(Document filter, Document amendments) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.findOneAndUpdate(filter, amendments);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public Document findOneAndUpdate(Document filter, Document amendments, Document options) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	if (options.getBoolean("upsert", false))
	    options.append("_id", UUID.randomUUID().toString());
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.findOneAndUpdate(filter, amendments, options);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public Document updateOne(Document filter, Document amendments) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.updateOne(filter, amendments);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public Document updateOne(Document filter, Document amendments, Document options) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	if (options.getBoolean("upsert", false))
	    options.append("_id", UUID.randomUUID().toString());
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.updateOne(filter, amendments, options);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public int updateMany(Document filter, Document amendments, Document options) {
	AtomicInteger count = new AtomicInteger(0);
	replicaConnections.values().parallelStream().forEach(collection -> {
	    count.addAndGet(collection.updateMany(filter,amendments,options));
	});

	return count.get();

    }

    @Override
    public int updateMany(Document filter, Document amendments) {
	AtomicInteger count = new AtomicInteger(0);
	replicaConnections.values().parallelStream().forEach(collection -> {
	    count.addAndGet(collection.updateMany(filter,amendments,new Document()));
	});

	return count.get();

    }

    @Override
    public int insertMany(List<Document> listRecs) {
	for (Document doc : listRecs) {
	    if (!doc.containsKey("_id")) {
		String _id = UUID.randomUUID().toString();
		doc.append("_id", _id);
	    }
	}


	AtomicInteger count = new AtomicInteger(0);
	replicaConnections.values().parallelStream().forEach(collection -> {
	    count.addAndGet(collection.insertMany(listRecs));
	});

	return count.get();

    }

    @Override
    public Document findOneAndDelete(Document filter) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.findOneAndDelete(filter);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public Document deleteOne(Document filter) {
	ConcurrentHashMap<String,Document> cursor = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document _cursor=collection.findOneAndDelete(filter);
	    if (_cursor!=null)
		cursor.put(_cursor.getString("_id"),_cursor);
	});
	if (cursor.size()>0)
	    return cursor.values().iterator().next();
	else return null;
    }

    @Override
    public int deleteMany(Document filter) {
	AtomicInteger count = new AtomicInteger(0);
	replicaConnections.values().parallelStream().forEach(collection -> {
	    count.addAndGet(collection.deleteMany(filter));
	});
	if (filter.keySet().size()==0 && schDoc!=null) {
	    getRelCollection().deleteMany(new Document());
	    getTaskCollection().deleteMany(new Document());
	}
	return count.get();
    }


    protected String getName() {
	return getDefinition().getString("Name");
    }

    /**
     * @return
     */
    protected Collection getRelCollection() {
	return (Collection) getDatabase().getRelCollection(getDefinition().getString("Name")+"_RELS");
    }

    /**
     * @return
     */
    protected Collection getTaskCollection() {
	return (Collection) getDatabase().getTaskCollection(getDefinition().getString("Name")+"_TASKS");
    }


    @Override
    public Stream<Document> aggregate(List<Document> pipeline, Document options) {
	return new AggregateIterable(this, pipeline, options).evaluateStream();
    }



    @Override
    public Stream<Document> aggregate(List<Document> pipeline) {
	return new AggregateIterable(this, pipeline, new Document()).evaluateStream();
    }

    @Override
    public Stream<Document> aggregate(List<Document> in, List<Document> pipeline) {
	return new AggregateIterable(this, in, pipeline, new Document()).evaluateStream();
    }

    @Override
    public Document aggregateMetadata(List<Document> in, List<Document> pipeline) {
	return new AggregateIterable(this, in, pipeline, new Document()).getMetadata();
    }

    @Override
    public Document aggregateMetadata(List<Document> pipeline) {
	return new AggregateIterable(this, pipeline, new Document()).getMetadata();
    }

    @Override
    public DBCursor listIndexes() {
	return localCollection.listIndexes();
    }


    @Override
    public void rebuildIndex(String name) {
	for (ICollection collection : replicaConnections.values()) {
	    collection.rebuildIndex(name);
	}
    }

    @Override
    public void dropIndex(String name) {
	for (ICollection collection : replicaConnections.values()) {
	    collection.dropIndex(name);
	}

    }

    @Override
    public int saveMany(List<Document> records) {
	//generate ID
	boolean isreplicated=this.localCollection.getReplicaType()==ReplicaType.Replicated;
	for (Document doc : records){
	    if (!doc.containsKey("_id")) {
		//add it
		Object _id = UUID.randomUUID().toString();
		doc.append("_id", _id);
	    } 
	    //docs.add( doc );
	}
	AtomicInteger count = new AtomicInteger(0);
	replicaConnections.keySet().parallelStream().forEach(collectionID -> {
	    ICollection collection = replicaConnections.get(collectionID);
	    int inc = collection.saveMany(records);
	    if (!isreplicated)
		count.addAndGet(inc);
	    else
		count.set(inc);
	});

	return count.get();
    }



    @Override
    public void disconnect() throws Exception {
	for (ICollection collection : replicaConnections.values()) {
	    collection.disconnect();
	}

    }

    @Override
    public void setExplaining(boolean explain) {
	this.explaining=explain;
	for (ICollection collection : replicaConnections.values()) {
	    collection.setExplaining(explain);
	}

    }

    @Override
    public DBCursor find(Document filter, Document explain) {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.find(filter, explain).stream().forEach(doc -> {
		if (doc!=null)
		    set.put(doc.getString("_id"),doc);
	    });
	});
	return new DBCursor(set.values());
    }


    public DBCursor executeCommand(String command, User user, Session session) {
	long start=System.currentTimeMillis();
	DBCursor ret = Container.executeCommand(this, command, user,session);
	if (System.currentTimeMillis()-start >10000) {
	    String c=command;
	    if (c.length()>40)
		c=c.substring(0, 37)+"...";
	    System.out.println("Command ("+c+") completed in " + (System.currentTimeMillis()-start)+"ms");
	}
	return ret;
    }


    @Override
    public void drop() {
	for (ICollection collection : replicaConnections.values()) {
	    collection.drop();
	}

    }

    @Override
    public <TResult> Iterable<TResult> distinct(String fieldName, Class<TResult> resultClass) {
	HashSet<TResult> set = new HashSet<TResult>();
	for (ICollection collection : replicaConnections.values()) {
	    Iterator<TResult> it = collection.distinct(fieldName, resultClass).iterator();
	    while(it.hasNext())
		set.add(it.next());
	}
	return set;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#distinctRelationship(java.lang.String, java.lang.Class)
     */
    @Override
    public <TResult> Iterable<TResult> distinctRelationship(String fieldName, Class<TResult> resultClass) {
	HashSet<TResult> set = new HashSet<TResult>();
	for (ICollection collection : replicaConnections.values()) {
	    Iterator<TResult> it = collection.distinctRelationship(fieldName, resultClass).iterator();
	    while(it.hasNext())
		set.add(it.next());
	}
	return set;
    }




    @Override
    public Document getStandardised(String indexName, String id) {
	HashSet<Document> set = new HashSet();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document interrim = collection.getStandardised(indexName, id);
	    if (interrim!=null)
		set.add(interrim);
	});
	if (set.size()>0)
	    return set.iterator().next();
	else 
	    return null;


    }


    @Override
    public Document getIndexPrefixSubMap(String indexName, String key, boolean b) {
	Document set = new Document();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    Document interrim = collection.getIndexPrefixSubMap(indexName, key, b);
	    set.putAll(interrim);
	});

	return set;
    }

    @Override
    public Document getDocument(Object pointer) {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	Optional<Document> ret = replicaConnections.values().parallelStream().map(collection -> {
	    return collection.getDocument(pointer);
	}).filter(f->f!=null).findFirst();

	if (ret.isPresent())
	    return (Document)ret.get();
	else
	    return null;
    }





    @Override
    public Document keyCount(String indexName) {
	ConcurrentHashSet<Document> mapout = new ConcurrentHashSet<Document>();
	if (localCollection.getReplicaType()==ReplicaType.Replicated)
	    return localCollection.keyCount(indexName);
	this.replicaConnections.values().parallelStream().forEach(collection -> {
	    Document o=collection.keyCount(indexName);
	    if (o!=null)
		mapout.add(o);
	});
	//aggregate
	Document out = new Document();
	for (Document in : mapout)
	    for (Object inKey : in.keySet())
		if (out.containsKey(inKey))
		    out.put(inKey, ((long)out.get(inKey))+((long)in.get(inKey)));
		else
		    out.put(inKey, in.get(inKey));
	return out;
    }


    @Override
    public Document conceptCount(String indexName) {
	ConcurrentHashSet<Document> mapout = new ConcurrentHashSet<Document>();
	if (localCollection.getReplicaType()==ReplicaType.Replicated)
	    return localCollection.conceptCount(indexName);

	this.replicaConnections.values().parallelStream().forEach(collection -> {
	    mapout.add(collection.conceptCount(indexName));
	});
	//aggregate
	Document out = new Document();
	for (Document in : mapout)
	    for (Object inKey : in.keySet())
		if (out.containsKey(inKey))
		    out.put(inKey, ((long)out.get(inKey))+((long)in.get(inKey)));
		else
		    out.put(inKey, in.get(inKey));
	return out;
    }

    public ICollection getLocalCollection() {
	return localCollection;
    }

    @Override
    public void saveTable(Document def) {
	for (ICollection collection : replicaConnections.values()) {
	    collection.saveTable(def);
	}
	getSchDoc().addTable(def.toObject(Table.class));
	updateDefinition(getSchDoc().toDocument());
    }

    public void setAutoMatch(boolean a) {
	if (schDoc!=null) {
	    for (ICollection collection : replicaConnections.values()) {
		collection.setAutoMatch(a);

		schDoc.setAutoMatch(a);
		updateDefinition(getSchDoc().toDocument());
	    }

	}
    }


    public void addTrigger(String name) {
	definition.append("Trigger", name);
	parent.storeCollection(getCollectionName(), this);
	for (ICollection collection : replicaConnections.values()) {
	    collection.addTrigger(name);
	}
    }



    public String getCollectionName() {
	return definition.getString("Name");
    }


    public void updateDefinition(Document doc) {

	if (doc!=null)
	    definition.append("Definition", doc);
	else
	    definition.remove("Definition");
	parent.storeCollection(getCollectionName(), this);
	for (ICollection collection : replicaConnections.values()) {
	    collection.updateDefinition(doc);
	}

    }

   

    public void updateDelta(Document doc) {

	if (doc!=null)
	    definition.append("Delta", doc);
	else
	    definition.remove("Delta");
	parent.storeCollection(getCollectionName(), this);
	for (ICollection collection : replicaConnections.values()) {
	    collection.updateDelta(doc);
	}

    }


    @Override
    public void deleteTable(String def) {

	getSchDoc().deleteTable(def);
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void saveConceptGroup(Document def) {

	ISchemaMeta schd = getSchDoc();
	schd.addPurpose((IPurpose) def.toObject(Purpose.class));
	for (Object o : def.getList("purposeColumns")) {
	    Document d = new Document((Map)o);
	    PurposeColumn pc = d.toObject(PurposeColumn.class);
	    if (d.containsKey("purposeColumnMaps")) {
		if (d.getList("purposeColumnMaps").size()>0) {
		    //find it inthe definition and blank it 
		    List<PurposeColumnMap> pcms =schd.getPurposeColumnMaps(pc);
		    for (PurposeColumnMap pcm : pcms)
			schd.deletePurposeColumnMap(pcm);

		    for (Object oo : d.getList("purposeColumnMaps")) {
			Document dd = new Document((Map)oo);
			schd.addPurposeColumnMap(dd.toObject(PurposeColumnMap.class));
		    }
		} else {
		    List<PurposeColumnMap> pcms =schd.getPurposeColumnMaps(pc);
		    for (PurposeColumnMap pcm : pcms)
			schd.deletePurposeColumnMap(pcm);
		}
	    } 
	}



	updateDefinition(schd.toDocument());
    }


    @Override
    public void deleteConceptGroup(String def) {

	getSchDoc().deletePurpose(def);
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void saveConcept(Document def) {

	getSchDoc().addPurposeColumn((PurposeColumn) def.toObject(PurposeColumn.class));
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void deleteConcept(String purposeName, String purposeColumn) {

	getSchDoc().deletePurposeColumn(purposeName, purposeColumn);
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void saveConceptMapping(Document def) {

	getSchDoc().addPurposeColumnMap(def.toObject(PurposeColumnMap.class));
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void deleteConceptMapping(Document def) {

	getSchDoc().deletePurposeColumnMap(def.toObject(PurposeColumnMap.class));
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void saveMatchRule(Document def) {
	getSchDoc().addRule((IRule) def.toObject(Rule.class));
	updateDefinition(getSchDoc().toDocument());
    }




    @Override
    public void saveMatchRules(List<Document> defs) {
	ISchemaMeta _schDoc = getSchDoc();
	for (IRule rule : _schDoc.getRules(null))
	    _schDoc.deleteMatchRule(rule.getOrder());
	for (Document def : defs)
	    _schDoc.addRule((IRule) def.toObject(Rule.class));
	updateDefinition(_schDoc.toDocument());
    }


    @Override
    public void deleteMatchRule(long order) {
	getSchDoc().deleteMatchRule(order);
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void saveFuzzyIndex(Document def) {
	getSchDoc().addIndex(def.toObject(Index.class));
	updateDefinition(getSchDoc().toDocument());
    }


    @Override
    public void saveFuzzyIndexes(List<Document> def) {
	ISchemaMeta _sch = getSchDoc();
	for (IIndex i : _sch.getIndexes())
	    _sch.deleteIndex(i.getIndexName());

	for (Document de : def)
	    _sch.addIndex(de.toObject(Index.class));
	updateDefinition(_sch.toDocument());
    }



    @Override
    public void deleteFuzzyIndex(String name) {
	getSchDoc().deleteIndex(name);
	updateDefinition(getSchDoc().toDocument());
    }

    @Override
    public void removeFuzzy() {
	schDoc=null;
	updateDefinition(null);

    }


    @Override
    public Document getDefinition() {

	return definition;
    }


    public Database getDatabase() {
	return this.parent;
    }

    @Override
    public Document getTable(String def) {
	return this.localCollection.getTable(def);
    }






    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#peekQueue()
     */
    @Override
    public DBCursor peekQueue() {
	DBCursor queue = null;
	for (ICollection collection : replicaConnections.values()) {
	    DBCursor ex = collection.peekQueue();
	    if (ex!=null) {
		if (queue==null)
		    ex=queue;
		else 
		    queue.extend(ex);
	    }
	}
	return queue;
    }



    /**
     * @return
     */
    public String getTrigger() {
	return definition.getString("Trigger");
    }



    /*
   	Access Right Example:
   	{
   		privileges: [
   			{ resource: { db: "products", collection: "inventory" }, actions: { "find": {field: 1}, "update": {field2: -1}, "insert": 1, "delete": {} } },
   			{ resource: { db: "products", collection: "orders" },  actions: { "find" : 1 } }
   		]
   	}
     */
    public Document filter(Document in, User user, String action){
	Document bestRole = null;
	int bestAction = 0;
	for (Document role : user.pullRoles(null)){

	    if (role!=null) {
		List<Document> privs = role.getList("privileges");
		for (Document priv : privs) {
		    Document res = priv.getAsDocument("resource");
		    Document actions = priv.getAsDocument("actions");
		    if (res!=null && actions != null) {
			Object raction = actions.get(action);
			if (res.getString("db").equalsIgnoreCase(getDatabase().getName()) &&
				res.getString("collection").equalsIgnoreCase(getCollectionName()) && 
				raction != null) {
			    if (raction instanceof Integer) {
				if ((Integer)raction > bestAction) {
				    return in;
				} 
			    } else {
				if (ICollection.filter(in, (Document)raction))
				    return in;

			    }
			}
		    }
		}
	    }
	}
	return null;

    }

    public Map<Integer, ICollection> getReplicaConnections() {
	return replicaConnections;
    }




    public Stream<Document> fuzzyMatch(Document fuzzyOptions, Stream<Document> in, Document options) {
	if (in == null)
	    in=find().stream();
	if (schDoc==null) {
	    System.out.println("No fuzzy artifacts found on collection: " +  definition.getString("Name") + ", bypassing $fuzzy stage");
	    return in;
	}
	final List<IIndex> _indexes = new ArrayList<IIndex>();

	if (fuzzyOptions.containsKey("Index"))
	    _indexes.add(getSchDoc().getIndex(fuzzyOptions.getString("Index")));
	else {
	    for (IIndex ind : getSchDoc().getIndexes())
		if(ind.isMatch()) {
		    _indexes.add(ind);
		    break;
		}
	}
	return in.parallel().map(doc -> {
	    return fuzzyMatch(doc, _indexes);
	})
		.filter(c -> c!=null);

    }

    public Stream<Document> fuzzySearch(String textQuery, Stream<Document> in, Document options) {
	if (schDoc==null) {
	    System.out.println("No fuzzy artifacts found on collection: " +  definition.getString("Name") + ", bypassing $fuzzy stage");
	    return in;
	}

	return this.findFuzzyStream(textQuery);

    }

    public Stream<Document> findFuzzyStream(String searchText) {
	Stream<Document> ret= null;
	for (IIndex matchI : getSchDoc().getIndexes()) {
	    if (matchI.isSearch()) {
		Stream<Document> t = searchInternal(searchText, matchI).collect(Collectors.toList()).stream();
		if (ret==null)
		    ret=t;
		else
		    ret=Stream.concat(ret, t);
	    }
	};
	if (ret!=null)
	    return ret;
	else return Stream.empty();
    }


    private void initialiseCollection() {



	if (definition.containsKey("Definition")) {
	    super.initialize(schDoc);

	    for (IIndex index : getSchDoc().getIndexes()) {
		if (!indexCatalogCache.containsKey(index.getIndexName())) {
		    String matchProcsString = new Gson().toJson(getSchDoc().getMatchProcs(index.getIndexName(), ""+index.getInstance(),""));
		    indexCatalogCache.put(index.getIndexName(), new Document("Options", new Document("fuzzy", true)
			    .append("matchProcs", matchProcsString)
			    .append("name", index.getIndexName())));
		}
	    }

	}




    }

    protected DBCursor findFuzzy(String searchText, Document explainPlan  ) {
	DBCursor cursor = new DBCursor();

	for (IIndex matchI : getSchDoc().getIndexes()) {

	    if (matchI.isSearch()) {


		List<Document> ret = searchInternal(searchText, matchI).collect(Collectors.toList());
		cursor.extend(new DBCursor(ret));

	    }


	}

	return cursor;
    }
    private Stream<Document> searchInternal(String searchText, IIndex matchI) {
	String indexName = matchI.getIndexName();
	java.util.Collection<String> useKeyList = getMatchKeys(searchText, matchI);

	return useKeyList.parallelStream().map( useKeys -> 
	{
	    String key=null;
	    if (matchI.isSearch() && useKeys.length()>0)
		key= ("S:"+ useKeys );

	    if (key!=null) {
		Document entry = getIndexPrefixSubMap(indexName, key, true);
		return entry.values();
	    }
	    return new ArrayList();
	})
		.parallel()
		.flatMap(c -> c.stream())
		.distinct()
		.map(pointer->{  
		    Document current = (Document)getDocument(pointer);
		    return current;
		})
		.filter(c -> c!=null);



    }

    @Override
    public DBCursor findFuzzy(String textQuery ) {
	Document explainPlan=new Document();
	long t = System.currentTimeMillis();
	DBCursor _return = findFuzzy(textQuery, explainPlan);
	t = System.currentTimeMillis()-t;
	if (explaining) {
	    explainPlan.append("Elapsed", t + "ms");
	    logger.info(explainPlan.toJson());
	}
	return _return;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findFuzzy(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findFuzzy(Document filter ) {
	Document explainPlan=new Document();
	long t = System.currentTimeMillis();
	DBCursor _return = findFuzzy(filter, explainPlan, true);
	t = System.currentTimeMillis()-t;
	if (explaining) {
	    explainPlan.append("Elapsed", t + "ms");
	    logger.info(explainPlan.toJson());
	}
	return _return;
    }

    protected DBCursor findFuzzy(Document filter, Document explainPlan, boolean isSearch  ) {
	String tableName = filter.getString("Table", definition.getString("Name"));
	DBCursor cursor = new DBCursor();
	if (tableName!=null) {

	    for (IIndex matchI : getSchDoc().getIndexes()) {

		if ((matchI.isSearch() && isSearch) || (matchI.isMatch() && !isSearch)) {
		    DBCursor temp = indexSeekFuzzy(matchI, filter, isSearch);
		    while (temp.hasNext()) {
			Document d = temp.next();
			d.append("_data", getDocument(d.getString("_id")));
			cursor.add(d);
		    }

		}


	    }

	}
	return cursor;
    }


    private DBCursor indexSeekFuzzy(IIndex matchI, Document filter, boolean isSearch) {
	String tableName = filter.getString("Table", definition.getString("Name"));

	if (tableName!=null) {
	    ITable table = getSchDoc().getTable(tableName);
	    if (table==null) {
		table=getSchDoc().getTables().iterator().next();
		System.out.println("Table was not defined, assuming "+table.getTableName());
	    }
	    if (table==null) 
		return new DBCursor(new Document("Error", "Fuzzy table was not specified and one could not be found"));

	    return new DBCursor((List<Document>) matchInternal(filter, matchI,isSearch, table)
		    .collect(Collectors.toList()));

	}
	return new DBCursor();
    }

    private Stream<Document> matchInternal(Document filter, IIndex matchI, boolean isSearch, ITable table) {
	String indexName = matchI.getIndexName();
	java.util.Collection<String> useKeyList = getMatchKeys(filter, table, matchI, false);
	if (!filter.containsKey("standardized")) 
	    filter.append("standardized", localCollection.standardise(filter));
	if (table != null)
	    filter.putIfAbsent("Table", table.getTableName());

	String _id = filter.getString("_id");
	return useKeyList.parallelStream().map( useKeys -> 
	{
	    String key=null;
	    if (matchI.isSearch() && useKeys.length()>0)
		key= ("S:"+ useKeys );

	    if ((matchI.isMatch() || (!matchI.isSearch() && !matchI.isMatch()))  && useKeys.length()>0)
		key= (matchI.getIndexName()+":"+ useKeys);

	    if (key!=null) {
		Document entry = getIndexPrefixSubMap(indexName, key, true);
		return entry.values();
	    }
	    return new ArrayList();
	})
		.parallel()
		.flatMap(c -> c.stream())
		.distinct()
		.map(pointer->{
		    if (!pointer.equals(_id)) {
			Document current = (Document)getDocument(pointer);
			if (current!=null) {
			    current.append("standardized", localCollection.standardise(current));
			    current.putIfAbsent("Table", table.getTableName());
			    Document result = score(filter, current, isSearch);

			    if (result!= null && (isSearch || (!isSearch && result.getDouble("score")>0))) {
				result.append("_id", current.getString("_id"));

				return result;
			    } 
			}

		    }

		    return null;
		}).filter(c -> c!=null);



    }

    public Stream<Document> arrf(Document options, Stream<Document> in, Document goptions){
	Document headerDoc = new Document();
	if (options==null)
	    options=new Document();
	if (options.containsKey("relation"))
	    headerDoc.append("relation", options.getString("relation"));
	else 
	    headerDoc.append("relation", "default");
	String className="";
	if (options.containsKey("className"))
	    className=options.getString("className");
	Document _definition=new Document();
	if (options.containsKey("definition"))
	    _definition=options.getAsDocument("definition");
	Document definition=_definition;
	List<Document> attrs = new ArrayList<Document>();
	headerDoc.append("attributes", attrs);
	List<Document> data = new ArrayList<Document>();
	Document header = new Document();
	//scan data and add it to the data, but also build up the header 
	in.sequential().forEach(doc -> {

	    Map<String, Object> ret= new HashMap<String, Object>();

	    Node.flattenDoc("", doc, ret, "");
	    ret.remove("_id");
	    for (Object key: ret.keySet()) {
		//process the header

		Object value = ret.get(key);

		Object type;
		if (header.containsKey(key)) {
		    type=header.get((String) key);
		    if (type instanceof HashSet) {
			((HashSet)type).add((String)value);
			header.append((String) key, type);
		    }
		}
		else {
		    if (definition.containsKey(key)) {
			type=definition.get(key);
			if (type instanceof String && ((String)type).equalsIgnoreCase("nominal"))
			    type=new HashSet<String>();
			if (type instanceof List) {
			    type = new HashSet<String>();
			    ((HashSet)type).addAll((List)definition.get(key));
			}
		    }
		    else 
			type=typeOf(value);
		    if (type instanceof HashSet)
			((HashSet)type).add((String)value);
		    header.append((String) key, type);
		}	
	    }
	    //process the data
	    List values = new ArrayList();
	    for (Object key: header.keySet()) {
		if (ret.containsKey(key))
		    values.add(ret.get(key));
		else
		    values.add("");
	    }
	    data.add(new Document("sparse", false).append("weight", 1.0).append("values", values));
	});

	int count=0;
	for (Object name : header.keySet()) {
	    count++;
	    boolean last=header.keySet().size()==count;
	    boolean isclass=className.equalsIgnoreCase((String)name) || last;
	    if(header.get(name) instanceof String)
		attrs.add(new Document("name", name).append("type", header.get(name)).append("weight", 1.0).append("class", isclass));
	    else 
		attrs.add(new Document("name", name).append("type", "nominal").append("weight", 1.0).append("class", isclass).append("labels", header.get(name)));
	}
	return Collections.singletonList(new Document("header", headerDoc).append("data", data)).stream();
    }

    /**
     * @param value
     * @return
     */
    private Object typeOf(Object value) {
	if (value instanceof Number)
	    return "numeric";

	if (value instanceof String) {
	    String temp= (String)value;
	    try {
		Double.parseDouble(temp);
		return "numeric";
	    } catch(Exception e) {}
	    HashSet<String> set = new HashSet<String>();
	    //set.add("");
	    return set;
	}

	if (value instanceof Date)
	    return "date " + DateFormat.getDateTimeInstance().toString();

	return "string";
    }

    public Stream<Document> classifierTree(Document definition, Stream<Document> in, Document options){
	if (definition.containsKey("modelFilter")) {
	    //find the model
	    definition = definition.getAsDocument("modelFilter");
	    String from = definition.getString("from");
	    Document filter = definition.getAsDocument("filter");
	    ICollection c2=this;
	    if (!this.getCollectionName().equalsIgnoreCase(from))
		c2=this.getDatabase().getCollection(from);
	    definition = c2.find(filter).first();

	}

	try {
	    Gson gson = new Gson();
	    Class clazz = Class.forName("weka.classifiers."+definition.getString("classifier"));

	    SerializedObject classifierSO = (SerializedObject) gson.fromJson(definition.getAsDocument("model").toJson(), SerializedObject.class);
	    AbstractClassifier classifier = (AbstractClassifier) classifierSO.getObject();
	    if (classifier instanceof Drawable && (((Drawable)classifier).graphType()!=Drawable.NOT_DRAWABLE)) {
		if (((Drawable)classifier).graphType()!=Drawable.BayesNet)
		    return Collections.singletonList(new Document("dotNotation", ((Drawable)classifier).graph())).stream();
		else
		    return Collections.singletonList(new Document("xmlBIFF", ((Drawable)classifier).graph())).stream();

	    }
	}
	catch (Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Object> evaluate(Document evalOptions, Stream<Document> in, Document options) {
	if (in != null){
	    return in
		    .parallel()
		    .map(doc -> Document.translate$(evalOptions,doc));
	} else return null;

    }


    public Stream<Document> sort(Document object, Stream<Document> in, Document options) {
	//object is a document as such: {salary:1, lastName:-1}, 1 is ascending -1 descending
	return ((List<Document>) in.sorted(new Comparator() {

	    @Override
	    public int compare(Object doc1, Object doc2) {
		int c=0;
		for (Object key: object.keySet()) {
		    Object v1=((Document)doc1).get(key);
		    Object v2=((Document)doc2).get(key);
		    int dir=object.getInteger((String)key, 1);
		    if (v1 instanceof String && v2 instanceof String) {
			c = v1.toString().compareTo(v2.toString())*dir;
		    } else if (v1 instanceof Number && v2 instanceof Number) {
			c = ((Double)((Number)v1).doubleValue()).compareTo(((Number)v2).doubleValue())*dir;
		    } else if (v1 instanceof Date && v2 instanceof Date) {
			c = ((Date)v1).compareTo((Date)v2)*dir;
		    } else if (v1 instanceof Boolean && v2 instanceof Boolean) {
			c = ((Boolean)v1).compareTo((Boolean)v2)*dir;
		    } else c=dir;
		    if (c!=0)
			return c;
		}
		return c;
	    }

	}).collect(Collectors.toList())).stream();

    }



    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#compare(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream compare(Document asDocument, Stream<Document> in, Document options) {
	if (in != null){
	    if (options==null)
		options=new Document();

	    Map<String, Document> results = new HashMap<String,Document>();
	    //look for matches amoungst the incoming set
	    //coerce them into concepts where the percentage is high enough
	    List<Document> docs = in.collect(Collectors.toList());
	    for (Document prep: docs) 
		prep.append("standardized", localCollection.standardise(prep));
	    for (Document filter: docs) {
		for (Document current: docs) {	
		    if (!results.containsKey(current.getString("_id")+"/"+filter.getString("_id"))) {
			Document result = score(filter, current, false);

			result.append("from", filter.getString("_id"));
			result.append("to", current.getString("_id"));
			if (result.getDouble("score")> 0.0d)
			    results.put(filter.getString("_id")+"/"+current.getString("_id"), result); 
		    }
		}
	    }

	    //coerce them into concepts where the percentage is high enough

	    return results.values().parallelStream();
	} else return null;

    }
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierBuild(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierBuild(Document definition, Stream<Document> in, Document options){
	List<Document> result = new ArrayList<Document>();
	Gson gson = new Gson();
	try {
	    JSONLoader loader = streamToLoader(in,null);

	    Map<String, Classifier> classifiers = new HashMap<String, Classifier>();


	    for (Object modelNameo : definition.keySet()) {
		String modelName=(String)modelNameo;
		Class clazz = Class.forName("weka.classifiers."+modelName);
		Constructor constructor = clazz.getConstructor();
		AbstractClassifier classifier=(AbstractClassifier) constructor.newInstance();
		classifiers.put(modelName, classifier);

	    }
	    Document header=null;

	    for (String classifiern : classifiers.keySet()) {
		Classifier classifier = classifiers.get(classifiern); 
		int classIndex=loader.getStructure().numAttributes()-1;
		int numFolds=10;
		if ((definition.getAsDocument(classifiern)!=null)) {
		    if 	(definition.getAsDocument(classifiern).containsKey("numFolds"))
			numFolds=definition.getAsDocument(classifiern).getInteger("numFolds");
		    String defClass=definition.getAsDocument(classifiern).getString("className"); 

		    for(int a=0;a<loader.getStructure().numAttributes();a++)
			if (loader.getStructure().attribute(a).name().equalsIgnoreCase(defClass))
			    classIndex=a;

		    loader.getStructure().setClassIndex(classIndex);
		    JSONNode json = JSONInstances.toJSON(loader.getStructure());
		    StringBuffer buffer = new StringBuffer();
		    json.toString(buffer);
		    header = (Document) Document.parse(buffer.toString());
		}
		Instances[] train = new Instances[numFolds];
		Instances[] test = new Instances[numFolds];
		for (int numFold =0; numFold<numFolds; numFold++) {
		    train[numFold]=loader.getDataSet().trainCV(numFolds, numFold);
		    test[numFold]=loader.getDataSet().testCV(numFolds, numFold);
		    train[numFold].setClassIndex(classIndex);
		    test[numFold].setClassIndex(classIndex);
		}

		FastVector predictions = new FastVector();
		for (int i=0;i<numFolds; i++) {
		    Evaluation evaluation = new Evaluation(train[i]);
		    classifier.buildClassifier(train[i]);

		    evaluation.evaluateModel(classifier, test[i]);
		    predictions.appendElements(evaluation.predictions());

		}
		double accuracy = calculateAccuracy(predictions);

		Document predResult = null;
		if ((definition.getAsDocument(classifiern)!=null))
		    predResult= definition.getAsDocument(classifiern);

		else predResult = new Document();

		predResult.append("classifier", classifiern)
		.append("accuracy", accuracy)
		.append("createDate", new Date())
		.append("header", header.getAsDocument("header"))
		.append("model", new Document(gson.toJson(new SerializedObject(classifier))));


		result.add(predResult);
	    }

	} catch (Exception e) {
	    e.printStackTrace();
	}
	return result.stream();
    }


    private double calculateAccuracy(FastVector predictions) {
	double correct = 0;

	for (int i = 0; i < predictions.size(); i++) {
	    NominalPrediction np = (NominalPrediction) predictions.elementAt(i);
	    if (np.predicted() == np.actual()) {
		correct++;
	    }
	}

	return 100 * correct / predictions.size();
    }

    private JSONLoader streamToLoader(Stream<Document> in, Document header) throws IOException {
	JSONLoader loader = new JSONLoader();

	Gson gson = new Gson();


	ByteArrayOutputStream bos = new ByteArrayOutputStream();
	AtomicBoolean first = new AtomicBoolean(true);
	in.sequential().forEach(doc -> {
	    try {
		if (header != null)
		    doc.append("header", header);
		byte[] bytes = doc.toJson().getBytes();
		if (!first.getAndSet(false))
		    bos.write(new byte[] {','});
		bos.write(bytes);

	    } catch (IOException e) {
		e.printStackTrace();
	    }

	});

	ByteArrayInputStream bis = new ByteArrayInputStream(bos.toByteArray());
	loader.setSource(bis);
	loader.getStructure();
	return loader;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierPredict(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierPredict(Document definition, Stream<Document> in, Document options) {
	if (definition.containsKey("modelFilter")) {
	    //find the model
	    definition = definition.getAsDocument("modelFilter");
	    String from = definition.getString("from");
	    Document filter = definition.getAsDocument("filter");
	    ICollection c2=this.getDatabase().getCollection(from);
	    definition = c2.find(filter).first();

	}




	try {
	    Gson gson = new Gson();
	    Class clazz = Class.forName("weka.classifiers."+definition.getString("classifier"));

	    SerializedObject classifierSO = (SerializedObject) gson.fromJson(definition.getAsDocument("model").toJson(), SerializedObject.class);
	    AbstractClassifier classifier = (AbstractClassifier) classifierSO.getObject();

	    JSONLoader loader = streamToLoader(in, definition.getAsDocument("header"));
	    String defClass=definition.getString("className"); 

	    int classIndex = -1;
	    for(int a=0;a<loader.getStructure().numAttributes();a++)
		if (loader.getStructure().attribute(a).name().equalsIgnoreCase(defClass))
		    classIndex=a;

	    loader.getStructure().setClassIndex(classIndex);

	    Instances dataSet = loader.getDataSet();

	    dataSet.setClassIndex(classIndex);
	    return dataSet.parallelStream().map( inst ->  {
		try {

		    double result = classifier.classifyInstance(inst);
		    Document doc = toJSON(inst);
		    String prediction=inst.classAttribute().value((int)result );
		    doc.append(defClass, prediction);
		    return doc;
		} catch (Exception e) {
		    e.printStackTrace();
		}
		return null;
	    });
	} catch (Exception e) {
	    e.printStackTrace();
	}

	return null;

    }
    protected static Document toJSON(Instance inst) {
	Document result = new Document();
	for (int a=0; a<inst.numAttributes(); a++)
	    if (!inst.attribute(a).isNumeric())
		result.append(inst.attribute(a).name(), inst.stringValue(a));
	    else
		result.append(inst.attribute(a).name(), inst.value(a));

	return result;
    }
    @Override
    public Document fuzzyMatch(Document doc, List<IIndex> indexes) {
	String tableName = doc.getString("Table", definition.getString("Name"));
	if (tableName==null || indexes==null || indexes.size()==0)
	    return null;
	Document r=new Document();
	r.append("_id", doc.getString("_id"));
	if (tableName!=null) {
	    ITable table = getSchDoc().getTable(tableName);
	    List<Document> matches = new ArrayList<Document>();
	    for (IIndex index : indexes) {
		List<Document> m = matchInternal(doc, index,false, table).collect(Collectors.toList());
		if (matches!=null)
		    matches.addAll(m);
	    }
	    if (matches.size()>0) {
		r.append("Matches", matches);
		r.append("Count", matches.size());
	    }
	    else return null; //we dont want 0 counts
	}


	return r;
    }






    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findRelationships(Document filter) {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.findRelationships(filter).stream().forEach(doc -> {
		if (doc!=null)
		    set.put(doc.getString("_id"),doc);
	    });
	});
	return new DBCursor(set.values());
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findTasks(Document filter) {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.findTasks(filter).stream().forEach(doc -> {
		if (doc!=null)
		    set.put(doc.getString("_id"),doc);
	    });
	});
	return new DBCursor(set.values());
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveRelationship(com.entitystream.monster.db.Document)
     */
    @Override
    public Document saveRelationship(Document relationship) {
	if (!relationship.containsKey("_id")) {
	    String _id = UUID.randomUUID().toString();
	    relationship.append("_id", _id);
	}
	Document retdoc = null;
	for (Integer key : replicaConnections.keySet()) {
	    ICollection collection = replicaConnections.get(key);
	    retdoc=collection.saveRelationship(relationship);
	}
	return retdoc;
    }

    @Override
    public Document saveTask(Document task) {
	if (!task.containsKey("_id")) {
	    String _id = UUID.randomUUID().toString();
	    task.append("_id", _id);
	}
	Document retdoc = null;
	for (Integer key : replicaConnections.keySet()) {
	    ICollection collection = replicaConnections.get(key);
	    retdoc=collection.saveTask(task);
	}
	return retdoc;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#resolveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public void resolveTask(Document taskData) {
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.resolveTask(taskData);
	});
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#aggregateTasks(java.util.ArrayList)
     */
    @Override
    public DBCursor aggregateTasks(ArrayList<Document> andlist) {
	ConcurrentHashMap<String,Document> set = new ConcurrentHashMap<String,Document>();
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.aggregateTasks(andlist).stream().forEach(doc -> {
		if (doc!=null)
		    set.put(doc.getString("_id"),doc);
	    });
	});
	return new DBCursor(set.values());
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateTask(java.lang.String, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateTask(String id, Document doc) {
	Document retdoc = null;
	for (Integer key : replicaConnections.keySet()) {
	    ICollection collection = replicaConnections.get(key);
	    Document _retdoc=collection.updateTask(id, doc);
	    if (_retdoc!=null) {
		retdoc=_retdoc;
		break;
	    }
	}
	return retdoc;
    }



    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateRelationship(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateRelationship(Document document, Document setUpdate, Document options) {
	Document retdoc = null;
	for (Integer key : replicaConnections.keySet()) {
	    ICollection collection = replicaConnections.get(key);
	    Document _retdoc=collection.updateRelationship(document, setUpdate, options);
	    if (_retdoc!=null) {
		retdoc=_retdoc;
		break;
	    }
	}
	return retdoc;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteTasks(Document document) {
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.deleteTasks(document);
	});

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteRelationships(Document document) {
	replicaConnections.values().parallelStream().forEach(collection -> {
	    collection.deleteRelationships(document);
	});


    }

    ///////////////////////////////////////////////////////////////////////////
    //Central Aggregate functions follow
    ///////////////////////////////////////////////////////////////////////////

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#limit(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> limit(long limit, Stream<Document> in, Document options) {

	return in.limit(limit);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#lookup(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> lookup(Document object, Stream<Document> in, Document options) {
	//from: <collection to join>,
	//localField: <field from the input documents>,
	//foreignField: <field from the documents of the "from" collection>,
	//as: <output array field>
	String from = object.getString("from");
	String localField = object.getString("localField");
	String foreignField = object.getString("foreignField");
	String as = object.getString("as");
	return in.parallel().map(doc-> {
	    Object value1 = doc.get(localField);
	    List values;
	    if (value1 instanceof List)
		values=(List)value1;
	    else
		values=Collections.singletonList(value1);

	    List l = new ArrayList();

	    for (Object value : values) {
		value=Document.translate$(value, doc);
		Document filter = new Document(foreignField, value);
		ICollection c2=this.parent.getCollection(from);

		if (c2!=null) {
		    DBCursor cur=c2.find(filter);

		    l.addAll(cur.stream().collect(Collectors.toList()));

		}
	    }
	    if (l.size()>0)
		doc.append(as, l);

	    return doc;
	});
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#join(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> join(Document object, Stream<Document> in, Document options) {
	//from: <collection to join>,
	//localField: <field from the input documents>,
	//foreignField: <field from the documents of the "from" collection>,
	//as: <output array field>
	String to = object.getString("with");

	Object joinon = object.get("on");
	String as = object.getString("into");
	return in.parallel().map(doc-> {

	    List l = new ArrayList();
	    if ((joinon instanceof Map)) {
		//toField is an expression, from field is not required
		Object filter=Document.translateJoin$(object.getAsDocument("on"),doc);
		if (filter!=null) {
		    ICollection c2=this.parent.getCollection(to);
		    if (c2!=null) {
			DBCursor cur=c2.find(new Document(filter));
			l.addAll(cur.stream().collect(Collectors.toList()));
		    }
		}
	    }



	    if (l.size()>0) {
		doc.append(as, l);
		return doc;
	    } else return null;
	}).filter(doc ->doc != null);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#minus(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> minus(Document object, Stream<Document> in, Document options) {
	//from: <collection to join>,
	//fields: {<field from the input documents>: <foreignName to use or 1/true for the same name>, ....},

	String from = object.getString("from");
	Document fields = object.getAsDocument("fields");

	return in.parallel().filter(doc-> {
	    Document filter = new Document();
	    for (Object localField : fields.keySet()) {
		Object value = doc.get(localField);
		String foreignName = (String) localField;
		if (fields.get(foreignName) instanceof String)
		    foreignName = fields.getString(foreignName);
		filter.append(foreignName, value);
	    }

	    ICollection c2=this.parent.getCollection(from);
	    boolean found=false;
	    if (c2!=null) {
		found=c2.find(filter).count()>0;
	    }
	    return !found;
	});
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#skip(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> skip(long skip, Stream<Document> in, Document options) {
	return in.skip(skip);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#group(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> group(Document object, Stream<Document> in, Document options) {

	Object _id = object.get("_id");

	//collection aggregates
	Map<Object, Document> out = new LinkedHashMap<Object, Document>();
	in.sequential().forEach(doc -> {
	    try {
		for (Object newfield : object.keySet()) {
		    String fieldName=(String)newfield;
		    if (!fieldName.equalsIgnoreCase("_id")) {
			Document aggFn = object.getAsDocument(fieldName);
			//evaluate the _id
			Object key=Document.translate$(_id,doc);
			if (key==null)
			    key="ALL";

			Document currKeyDoc=out.get(key.hashCode());
			if (currKeyDoc==null) {
			    currKeyDoc=new Document();
			    if (!key.equals("ALL"))
				currKeyDoc.append("_id", key);
			}
			Document currentValue=currKeyDoc.getAsDocument(fieldName);
			//if (currentValue!=null)
			//System.out.println("CV Before: " + currentValue.toJson());

			Object additive =  Document.translate$(aggFn, doc);
			currentValue=(Document) Document.translateFn$((String)aggFn.keySet().iterator().next()+"Map", additive, currentValue);

			currKeyDoc.append(fieldName, currentValue);

			//System.out.println("CV After: " + currentValue.toJson());
			//System.out.println("Add was: " + additive.toString());

			out.put(key.hashCode(), currKeyDoc);


		    }
		}
	    }catch (Exception e) {
		e.printStackTrace();
	    }
	});
	//finalize aggregates
	return out.keySet().stream().map(key -> {
	    Document doc = out.get(key);

	    for (Object newfield : object.keySet()) {
		String fieldName=(String)newfield;
		if (!fieldName.equalsIgnoreCase("_id")) {
		    Document aggFn = object.getAsDocument(fieldName);
		    Document agg = doc.getAsDocument(fieldName);

		    doc.append(fieldName, Document.translateFn$((String)aggFn.keySet().iterator().next()+"Finalize", aggFn, agg));


		}
	    }
	    return doc;
	});
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#bucket(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream bucket(Document bucketDef, Stream<Document> in, Document options) {
	/*
   		  {
         		groupBy: <expression>,
         		boundaries: [ <lowerbound1>, <lowerbound2>, ... ],
         		default: <literal>,
         		output: {
            		<output1>: { <$accumulator expression> },
            		...
            		<outputN>: { <$accumulator expression> }
         		}
      		  }
	 */
	//initialise the output
	NavigableMap<Double, BucketRange> output = new TreeMap<Double, BucketRange>();
	Double last=null;
	for (Number boundary : ((List<Number>)bucketDef.get("boundaries"))) {
	    if (last!=null)
		output.put(last, new BucketRange(boundary.doubleValue(), new Document("_id", boundary.doubleValue())));
	    last=boundary.doubleValue();
	}
	Document defaultDoc = new Document("_id", bucketDef.get("default"));
	Document outputDef = bucketDef.getAsDocument("output");
	Object groupByField=bucketDef.get("groupBy");
	in.sequential().forEach(doc -> {
	    //calculate the bucket
	    Double groupByValue = ((Number) Document.translate$(groupByField, doc)).doubleValue();
	    Map.Entry<Double,BucketRange> entry = (Entry<Double, BucketRange>) output.floorEntry(groupByValue);
	    Document doc2use = defaultDoc;
	    if (entry != null && groupByValue < entry.getValue().upper) {
		doc2use = entry.getValue().document;
	    }

	    //add the output
	    for (Object outkey : outputDef.keySet()) {
		Object outvaluedef = outputDef.get(outkey);
		bucket_merge(outvaluedef, doc, doc2use, (String)outkey);


		if (entry!=null) {
		    entry.getValue().document=doc2use;
		    output.put(entry.getKey(), entry.getValue());
		}
	    }



	});
	return output.values().stream().map(br -> {
	    for (Object outkey : outputDef.keySet()) {
		Object outvaluedef = outputDef.get(outkey);
		bucket_finalize(outvaluedef, br.document, (String)outkey);
	    }
	    return br.document;
	});
    }


    public void bucket_merge(Object definition, Document evaluatedDoc, Document bucketDoc, String bucketKey) {
	Object additive=Document.translate$(definition, evaluatedDoc);
	Object out = null;
	if (definition instanceof Document) {
	    for (Object inner : ((Document)definition).keySet()) {
		out = Document.translateFn$(((String)inner)+"Map", additive, bucketDoc.getAsDocument(bucketKey));
	    }
	} else {
	    out=Document.translateFn$(((String)definition)+"Map", additive, bucketDoc.getAsDocument(bucketKey));
	}

	bucketDoc.append(bucketKey, out);

    }

    public void bucket_finalize(Object definition,Document bucketDoc, String bucketKey) {

	Object out = null;
	if (definition instanceof Document) {
	    for (Object inner : ((Document)definition).keySet()) {
		out = Document.translateFn$(((String)inner)+"Finalize", null, bucketDoc.getAsDocument(bucketKey));
	    }
	} else {
	    out = Document.translateFn$(((String)definition)+"Finalize", null, bucketDoc.getAsDocument(bucketKey));
	}

	bucketDoc.append(bucketKey, out);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#unwind(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Object> unwind(Document unwindOptions, Stream<Document> in, Document options) {
	if (in != null){
	    return in
		    .parallel()
		    .map(doc -> unwind(doc, unwindOptions))
		    .flatMap(c -> c.stream());
	} else return null;
    }

    private List<Document> unwind(Document doc, Object unwindOptions) {
	//pull the projection out into a list.
	String path = "";
	if(unwindOptions instanceof Document)
	    path = ((Document)unwindOptions).getString("path");
	else if (unwindOptions instanceof String)
	    path=(String) unwindOptions;
	String includeArrayIndex=((Document)unwindOptions).getString("includeArrayIndex");
	boolean preserveNullAndEmptyArrays = ((Document)unwindOptions).getBoolean("preserveNullAndEmptyArrays", false);
	List<Document> out = new ArrayList<Document>();

	if (path!=null) {
	    Object proj = doc.getProjection(path);
	    if (proj != null && proj instanceof List) {
		int count=0;
		for (Object item : (List)proj) {
		    Document newDoc = new Document(doc);
		    newDoc.setProjection(path, item);
		    if (includeArrayIndex!=null)
			newDoc.append(includeArrayIndex, count);
		    out.add(newDoc);
		    count++;
		}
		if (count==0 && preserveNullAndEmptyArrays)
		    out.add(doc);
	    } else if (preserveNullAndEmptyArrays)
		out.add(doc);
	}

	return out;
    }



    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#count(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> count(Document asDocument, Stream<Document> in, Document options) {
	return Collections.singletonList(new Document("count", in.count())).stream();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#first(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> first(Document asDocument, Stream<Document> in, Document options) {
	Optional<Document> o = in.findFirst();
	if (o.isPresent())
	    return Collections.singletonList(o.get()).stream();
	else return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#last(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> last(Document asDocument, Stream<Document> in, Document options) {
	Document o = in.reduce((first, second) -> second).orElse(null);
	if (o!=null)
	    return Collections.singletonList(o).stream();
	else return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#between(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> between(Document asDocument, Stream<Document> in, Document options) {
	if (options!=null) {
	    long start=0;
	    if (options.containsKey("start"))
		start=options.getLong("start");
	    long end=Long.MAX_VALUE;
	    if (options.containsKey("end"))
		end=options.getLong("end");
	    return in.skip(start).limit(end-start+1);
	}
	return in;
    }
    public Document score(Document base, Document comparitor, boolean forSearch) {

	try {
	    MatchRecordInterface mr = calculateScore(base, comparitor, forSearch);
	    return mr.toDocument();
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return null;
    }


    public Document validate(ITable table, Document doc) {
	boolean failed=false;
	boolean warning=false;
	StringBuilder error=new StringBuilder();
	try {
	    for (ITableColumn col : table.getColumns()){
		Object v=doc.getProjection(col.getColName());
		Object vo=v;
		//data type validation
		String lu=col.getDisplayType();
		String su = col.getSpecialUse();
		boolean validationMandatory = col.isSensitivityField();
		if (col.isNotNull() && (v==null || v.toString().isEmpty())) {
		    failed=true;
		    error.append(col.getColName()+" is Mandatory and is blank or empty");
		} else {

		    if (lu!=null && lu.length()!=0) {
			if (lu.equals("string")) {
			    //pattern validation
			    if (v!=null && v instanceof String) {
				if (su!=null && su.length()>0){
				    if (v!=null && !((String)v).matches(su)){
					if (validationMandatory) {
					    failed=true;  
					    error.append("[Error] "+col.getColName() + " does not match required text pattern, ");
					} else {
					    warning=true;
					    error.append("[Warning] "+col.getColName() + " does not match required text pattern, ");
					}
				    } 
				    
				}
			    } else if (v!=null){
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a text value, ");
			    } else if (v==null) {
				if (validationMandatory) {
				    failed=true;  
				    error.append("[Error] "+col.getColName() + " is blank, ");
				} else {
				    warning=true;
				    error.append("[Warning] "+col.getColName() + " is blank, ");
				}
			    }
			} else if (lu.equals("List")) {
			    if (!(v instanceof List)) {
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a list, ");

			    }
			} else if (lu.equals("Structure")) {
			    if (!(v instanceof Map)) {
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a structure, ");

			    }
			} else if (lu.equals("date")) {

			    Date vn = tryParseDate(table.getDateFormat(),v);
			    if (v!=null && vn==null) {
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a date, ");
			    }
			    v=vn;

			} else if (lu.equals("number")) {
			    Number vn=tryParseNumber(v);
			    if (v!=null && vn==null) {
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a number, ");
			    }
			    v=vn;
			    if (v!=null && su!=null && su.length()>0){
				//validate number range

				String[] parts= {};

				if (su.contains(">")) {
				    //>9999 or 0>9999
				    parts=su.split(">");


				    if (parts.length==1) {
					//>9999 
					Number n1=tryParseNumber(parts[0]);
					if (!(((Number)v).doubleValue() < n1.doubleValue())) {
					    if (validationMandatory) {
						failed=true;  
						error.append("[Error] "+col.getColName() + " is too small, ");
					    } else {
						warning=true;
						error.append("[Warning] "+col.getColName() + " is too small, ");
					    }
					}
				    } else 
					if (parts.length==2) {
					    //0>9999
					    Number n1=tryParseNumber(parts[0]);
					    Number n2=tryParseNumber(parts[1]);
					    if (!(((Number)v).doubleValue() > n1.doubleValue())) {
						if (validationMandatory) {
						    failed=true;  
						    error.append("[Error] "+col.getColName() + " is out of range, too small ");
						} else {
						    warning=true;
						    error.append("[Warning] "+col.getColName() + " is out of range, too small, ");
						}
					    }
					    if (!(((Number)v).doubleValue() < n2.doubleValue())) {
						if (validationMandatory) {
						    failed=true;  
						    error.append("[Error] "+col.getColName() + " is out of range, too large, ");
						} else {
						    warning=true;
						    error.append("[Warning] "+col.getColName() + " is out of range, too large, ");
						}
					    }
					}

				}

				else if (su.contains("<")) {
				    //<9999 or 0<9999
				    parts=su.split("<");


				    if (parts.length==1) {
					//<9999 
					Number n1=tryParseNumber(parts[0]);
					if (!(((Number)v).doubleValue() > n1.doubleValue())) {
					    if (validationMandatory) {
						failed=true;  
						error.append("[Error] "+col.getColName() + " is too large, ");
					    } else {
						warning=true;
						error.append("[Warning] "+col.getColName() + " is too large, ");
					    }
					}
				    } else 
					if (parts.length==2) {
					    //0<9999 ie less than 0 and greater than 9999
					    Number n1=tryParseNumber(parts[0]);
					    Number n2=tryParseNumber(parts[1]);
					    if (((Number)v).doubleValue() > n1.doubleValue() && ((Number)v).doubleValue() < n2.doubleValue()) {
						if (validationMandatory) {
						    failed=true;  
						    error.append("[Error] "+col.getColName() + " is not out of range, ");
						} else {
						    warning=true;
						    error.append("[Warning] "+col.getColName() + " is not in out of range, ");
						}
					    }
					}

				}


			    }

			} else if (v!=null && lu.equals("boolean") && !tryParseBoolean(v)) {
			    Boolean vn =tryParseBoolean(v);
			    if (v!=null && vn==null) {
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a boolean, ");
			    }
			    v=vn;
			} else if (v!=null && lu.equals("image")) {
			    try {
				new URL((String) v);
			    }	catch (Exception e) {
				failed=true;
				error.append("[Error] "+col.getColName() + " is not a url, ");
			    }


			} else if (lu.startsWith("table:")) {
			    if (v!=null) {
				//reference validation
				String[] bits=lu.substring(6).split("!");
				if (bits.length==2) {
				    ICollection icoll=this;
				    if (!bits[0].trim().equalsIgnoreCase("undefined") && !bits[0].trim().equalsIgnoreCase(getCollectionName()))
					icoll=this.parent.getCollection(bits[0].trim());
				    String k=getSchDoc().getTable(bits[1].trim()).getKeyField();
				    DBCursor c = icoll.find(new Document(k, v));
				    if (c.count()==0) {
					failed=true;
					error.append("[Error] "+col.getColName() + " is not a valid "+bits[1]+", ");
				    }
				}
			    } else {
				if (validationMandatory) {
				    failed=true;  
				    error.append("[Error] "+col.getColName() + " is a blank link, ");
				} else {
				    error.append("[Warning] "+col.getColName() + " is a blank link, ");
				    warning=true;
				}
			    }
			}
		    }
		}
		doc.setProjection(col.getColName(), v);


	    }
	} catch (Exception e) {
	    failed=true;
	    error.append(e.toString());
	} finally {

	    if (error.length()>0)
		doc.append("Error", error.toString());
	    else 
		doc.append("Error","");
	    if (failed)
		doc.append("Status", "Invalid");
	    else if (warning)
		doc.append("Status", "Warning");
	    else 
		doc.append("Status", "Valid");

	    doc.append("Table", table.getTableName());
	}
	return doc;
    }

    private Date tryParseDate(String dateformat, Object in) {
	try {
	    if (in instanceof String) {
		if (dateformat==null)
		    try {
			return new Date(Date.parse((String)in));
		    } catch (Exception e) {
			return null;
		    }
		else if (dateformat!=null ){
		    DateFormat df = new SimpleDateFormat(dateformat);
		    return df.parse((String)in);
		}
	    } else if (in instanceof Long) {
		return Date.from(Instant.ofEpochMilli((Long)in));
	    } else if (in instanceof Date)
		return (Date) in;
	} catch (Exception e) {
	    return null;
	}
	return null;
    }

    private Number tryParseNumber(Object in) {

	if (in instanceof String) {

	    try {
		return Double.parseDouble((String)in);

	    } catch (Exception e) {

	    }
	    try {
		return Float.parseFloat((String)in);

	    } catch (Exception e) {

	    }
	    try {
		return Long.parseLong((String)in);

	    } catch (Exception e) {

	    }
	    try {
		return Integer.parseInt((String)in);

	    } catch (Exception e) {

	    }

	} else if (in instanceof Long)
	    return (Long) in;
	else if (in instanceof Integer)
	    return (Integer) in;
	else if (in instanceof Double)
	    return (Double) in;
	else if (in instanceof Float)
	    return (Float) in;

	return null;
    }
    private Boolean tryParseBoolean(Object in) {
	if (in instanceof String) {

	    try {
		return Boolean.parseBoolean((String)in);

	    } catch (Exception e) {

	    }
	} else if (in instanceof Boolean)
	    return (Boolean) in;
	return null;
    }
    public Stream<Document> validate(String options, Stream<Document> in, Document globalOptions) {
	if (in != null){
	    ITable table=this.getSchDoc().getTable(options.replaceAll("\\\"|\\\\|\\'", ""));
	    if (table!=null) {
		return in.parallel()
			.map(doc -> {
			    return validate(table, doc);
			});
	    }
	}
	return null;
    }

    public Stream<Document> task(Document options, Stream<Document> in, Document globalOptions) {
	if (in != null){
	    return in.map(doc ->{
		if (doc.containsKey("Status")) {
		    String status=doc.getString("Status");
		    if (!status.equalsIgnoreCase("Valid")) {
			Document task=new Document();
			String attr=options.getString("Message");
			if (attr.contains("$")) {
			    StringBuilder sb = new StringBuilder();
			    for (String token : tokenise(attr)) {
				if (token.startsWith("$"))
				    sb.append(doc.getProjection(token.substring(1)));
				else sb.append(token);
			    }
			    attr=sb.toString();
			}
			task.append("Task", attr);
			Calendar c = Calendar.getInstance();
			task.append("Created", c.getTime().toGMTString());
			Number days=tryParseNumber(options.getString("TimeFrameDays"));
			if (days!=null)
			    c.add(Calendar.DAY_OF_MONTH,days.intValue());
			task.append("Due", c.getTime().toGMTString());
			task.append("Tablename", "["+doc.getString("Table")+"]");
			task.append("Resolved", "false");
			task.append("Table", "Tasks");
			task.append("Type", status);
			task.append("Comment", doc.getString("Error"));
			if (doc.get("_id")!=null) {
			    task.append("Nodes", List.of(doc.getString("_id")));
			    task.append("_id", doc.getString("_id")+doc.hashCode());
			} else 
			    task.append("_id", ""+doc.hashCode());



			if (options.containsKey("overrite") && options.getBoolean("overrite", false)) {
			    this.saveTask(task);
			} else {
			    DBCursor cc = this.findTasks(new Document("_id", task.getString("_id")));
			    if (!cc.hasNext()) 
				this.saveTask(task);
			    else { /////check to see if it has been resolved before
				Document d = cc.first();
				if (!d.getBoolean("Resolved", false))
				    this.saveTask(task);

			    }
			}
		    }
		}
		return doc;
	    });
	}
	return null;
    }

    private List<String> tokenise(String attr) {
	char[] ca = attr.toCharArray();
	List<String> tokens=new ArrayList<String>();
	StringBuilder sb=new StringBuilder();
	boolean intoken=false;
	for (int i=0; i<ca.length; i++) {
	    if (ca[i]=='$') {
		tokens.add(sb.toString());
		sb.setLength(0);
		intoken=true;
	    }
	    if (intoken && (ca[i]==' ' || ca[i]==',' ||  ca[i]==';' || ca[i]==')' || ca[i]=='(' || ca[i]==':')) {
		intoken=false;
		tokens.add(sb.toString());
		sb.setLength(0);
	    }
	    sb.append(ca[i]);

	}
	if (sb.length()>0)
	    tokens.add(sb.toString());
	return tokens;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#cluster(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> cluster(Document options, Stream<Document> in, Document globalOptions) {
	if (in != null){



	    if (options==null)
		options=new Document();
	    double score=0d;
	    if (options.containsKey("score"))
		score=options.getDouble("score");

	    Map<String, Document> results = new HashMap<String,Document>();
	    //look for matches amoungst the incoming set
	    //coerce them into concepts where the percentage is high enough
	    List<Document> docs = in.collect(Collectors.toList());
	    for (Document prep: docs) 
		prep.append("standardized", localCollection.standardise(prep));
	    for (Document filter: docs) {
		for (Document current: docs) {	
		    if (!current.getString("_id").equalsIgnoreCase(filter.getString("_id"))){
			if (!results.containsKey(current.getString("_id")+"/"+filter.getString("_id"))) {
			    Document result = score(filter, current, false);

			    result.append("from", filter.getString("_id"));
			    result.append("to", current.getString("_id"));
			    if (result.getDouble("score")>= score)
				results.put(filter.getString("_id")+"/"+current.getString("_id"), result); 
			}
		    }
		}
	    }

	    //cluster them based on score
	    Map<String, Long> clusters = new HashMap<String, Long>();

	    //find existing cluster that one or the other node might be in
	    //add this result to both clusters
	    long clusterID=0;
	    for (Document result : results.values()) {
		long currentClusterFrom=-1;
		long currentClusterTo=-1;
		if (clusters.containsKey(result.getString("from"))) {
		    currentClusterFrom=clusters.get(result.getString("from"));
		}

		if (clusters.containsKey(result.getString("to"))) {
		    currentClusterTo=clusters.get(result.getString("to"));
		} 

		if (currentClusterTo != -1 && currentClusterFrom != -1) {
		    //move the from cluster and all other members to the to cluster

		    for (String key : clusters.keySet())
			if (clusters.get(key) == currentClusterFrom )
			    clusters.replace(key, currentClusterTo);
		    clusters.put(result.getString("to"), currentClusterTo);
		    clusters.put(result.getString("from"), currentClusterTo);
		} else if (currentClusterTo != -1) {
		    clusters.put(result.getString("to"), currentClusterTo);
		    clusters.put(result.getString("from"), currentClusterTo);
		} else if (currentClusterFrom != -1) {
		    clusters.put(result.getString("to"), currentClusterFrom);
		    clusters.put(result.getString("from"), currentClusterFrom);
		} else {
		    clusterID++;
		    clusters.put(result.getString("to"), clusterID);
		    clusters.put(result.getString("from"), clusterID);
		}

	    }

	    // switch the map around so its clusterid -» list (results) -» list(documents)
	    // sort based on value
	    ValueComparator bvc =  new ValueComparator(clusters);		
	    TreeMap<String, Long> sortedmap=new TreeMap<String, Long>(bvc);
	    sortedmap.putAll(clusters);


	    long lastCluster=-1;
	    Set<Document> ret = new HashSet<Document>();
	    ArrayList<Document> cluster=new ArrayList<Document>();;
	    for (String node : sortedmap.keySet()){			
		long clusterNo = clusters.get(node);			
		if (clusterNo!=lastCluster && lastCluster!=-1){
		    ret.add(new Document("_id", clusterNo)
			    .append("size", cluster.size())
			    .append("cluster", cluster));
		    cluster=new ArrayList<Document>();

		}
		cluster.add(getDocument(node)); //
		lastCluster=clusterNo;
	    }
	    if (lastCluster!=-1 && cluster.size()>0)
		ret.add(new Document("_id", lastCluster)
			.append("size", cluster.size())
			.append("cluster", cluster));

	    return ret.parallelStream();
	} else return null;

    }
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#out(java.lang.String, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> out(String string, Stream<Document> in, Document options) {
	ICollection c2=parent.createCollection(string);
	ArrayList<Document> out=new ArrayList<Document>();
	in.forEach(doc -> {
	    c2.save((Document)doc);
	    if (!options.containsKey("noOutput") || options.getBoolean("noOutput", false))
	        out.add(doc);
	});
	return out.stream();

    }


    @Override
    public Stream<Document> writeRel(Document options, Stream<Document> in, Document goptions) {

	String direction=options.getString("direction");
	if (direction==null)
	    direction="BOTH";
	String _direction=direction;
	String idField=options.getString("idField");
	String parentField=options.getString("parentField");
	String relTypeField="relType";
	if (options.containsKey("relTypeField"))
	    relTypeField=options.getString("relTypeField");
	String relType=options.getString("relType");
	String _relTypeField=relTypeField;
	return in.map(doc -> {
	    String id=doc.getString(idField);
	    String parent=doc.getString(parentField);

	    //find both nodes
	    if (id!=null && parent !=null) {
		Document thisDoc = this.find(new Document(idField, id)).first();
		Document parentDoc = this.find(new Document(idField, parent)).first();
		String _relType=relType;
		if (thisDoc!=null && parentDoc!=null) {
		    if (doc.containsKey(_relTypeField))
			_relType=doc.getString(_relTypeField);
		    Set<Document> related=new HashSet<Document>();
		    if (_direction.equalsIgnoreCase("BOTH") || _direction.equalsIgnoreCase("TO")) {
			Document relationship = new Document("relType", _relType);
			relationship.append("fromCol", thisDoc.getString("_id"));
			relationship.append("toCol", parentDoc.getString("_id"));
			related.add(saveRelationship(relationship));
		    }
		    if (_direction.equalsIgnoreCase("BOTH") || _direction.equalsIgnoreCase("FROM")) {
			Document relationship = new Document("relType", _relType);
			relationship.append("fromCol", parentDoc.getString("_id"));
			relationship.append("toCol", thisDoc.getString("_id"));
			related.add(saveRelationship(relationship));
		    }
		    doc.append(relType, related);
		}
	    }
	    return doc;

	});

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#match(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> match(Document filter, Stream<Document> in, Document options) {
	if (in == null)
	    return findStream(filter);
	else {
	    return in.parallel().filter(doc -> ICollection.filter(doc, filter));
	}
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#analyse(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream analyse(Document fuzzyOptions, Stream<Document> in, Document options) {
	//consume the documents - this is a terminal operation on this stream
	if (in!=null)
	    in.collect(Collectors.toList());
	ArrayList<Document> ret = new ArrayList<Document>();
	IIndex _index = null;
	if (fuzzyOptions.containsKey("Index")) {
	    _index=getSchDoc().getIndex(fuzzyOptions.getString("Index"));

	    final IIndex index = _index;
	    if (fuzzyOptions.containsKey("InspectKey")) {

		Document entries = this.getIndexPrefixSubMap(index.getIndexName(), (index.getIndexName()+":"+fuzzyOptions.getString("InspectKey")+"|"), true);
		//BTreeMap collIndex = (BTreeMap)this.indexes.get(index.getIndexName());

		//

		//ConcurrentNavigableMap entries = collIndex.prefixSubMap(((index.getIndexName()+":"+fuzzyOptions.getString("InspectKey")+"|")).getBytes(), true);
		return entries
			.values()
			.stream()
			.map(id->{
			    return getDocument(id);
			});

	    } else if (fuzzyOptions.containsKey("GenerateKeyForID")) {
		Document doc = (Document) getDocument(fuzzyOptions.get("GenerateKeyForID"));
		String tableName = doc.getString("Table");
		ITable table=getSchDoc().getTable(tableName);
		java.util.Collection<String> useKeyList = getMatchKeys(doc, table, index, false);
		for (String k: useKeyList) {
		    ret.add(new Document ("Key", k));
		}

		return ret.stream();
	    }
	    else {

		Document mapout=keyCount(fuzzyOptions.getString("Index"));

		long mcount=0;
		if (mapout!=null)
		    for (Object key: mapout.keySet()) {
			if (((Long)mapout.get(key))>1)
			    ret.add(new Document("key", (String)key).append("count", mapout.get(key)));
		    }
		ret.add(new Document("key","ALLOTHERS").append("count", mcount));
		return sort(new Document("count", 1), ret.stream(), new Document());
	    } 
	} else if (fuzzyOptions.containsKey("Standardize")) {
	    String std=fuzzyOptions.getString("Standardize");
	    //BTreeMap collIndex = (BTreeMap)this.indexes.get(std);
	    if (fuzzyOptions.containsKey("StandardizeForID")) {
		Document doc = (Document) getDocument(fuzzyOptions.get("StandardizeForID"));
		Document standardised = localCollection.standardise(doc);
		ret.add(standardised);
		return ret.stream();
	    }  else if (fuzzyOptions.containsKey("InspectStandardized")) {
		Document doc = (Document) getDocument(fuzzyOptions.get("InspectStandardized"));
		if (doc==null) {
		    ret.add(new Document("Cause", "Record not found by ID"));

		} else {
		    Document storedStandardised = (Document) getStandardised("STD_1",fuzzyOptions.getString("InspectStandardized"));
		    Document standardised = localCollection.standardise(doc);
		    ret.add(standardised.append("Origin", "CALCULATED"));
		    if (storedStandardised!=null)
			ret.add(storedStandardised.append("Origin", "STORED"));

		    Document mr = score(doc.append("standardized", standardised), new Document("standardized", storedStandardised).append("Table", doc.getString("Table")), false);

		    if (mr==null  || mr.getDouble("score")!=100.0) {
			Document r=new Document("Action", "db."+this.definition.getString("Name")+".rebuildIndex(\""+std+"\")");
			if (mr==null || storedStandardised==null)
			    r.append("Cause", "Stored Standardised value was not found");
			else 
			    r.append("Cause", "Stored Standardised value was out of date, score="+mr.getDouble("score"));
			ret.add(r);
		    } else {
			if (mr!=null  || mr.getDouble("score")==100.0) {
			    Document r=new Document("Action", "NONE");
			    r.append("Cause", "Stored Standardised value was correct, score="+mr.getDouble("score"));
			    ret.add(r);


			}
		    }

		}


		return ret.stream();
	    }  else {
		//analyse std index - report existance of concept count
		Document retSumm = new Document();
		Document mapout = this.conceptCount(std);
		long mcount=0;
		for (Object key: mapout.keySet()) {
		    ret.add(new Document("concept", (String)key).append("count", mapout.get(key)));
		}

		ret.add(new Document("concept", "total_documents").append("count",count(new Document())));
		return sort(new Document("count", 1), ret.stream(), new Document());
	    }
	}
	return null;
    }

    public Stream<Document> spinOut(Document spinOptions, Stream<Document> in, Document options) {
	if (in != null){
	    return in
		    .parallel()
		    .map(doc -> {return spinOut(doc, spinOptions);})
		    .filter(doc -> doc!=null);
	} else return null;

    }

    public Stream<Document> getRelated(Document spinOptions, Stream<Document> in, Document options) {
	if (in != null){
	    return in
		    .parallel()
		    .map(doc -> {return getRelated(doc, spinOptions);})
		    .filter(doc -> doc!=null);
	} else return null;

    }


    @Override
    public DBCursor traverseTop(Document from, String relType){
	if (relType==null)
	    relType="parent";
	Document relFilter=new Document("relType", relType).append("fromCol", from.get("_id"));

	List<Document> list = new ArrayList<Document>();
	list.add(getDocument(from.getString("_id")));
	DBCursor c=null;
	do {
	    c = this.findRelationships(relFilter);
	    Optional<Document> first = c.stream().findFirst();
	    if (first.isPresent()) {
		Document doc=first.get();
		relFilter.append("fromCol", doc.get("toCol"));
		//get the actual doc
		list.add(getDocument(doc.getString("toCol")));
	    }
	    else 
		c=null;
	} while (c!=null);
	return new DBCursor(list);
    }

    private Document getRelated(Document doc, Document relOptions) {

	String relType=null;
	String into="children";
	String idField="_id";
	String nameField=null;

	int recurseLevels=1;
	int currentLevel=0;
	int direction=0;

	if (relOptions!=null) {
	    if (relOptions.containsKey("relType"))
		relType=relOptions.getString("relType");
	    if (relOptions.containsKey("_idName"))
		idField=relOptions.getString("_idName");
	    if (relOptions.containsKey("into"))
		into=relOptions.getString("into");
	    if (relOptions.containsKey("nameField"))
		nameField=relOptions.getString("nameField");
	    if (relOptions.containsKey("recurse"))
		recurseLevels=relOptions.getInteger("recurse");
	    if (relOptions.containsKey("level"))
		currentLevel=relOptions.getInteger("level");
	    if (relOptions.containsKey("direction"))
		direction=relOptions.getInteger("direction");
	} else relOptions=new Document();

	Set<Document> list = new HashSet<Document>();
	//for each related doc
	List<Document> orlist = new ArrayList<Document>();
	if (direction<=0)
	    orlist.add(new Document("fromCol", doc.get("_id")));
	if (direction>=0)
	    orlist.add(new Document("toCol", doc.get("_id")));
	Document filter = new Document("$or", orlist);

	if (relType!=null)
	    filter.append("relType", relType);
	DBCursor c = this.findRelationships(filter);

	relOptions.append("level", ++currentLevel);

	while (c.hasNext()) {
	    Document rel = c.next();
	    String toID=rel.getString("toCol");
	    if (toID.equalsIgnoreCase(doc.getString("_id")))
		toID=rel.getString("fromCol");
	    Document toDoc = getDocument(toID);
	    if(toDoc!=null) {
		if (currentLevel<recurseLevels) {

		    toDoc=getRelated(toDoc, relOptions.append("level", ++currentLevel));
		}
		if (!idField.equalsIgnoreCase("_id")) {
		    toDoc.remove("_id");
		    toDoc.append(idField, toID);
		}
		if (nameField!=null)
		    toDoc.append("name", toDoc.getString(nameField));
		list.add((Document) toDoc);
	    }


	}

	doc.append(into, list);
	if (!idField.equalsIgnoreCase("_id")) {
	    doc.append(idField, doc.get("_id"));
	    doc.remove("_id");
	}
	if (nameField!=null)
	    doc.append("name", doc.getString(nameField));
	return doc;



    }



    private Document rematch(Document doc, Document rematchOptions) {
	this.localCollection.match(doc);
	return doc;
    }


    public Stream<Document> rematch(Document rematchOptions, Stream<Document> in, Document options) {
	if (in != null){
	    return in
		    .parallel()
		    .map(doc -> {
			return rematch(doc, rematchOptions);
		    });
	} else return null;

    }

    private Document spinOut(Document doc, Document spinOptions) {


	if (spinOptions!=null) {
	    String type="Inner";
	    if (spinOptions.containsKey("joinType"))
		type=spinOptions.getString("joinType");
	    String relType=spinOptions.getString("relationshipType");
	    String into=spinOptions.getString("into");
	    List<String> projectors=spinOptions.getList("fields");
	    List<Document> list = new ArrayList<Document>();
	    //for each related doc
	    Document filter = new Document("fromCol", doc.get("_id"));
	    if (relType!=null)
		filter.append("relType", relType);
	    DBCursor c = this.findRelationships(filter);

	    if (into==null) {
		into=relType;
	    }


	    while (c.hasNext()) {
		Document rel = c.next();
		Document to = new Document();

		for (String projField : projectors) {
		    String toID=rel.getString("toCol");
		    Document toDoc = getDocument(toID);
		    if(toDoc!=null) {
			Object proj = toDoc.getProjection(projField);
			to.append(projField, proj);
		    }
		}
		list.add(to);
	    }

	    doc.append(into, list);
	    if ((type.equalsIgnoreCase("Inner") && list.size()>0) || type.equalsIgnoreCase("Outer"))
		return doc;
	}

	return null;
    }
    public Stream<Document> coerce(Document options, Stream<Document> in, Document globalOptions) {
	//coerce has to be after cluster and each document must contain a list called cluster

	if (in != null){
	    if (options==null)
		options=new Document();
	    final Document _options = options;
	    return in
		    .parallel()
		    .map(doc -> {
			if (doc.containsKey("cluster")) {
			    List<Document> cluster = doc.getList("cluster");
			    doc.remove("cluster");
			    for (Document cldoc : cluster) {
				//pull out all the concepts
				Document std = localCollection.standardise(cldoc);
				Set keySet = std.keySet();
				if (_options.containsKey("concepts"))
				    keySet=_options.getAsDocument("concepts").keySet();
				for (Object purpose : keySet) {
				    for (Object stdy : MatchRule.deserialise(std.getList((String)purpose))) {
					StringBuilder value = new StringBuilder();
					for (String v : ((Standardized)stdy).getComparitorWords())
					    value.append(v + " ");
					doc.append((String)purpose,value.toString().trim());
				    }
				}
				if (_options.containsKey("projection")) {
				    Document projections = _options.getAsDocument("projection");
				    for (Object key : projections.keySet()) {
					String projType=projections.getString((String)key);
					if (projType==null || projType.equalsIgnoreCase("1"))
					    projType="first";
					if (projType.equalsIgnoreCase("first")) {
					    if (cldoc.get(key)!=null) {
						doc.append((String)key, cldoc.get(key));
						projections.replace(key, "done");
						_options.append("projection", projections);
					    }

					} else if (projType.equalsIgnoreCase("last")) {
					    if (cldoc.get(key)!=null) {
						doc.append((String)key, cldoc.get(key));

					    }

					}

				    }
				}
			    }

			    return doc;
			} else return null;

		    });
	} 
	return null;

    }
    public Map<Document, List<Document>> split(List<Document> splitter, Stream<Document> in, Document globalOptions) {
	if (in != null){
	    /* options need to look like this:
	     * [ { 'pipeline': [ ... ], 'filter' : { 'field': 'value' } }, { 'pipeline': [ ... ], 'filter': { 'field': 'value' } } ]
	     */


	    if (splitter==null) {
		splitter=new ArrayList<Document>();
		splitter.add(new Document("filter", new Document())); //does nothing
	    }

	    Map<Document, Document> groupBy = new HashMap<Document, Document>();
	    for (Map splitStepo: splitter) {
		Document splitStep = new Document(splitStepo);
		Document pipeline=null;
		Document filter=null;
		if (splitStep.containsKey("pipeline"))
		    pipeline=new Document("pipeline", splitStep.getList("pipeline")).append("options", globalOptions);
		else
		    pipeline=new Document("pipeline", new ArrayList()).append("options", globalOptions);
		if (splitStep.containsKey("filter"))
		    filter=splitStep.getAsDocument("filter");
		else
		    filter=new Document();


		groupBy.put(pipeline, filter);
	    }


	    Map<Document, List<Document>> ret = new HashMap<Document, List<Document>> ();
	    for (Document pipeline : groupBy.keySet()) 
		ret.put(pipeline, new ArrayList<Document>());
	    in.forEach(doc -> {
		for (Document pipeline : groupBy.keySet()) {
		    Document filter = groupBy.get(pipeline);
		    if (ICollection.filter(doc, filter)) {
			ret.get(pipeline).add(doc);
		    }
		}
	    });
	    return ret;
	    //return should be via pipeline not filter.
	}
	return null;
    }

    private Document groupingBy(Document doc, ArrayList<Document> groupBy) {
	Document key = new Document();
	// returns the pipeline for the successful filter
	for (Document filter : groupBy)
	    if (ICollection.filter(doc, filter)) {
		key=filter;
		break;
	    }
	return key;
    }
    public Stream<Document> project(Document options, Stream<Document> in, Document globalOptions) {
	if (in != null){
	    if (options==null)
		options=new Document();
	    final Document _options = options;
	    return in
		    .parallel()
		    .map(doc -> {
			Document ret = new Document();
			for (Object member : _options.keySet())
			    if (_options.getInteger((String)member, 0)==1)
				ret.append((String)member, doc.getProjection((String)member));
			return ret;
		    });
	} 
	return null;
    }

    

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#script(java.util.Map, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> script(Document scriptStatements, Stream<Document> in, Document metadata) {
	StringBuilder sb = new StringBuilder("var target={};\n");
	for (Object target : scriptStatements.keySet()) {
	    sb.append("target[\'"+target + "\'] =" + scriptStatements.get(target)+";\n");
	}
	sb.append("JSON.stringify(target);"); //returns the target object

	ScriptEngine engine = new ScriptEngineManager(null).getEngineByName("nashorn");;

	final String script=sb.toString();

	return in.map(doc -> {
	    try {
		Bindings bindings= new SimpleBindings();
		for (Object source : doc.keySet()) {
		    if (source!=null && ((String)source).length()>0 && doc.get(source) !=null)
			bindings.put( (String)source, doc.get(source));
		}

		String output =(String) engine.eval(script, bindings);
		Document dout= Document.parse(output);

		return expandDoc(dout, metadata);
	    } catch (ScriptException e) {
		e.printStackTrace();
	    }
	    return doc;
	});
    }


    private static Document expandDoc(Document flatdoc, Document metadata) {
	Document temp = new Document();


	for (Object key : flatdoc.keySet()){
	    Object value =  flatdoc.get(key);
	    if (key!=null && value!=null && !(value instanceof String && ((String)value).length()==0)){
		//key==LEI.Entity.Registration.regDate
		String[] parts = ((String)key).split("\\.");
		Object parent=temp;
		String partInc="";
		for (String part : parts){
		    partInc+=part;

		    String colName = partInc.replaceAll("\\[[0-9]*?\\]", "");
		    String type = null;
		    if (metadata.getProjection(colName) instanceof List)
			type="List";
		    else if (metadata.getProjection(colName) instanceof Map)
			type="Structure";

		    if (type==null || type.length()==0)
			type="text";

		    int levelinstance=0;
		    if (part.contains("[")){
			String[] partSplit= part.split("\\[");
			part=partSplit[0];
			levelinstance=Integer.parseInt(partSplit[1].replace("]", ""));
		    }
		    if (type.equalsIgnoreCase("Structure")){
			Object newParent=new Document();
			if (parent instanceof Document){
			    if (!((Document)parent).containsKey(part))
				((Document)parent).append(part, newParent);
			    else newParent=((Document)parent).get(part);
			}
			else if (parent instanceof ArrayList){
			    //expand the array to enable us to place an item at the level
			    if (levelinstance>((ArrayList)parent).size()-1)
				for (int expand=((ArrayList)parent).size()-1; expand<levelinstance; expand++)
				    ((ArrayList)parent).add(new Document());
			    //get the current document?
			    Document listparent = ((Document) ((ArrayList)parent).get(levelinstance));
			    Document listitem = (Document) listparent.get(part);
			    if (listitem==null)
				listitem=new Document();
			    if (!(listparent).containsKey(part))
				listparent.append(part, newParent);
			    else 
				newParent=listitem;
			    //((ArrayList)parent).set(levelinstance,listparent);
			}
			parent=newParent;

		    } else if (type.equalsIgnoreCase("List")){
			Object newParent=new ArrayList();
			if (parent instanceof Document)
			    if (!((Document)parent).containsKey(part))
				((Document)parent).append(part, newParent);
			    else newParent=((Document)parent).get(part);
			else if (parent instanceof ArrayList){
			    if (levelinstance>((ArrayList)parent).size()-1)
				for (int expand=((ArrayList)parent).size()-1; expand<levelinstance-1; expand++)
				    ((ArrayList)parent).add(new Document());
			    ((ArrayList)parent).set(levelinstance, new Document(part,newParent));
			}
			parent=newParent;
		    } else {///text-date-etc
			if (parent instanceof Document)
			    ((Document)parent).append(part, value);
			else if (parent instanceof ArrayList){
			    //get the position in the array defined by instance - it will be a document
			    //add this value to it.
			    Object instancearray = null;
			    if (((ArrayList)parent).size()>levelinstance)
				instancearray = ((ArrayList)parent).get(levelinstance);
			    if (instancearray==null)
				instancearray=new Document();
			    if (instancearray instanceof Document)
				((Document)instancearray).append(part, value);
			}
		    }
		    partInc+=".";
		}
	    }
	}
	return temp;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getMatchTypes()
     */
    @Override
    public DBCursor getMatchTypes() {
	Map<String, Document> mt = getSchDoc().getMatchTypes();
	List<Document> out = new ArrayList<Document>();
	for (String k : mt.keySet())
	    out.add(mt.get(k).append("Name", k));
	return new DBCursor(out);    
    }

    @Override 
    public DBCursor getFlows() {
	return getResources("FLW");

    }

    @Override 
    public DBCursor getFlow(String name) {
	return getResource(name);

    }

    @Override 
    public DBCursor getBooks() {
	return getResources("GML");

    } 

    private DBCursor getResources(String suffix) {
	File  file=new File(IdentizaSettings.getApplicationRootPath("")+File.separator+"root"+File.separator+getDatabase().getName()+File.separator+getName());
	List<Document> out = new ArrayList<Document>();
	if (file.isDirectory())
	    for (File d : file.listFiles()) {
		if (d.isFile() && (d.getName().toUpperCase().endsWith("."+suffix))){
		    Document metadoc=this.parent.getCollection("$custom").find(new Document("collection", this.getName())
			    .append("database", parent.getName()).append("name", d.getName())).first();
		    if (metadoc==null)
			metadoc=new Document();
		    metadoc.append("Name", d.getName()).append("Path", d.getAbsolutePath());

		    out.add(metadoc);
		}

	    }
	return new DBCursor(out);
    }

    private DBCursor getResource(String name) {
	File  file=new File(IdentizaSettings.getApplicationRootPath("")+File.separator+"root"+File.separator+getDatabase().getName()+File.separator+getName()+File.separator+name);

	if (file.exists() && file.isFile()) {
	    JsonParser parser = new JsonParser();
	    JsonElement json;
	    try {
		StringBuilder sb = new StringBuilder();

		BufferedReader br = (new BufferedReader(new FileReader(file)));
		br.lines().sequential().forEach(line -> {
		    sb.append(line);
		});
		json = parser.parse(sb.toString());
		return new DBCursor(new Document(json));
	    } catch (JsonIOException | JsonSyntaxException | FileNotFoundException e) {
		e.printStackTrace();
	    }

	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getDeltas()
     */
    @Override
    public Stream<Document> getDeltas() {
	return replicaConnections
		.values()
		.stream()
		.map(collection -> collection.getDeltas())
		.flatMap(s -> s);
    }

    

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getNodes()
     */
    @Override
    public Stream<Document> getNodes() {
	Document out=new Document();
	for (Integer nodenum : replicaConnections.keySet()) {
	    out.append(""+nodenum, replicaConnections.get(nodenum).getDefinition());
	}
	return Stream.of(out);
    }



}
