/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;


import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.util.ArrayList;


import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jms.JMSException;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;

import org.apache.activemq.command.ActiveMQTextMessage;
import org.apache.commons.io.FileUtils;

import org.mapdb.BTreeMap;
import org.mapdb.DB;
import org.mapdb.HTreeMap;
import org.mapdb.IndexTreeList;
import org.mapdb.Serializer;
import org.mapdb.serializer.SerializerCompressionWrapper;
import org.mapdb.DBMaker.Maker;

import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import com.entitystream.monster.geo.GeoType;

import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.entity.resolve.match.Indexable;
import com.entitystream.identiza.entity.resolve.match.MatchRecordInterface;
import com.entitystream.identiza.entity.resolve.match.Matchable;
import com.entitystream.identiza.entity.resolve.metadata.IIndex;
import com.entitystream.identiza.entity.resolve.metadata.IPurpose;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumn;
import com.entitystream.identiza.entity.resolve.metadata.PurposeColumnMap;
import com.entitystream.identiza.entity.resolve.metadata.Rule;
import com.entitystream.identiza.entity.resolve.metadata.IRule;
import com.entitystream.identiza.entity.resolve.metadata.SchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.Table;
import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.Index;
import com.entitystream.identiza.entity.resolve.metadata.Purpose;
import com.entitystream.identiza.entity.resolve.types.MatchProcInterface;
import com.entitystream.identiza.entity.resolve.types.MatchProcDefinition;
import com.entitystream.identiza.entity.resolve.types.Standardized;


public class CollectionLocal extends Indexable implements Serializable, ICollection, Matchable {

    private static final String STANDARDINDEX = "STD_1";
    private transient DB _catalog = null;
    private HTreeMap indexCatalogStore;
    private HTreeMap data;
    private Map indexes;
    private Document definition;
    private Database owningDB;
    private Collection parent;
    private Map<Object, Lock> indexLock;
    private boolean explaining=false;
    private Logger logger = Logger.getAnonymousLogger();
    private ISchemaMeta schDoc;
    private String schemaName;
    private MatchCollection matchCollection;



    private Document replicaRouter = new Document();
    private ReplicaType replicaType =ReplicaType.Replicated;
    private int nodeNum=0;
    private String myKey;
    private List<String> replicaSet=new ArrayList<String>();
    private String hashKey="_id";

    private Map<String, IIndex> indexs = new HashMap<String, IIndex>();
    private Map<String, ITable> tables = new HashMap<String, ITable>();

    private Map rollupCubeIO=new HashMap();
    private Map rollupCube=new HashMap();
    private Map<String, String> formulaMap=new HashMap();
    private ScriptEngine engine;
    private IndexTreeList delta;
    private Lock deltaLock;


    private DB getData() {
	return this.getData(owningDB.isReadOnly());
    }

    private DB getData(boolean allowRO) {
	if (_catalog==null) {

	    File fpath = new File(definition.getString("DBPath")+definition.getString("Path"));
	    if (!fpath.exists())
		fpath.mkdirs();
	    String path=fpath.getAbsolutePath()+File.separator+definition.getString("Name");
	    Maker _data = org.mapdb.DBMaker.fileDB(path).fileMmapEnable().checksumHeaderBypass().closeOnJvmShutdown();
	    try {

		if (allowRO)
		    _data=_data.readOnly();
		if (_data!=null)
		    _catalog = _data.make();
	    } catch (Exception e) {
		logger.severe("Collections can't be created on a non existent database, " + e.toString());
		//e.printStackTrace();
	    }

	}

	return _catalog;
    }

    protected CollectionLocal(Collection parent, Document collDoc, boolean initialised) {
	this.definition=collDoc;
	this.myKey=definition.getString("HostKey");
	this.parent=parent;
	this.owningDB=parent.getDatabase();

	definition.append("DBPath", owningDB.getDataPath());



	this.nodeNum=this.owningDB.getNodeNum();

	if (definition.containsKey("Ranges")) {
	    replicaSet=owningDB.getReplicaSet();
	    definition.append("ReplicaSet", replicaSet);

	    replicaType =ReplicaType.Range;
	    replicaRouter = definition.getAsDocument("Ranges").getAsDocument(""+nodeNum);	
	    if (replicaRouter==null)
		replicaRouter=new Document();

	}

	if (definition.containsKey("HashKey")) {
	    replicaSet=owningDB.getReplicaSet();
	    definition.append("ReplicaSet", replicaSet);
	    replicaType =ReplicaType.Hash;
	    hashKey=definition.getString("HashKey");
	}


	try {
	    indexLock = new LinkedHashMap<Object, Lock>();
	    indexCatalogStore=getData().hashMap(definition.getString("Name")+"_Cat").keySerializer(Serializer.STRING).createOrOpen();
	    data=getData().hashMap(definition.getString("Name")+"_Data").keySerializer(Serializer.STRING).valueSerializer(SerializerCompressionWrapper.JAVA).createOrOpen();
	    indexes= new LinkedHashMap<String, BTreeMap>();


	    //open the maps required
	    if (!initialised)
		initialiseCollection();
	    store(owningDB);

	    //because data can be stored in a lambda - then we must get this collection ahead of time to avoid multiple threads opening it

	    getRelCollection();
	    getTaskCollection();

	} catch (Exception e) {
	    e.printStackTrace();
	}
    }

    private boolean isSpecialCollection() {
	return definition.getString("Name").endsWith("_TASKS") || definition.getString("Name").endsWith("_RELS") ;
    }


    private void initialiseCollection() {

	if (!indexCatalogStore.containsKey("_id_1")) {
	    indexCatalogStore.put("_id_1", new Document("Fields", Collections.singletonList(new Document("Name", "_id").append("Order", 1))).append("Options", new Document("name", "_id_1").append("dirty", 1)));
	}
	if (definition.containsKey("Delta")) {
	    this.deltaLock=new ReentrantLock();
	    this.delta=(IndexTreeList) getData().indexTreeList("Delta", SerializerCompressionWrapper.STRING).createOrOpen();
	}

	if (definition.containsKey("Definition")) {
	    if (!indexes.containsKey(STANDARDINDEX))
		indexes.put(STANDARDINDEX, getData().treeMap(STANDARDINDEX).keySerializer(Serializer.BYTE_ARRAY).valueSerializer(Serializer.JAVA).createOrOpen());


	    this.schDoc = SchemaMeta.createSchemaMeta(definition.getAsDocument("Definition"));
	    super.initialize(schDoc);
	    this.schemaName=getSchDoc().getName();
	    for (IIndex index : getSchDoc().getIndexes()) {
		if (!indexCatalogStore.containsKey(index.getIndexName())) {
		    String matchProcsString = new Gson().toJson(getSchDoc().getMatchProcs(index.getIndexName(), ""+index.getInstance(),""));
		    indexCatalogStore.put(index.getIndexName(), new Document("Options", new Document("fuzzy", true)
			    .append("matchProcs", matchProcsString)
			    .append("name", index.getIndexName())));
		}
	    }
	    indexLock.put(STANDARDINDEX, new ReentrantLock());
	    if (schDoc.isAutoMatch()) {
		if (matchCollection==null)
		    matchCollection = new MatchCollection(parent, getRelCollection(), getTaskCollection(), schDoc);
		else
		    matchCollection.initialize(schDoc);
	    }
	}

	

	for (Object id : indexCatalogStore.keySet()) {
	    indexLock.put(id, new ReentrantLock());
	    if (!indexes.containsKey(id)) {
		Document indexDoc = (Document) indexCatalogStore.get(id);
		List<Map> fieldsList = indexDoc.getList("Fields");
		Document cidoc = new Document();
		for (Map fldoc : fieldsList){
		    cidoc.append((String)fldoc.get("Name"), fldoc.get("Order"));
		}
		createIndex(cidoc, indexDoc.getAsDocument("Options"));

	    }
	}


	indexs = new HashMap<String, IIndex>();
	tables = new HashMap<String, ITable>();
	indexCatalogCache.putAll(indexCatalogStore);

    }



    protected CollectionLocal(Collection parent, Document collDoc) {
	this(parent, collDoc, false);

    }



    public void disconnect() {
	if (matchCollection!=null)
	    matchCollection.disconnect();
	matchCollection=null;
	if(data!=null && !data.isClosed())
	    data.close();

    }


    public void setAutoMatch(boolean a) {
	if (schDoc!=null) {
	    schDoc.setAutoMatch(a);
	    if (!a){
		if (matchCollection!=null)
		    matchCollection.disconnect();
		matchCollection=null;

	    }
	    updateDefinition();
	}
    }

    public void addTrigger(String name) {

	if(matchCollection !=null) {
	    matchCollection.addTrigger(name);
	    definition.append("Trigger", name);
	}
	else
	    System.out.println("Trigger ("+name+") was attempted to be applied to non fuzzy collection " + this.getCollectionName());
    }

    public String getTrigger() {
	return definition.getString("Trigger");
    }

    private void store(Database parent) throws Exception {
	if (owningDB !=null)
	    owningDB.storeCollection(definition.getString("Name"), this.parent);
	else 
	    throw new Exception("Collection could not be altered");
    }


    @Override
    public void createUniqueIndex(Document fields) {
	Document options = new Document("unique",true);
	createIndex(fields,options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#createIndex(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public void createIndex(Document fields, Document options) {
	try {
	    String name = "";
	    Boolean unique = options.getBoolean("unique", false);
	    Boolean fuzzy = options.getBoolean("fuzzy", false);
	    List<Document> newfields = new ArrayList<Document>();

	    for(Object k : fields.keySet()) {
		Document df = new Document("Name", k);
		df.append("Order", 1);
		if (fields.get((String)k) instanceof Integer)
		    df.append("Order", fields.getInteger((String)k));
		else if (fields.get((String)k) instanceof String){
		    if (fields.getString((String)k).equalsIgnoreCase("2d"))
			options.append("geo", "2d");
		    else if (fields.getString((String)k).equalsIgnoreCase("2dsphere"))
			options.append("geo", "2dsphere");
		}

		newfields.add(df);

		if (name.length()!=0)
		    name+="_";
		name+=(String)k;
	    }
	    name+="_1";

	    if (options.getString("name")!=null)
		name=options.getString("name");

	    if (fuzzy) {
		String matchProcsJson = options.getString("matchProcs");
		Type listType = new TypeToken<List<MatchProcDefinition>>() {}.getType();
		List<MatchProcDefinition> defs = new Gson().fromJson(matchProcsJson, listType);

		if (!matchProcs.containsKey(name))
		    matchProcs.put(name, new LinkedHashMap<String, MatchProcInterface>());
		for (MatchProcDefinition def : defs) {
		    String procname = def.getProcName();
		    matchProcs.get(name).put(procname, def.build());
		}
	    }

	    if (!indexCatalogStore.containsKey(name))
		if (!owningDB.isReadOnly())
		    indexCatalogStore.put(name, new Document("Fields", newfields).append("Options", options).append("dirty", 1));

	    indexLock.put(name, new ReentrantLock());

	    if (!indexes.containsKey(name)) {
		BTreeMap<byte[], String> datIndex = getData().treeMap(name).keySerializer(Serializer.BYTE_ARRAY).valueSerializer(Serializer.STRING).createOrOpen();
		indexes.put(name, datIndex);
		Document indexC= ((Document)indexCatalogStore.get(name));
		if(indexC.getInteger("dirty",0)==1) {
		    //clear the flag to mallow it
		    indexC.remove("dirty");
		    indexCatalogStore.put(name, indexC);
		    indexCatalogCache.putAll(indexCatalogStore);
		    rebuildIndex(name);
		}
	    }

	    store(owningDB);
	} catch (Exception e) {
	    e.printStackTrace();

	}



    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#createIndex(com.entitystream.monster.db.Document)
     */
    @Override
    public void createIndex(Document document) {
	createIndex(document, new Document());

    }

    @Override
    public void dropIndex(String name) {
	if (!owningDB.isReadOnly()) {
	    indexLock.get(name).lock();

	    if (indexCatalogStore.containsKey(name))
		indexCatalogStore.remove(name);

	    indexCatalogCache.clear();
	    indexCatalogCache.putAll(indexCatalogStore);

	    if (indexes.containsKey(name)) {
		((BTreeMap)indexes.get(name)).clear();
		indexes.remove(name);
	    }

	    indexLock.get(name).unlock();

	    if (indexLock.containsKey(name))
		indexLock.remove(name);
	}
    }

    @Override
    public void rebuildIndex(String indexName) {

	Document indexC=(Document) indexCatalogStore.get(indexName);
	if (indexC!=null && indexC.getInteger("dirty",0)==1) {
	    System.out.println("Index rebuild already in progress, request ignored " + indexName);
	    indexLock.get(indexName).unlock();
	    return;
	}


	//Executor exec = Executors.newFixedThreadPool(1);
	//exec.execute(new Runnable() {

	//	@Override
	//	public void run() {
	if (!owningDB.isReadOnly()) {
	    if (indexC!=null) {
		indexC.append("dirty", 1);
		indexCatalogStore.put(indexName,indexC);
	    }
	    if (indexLock.containsKey(indexName)) {
		indexLock.get(indexName).lock();
		BTreeMap index = (BTreeMap) indexes.get(indexName);
		index.clear();
		indexLock.get(indexName).unlock();

		bulkIndex(indexName, data.values().stream());

		if (indexC!=null) {
		    indexC.remove("dirty");
		    indexCatalogStore.put(indexName,indexC);
		}
	    }
	}

	//		}

	//	});




    }

    public String getCollectionName() {
	return definition.getString("Name");
    }



    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#find(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor find(Document filter ) {
	Document explainPlan=new Document();
	long t = System.currentTimeMillis();
	DBCursor _return = find(filter, explainPlan);
	t = System.currentTimeMillis()-t;
	if (explaining) {
	    explainPlan.append("Elapsed", t + "ms");
	    logger.info(explainPlan.toJson());
	}
	return _return;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#find()
     */
    @Override
    public DBCursor find() {
	Document explainPlan=new Document();
	long t = System.currentTimeMillis();
	DBCursor _return = find(explainPlan);
	t = System.currentTimeMillis()-t;
	if (explaining ){
	    explainPlan.append("Elapsed", t + "ms");
	    logger.info(explainPlan.toJson());
	}
	return _return;
    }







    public Stream<Document> findStream(Document filter, Document explainPlan ) {
	int eval=evaluate(filter,explainPlan);
	List hints = explainPlan.getList("Hint");
	explainPlan.append("Streamed", "true");

	if(eval>0 && hints.size()==0) {
	    return fullScanStream(filter, explainPlan);
	} else if (eval>0 && hints.size()>0){
	    return indexScan(filter, explainPlan, hints).stream();
	} else {

	    List<Document> results = new ArrayList();
	    boolean basic=true;
	    if (filter==null)
		filter=new Document();
	    for (Object k : filter.keySet())
		if (((String)k).startsWith("$")) {
		    ArrayList<Document> innerExplainList = new ArrayList<Document>();
		    explainPlan.append((String)k,innerExplainList);
		    basic=false;
		    if (((String)k).equalsIgnoreCase("$or")) {
			//iterate over the list

			for (Document innerFilter : (List<Document>)filter.get(k)) {
			    Document innerExplain = new Document();
			    innerExplainList.add(innerExplain);
			    for (Document martin : find(innerFilter, innerExplain))
				results.add(martin);

			}
		    }
		    if (((String)k).equalsIgnoreCase("$and")) {
			//iterate over the list finding the intersection at the end

			boolean first=true;
			HashSet keyList=new HashSet();
			for (Document innerFilter : (List<Document>)filter.get(k)) {
			    Document innerExplain = new Document();
			    innerExplainList.add(innerExplain);
			    if (first) {
				for (Document martin: find(innerFilter, innerExplain)) {
				    keyList.add(martin.getString("_id"));
				    results.add(martin);
				}
			    }
			    else {

				Iterable<Document> newSet = find(innerFilter, innerExplain);
				HashSet newKeyList=new HashSet();
				for (Document martin: newSet) 
				    newKeyList.add(martin.getString("_id"));

				///{a,b,c} + {b + c} = {a}
				HashSet deleteKeyList = new HashSet();
				deleteKeyList.addAll(keyList);
				deleteKeyList.removeAll(newKeyList);
				//remove all removed keys from the results
				HashSet deleteDocs = new HashSet();
				for (Document martin : results) {
				    if (deleteKeyList.contains(martin.getString("_id")))
					deleteDocs.add(martin);

				}

				///{a,b,c} + {b + c} = {b + c }
				results.removeAll(deleteDocs);
			    }
			    first=false;
			}
		    }
		}
	    if (basic) {
		return findInternal(filter, explainPlan);
	    }
	    else
		return results.stream();
	}


    }


    public DBCursor find(Document filter, Document explainPlan ) {
	int eval=evaluate(filter,explainPlan);
	List hints = explainPlan.getList("Hint");

	if(eval>0 && hints.size()==0) {
	    return fullScan(filter, explainPlan);
	} else if (eval>0 && hints.size()>0){
	    return indexScan(filter, explainPlan, hints);
	} else {

	    List<Document> results = new ArrayList();
	    boolean basic=true;
	    for (Object k : filter.keySet())
		if (((String)k).startsWith("$")) {
		    ArrayList<Document> innerExplainList = new ArrayList<Document>();
		    explainPlan.append((String)k,innerExplainList);
		    basic=false;
		    if (((String)k).equalsIgnoreCase("$or")) {
			//iterate over the list

			for (Document innerFilter : (List<Document>)filter.get(k)) {
			    Document innerExplain = new Document();
			    innerExplainList.add(innerExplain);
			    for (Document martin : find(innerFilter, innerExplain))
				results.add(martin);

			}
		    }
		    if (((String)k).equalsIgnoreCase("$and")) {
			//iterate over the list finding the intersection at the end

			boolean first=true;
			HashSet keyList=new HashSet();
			for (Document innerFilter : (List<Document>)filter.get(k)) {
			    Document innerExplain = new Document();
			    innerExplainList.add(innerExplain);
			    if (first) {
				for (Document martin: find(innerFilter, innerExplain)) {
				    keyList.add(martin.getString("_id"));
				    results.add(martin);
				}
			    }
			    else {

				Iterable<Document> newSet = find(innerFilter, innerExplain);
				HashSet newKeyList=new HashSet();
				for (Document martin: newSet) 
				    newKeyList.add(martin.getString("_id"));

				///{a,b,c} + {b + c} = {a}
				HashSet deleteKeyList = new HashSet();
				deleteKeyList.addAll(keyList);
				deleteKeyList.removeAll(newKeyList);
				//remove all removed keys from the results
				HashSet deleteDocs = new HashSet();
				for (Document martin : results) {
				    if (deleteKeyList.contains(martin.getString("_id")))
					deleteDocs.add(martin);

				}

				///{a,b,c} + {b + c} = {b + c }
				results.removeAll(deleteDocs);
			    }
			    first=false;
			}
		    }
		}
	    if (basic) {


		return new DBCursor(findInternal(filter, explainPlan).collect(Collectors.toList()));
	    }
	    else
		return new DBCursor(results);
	}


    }

    private DBCursor indexScan(Document filter, Document explainPlan, List hints) {
	//Uses the hints from the evaluator to scan 


	explainPlan.append("Collection", this.definition.getString("Name")).append("Filter", filter).append("Plan", "IndexScan");
	//pick the best index
	//first is unique
	//then smaller
	//ohh fuck it - just use the first
	String indexName = (String) hints.get(0);
	BTreeMap indx = (BTreeMap)(indexes.get(indexName));


	return new DBCursor(((java.util.Collection<String>)indx.values())
		.stream()
		.parallel()
		.map(pointer->(Document)data.get(pointer))
		.filter(doc->ICollection.filter((Document)doc, filter))
		.collect(
			Collectors.toList()
			));


    }

    public int evaluate(Document filter, Document explainPlan) {
	int fullscans=0;

	Document bestIndex=null;
	if (filter!=null)
	    for (Object k : filter.keySet()) {
		if (((String)k).startsWith("$")) {

		    if (((String)k).equalsIgnoreCase("$or")) {
			for (Document innerFilter : (List<Document>)filter.get(k)) {
			    fullscans +=evaluate(innerFilter,explainPlan);
			}
		    }
		    if (((String)k).equalsIgnoreCase("$and")) {
			//if any hit an index, we should direct the query to it.
			List<Document> filters = (List<Document>)filter.get(k);
			for (Document innerFilter : filters) {
			    int _fullscans =evaluate(innerFilter,explainPlan);
			    if (_fullscans >= filters.size())
				fullscans++;

			}
		    }
		} else {
		    //if any hit an index, we should direct the query to it. same as the and
		    Document _index=getIndex(filter);
		    if (_index.getString("IndexName").equalsIgnoreCase("DataScan")) 
			fullscans++;
		    else {
			if (!explainPlan.containsKey("Hint"))
			    explainPlan.append("Hint", new ArrayList());
			explainPlan.getList("Hint").add(_index.getString("IndexName"));
		    }
		}
	    }

	if (fullscans>0)
	    explainPlan.append("ScanCostReduction", 100-(100/fullscans)+ "%");
	else
	    explainPlan.append("ScanCostReduction", "0%");

	return fullscans;

    }



    private Set indexSeek(org.mapdb.BTreeMap indx, Set<Object> keys, Document indexPlan) {

	double hitRatio = indexPlan.getDouble("HitRatio");
	boolean uniqueIndex=indexPlan.getBoolean("Unique", false);
	String indexMethod=indexPlan.getString("IndexMethod");
	HashSet output = new HashSet();

	for (Object key : keys) {
	    Object entry = null;

	    //is this an IndexLookup and a perfect score?
	    if (indexMethod.equalsIgnoreCase("IndexLookup")) {
		//key is always a string
		if (hitRatio==1.0d) {
		    if (uniqueIndex) {
			entry = indx.get(((String)key).getBytes());
		    } else {
			entry = indx.prefixSubMap((((String)key)+"|").getBytes(), true);
		    }



		} //its a IndexLookup, but using a range Prefix
		else {
		    entry = indx.prefixSubMap(((String)key).getBytes(), true);
		}
	    } else if (indexMethod.equalsIgnoreCase("IndexRangeScan")) {
		//TODO - key object is always a prefix string
		entry = indx.prefixSubMap((((String)key)).getBytes(), true);


	    } else if (indexMethod.equalsIgnoreCase("IndexScan")) {
		//TODO - key object could be a pattern
		indx.getKeys().parallelStream().forEach(ikeyb -> {
		    String ikey = new String((byte[])ikeyb);
		    if (key instanceof Pattern) {
			if(((Pattern)key).matcher(ikey).find())
			    output.add(indx.get(ikeyb));
		    } else if (ikey.startsWith(((String)key)))
			output.add(indx.get(ikeyb));

		});

	    }
	    if (entry!=null) {

		if (entry instanceof Map) {
		    for (Object set : ((Map)entry).values()) {
			if (set instanceof Set) {
			    output.addAll((Set)set);
			} else if (set instanceof String) {
			    output.add(set);
			}
		    }
		} else if (entry instanceof Set) {
		    output.addAll((Set)entry);
		} else if (entry instanceof String)
		    output.add(entry);

	    }
	}
	return output;
    }



    private Stream<Document> findInternal(Document filter, Document explainPlan) {


	Document bestIndex=getIndex(filter);

	String indexName = (String) bestIndex.get("IndexName");
	Set<Object> ifind = calculateForIndex(indexName, filter, bestIndex.getString("IndexMethod"));

	explainPlan.append("Collection", this.definition.getString("Name")).append("Filter", filter).append("Plan", bestIndex);



	if (ifind!=null && ifind.size()>0) {

	    BTreeMap idx = (BTreeMap)(indexes.get(indexName));
	    Object o = indexSeek(idx, ifind, bestIndex);

	    if (o!=null) {
		if (!(o instanceof HashSet))
		    o=Collections.singleton(o);
		return ((Set<Object>)o)
			.stream()
			.parallel()
			.map(pointer->(Document)data.get(pointer))
			.filter(doc->ICollection.filter(doc, filter));
	    }


	} else { //fullscan
	    return fullScanStream(filter, explainPlan);
	}
	return Stream.empty();

    }

    private DBCursor fullScan(Document filter, Document explainPlan) {
	explainPlan.append("Collection", this.definition.getString("Name")).append("Filter", filter).append("Plan", "DataScan");
	//System.out.println(((Map<Object,Document>)data).values().size());
	return new DBCursor(fullScanStream(filter, explainPlan)
		.collect(
			Collectors.toList()
			));
    }


    private Stream<Document> fullScanStream(Document filter, Document explainPlan) {
	explainPlan.append("Collection", this.definition.getString("Name")).append("Filter", filter).append("Plan", "DataScan");
	return ((Set<Object>)data.getKeys())
		.stream()
		.parallel()
		.map(key -> {
		    return (Document)data.get(key);
		})
		.filter(doc->{
		    return ICollection.filter(doc, filter);
		});
    }


    private Set<Object> calculateForIndex(String ikey, Document document, String indexMethod) {
	//search function to decode filter
	if (ikey.equalsIgnoreCase("DataScan"))
	    return null;
	String geoType=null;
	int geoAccuracy=12;
	List<Document> ifields = null;
	boolean caseInsensitive=false;
	Document ic=(Document)indexCatalogCache.get(ikey);

	if (ic!=null){
	    ifields = ic.getList("Fields");
	    Document options = ic.getAsDocument("Options");
	    if (options!=null){
		if (options.get("geo")!=null)
		    geoType=options.getString("geo");
		if (options.get("accuracy")!=null)
		    geoAccuracy=options.getInteger("accuracy");
		if (options.get("caseInsensitive")!=null)
		    caseInsensitive=options.getBoolean("caseInsensitive", false);
	    }
	}

	StringBuilder key = new StringBuilder();


	ArrayList<java.util.Collection> fieldValues = new ArrayList();
	if (ifields!=null){
	    for (Map ifieldName : ifields) {
		if (document.containsProjection((String)ifieldName.get("Name"))) {
		    Object val=document.getProjection((String)ifieldName.get("Name"));
		    if (val==null)
			break;
		    if (geoType==null){
			if (val instanceof Set || val instanceof List)
			    fieldValues.add((java.util.Collection) val);
			else {
			    if((val instanceof String)) {
				if (caseInsensitive)
				    fieldValues.add((java.util.Collection)java.util.Collections.singleton(((String)val).toUpperCase()));
				else
				    fieldValues.add((java.util.Collection)java.util.Collections.singleton(val));
			    } else fieldValues.add((java.util.Collection)java.util.Collections.singleton(val));
			}
			if(isPattern(val)) // pattern is always a terminator element in a key
			    break;
		    } else {
			if (geoType.equalsIgnoreCase("2d")){
			    if (val instanceof Set || val instanceof List){
				java.util.Set newVal = new java.util.HashSet();
				for (Object obj : (Iterable)val){
				    //convert obj to a point
				    if (obj instanceof Document){
					GeoType point=GeoType.fromDoc((Document)obj);
					if (point !=null){
					    Set<String> hash=GeoType.encodeGeohash(point,geoAccuracy);
					    if (hash!=null)
						newVal.addAll(hash);
					}
				    }
				}
				fieldValues.add(newVal);
			    } else {
				if (val instanceof Map)
				    val=new Document(val);
				if (val instanceof Document){
				    GeoType point=GeoType.fromDoc((Document)val);

				    if (point !=null){
					Set<String> hash=GeoType.encodeGeohash(point,geoAccuracy);
					if (hash!=null)
					    fieldValues.add(hash);
				    }
				}
			    }

			}
		    }
		}  else break;
	    }

	    return processKey(fieldValues, 0, "", indexMethod);
	}
	return null;
    }




    private Document getIndex(Document filter) {
	Document bestIndex=new Document("HitRatio", 0.0).append("IndexName", "DataScan").append("IndexMethod", "DataScan");
	String indexMethod = "IndexLookup";
	for (Object ikey: indexCatalogCache.keySet()) {


	    Document indexDoc=indexCatalogCache.get(ikey);
	    if (indexDoc.getInteger("dirty",0)==0) {
		double indexHit=0;
		Set<String> indexFields = new HashSet();
		List<Document> ifields = indexDoc.getList("Fields");
		boolean unique = indexDoc.getAsDocument("Options").getBoolean("unique", false);
		for (Map ifieldName : ifields) {
		    for (Object fkeyo: filter.keySet()) {
			String fkey=(String)fkeyo;
			if (!fkey.startsWith("$")) {
			    if (fkey.equalsIgnoreCase((String)ifieldName.get("Name"))) {
				if (isPattern(filter.getProjection(fkey))) {
				    if (isPrefixPattern(filter.getProjection(fkey))){
					//stop using fields, take the prefix element of the pattern and force an index range lookup
					indexHit++;
					indexFields.add((String)ifieldName.get("Name"));
					indexMethod = "IndexRangeScan";
					break;
				    }
				    else {
					//if there is a full pattern, then use the index with the current found fields - ie index scan
					indexMethod = "IndexScan";
					indexFields.add((String)ifieldName.get("Name"));
					indexHit++;
					break;
				    }

				} else {

				    indexHit++;
				    indexFields.add((String)ifieldName.get("Name"));
				}
			    } 

			}

			if (filter.get(fkey) instanceof Document && filter.getAsDocument(fkey).containsKey("$geoWithin")){
			    indexMethod = "IndexRangeScan";
			}
		    }
		}
		double thisHit=indexHit/((double)ifields.size());
		if (thisHit>bestIndex.getDouble("HitRatio")) {
		    bestIndex.append("IndexName", (String)ikey);
		    bestIndex.append("IndexMethod", indexMethod);
		    bestIndex.append("HitRatio",thisHit);
		    bestIndex.append("IndexFields",indexFields);
		    bestIndex.append("Unique",unique);
		}
	    }
	}
	return bestIndex;
    }


    private boolean hash(Document doc, String hashKey) {
	Object o = doc.get(hashKey);
	int value;
	if (o instanceof Integer)
	    value=(Integer)o;
	else value=o.hashCode();

	return this.nodeNum==(Math.abs(value) % (replicaSet.size()+1));
    }

    private Document routerFilter(Document doc) {
	if (replicaType==ReplicaType.Range) {
	    if (ICollection.filter(doc, this.replicaRouter))
		return doc;
	    else return null;
	} else if (replicaType==ReplicaType.Hash) {
	    if (hash(doc, this.hashKey))
		return doc;
	    else return null;
	} else if (replicaType==ReplicaType.Replicated)
	    return doc;
	return doc;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#save(com.entitystream.monster.db.Document)
     */
    @Override
    public Document save(Document docIn) {
	Object _id=docIn.get("_id");
	Document doc = routerFilter(docIn);
	if (doc!=null) {
	    if (index(doc))
		data.put(_id, doc);
	    preProcess(doc);
	    match(doc);
	} else {
	    //System.out.println(this.nodeNum + " is not saving " + docIn + " due to hash filter " + this.hashKey + " and range filter " + this.replicaRouter);
	    //may need to delete it - it has moved nodes
	    if (index(docIn)) {
		data.remove(_id);
		deindex(docIn);
	    }
	}



	return doc;
    }

    private void preProcessReplacement(Document before, Document node) {

    }

    private void preProcess(Document node) {
	preProcess(Stream.of(node));
    }

    private void preProcess(Stream<Document> in) {
	//update the rollups
	try {

	    in.forEach(node -> {

		//add to delta queue
		if (definition.containsKey("Delta")) {
		    if (ICollection.filter(node, definition.getAsDocument("Delta"))) {
			delta.add(node.getString("_id"));
		    }
		}


		
	    });
	} catch (Exception e) {
	    e.printStackTrace();
	}

    }

   
    @Override 
    public Stream<Document> getDeltas(){
	
	return Stream.generate(new Supplier<Document>() {
		@Override
		public Document get() {
		    try {
			
			String id=null;			
			
			if (delta!=null && delta.getSize()>0)
			   id=(String)delta.removeAt(delta.getSize()-1);
			if (id !=null)
			    return getDocument(id);
			else 
			    Thread.sleep(100);    

		    } catch (StopIterationException ex) {
			
			  
			
			throw ex;

		    }  catch (InterruptedException ex2) {
			
			  
			
			throw new StopIterationException(ex2.toString());

		    }catch (Exception ex) {
			
			
			
			ex.printStackTrace();
		    }
		    return null;
		}

	    }).filter(o -> {return o!=null;});
	
	
    }

    @Override
    public int saveMany(List<Document> docs) {


	insertMany(docs);
	return docs.size();

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#insertOne(com.entitystream.monster.db.Document)
     */
    @Override
    public Document insertOne(Document doc) {
	return save(doc);
    }



    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#count(com.entitystream.monster.db.Document)
     */
    @Override
    public long count(Document query) {
	if (query==null)
	    return data.sizeLong();
	else return find(query).count();
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findOneAndReplace(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.FindOneAndReplaceOptions)
     */
    @Override
    public Document findOneAndReplace(Document filter, Document replacement, Document options) {

	Document docIn = find(filter).first();

	if (docIn!=null) {
	    Document doc = routerFilter(docIn);

	    if (doc!=null) {
		Object _id = doc.getString("_id");
		replacement.append("_id", _id);
		deindex(doc);
		if (index(replacement))
		    data.put(_id, replacement);
		preProcess(replacement);
		match(replacement);
		return replacement;
	    } else {
		deindex(docIn);
		Object _id = docIn.getString("_id");
		data.remove(_id);
	    }
	} else if (options.getBoolean("upsert", false)) {
	    if (!replacement.containsKey("_id"))
		replacement.append("_id", options.get("_id"));
	    save(replacement);
	}
	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findOneAndReplace(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document findOneAndReplace(Document filter, Document replacement ) {
	return findOneAndReplace(filter, replacement, new Document());

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findOneAndUpdate(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document findOneAndUpdate(Document filter, Document amendments) {
	Document doc = find(filter).first();
	updateInternal(doc, amendments, new Document());

	return doc;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findOneAndUpdate(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.UpdateOptions)
     */
    @Override
    public Document findOneAndUpdate(Document filter, Document amendments, Document updateOpts) {
	Document doc = find(filter).first();

	updateInternal(doc, amendments, updateOpts);

	return doc;
    }


    private void updateInternal(Document doc, Document amendments,  Document options) {
	if (doc==null && options.getBoolean("upsert", false)) {
	    doc = new Document("_id", options.get("_id"));
	}
	if (doc!=null) {
	    doc.applyUpdate(amendments);
	    save(doc);
	    match(doc);
	} else System.out.println("not updating " + amendments.toJson());

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateOne(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateOne(Document filter, Document amendments) {
	return findOneAndUpdate(filter, amendments, new Document());
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateOne(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.UpdateOptions)
     */
    @Override
    public Document updateOne(Document filter, Document amendments, Document options) {
	return findOneAndUpdate(filter, amendments, options);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateMany(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.UpdateOptions)
     */
    @Override
    public int updateMany(Document filter, Document amendments, Document options) {
	AtomicInteger count= new AtomicInteger(0);
	find(filter)
	.stream()
	.forEach(doc -> { 
	    count.incrementAndGet();
	    updateInternal(doc, amendments, options);
	});
	return count.get();
    }


    @Override
    public int updateMany(Document filter, Document amendments) {
	AtomicInteger count= new AtomicInteger(0);
	find(filter)
	.stream()
	.forEach(doc -> { 
	    count.incrementAndGet();
	    updateInternal(doc, amendments, new Document());
	});
	return count.get();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#insertMany(java.util.List)
     */
    @Override
    public int insertMany(List<Document> listRecs) {
	AtomicInteger count=new AtomicInteger(0);
	bulkIndex(listRecs.stream()).parallelStream().forEach(doc -> {
	    String _id=(String)doc.get("_id");
	    Document d = routerFilter(doc);
	    if (d!=null) {
		data.put(_id, d);
		preProcess(d);
		match(d);
		count.incrementAndGet();
	    }

	});

	return count.get();
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findOneAndDelete(com.entitystream.monster.db.Document)
     */
    @Override
    public Document findOneAndDelete(Document filter) {
	Document doc = find(filter).first();
	if (doc!=null) {
	data.remove(doc.getString("_id"));
	deindex(doc);
	}
	return doc;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteOne(com.entitystream.monster.db.Document)
     */
    @Override
    public Document deleteOne(Document filter) {
	return findOneAndDelete(filter);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteMany(com.entitystream.monster.db.Document)
     */
    @Override
    public int deleteMany(Document filter) {
	if (filter.keySet().size()>0) {
	    AtomicInteger count= new AtomicInteger(0);
	    find(filter)
	    .stream()
	    .forEach(doc -> { 
		if (data.remove(doc.getString("_id"))!=null) {
		    count.incrementAndGet();
		    deindex(doc);
		}
	    });
	    return count.get();
	} else {
	    //truncate

	    int s=data.getSize();
	    System.out.println("Truncating " + s + " records from " + this.definition.getString("Name") + "...");
	    data.clear();
	    for (Object indexName:indexes.keySet()) {
		((BTreeMap)indexes.get(indexName)).clear();
	    }
	    if (indexes.containsKey(STANDARDINDEX))
		((BTreeMap)indexes.get(this.STANDARDINDEX)).clear();

	    return s;
	}
    }







    private IIndex getCacheIndex(String indexName) {
	if (!indexs.containsKey(indexName))
	    indexs.put(indexName, getSchDoc().getIndex(indexName));
	return indexs.get(indexName);
    }

    private ITable getCacheTable(String tableName) {
	if (!tables.containsKey(tableName))
	    tables.put(tableName, getSchDoc().getTable(tableName));
	return tables.get(tableName);
    }





    public Document standardise(Document record)  {
	BTreeMap stdIndex = ((BTreeMap)indexes.get(STANDARDINDEX));
	if (stdIndex!=null) {
	    Document stds=null;
	    if(record.getString("_id")!=null)
		stds = (Document) stdIndex.get(record.getString("_id").getBytes());
	    stds=null;
	    if (stds==null || stds.size()==0) {
		String tableName=record.getString("Table", definition.getString("Name"));
		//standardise
		stds = new Document();

		List<IPurpose> tp = getSchDoc().getPurposes(tableName);
		for (IPurpose p : tp){
		    //if (p!=null && !p.getPurposeType().equalsIgnoreCase("Data")) {
		    List<PurposeColumn> pcs = getSchDoc().getPurposeColumns(p.getPurposeName(),tableName);
		    for (PurposeColumn pc : pcs){
			List<PurposeColumnMap> pcms = getSchDoc().getPurposeColumnMaps(pc.getPurposeName(), pc.getColumn(), tableName);
			MatchProcInterface proc = pc.createMatchProc();

			ArrayList<Standardized> sa=null;
			try {
			    sa = new ArrayList<Standardized>();

			    Object[] objects= INode.getWords(record, pcms,  pc.popRuleSet().getRuleAnon(), new Date());
			    ArrayList<ArrayList<String>> baseTokensALL = (ArrayList<ArrayList<String>>) objects[0];
			    String originalText = (String) objects[1];
			    for (ArrayList<String> baseTokens : baseTokensALL){
				Standardized stdItem = proc.standardise(originalText,baseTokens.toArray(new String[baseTokens.size()]));
				if (stdItem.getCalculatedWords()!=null)
				    sa.add(stdItem);

			    }
			} catch (Exception e) {
			    e.printStackTrace();
			}

			if (sa.size()>0)
			    stds.append(pc.getColumn(),INode.serialize(sa));
		    }
		}
	    }
	    return stds;
	}
	return null;

    }

    private boolean checkDocumentForAllIndexes(Document doc) {
	boolean allowUpdate=true;
	for (Object ikey: indexCatalogCache.keySet()) {
	    String indexName= (String)ikey;
	    boolean unique = isUniqueIndex(indexName);

	    if (unique) {
		String _id = doc.getString("_id");
		boolean found=false;
		String _cid=null;
		Set s = existsInIndex(indexName, doc);
		if (s.size()>0) {
		    if (_id!=null)
			for (Object _cid1 : s) {
			    if (((String)_cid1).equalsIgnoreCase(_id)) {
				_cid=(String) _cid1;
				found=true;
				break;
			    }
			}


		}
		if (!found) {//id is different
		    if (_id==null) {
			//no id - but still matched - add the id to the doc
			doc.append("_id", _cid);
		    } else if (s.size()>0){
			//reject the record change, it has a different Id and contravenes unique index
			allowUpdate=false;
		    }
		}
	    }
	}
	return allowUpdate;
    }


    private List<Document> bulkIndex(Stream<Document> docs) {

	List<Document> idocs = docs.parallel().filter(doc->checkDocumentForAllIndexes(doc)).collect(Collectors.toList());	

	for (Object ikey: indexCatalogCache.keySet()) {
	    //long c=System.currentTimeMillis();
	    String indexName= (String)ikey;
	    bulkIndex(indexName,idocs.stream());
	    //long d=System.currentTimeMillis();
	    //System.out.println(indexName + ":"+ (d-c));
	}

	if (indexes.containsKey(STANDARDINDEX)) {
	    //long c=System.currentTimeMillis();
	    bulkIndex(STANDARDINDEX,idocs.stream());
	    //long d=System.currentTimeMillis();
	    //System.out.println(STANDARDINDEX + ":"+ (d-c));
	}

	return idocs;

    }


    private boolean index(Document doc) {

	if (checkDocumentForAllIndexes(doc)) {
	    for (Object ikey: indexCatalogCache.keySet()) {
		String indexName= (String)ikey;
		index(indexName,doc);
	    }
	    if (indexes.containsKey(STANDARDINDEX))
		index(STANDARDINDEX,doc);
	    return true;
	}
	else return false;

    }

    private boolean deindex(Document doc) {
	for (Object ikey: indexCatalogCache.keySet()) {
	    String indexName= (String)ikey;
	    if (existsInIndex(indexName, doc).size()>0)
		deindex(indexName,doc);
	}
	if (indexes.containsKey(STANDARDINDEX))
	    deindex("STD_1",doc);
	return true;
    }

    private boolean deindex(String indexName, Document doc) {
	if (indexName.equalsIgnoreCase(STANDARDINDEX)) {
	    BTreeMap index =  (BTreeMap)indexes.get(indexName);
	    index.remove((doc.getString("_id")).getBytes());
	} else {
	    boolean unique = isUniqueIndex(indexName);
	    Set<Object> indexky = calculateForIndex(indexName, doc, "IndexLookup");
	    if (indexky!=null && indexky.size()>0) {
		BTreeMap index =  (BTreeMap)indexes.get(indexName);
		if (unique) {
		    for (Object indexkey : indexky)
			index.remove(((String)indexkey).getBytes());
		} else {
		    for (Object indexkey : indexky)
			for (Object pointer :  (Set) indexSeek(index, Collections.singleton(indexkey), new Document("HitRatio",1.0d).append("Unique", unique).append("IndexMethod","IndexLookup")))
			    index.remove((indexkey +"|" + pointer).getBytes());
		}

	    }
	}
	return true;
    }

    private boolean isUniqueIndex(String indexName) {
	Object ic = indexCatalogCache.get(indexName);
	if (ic!=null) {
	    Document options = ((Document)ic).getAsDocument("Options");
	    return options.getBoolean("unique", false);
	}
	return false;
    }

    private boolean isFuzzyIndex(String indexName) {
	Object ic = indexCatalogCache.get(indexName);
	if (ic!=null) {
	    Document options = ((Document)ic).getAsDocument("Options");
	    return options.getBoolean("fuzzy", false);
	}
	return false;
    }

    private String geoIndexType(String indexName) {
	Object ic = indexCatalogCache.get(indexName);
	if (ic!=null) {
	    Document options = ((Document)ic).getAsDocument("Geo");
	    return options.getString("geo");
	}
	return null;
    }


    private boolean bulkIndex(String indexName, Stream<Document> docs) {
	if (!indexes.containsKey(indexName)) {
	    System.out.println("Index is not found " + indexName);
	    return false;
	}
	if (docs!=null) {
	    ///System.out.println("Bulk indexing " + indexName);
	    long start = System.currentTimeMillis();
	    indexLock.get(indexName).lock();

	    BTreeMap index = ((BTreeMap)indexes.get(indexName));
	    boolean isFuzzy = isFuzzyIndex(indexName);
	    boolean unique = isUniqueIndex(indexName);
	    IIndex matchI = getCacheIndex(indexName);

	    docs.parallel().forEach(doc->{
		if (routerFilter(doc)!=null) {
		    String id = doc.getString("_id");
		    if (indexName.equalsIgnoreCase(STANDARDINDEX) && id !=null) {
			//standardise record
			index.put(id.getBytes(), standardise(doc));
		    } else 	if (isFuzzy && id !=null) {
			String tableName = doc.getString("Table", definition.getString("Name"));
			if (tableName!=null) {

			    ITable table = getCacheTable(tableName);

			    if (table!=null) {

				Object pkey = doc.getProjection(table.getKeyField());




				java.util.Collection<String> useKeyList = getMatchKeys(doc, table, matchI, false);


				//ss=System.currentTimeMillis();
				for (String useKeys : useKeyList ){
				    try {
					if (matchI.isSearch() && useKeys.length()>0)
					    index.put(("S:"+ useKeys + "|" + id).getBytes(), id);

					if ((matchI.isMatch() || (!matchI.isSearch() && !matchI.isMatch()))  && useKeys.length()>0)
					    index.put((matchI.getIndexName()+":"+ useKeys+ "|" + id).getBytes(), id);
				    } catch (Exception e) {
					if(index==null )
					    System.out.println(matchI.getIndexName());

					e.printStackTrace();
				    }

				}
				//System.out.println("useKeyList:"+(System.currentTimeMillis()-ss)+"ms");

			    }

			}
		    } else {


			Set<Object> indexky = calculateForIndex(indexName, doc, "IndexLookup");
			if (indexky.size()>0) {



			    for (Object indexkey : indexky) {
				if (!unique) {
				    //if not unique then the index value is the key, plus the ID
				    index.put((indexkey+"|"+id).getBytes(), id);
				} else {
				    byte[] keyBytes = ((String)indexkey).getBytes();
				    index.put(keyBytes, id);

				}
			    }



			}
		    }
		}});
	    try {



		indexLock.get(indexName).unlock();
	    } catch (Exception ee) {

	    }

	    //System.out.println("Finished: Bulk indexing " + indexName + " in "+ (System.currentTimeMillis()-start)+"ms");
	    return true;
	}
	return false;
    }

    private boolean index(String indexName, Document doc) {
	if (doc!=null) {
	    String id = doc.getString("_id");
	    boolean normalIndex=true;

	    if (indexName.equalsIgnoreCase(STANDARDINDEX)) {
		//standardise record
		BTreeMap index =  ((BTreeMap)indexes.get(STANDARDINDEX));
		if (index!=null) {
		    normalIndex=false;
		    indexLock.get(indexName).lock();
		    index.put(id.getBytes(), standardise(doc));
		    indexLock.get(indexName).unlock();
		}
	    } else if (isFuzzyIndex(indexName) && id !=null) {
		String tableName = doc.getString("Table", definition.getString("Name"));
		if (tableName!=null && doc!=null) {
		    ITable table = getSchDoc().getTable(tableName);
		    if (table!=null) {
			normalIndex=false;
			Object pkey = doc.getProjection(table.getKeyField());

			IIndex matchI = getSchDoc().getIndex(indexName);
			java.util.Collection<String> useKeyList = getMatchKeys(doc, table, matchI, false);

			BTreeMap index =  ((BTreeMap)indexes.get(indexName));
			indexLock.get(indexName).lock();

			for (String useKeys : useKeyList ){
			    try {
				if (matchI.isSearch() && useKeys.length()>0)
				    index.put(("S:"+ useKeys + "|" + id).getBytes(), id);

				if ((matchI.isMatch() || (!matchI.isSearch() && !matchI.isMatch()))  && useKeys.length()>0)
				    index.put((matchI.getIndexName()+":"+ useKeys+ "|" + id).getBytes(), id);
			    } catch (Exception e) {
				if(index==null )
				    System.out.println(matchI.getIndexName());

				e.printStackTrace();
			    }

			}
			indexLock.get(indexName).unlock();

		    }
		    return true; 
		}
	    } 
	    if (normalIndex){

		boolean unique = isUniqueIndex(indexName);
		Set<Object> indexky = calculateForIndex(indexName, doc, "IndexLookup");


		if (indexky.size()>0) {
		    BTreeMap index =  ((BTreeMap)indexes.get(indexName));
		    indexLock.get(indexName).lock();

		    for (Object indexkey : indexky) {
			if (!unique) {
			    //if not unique then the index value is the key, plus the ID
			    index.put((indexkey+"|"+id).getBytes(), id);
			} else {
			    index.put(((String)indexkey).getBytes(), id);
			}
		    }
		    try {
			indexLock.get(indexName).unlock();
		    } catch (Exception e) {}
		    return true; 
		}
	    }
	}
	return false; 

    }

    private Set existsInIndex(String indexName, Document doc) {
	BTreeMap index = ((BTreeMap)indexes.get(indexName));
	Set<Object> indexky = calculateForIndex(indexName, doc, "IndexLookup");
	Document indexdef = (Document)indexCatalogCache.get(indexName);
	Set s = indexSeek(index, indexky, new Document("HitRatio",0.0d).append("Unique", indexdef.getAsDocument("Options").getBoolean("unique", false)).append("IndexMethod","IndexLookup"));
	return s;
    }


    public static void drop(Document definition) {
	List<File> dirs = new ArrayList<File>();
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")));
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")+"_RELS"));
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")+"_TASKS"));
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")+"_Queue"));
	for (File dir : dirs)
	    try {
		if (dir.exists())
		    FileUtils.forceDelete(dir);
	    } catch (IOException e) {}


    }

    public void drop() {


	List<File> dirs = new ArrayList<File>();
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")));
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")+"_RELS"));
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")+"_TASKS"));
	dirs.add(new File(definition.getString("DBPath")+definition.getString("Path") + File.separator + definition.getString("Name")+"_Queue"));
	for (File dir : dirs)
	    try {
		if (dir.exists())
		    FileUtils.forceDelete(dir);
	    } catch (IOException e) {}




	try {
	    if (indexes!=null)
		for (Object index : indexes.values())
		    ((BTreeMap)index).clear();
	} catch (Exception e) {

	    e.printStackTrace();
	}
	try {
	    if(indexes!=null) 
		indexes.clear();
	    if (indexCatalogStore!=null) {
		if(!indexCatalogStore.isClosed())
		    indexCatalogStore.clear();
	    }
	    disconnect();

	} catch (Exception e) {

	    e.printStackTrace();
	}



    }
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#aggregate(java.util.List, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> aggregate(List<Document> in, List<Document> pipeline) {
	try {
	    throw new Exception("Aggregate iterable cannot be called at the local level");
	} catch (Exception e) {

	    e.printStackTrace();
	}
	//return new DBCursor(new AggregateIterable(this, pipeline, options));
	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#aggregate(java.util.List, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> aggregate(List<Document> pipeline, Document options) {
	try {
	    throw new Exception("Aggregate iterable cannot be called at the local level");
	} catch (Exception e) {

	    e.printStackTrace();
	}
	//return new DBCursor(new AggregateIterable(this, pipeline, options));
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#aggregate(java.util.List)
     */
    @Override
    public Stream<Document> aggregate(List<Document> pipeline) {
	try {
	    return parent.aggregate(pipeline);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	//return new DBCursor(new AggregateIterable(this, pipeline, new Document()));
	return null;
    }

    @Override
    public Document aggregateMetadata(List<Document> in,List<Document> pipeline) {
	try {
	    return parent.aggregateMetadata(in,pipeline);
	} catch (Exception e) {

	    e.printStackTrace();
	}

	return null;
    }

    @Override
    public Document aggregateMetadata(List<Document> pipeline) {
	try {
	    return parent.aggregateMetadata(pipeline);
	} catch (Exception e) {

	    e.printStackTrace();
	}

	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#listIndexes()
     */
    @Override
    public DBCursor listIndexes() {
	HashSet<Document> indexOut = new HashSet<Document>();
	for (Object name : indexCatalogCache.keySet()) 
	    indexOut.add(new Document("name", name).append("Fields", indexCatalogCache.get(name)));
	return new DBCursor(indexOut);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#distinct(java.lang.String, java.lang.Class)
     */







    public boolean isExplaining() {
	return explaining;
    }

    public void setExplaining(boolean explaining) {
	this.explaining = explaining;
    }




    public DBCursor executeCommand(String command, User user, Session session) {
	long start=System.currentTimeMillis();
	DBCursor ret = Container.executeCommand(this, command, user, session);
	if (System.currentTimeMillis()-start >10000) {
	    String c=command;
	    if (c.length()>40)
		c=c.substring(0, 37)+"...";
	    System.out.println("Command ("+c+") completed in " + (System.currentTimeMillis()-start)+"ms");
	}
	return ret;
    }

    public <TResult> Iterable<TResult> distinct(String fieldName, Class<TResult> resultClass){
	HashSet<TResult> set = new HashSet<TResult>();
	DBCursor f = find();
	while(f.hasNext())
	    set.add((TResult) f.next().getProjection(fieldName));
	return set;

    }










    public Stream bucket(Document bucketDef, Stream<Document> in, Document options) {
	try {
	    throw new Exception("bucket iterable cannot be called at the local level");
	} catch (Exception e) {

	    e.printStackTrace();
	}
	//return new DBCursor(new AggregateIterable(this, pipeline, options));
	return null;
    }




    @Override
    public Document getDocument(Object pointer) {
	return (Document) data.get(pointer);
    }

    @Override
    public Document getIndexPrefixSubMap(String indexName, String key, boolean b) {
	Document d = new Document();
	BTreeMap i = ((BTreeMap)indexes.get(indexName));
	if (i!=null)
	   d.putAll( i.prefixSubMap(key.getBytes(), b));
	return d;

    }

    @Override
    public Document keyCount(String indexName) {
	BTreeMap collIndex = (BTreeMap) indexes.get(indexName);
	if (collIndex!=null)
	    return new Document((HashMap) collIndex
		    .keySet()
		    .stream()
		    .map(key->{
			String keyValue = new String((byte[])key);
			keyValue=keyValue.split(":|\\|")[1];
			return keyValue;
		    })
		    .collect(Collectors.groupingBy(Function.identity(), Collectors.counting())));
	else return null;

    }

    @Override
    public Document conceptCount(String indexName) {
	BTreeMap collIndex = (BTreeMap) indexes.get(indexName);
	return new Document( (HashMap)((BTreeMap<Object, Document>) collIndex)
		.values()
		.stream()
		.map(doc->{
		    return doc.keySet();
		})
		.flatMap(c -> c.stream())
		.collect(Collectors.groupingBy(Function.identity(), Collectors.counting())));
    }

    @Override
    public Document getStandardised(String indexName, String id) {
	return (Document) ((BTreeMap)indexes.get(indexName)).get(id.getBytes());
    }

    @Override
    public void saveTable(Document def) {
	getSchDoc().addTable((ITable) def.toObject(Table.class));
	updateDefinition();
    }

    @Override
    public Document getTable(String def) {
	Gson gson = new Gson();
	return Document.parse(gson.toJson(getSchDoc().getTable(def), Table.class));
    }


    @Override
    public void deleteTable(String def) {
	getSchDoc().deleteTable(def);
	updateDefinition();

    }

    @Override
    public void saveConceptGroup(Document def) {
	getSchDoc().addPurpose((IPurpose) def.toObject(Purpose.class));
	updateDefinition();
    }

    @Override
    public void deleteConceptGroup(String def) {
	getSchDoc().deletePurpose(def);
	updateDefinition();
    }

    @Override
    public void saveConcept(Document def) {
	getSchDoc().addPurpose((IPurpose) def.toObject(Purpose.class));
	updateDefinition();
    }

    @Override
    public void deleteConcept(String purposeName, String purposeColumn) {
	getSchDoc().deletePurposeColumn(purposeName, purposeColumn);
	updateDefinition();
    }

    @Override
    public void saveConceptMapping(Document def) {
	getSchDoc().addPurposeColumnMap(def.toObject(PurposeColumnMap.class));
	updateDefinition();
    }

    @Override
    public void deleteConceptMapping(Document def) {
	getSchDoc().deletePurposeColumnMap(def.toObject(PurposeColumnMap.class));
	updateDefinition();
    }



    @Override
    public void saveMatchRule(Document def) {
	getSchDoc().addRule((IRule) def.toObject(Rule.class));

	updateDefinition();
    }

    @Override
    public void saveMatchRules(List<Document> defs) {
	ISchemaMeta _schDoc = getSchDoc();
	for (IRule rule : _schDoc.getRules(null))
	    _schDoc.deleteMatchRule(rule.getOrder());
	for (Document def : defs)
	    _schDoc.addRule((IRule) def.toObject(Rule.class));
	schDoc=_schDoc;
	updateDefinition();
    }



    @Override
    public void deleteMatchRule(long order) {
	getSchDoc().deleteMatchRule(order);
	updateDefinition();
    }

    @Override
    public void saveFuzzyIndex(Document def) {
	getSchDoc().addIndex((IIndex) def.toObject(Index.class));
	updateDefinition();
    }

    @Override
    public void deleteFuzzyIndex(String def) {
	getSchDoc().deleteIndex(def);
	updateDefinition();
    }
    @Override
    public void removeFuzzy() {
	schDoc=null;
	updateDefinition();
    }

    private ISchemaMeta getSchDoc() {
	if (schDoc==null)
	    schDoc= SchemaMeta.createSchemaMeta(new Document());
	return schDoc;
    }
    @Override
    public Document getDefinition() {
	return parent.getDefinition();
    }
    private void updateDefinition() {

	//method pushes the changes whcih came from ta remote server call to the owning collection
	if(schDoc!=null) {
	    Document doc = schDoc.toDocument();
	    definition.append("Definition",doc);
	    parent.updateDefinition(doc);
	} else {
	    definition.remove("Definition");
	    parent.updateDefinition(null);

	}



	//reinitialise the indexes.

	initialiseCollection();
    }

    public ReplicaType getReplicaType() {
	return replicaType;
    }

    public void setReplicaType(ReplicaType replicaType) {
	this.replicaType = replicaType;
    }


    public ICollection getParent() {

	return parent;
    }


    @Override
    public void updateDefinition(Document def) {
	this.definition.append("Definition",def);
	schDoc= SchemaMeta.createSchemaMeta(def);
	initialiseCollection();
    }


   

    //match processing
    @Override
    public boolean match(Document node) {


	if (matchCollection!=null && node.containsKey("Table")) {
	    //add to match queue
	    matchCollection.offer(node);
	    return true;
	} else return false;
    }

    @Override
    public DBCursor peekQueue() {
	if (matchCollection!=null) {

	    return matchCollection.peekAll();

	} 
	return new DBCursor();
    }



    /* (non-Javadoc)
     * @see com.entitystream.identiza.entity.resolve.match.Matchable#resolveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public void resolveTask(Document mr) {
	if (matchCollection !=null)
	    matchCollection.resolveTask(mr);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findStream(com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> findStream(Document filter) {
	Document explainPlan=new Document();
	long t = System.currentTimeMillis();
	Stream<Document> _return = findStream(filter, explainPlan);
	t = System.currentTimeMillis()-t;
	if (explaining) {
	    explainPlan.append("Elapsed", t + "ms");
	    logger.info(explainPlan.toJson());
	}
	return _return;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getReplicaConnections()
     */
    @Override
    public Map<Integer, ICollection> getReplicaConnections() {
	return this.parent.getReplicaConnections();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#limit(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */

    @Override
    public Stream<Document> skip(long skip, Stream in, Document options) {
	try {
	    throw new Exception("skip is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;

    }
    @Override
    public Stream<Document> limit(long limit, Stream in, Document options) {
	try {
	    throw new Exception("limit is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;

    }










    public Stream out(String target, Stream<Document> in, Document options) {

	try {
	    throw new Exception("out is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }




    public Stream<Document> group(Document object, Stream<Document> in, Document options) {
	try {
	    throw new Exception("group is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Document> lookup(Document object, Stream<Document> in, Document options) {
	try {
	    throw new Exception("lookup is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Document> join(Document object, Stream<Document> in, Document options) {
	try {
	    throw new Exception("join is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Document> minus(Document object, Stream<Document> in, Document options) {
	try {
	    throw new Exception("minus is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }




    public Stream<Object> unwind(Document unwindOptions, Stream<Document> in, Document options) {
	try {
	    throw new Exception("unwind is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;

    }



    public Stream<Document> count(Document options, Stream<Document> in, Document globalOptions) {
	try {
	    throw new Exception("count is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Document> first(Document options, Stream<Document> in, Document globalOptions) {
	try {
	    throw new Exception("first is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Document> last(Document options, Stream<Document> in, Document globalOptions) {
	try {
	    throw new Exception("last is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    public Stream<Document> between(Document options, Stream<Document> in, Document globalOptions) {
	try {
	    throw new Exception("between is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }



    public Stream<Document> cluster(Document options, Stream<Document> in, Document globalOptions) {
	try {
	    throw new Exception("cluster is not to be called at the local level");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;

    }
















    /**
     * @return
     */
    protected Collection getRelCollection() {
	if (!isSpecialCollection())
	    return (Collection) getDatabase().getRelCollection(getDefinition().getString("Name")+"_RELS");
	else
	    return null;
    }

    /**
     * @return
     */
    protected Collection getTaskCollection() {
	if (!isSpecialCollection())
	    return (Collection) getDatabase().getTaskCollection(getDefinition().getString("Name")+"_TASKS");
	else return null;
    }

    @Override
    public DBCursor findRelationships(Document filter) {
	return getRelCollection().find(filter);
    }

    @Override
    public Document saveRelationship(Document relationship) {
	Document filter = new Document("fromCol", relationship.get("fromCol"))
		.append("toCol", relationship.get("toCol"))
		.append("relType", relationship.get("relType"));

	return getRelCollection().findOneAndReplace(filter, relationship, new Document("upsert", true));
    }


    @Override
    public DBCursor findTasks(Document filter) {
	return getTaskCollection().find(filter);
    }






    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#aggregateTasks(java.util.ArrayList)
     */
    @Override
    public DBCursor aggregateTasks(ArrayList<Document> andlist) {

	return new DBCursor(getTaskCollection().aggregate(andlist));
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateTask(java.lang.String, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateTask(String id, Document doc) {
	return getTaskCollection().findOneAndUpdate(new Document("_id",id),  doc);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateRelationship(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateRelationship(Document document, Document setUpdate, Document options) {
	return getRelCollection().findOneAndUpdate(document,  setUpdate, options);
    }

    @Override
    public <TResult> Iterable<TResult> distinctRelationship(String fieldName, Class<TResult> resultClass) {

	return getRelCollection().distinct(fieldName, resultClass);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteTasks(Document document) {
	this.getTaskCollection().deleteMany(document);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteRelationships(Document document) {
	this.getRelCollection().deleteMany(document);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getDatabase()
     */
    @Override
    public Database getDatabase() {
	return owningDB;
    }




    static class BucketRange {
	public Double upper;
	public Document document;
	public BucketRange(Double u, Document d) {
	    this.upper=u;
	    this.document=d;
	}
    }



    public Document score(Document base, Document comparitor, boolean forSearch) {

	try {
	    MatchRecordInterface mr = calculateScore(base, comparitor, forSearch);
	    return mr.toDocument();
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findFuzzy(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findFuzzy(Document filter) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findFuzzy(java.lang.String)
     */
    @Override
    public DBCursor findFuzzy(String textQuery) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#match(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream match(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }
    public Stream match(Document asDocument, long nodeid, long connectionid, long cursorid, Document options) {
	//get the stream from the remote connection, then apply it to the parent collection
	ICollection remote = getReplicaConnections().get((int)nodeid);
	if (remote instanceof CollectionRemote) {
	    Stream<Document> in = ((CollectionRemote) remote).getClient().remoteGetStream(connectionid, cursorid);
	    return getParent().match(asDocument, in, options);
	} else	return Stream.empty();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#analyse(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream analyse(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#sort(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream sort(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzySearch(java.lang.String, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream fuzzySearch(String string, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzyMatch(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream fuzzyMatch(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#spinOut(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream spinOut(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getRelated(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream getRelated(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#rematch(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream rematch(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierBuild(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream classifierBuild(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierPredict(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream classifierPredict(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    @Override
    public DBCursor traverseTop(Document from, String relType){
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#arrf(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream arrf(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierTree(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream classifierTree(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#coerce(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream coerce(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#compare(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream compare(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#project(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream project(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#evaluate(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream evaluate(Document asDocument, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#split(java.util.List, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Map<Document, List<Document>> split(List<Document> list, Stream<Document> in, Document options) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzyMatch(com.entitystream.monster.db.Document, java.util.List)
     */
    @Override
    public Document fuzzyMatch(Document matchNode, List<IIndex> matchIndexes) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateRollup(com.entitystream.monster.db.Document)
     */
    
    
    
    @Override
    public void updateDelta(Document def) {
	this.definition.append("Delta",def);
	initialiseCollection();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getRollup()
     */
  
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#writeRel(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> writeRel(Document options, Stream<Document> in, Document goptions) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#script(java.util.Map, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> script(Document scriptStatements, Stream<Document> in, Document stageoptions) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getMatchTypes()
     */
    @Override
    public DBCursor getMatchTypes() {
	Map<String, Document> mt = getSchDoc().getMatchTypes();
	List<Document> out = new ArrayList<Document>();
	for (String k : mt.keySet())
	    out.add(mt.get(k).append("Name", k));
	return new DBCursor(out);    
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveFuzzyIndexes(java.util.List)
     */
    @Override
    public void saveFuzzyIndexes(List<Document> def) {
	ISchemaMeta _sch = getSchDoc();
	for (IIndex i : _sch.getIndexes())
	    _sch.deleteIndex(i.getIndexName());

	for (Document de : def)
	    _sch.addIndex(de.toObject(Index.class));
	updateDefinition(_sch.toDocument());
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getFlows()
     */
    @Override
    public DBCursor getFlows() {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getBooks()
     */
    @Override
    public DBCursor getBooks() {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getFlow(java.lang.String)
     */
    @Override
    public DBCursor getFlow(String name) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#validate(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> validate(String string, Stream<Document> in, Document stageoptions) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#task(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> task(Document options, Stream<Document> in, Document globalOptions) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public Document saveTask(Document task) {
	return getTaskCollection().save(task);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getNodes()
     */
    @Override
    public Stream<Document> getNodes() {
	// TODO Auto-generated method stub
	return null;
    }
}
